# Can - Algorithme de traitement de requêtes universelles

Can est un système intelligent qui permet d'exécuter des requêtes sur différentes sources de données (bases de données, fichiers, APIs) via une syntaxe unifiée.

## Installation

```bash
git clone https://github.com/votre-repo/can.git
cd can
```

## Structure

```
can/
├── cli.php                 # Interface console
├── api.php                 # Endpoint API REST
├── core/
│   ├── parser.php         # Analyseur de requêtes
│   ├── router.php         # Routeur vers connectors
│   └── response.php       # Gestionnaire de réponses
└── connectors/
    ├── mysql.php          # Connector bases de données
    ├── file.php           # Connector fichiers
    ├── api.php            # Connector APIs externes
    └── config/            # Configurations des ressources
```

## Syntaxe des requêtes

```
[action] [paramètres] in '[ressource]', conn to '[connector]'
```

### Actions supportées
- `add/insert` - Ajouter des données
- `get/select` - Récupérer des données  
- `update/modify` - Modifier des données
- `delete/remove` - Supprimer des données
- `search/find` - Chercher dans les données
- `count` - Compter des éléments
- `send` - Envoyer vers APIs

## Configuration

### Base de données (MySQL)
Créer `connectors/config/database_users.json` :
```json
{
    "host": "localhost",
    "database": "myapp",
    "username": "user",
    "password": "pass"
}
```

### API externe
Créer `connectors/config/inventory_api.json` :
```json
{
    "base_url": "https://api.inventory.com/v1",
    "auth_type": "bearer",
    "token": "your_token_here",
    "search_endpoint": "search"
}
```

### Fichiers
Placer les fichiers dans `connectors/files/`

## Utilisation

### Console
```bash
php cli.php "get all users where status='active' in 'database_users', conn to 'mysql.php'"
php cli.php "add line 'john,doe,30' to 'users.csv', conn to 'file.php'"
php cli.php "search user 'john@doe.com' in 'crm_api', conn to 'api.php'"
```

### API REST

#### JavaScript
```javascript
fetch('http://localhost/can/api', {
    method: 'POST',
    headers: {
        'Content-Type': 'application/json'
    },
    body: JSON.stringify({
        query: "get all users where status='active' in 'database_users', conn to 'mysql.php'"
    })
})
.then(response => response.json())
.then(data => console.log(data));
```

#### PHP
```php
$query = "add 1.92 value to colonne 'price' in 'products_db', conn to 'mysql.php'";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://localhost/can/api');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['query' => $query]));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

$response = curl_exec($ch);
$data = json_decode($response, true);
```

#### Go
```go
package main

import (
    "bytes"
    "encoding/json"
    "net/http"
    "io/ioutil"
)

func callCan(query string) {
    payload, _ := json.Marshal(map[string]string{"query": query})
    
    resp, _ := http.Post("http://localhost/can/api", "application/json", bytes.NewBuffer(payload))
    defer resp.Body.Close()
    
    body, _ := ioutil.ReadAll(resp.Body)
    var result map[string]interface{}
    json.Unmarshal(body, &result)
}
```

## Intégration dans un service

### Service PHP existant
```php
class MonService {
    private $can_url = 'http://localhost/can/api';
    
    public function getUsersActive() {
        return $this->callCan("get all users where status='active' in 'database_users', conn to 'mysql.php'");
    }
    
    public function updateUserEmail($userId, $email) {
        return $this->callCan("update user_id={$userId} set email='{$email}' in 'database_users', conn to 'mysql.php'");
    }
    
    public function getAnalyticsData() {
        return $this->callCan("get value from key 'total_sales' in 'analytics.json', conn to 'file.php'");
    }
    
    private function callCan($query) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->can_url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['query' => $query]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        $response = curl_exec($ch);
        return json_decode($response, true);
    }
}
```

### Service JavaScript/Node.js
```javascript
class CanService {
    constructor() {
        this.canUrl = 'http://localhost/can/api';
    }
    
    async getUsersActive() {
        return await this.callCan("get all users where status='active' in 'database_users', conn to 'mysql.php'");
    }
    
    async searchProduct(sku) {
        return await this.callCan(`search product where sku='${sku}' in 'inventory_api', conn to 'api.php'`);
    }
    
    async callCan(query) {
        const response = await fetch(this.canUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ query })
        });
        
        return await response.json();
    }
}
```

## Exemples de requêtes

### Base de données
```
"add 1.92 value to colonne 'price' in 'products_db', conn to 'mysql.php'"
"get all users where status='active' in 'users_db', conn to 'mysql.php'"
"update user_id=123 set email='new@email.com' in 'users_db', conn to 'mysql.php'"
"delete record where id=456 in 'logs_db', conn to 'mysql.php'"
"count rows where date>'2024-01-01' in 'analytics_db', conn to 'mysql.php'"
```

### Fichiers
```
"get value from key 'total_sales' in 'report.json', conn to 'file.php'"
"add line 'new,data,row' to 'export.csv', conn to 'file.php'"
"search text 'error' in 'logs.txt', conn to 'file.php'"
"update key 'status' with 'completed' in 'config.json', conn to 'file.php'"
```

### APIs externes
```
"get product_info where sku='ABC123' in 'inventory_api', conn to 'api.php'"
"send data '{\"name\":\"John\"}' to endpoint 'users' in 'crm_api', conn to 'api.php'"
"search user 'john@doe.com' in 'crm_api', conn to 'api.php'"
"update users id=123 with '{\"status\":\"active\"}' in 'api_service', conn to 'api.php'"
```

## Format de réponse

```json
{
    "status": "success",
    "data": "résultat ou données",
    "message": "Query executed successfully",
    "execution_time": "15.2ms",
    "parsed_query": {
        "action": "get",
        "params": "all users where status='active'",
        "resource": "database_users",
        "connector": "mysql.php"
    }
}
```

## Création d'un nouveau connector

Pour ajouter un nouveau type de source, créer `connectors/nouveau.php` :

```php
<?php

class NouveauConnector {
    
    public function get($params, $resource) {
        // Logique pour récupérer des données
    }
    
    public function add($params, $resource) {
        // Logique pour ajouter des données
    }
    
    // Autres méthodes...
}
```

Puis utiliser : `conn to 'nouveau.php'`

## Sécurité

- Valider toutes les entrées utilisateur
- Configurer les accès aux bases de données avec des utilisateurs limités
- Utiliser HTTPS pour les APIs
- Implémenter une authentification pour l'API Can en production