<?php

class MysqlConnector {
    
    private $connections = [];
    
    private function getConnection($resource) {
        if (!isset($this->connections[$resource])) {
            $config = $this->getConfig($resource);
            
            try {
                $this->connections[$resource] = new PDO(
                    "mysql:host={$config['host']};dbname={$config['database']};charset=utf8",
                    $config['username'],
                    $config['password'],
                    [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
                );
            } catch (PDOException $e) {
                throw new Exception("Database connection failed: " . $e->getMessage());
            }
        }
        
        return $this->connections[$resource];
    }
    
    private function getConfig($resource) {
        $config_file = __DIR__ . '/config/' . $resource . '.json';
        
        if (!file_exists($config_file)) {
            throw new Exception("Database config not found for: " . $resource);
        }
        
        return json_decode(file_get_contents($config_file), true);
    }
    
    public function add($params, $resource) {
        return $this->insert($params, $resource);
    }
    
    public function insert($params, $resource) {
        $pdo = $this->getConnection($resource);
        
        if (preg_match("/(.+?)\s+value\s+to\s+colonne\s+['\"]([^'\"]+)['\"]/i", $params, $matches)) {
            $value = trim($matches[1]);
            $column = $matches[2];
            
            $table = $this->extractTable($params);
            
            $sql = "INSERT INTO {$table} ({$column}) VALUES (?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$value]);
            
            return $pdo->lastInsertId();
        }
        
        throw new Exception("Invalid insert syntax");
    }
    
    public function get($params, $resource) {
        return $this->select($params, $resource);
    }
    
    public function select($params, $resource) {
        $pdo = $this->getConnection($resource);
        
        if (preg_match("/all\s+(\w+)(?:\s+where\s+(.+))?/i", $params, $matches)) {
            $table = $matches[1];
            $where = isset($matches[2]) ? $matches[2] : '';
            
            $sql = "SELECT * FROM {$table}";
            if ($where) {
                $sql .= " WHERE " . $this->sanitizeWhere($where);
            }
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute();
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        
        throw new Exception("Invalid select syntax");
    }
    
    public function update($params, $resource) {
        $pdo = $this->getConnection($resource);
        
        if (preg_match("/(\w+)=(\d+)\s+set\s+(.+)/i", $params, $matches)) {
            $id_field = $matches[1];
            $id_value = $matches[2];
            $set_clause = $matches[3];
            
            $table = $this->extractTable($params);
            
            $sql = "UPDATE {$table} SET {$set_clause} WHERE {$id_field} = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$id_value]);
            
            return $stmt->rowCount();
        }
        
        throw new Exception("Invalid update syntax");
    }
    
    public function delete($params, $resource) {
        $pdo = $this->getConnection($resource);
        
        if (preg_match("/record\s+where\s+(.+)/i", $params, $matches)) {
            $where = $matches[1];
            $table = $this->extractTable($params);
            
            $sql = "DELETE FROM {$table} WHERE " . $this->sanitizeWhere($where);
            $stmt = $pdo->prepare($sql);
            $stmt->execute();
            
            return $stmt->rowCount();
        }
        
        throw new Exception("Invalid delete syntax");
    }
    
    public function count($params, $resource) {
        $pdo = $this->getConnection($resource);
        
        if (preg_match("/rows(?:\s+where\s+(.+))?/i", $params, $matches)) {
            $where = isset($matches[1]) ? $matches[1] : '';
            $table = $this->extractTable($params);
            
            $sql = "SELECT COUNT(*) as count FROM {$table}";
            if ($where) {
                $sql .= " WHERE " . $this->sanitizeWhere($where);
            }
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute();
            
            return $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        }
        
        throw new Exception("Invalid count syntax");
    }
    
    private function extractTable($params) {
        if (preg_match("/(?:from|into|table)\s+['\"]?(\w+)['\"]?/i", $params, $matches)) {
            return $matches[1];
        }
        
        throw new Exception("No table specified");
    }
    
    private function sanitizeWhere($where) {
        return preg_replace("/['\"]([^'\"]+)['\"]/", "'$1'", $where);
    }
}

?>