<?php
header('Content-Type: text/html; charset=UTF-8');

function getMediaFiles($dir) {
    $files = [];
    $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp', 'tiff', 'heic', 'avif'];
    $videoExtensions = ['mp4', 'mov', 'avi', 'mkv', 'webm', 'flv', 'm4v', 'wmv', '3gp', 'ogv'];
    $audioExtensions = ['mp3', 'wav', 'aac', 'ogg', 'flac', 'm4a', 'wma', 'opus'];
    
    $allowedExtensions = array_merge($imageExtensions, $videoExtensions, $audioExtensions);
    
    if (!file_exists($dir)) return $files;
    
    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        $path = $dir . '/' . $item;
        if (!is_dir($path)) {
            $ext = strtolower(pathinfo($item, PATHINFO_EXTENSION));
            if (in_array($ext, $allowedExtensions)) {
                $type = 'unknown';
                if (in_array($ext, $imageExtensions)) $type = 'image';
                elseif (in_array($ext, $videoExtensions)) $type = 'video';
                elseif (in_array($ext, $audioExtensions)) $type = 'audio';
                
                $files[] = [
                    'name' => $item,
                    'path' => str_replace(__DIR__ . '/assets/', 'assets/', $path),
                    'type' => $type,
                    'size' => filesize($path),
                    'modified' => filemtime($path)
                ];
            }
        }
    }
    
    usort($files, function($a, $b) {
        return $b['modified'] - $a['modified'];
    });
    
    return $files;
}

function formatFileSize($bytes) {
    if ($bytes >= 1073741824) return number_format($bytes / 1073741824, 2) . ' GB';
    elseif ($bytes >= 1048576) return number_format($bytes / 1048576, 2) . ' MB';
    elseif ($bytes >= 1024) return number_format($bytes / 1024, 2) . ' KB';
    else return $bytes . ' B';
}

$folderName = isset($_GET['folder']) ? $_GET['folder'] : '';
$mediaDir = __DIR__ . '/assets';
if (!empty($folderName)) $mediaDir .= '/' . $folderName;
$mediaFiles = getMediaFiles($mediaDir);

foreach ($mediaFiles as $file) {
    $fileSize = formatFileSize($file['size']);
    ?>
    <div class="media-item aspect-portrait bg-white rounded-xl overflow-hidden shadow-sm hover:shadow-lg transition-all duration-300 cursor-pointer relative group border border-gray-100"
         onclick="openModal('<?= htmlspecialchars($file['path']) ?>', '<?= $file['type'] ?>', '<?= htmlspecialchars($file['name']) ?>')"
         data-type="<?= $file['type'] ?>">

        <div class="relative w-full h-full">
            <?php if ($file['type'] === 'image'): ?>
                <img src="<?= htmlspecialchars($file['path']) ?>" 
                     class="w-full h-full object-cover"
                     loading="lazy"
                     alt="<?= htmlspecialchars($file['name']) ?>">
                <div class="absolute top-2 left-2 bg-black/60 text-white text-xs px-2 py-1 rounded-full opacity-0 group-hover:opacity-100 transition-opacity">
                    🖼️ Image
                </div>
            <?php elseif ($file['type'] === 'video'): ?>
                <div class="w-full h-full bg-gradient-to-br from-gray-100 to-gray-200 flex items-center justify-center relative">
                    <video class="w-full h-full object-cover" preload="metadata" muted>
                        <source src="<?= htmlspecialchars($file['path']) ?>#t=1" type="video/<?= pathinfo($file['path'], PATHINFO_EXTENSION) ?>">
                    </video>
                    <div class="absolute inset-0 bg-black/20 flex items-center justify-center">
                        <div class="bg-white/90 rounded-full p-3 shadow-lg">
                            <svg class="w-8 h-8 text-gray-700" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M8 5v14l11-7z"/>
                            </svg>
                        </div>
                    </div>
                    <div class="absolute top-2 left-2 bg-black/60 text-white text-xs px-2 py-1 rounded-full">
                        🎬 Vidéo
                    </div>
                </div>
            <?php elseif ($file['type'] === 'audio'): ?>
                <div class="w-full h-full bg-gradient-to-br from-blue-50 to-purple-50 flex flex-col items-center justify-center p-4">
                    <div class="bg-gradient-to-r from-blue-500 to-purple-600 rounded-full p-4 mb-3 shadow-lg">
                        <svg class="w-8 h-8 text-white" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M12 3v10.55c-.59-.34-1.27-.55-2-.55-2.21 0-4 1.79-4 4s1.79 4 4 4 4-1.79 4-4V7h4V3h-6z"/>
                        </svg>
                    </div>
                    <div class="text-center">
                        <div class="text-xs font-medium text-gray-700 mb-1 truncate w-full">
                            <?= htmlspecialchars(pathinfo($file['name'], PATHINFO_FILENAME)) ?>
                        </div>
                        <div class="text-xs text-gray-500">🎵 Audio</div>
                    </div>
                </div>
            <?php endif; ?>
        </div>

        <div class="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/80 to-transparent p-3 opacity-0 group-hover:opacity-100 transition-opacity">
            <div class="text-white text-xs truncate font-medium mb-1">
                <?= htmlspecialchars(pathinfo($file['name'], PATHINFO_FILENAME)) ?>
            </div>
            <div class="flex justify-between items-center">
                <span class="text-white/80 text-xs"><?= $fileSize ?></span>
                <a href="<?= htmlspecialchars($file['path']) ?>" 
                   download 
                   onclick="event.stopPropagation()"
                   class="bg-white/20 hover:bg-white/30 rounded-full p-1.5 transition-colors">
                    <svg class="w-4 h-4 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4" />
                    </svg>
                </a>
            </div>
        </div>
    </div>
    <?php
}

if (empty($mediaFiles)) {
    echo '<div class="col-span-full text-center py-16 text-gray-500">';
    echo '<div class="text-4xl mb-4">📁</div>';
    echo '<p>Aucun média dans ce dossier</p>';
    echo '</div>';
}
?>