<?php
session_start();
require_once 'db.php';
require_once 'cpanel-api.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: /login');
    exit;
}

if(isset($_SESSION['andweare_member']) && $_SESSION['andweare_member'] == 1 && !empty($_SESSION['family_site_url'])) {
    header('Location: ' . $_SESSION['family_site_url']);
    exit;
}

$error = '';
$success = '';
$is_loading = false;
$current_step = 1;

if (isset($_POST['step'])) {
    $current_step = intval($_POST['step']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_family'])) {
    $family_name = htmlspecialchars(trim($_POST['family_name'] ?? ''));
    $site_type = $_POST['site_type'] ?? 'subdomain';
    $language = $_POST['language'] ?? 'en';
    
    if(empty($family_name)) {
        $error = "Please enter a family name";
    } else {
        $db = new Database();
        $conn = $db->connect();
        
        try {
            $family_slug = strtolower(preg_replace('/[^a-z0-9]+/i', '-', $family_name));
            
            $stmt = $conn->prepare("SELECT COUNT(*) FROM family_sites WHERE family_slug = ?");
            $stmt->execute([$family_slug]);
            $exists = $stmt->fetchColumn();
            
            if($exists) {
                $error = "This family name is already taken. Please choose another one.";
            } else {
                $is_loading = true;
                
                $cpanel = new CpanelAPI();
                $site_url = '';
                
                if($site_type == 'subdomain') {
                    $subdomain = $family_slug . '.andweare.com';
                    $result = $cpanel->createSubdomain($family_slug, 'andweare.com', '/public_html/' . $family_slug);
                    
                    if($result['status'] == 'success') {
                        $site_url = 'https://' . $subdomain;
                    } else {
                        $error = "Failed to create subdomain: " . $result['message'];
                    }
                } else {
                    $directory = 'andweare.com/' . $family_slug;
                    $result = $cpanel->createDirectory('/public_html/' . $family_slug);
                    
                    if($result['status'] == 'success') {
                        $site_url = 'https://' . $directory;
                    } else {
                        $error = "Failed to create directory: " . $result['message'];
                    }
                }
                
                if($site_url) {
                    $app_source = '/app-' . $language;
                    $site_destination = '/public_html/' . $family_slug;
                    
                    $copy_result = $cpanel->copyFiles($app_source, $site_destination);
                    
                    if($copy_result['status'] == 'success') {
                        $stmt = $conn->prepare("UPDATE utilisateurs SET andweare_member = 1, family_site_url = ?, family_role = 'admin' WHERE id = ?");
                        $stmt->execute([$site_url, $_SESSION['user_id']]);
                        
                        $stmt = $conn->prepare("INSERT INTO family_sites (family_name, family_slug, site_url, created_by, site_type, language) VALUES (?, ?, ?, ?, ?, ?)");
                        $stmt->execute([
                            $family_name,
                            $family_slug,
                            $site_url,
                            $_SESSION['user_id'],
                            $site_type,
                            $language
                        ]);
                        
                        $family_site_id = $conn->lastInsertId();
                        $stmt = $conn->prepare("INSERT INTO family_members (family_site_id, user_id, role) VALUES (?, ?, 'admin')");
                        $stmt->execute([$family_site_id, $_SESSION['user_id']]);
                        
                        $_SESSION['andweare_member'] = 1;
                        $_SESSION['family_site_url'] = $site_url;
                        $_SESSION['family_role'] = 'admin';
                        
                        header('Location: ' . $site_url);
                        exit;
                    } else {
                        $error = "Failed to copy application files: " . $copy_result['message'];
                    }
                }
            }
        } catch(PDOException $e) {
            $error = "Database error: " . $e->getMessage();
        } catch(Exception $e) {
            $error = "Error: " . $e->getMessage();
        }
    }
}

if (isset($_POST['next_step'])) {
    $current_step++;
} elseif (isset($_POST['prev_step'])) {
    $current_step--;
}

$current_step = max(1, min(3, $current_step));

function sanitize_slug($input) {
    return strtolower(preg_replace('/[^a-z0-9]+/i', '-', $input));
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Create Your Family Hub - AndWeare</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+KR:wght@300;400;500;700&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        body {
            font-family: 'Noto Sans KR', sans-serif;
        }
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(255, 255, 255, 0.9);
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            z-index: 50;
        }
        .loading-dots span {
            animation: loadingDots 1.4s infinite ease-in-out both;
        }
        .loading-dots span:nth-child(1) {
            animation-delay: -0.32s;
        }
        .loading-dots span:nth-child(2) {
            animation-delay: -0.16s;
        }
        @keyframes loadingDots {
            0%, 80%, 100% { transform: scale(0); }
            40% { transform: scale(1); }
        }
        .faq-item.active .faq-content {
            display: block;
        }
        .faq-item.active .faq-icon {
            transform: rotate(180deg);
        }
    </style>
</head>
<body class="bg-white min-h-screen">
    <?php if ($is_loading): ?>
        <div class="loading-overlay">
            <div class="mb-4">
                <svg class="animate-spin h-12 w-12 text-black" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                    <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                </svg>
            </div>
            <h3 class="text-xl font-medium mb-2">Creating your family hub...</h3>
            <p class="text-gray-600 mb-4">This may take a few moments. Please don't close this window.</p>
            <div class="loading-dots flex space-x-2">
                <span class="w-3 h-3 bg-black rounded-full"></span>
                <span class="w-3 h-3 bg-black rounded-full"></span>
                <span class="w-3 h-3 bg-black rounded-full"></span>
            </div>
        </div>
    <?php endif; ?>

    <div class="max-w-4xl mx-auto px-4 py-8" <?php if ($is_loading): ?>style="display: none;"<?php endif; ?>>
        <div class="flex justify-between items-center mb-8">
            <a href="/" class="block">
                <img src="logo.png" alt="AndWeare Logo" class="h-10">
            </a>
            <a href="/login/logout.php" class="text-gray-600 hover:text-black transition-colors">
                Sign Out
            </a>
        </div>
        
        <div class="text-center mb-12">
            <h1 class="text-3xl font-light mb-2">Create Your Family Hub</h1>
            <p class="text-gray-600">Welcome <?php echo htmlspecialchars($_SESSION['username']); ?>, let's set up your family's online space</p>
        </div>
        
        <?php if($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>
        
        <div class="bg-gray-50 rounded-lg shadow-sm p-6 mb-8">
            <div class="flex justify-between mb-8">
                <div class="flex items-center <?php echo $current_step >= 1 ? 'text-blue-600' : 'text-gray-400'; ?>">
                    <div class="w-8 h-8 rounded-full flex items-center justify-center mr-2 <?php echo $current_step >= 1 ? 'bg-blue-600 text-white' : 'bg-gray-200'; ?>">1</div>
                    <span>Family Details</span>
                </div>
                <div class="flex items-center <?php echo $current_step >= 2 ? 'text-blue-600' : 'text-gray-400'; ?>">
                    <div class="w-8 h-8 rounded-full flex items-center justify-center mr-2 <?php echo $current_step >= 2 ? 'bg-blue-600 text-white' : 'bg-gray-200'; ?>">2</div>
                    <span>Site Type</span>
                </div>
                <div class="flex items-center <?php echo $current_step >= 3 ? 'text-blue-600' : 'text-gray-400'; ?>">
                    <div class="w-8 h-8 rounded-full flex items-center justify-center mr-2 <?php echo $current_step >= 3 ? 'bg-blue-600 text-white' : 'bg-gray-200'; ?>">3</div>
                    <span>Language</span>
                </div>
            </div>
            
            <form method="POST" id="familyForm">
                <input type="hidden" name="step" value="<?php echo $current_step; ?>">
                
                <!-- Step 1: Family Details -->
                <div id="step1" <?php if ($current_step != 1): ?>style="display: none;"<?php endif; ?>>
                    <h2 class="text-xl font-medium mb-4">What's your family's name?</h2>
                    <p class="text-gray-600 mb-4">This will be the name of your family hub.</p>
                    
                    <div class="mb-6">
                        <label for="family_name" class="block text-sm font-medium text-gray-700 mb-1">Family Name</label>
                        <input 
                            type="text" 
                            id="family_name" 
                            name="family_name" 
                            class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            placeholder="e.g. The Smiths, Johnson Family"
                            required
                            value="<?php echo isset($_POST['family_name']) ? htmlspecialchars($_POST['family_name']) : ''; ?>"
                            onkeyup="updatePreview()"
                        >
                        <p class="mt-2 text-sm text-gray-500">
                            This will be used to create your family's URL.
                        </p>
                    </div>
                    
                    <div class="flex justify-end">
                        <button 
                            type="submit" 
                            name="next_step" 
                            class="px-4 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors"
                            id="nextButton"
                            disabled
                        >
                            Next
                        </button>
                    </div>
                </div>
                
                <!-- Step 2: Site Type -->
                <div id="step2" <?php if ($current_step != 2): ?>style="display: none;"<?php endif; ?>>
                    <h2 class="text-xl font-medium mb-4">Choose your site type</h2>
                    <p class="text-gray-600 mb-4">How would you like your family hub to be accessed?</p>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all site-type-option <?php echo (!isset($_POST['site_type']) || $_POST['site_type'] == 'subdomain') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            data-type="subdomain"
                            onclick="selectSiteType('subdomain')"
                        >
                            <div class="flex items-center mb-2">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (!isset($_POST['site_type']) || $_POST['site_type'] == 'subdomain') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (!isset($_POST['site_type']) || $_POST['site_type'] == 'subdomain') ? '' : 'style="display: none;"'; ?>></div>
                                </div>
                                <span class="font-medium">Subdomain</span>
                            </div>
                            <p class="text-sm text-gray-600 ml-7">
                                Your family hub will be available at:<br>
                                <strong id="subdomain-preview"></strong>
                            </p>
                        </div>
                        
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all site-type-option <?php echo (isset($_POST['site_type']) && $_POST['site_type'] == 'directory') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            data-type="directory"
                            onclick="selectSiteType('directory')"
                        >
                            <div class="flex items-center mb-2">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (isset($_POST['site_type']) && $_POST['site_type'] == 'directory') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (isset($_POST['site_type']) && $_POST['site_type'] == 'directory') ? '' : 'style="display: none;"'; ?>></div>
                                </div>
                                <span class="font-medium">Directory</span>
                            </div>
                            <p class="text-sm text-gray-600 ml-7">
                                Your family hub will be available at:<br>
                                <strong id="directory-preview"></strong>
                            </p>
                        </div>
                    </div>
                    
                    <input type="hidden" name="site_type" id="site_type" value="<?php echo isset($_POST['site_type']) ? htmlspecialchars($_POST['site_type']) : 'subdomain'; ?>">
                    
                    <div class="flex justify-between">
                        <button 
                            type="submit" 
                            name="prev_step" 
                            class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                        >
                            Back
                        </button>
                        <button 
                            type="submit" 
                            name="next_step" 
                            class="px-4 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors"
                        >
                            Next
                        </button>
                    </div>
                </div>
                
                <!-- Step 3: Language -->
                <div id="step3" <?php if ($current_step != 3): ?>style="display: none;"<?php endif; ?>>
                    <h2 class="text-xl font-medium mb-4">Choose a language</h2>
                    <p class="text-gray-600 mb-4">Select the primary language for your family hub.</p>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all language-option <?php echo (!isset($_POST['language']) || $_POST['language'] == 'en') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            data-language="en"
                            onclick="selectLanguage('en')"
                        >
                            <div class="flex items-center">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (!isset($_POST['language']) || $_POST['language'] == 'en') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (!isset($_POST['language']) || $_POST['language'] == 'en') ? '' : 'style="display: none;"'; ?>></div>
                                </div>
                                <span class="font-medium">English</span>
                            </div>
                        </div>
                        
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all language-option <?php echo (isset($_POST['language']) && $_POST['language'] == 'fr') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            data-language="fr"
                            onclick="selectLanguage('fr')"
                        >
                            <div class="flex items-center">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (isset($_POST['language']) && $_POST['language'] == 'fr') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (isset($_POST['language']) && $_POST['language'] == 'fr') ? '' : 'style="display: none;"'; ?>></div>
                                </div>
                                <span class="font-medium">Français</span>
                            </div>
                        </div>
                        
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all language-option <?php echo (isset($_POST['language']) && $_POST['language'] == 'de') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            data-language="de"
                            onclick="selectLanguage('de')"
                        >
                            <div class="flex items-center">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (isset($_POST['language']) && $_POST['language'] == 'de') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (isset($_POST['language']) && $_POST['language'] == 'de') ? '' : 'style="display: none;"'; ?>></div>
                                </div>
                                <span class="font-medium">Deutsch</span>
                            </div>
                        </div>
                        
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all language-option <?php echo (isset($_POST['language']) && $_POST['language'] == 'nl') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            data-language="nl"
                            onclick="selectLanguage('nl')"
                        >
                            <div class="flex items-center">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (isset($_POST['language']) && $_POST['language'] == 'nl') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (isset($_POST['language']) && $_POST['language'] == 'nl') ? '' : 'style="display: none;"'; ?>></div>
                                </div>
                                <span class="font-medium">Nederlands</span>
                            </div>
                        </div>
                    </div>
                    
                    <input type="hidden" name="language" id="language" value="<?php echo isset($_POST['language']) ? htmlspecialchars($_POST['language']) : 'en'; ?>">
                    
                    <div class="bg-white p-4 rounded-lg border border-gray-200 mb-6">
                        <h3 class="font-medium mb-2">Your family hub summary</h3>
                        <ul class="space-y-2 text-sm">
                            <li class="flex justify-between">
                                <span class="text-gray-600">Family name:</span>
                                <span class="font-medium" id="summary-name"></span>
                            </li>
                            <li class="flex justify-between">
                                <span class="text-gray-600">URL:</span>
                                <span class="font-medium" id="summary-url"></span>
                            </li>
                            <li class="flex justify-between">
                                <span class="text-gray-600">Language:</span>
                                <span class="font-medium" id="summary-language"></span>
                            </li>
                        </ul>
                    </div>
                    
                    <div class="flex justify-between">
                        <button 
                            type="submit" 
                            name="prev_step" 
                            class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                        >
                            Back
                        </button>
                        <button 
                            type="submit" 
                            name="create_family" 
                            class="px-4 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors"
                        >
                            Create Family Hub
                        </button>
                    </div>
                </div>
            </form>
        </div>
        
        <div class="mt-8">
            <h3 class="text-lg font-medium mb-4">Frequently Asked Questions</h3>
            
            <div class="space-y-4">
                <div class="border border-gray-200 rounded-lg overflow-hidden faq-item">
                    <button 
                        class="flex justify-between items-center w-full p-4 text-left bg-white hover:bg-gray-50 transition-colors"
                        onclick="toggleFAQ(this)"
                    >
                        <span class="font-medium">Can I change my family hub URL later?</span>
                        <svg class="h-5 w-5 text-gray-500 faq-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd" />
                        </svg>
                    </button>
                    <div class="p-4 border-t border-gray-200 bg-white faq-content" style="display: none;">
                        <p class="text-gray-600">
                            Once you've created your family hub URL, it cannot be changed. This ensures the stability of your family's online home. Choose a name that will stand the test of time!
                        </p>
                    </div>
                </div>
                
                <div class="border border-gray-200 rounded-lg overflow-hidden faq-item">
                    <button 
                        class="flex justify-between items-center w-full p-4 text-left bg-white hover:bg-gray-50 transition-colors"
                        onclick="toggleFAQ(this)"
                    >
                        <span class="font-medium">How do I invite family members?</span>
                        <svg class="h-5 w-5 text-gray-500 faq-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd" />
                        </svg>
                    </button>
                    <div class="p-4 border-t border-gray-200 bg-white faq-content" style="display: none;">
                        <p class="text-gray-600">
                            After creating your family hub, you can invite family members by sending them an email invitation directly from your dashboard. They'll receive a link to join your family hub, and can create their own account if they don't already have one.
                        </p>
                    </div>
                </div>
                
                <div class="border border-gray-200 rounded-lg overflow-hidden faq-item">
                    <button 
                        class="flex justify-between items-center w-full p-4 text-left bg-white hover:bg-gray-50 transition-colors"
                        onclick="toggleFAQ(this)"
                    >
                        <span class="font-medium">Can I change the language later?</span>
                        <svg class="h-5 w-5 text-gray-500 faq-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd" />
                        </svg>
                    </button>
                    <div class="p-4 border-t border-gray-200 bg-white faq-content" style="display: none;">
                        <p class="text-gray-600">
                            Yes, you can change the language of your family hub after creation. You can do this in the settings section of your family hub dashboard.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        let familyName = "<?php echo isset($_POST['family_name']) ? htmlspecialchars($_POST['family_name']) : ''; ?>";
        let familySlug = "<?php echo isset($_POST['family_name']) ? sanitize_slug($_POST['family_name']) : ''; ?>";
        let siteType = "<?php echo isset($_POST['site_type']) ? htmlspecialchars($_POST['site_type']) : 'subdomain'; ?>";
        let language = "<?php echo isset($_POST['language']) ? htmlspecialchars($_POST['language']) : 'en'; ?>";
        
        function updatePreview() {
            let nameInput = document.getElementById('family_name');
            let nextButton = document.getElementById('nextButton');
            familyName = nameInput.value.trim();
            
            if (familyName) {
                nextButton.disabled = false;
                familySlug = familyName.toLowerCase().replace(/[^a-z0-9]+/g, '-');
                if (familySlug.length > 30) {
                    familySlug = familySlug.substring(0, 30);
                }
            } else {
                nextButton.disabled = true;
            }
            
            document.getElementById('subdomain-preview').textContent = familySlug + '.andweare.com';
            document.getElementById('directory-preview').textContent = 'andweare.com/' + familySlug;
            
            updateSummary();
        }
        
        function selectSiteType(type) {
            siteType = type;
            document.getElementById('site_type').value = type;
            
            let options = document.querySelectorAll('.site-type-option');
            options.forEach(option => {
                let isSelected = option.dataset.type === type;
                option.className = isSelected 
                    ? 'border rounded-lg p-4 cursor-pointer transition-all site-type-option border-blue-500 bg-blue-50'
                    : 'border rounded-lg p-4 cursor-pointer transition-all site-type-option border-gray-200 hover:border-gray-300';
                
                let indicator = option.querySelector('.w-3');
                if (indicator) {
                    indicator.style.display = isSelected ? 'block' : 'none';
                }
                
                let border = option.querySelector('.w-5');
                if (border) {
                    border.className = isSelected 
                        ? 'w-5 h-5 rounded-full border border-blue-500 mr-2 flex items-center justify-center'
                        : 'w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center';
                }
            });
            
            updateSummary();
        }
        
        function selectLanguage(lang) {
            language = lang;
            document.getElementById('language').value = lang;
            
            let options = document.querySelectorAll('.language-option');
            options.forEach(option => {
                let isSelected = option.dataset.language === lang;
                option.className = isSelected 
                    ? 'border rounded-lg p-4 cursor-pointer transition-all language-option border-blue-500 bg-blue-50'
                    : 'border rounded-lg p-4 cursor-pointer transition-all language-option border-gray-200 hover:border-gray-300';
                
                let indicator = option.querySelector('.w-3');
                if (indicator) {
                    indicator.style.display = isSelected ? 'block' : 'none';
                }
                
                let border = option.querySelector('.w-5');
                if (border) {
                    border.className = isSelected 
                        ? 'w-5 h-5 rounded-full border border-blue-500 mr-2 flex items-center justify-center'
                        : 'w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center';
                }
            });
            
            updateSummary();
        }
        
        function updateSummary() {
            document.getElementById('summary-name').textContent = familyName;
            document.getElementById('summary-url').textContent = siteType === 'subdomain' 
                ? familySlug + '.andweare.com' 
                : 'andweare.com/' + familySlug;
                
            let langDisplay = '';
            switch(language) {
                case 'en': langDisplay = 'English'; break;
                case 'fr': langDisplay = 'Français'; break;
                case 'de': langDisplay = 'Deutsch'; break;
                case 'nl': langDisplay = 'Nederlands'; break;
                default: langDisplay = 'English';
            }
            document.getElementById('summary-language').textContent = langDisplay;
        }
        
        function toggleFAQ(button) {
            const faqItem = button.closest('.faq-item');
            const content = faqItem.querySelector('.faq-content');
            
            if (content.style.display === 'none' || content.style.display === '') {
                // Close all other FAQs
                document.querySelectorAll('.faq-content').forEach(item => {
                    item.style.display = 'none';
                });
                document.querySelectorAll('.faq-item').forEach(item => {
                    item.classList.remove('active');
                });
                
                // Open this FAQ
                content.style.display = 'block';
                faqItem.classList.add('active');
            } else {
                content.style.display = 'none';
                faqItem.classList.remove('active');
            }
        }
        
        // Initialize the page
        document.addEventListener('DOMContentLoaded', function() {
            updatePreview();
            updateSummary();
        });
    </script>
</body>
</html>