document.addEventListener('DOMContentLoaded', () => {
    const app = {
        map: null,
        mapbox: null,
        mapkit: null,
        markers: {},
        userMarker: null,
        tempMarker: null,
        selectedLightId: null,
        userWatchId: null,
        refreshTimerId: null,
        rafId: null,
        lastTimestamp: 0,
        settings: {
            darkMode: false,
            notifications: true,
            autoRefresh: true,
            predictions: true,
            autoMeasure: true,
            mapStyle: 'navigation-day-v1',
            defaultTransport: 'driving'
        },
        measure: {
            timer: 0,
            startTime: 0,
            rafId: null,
            mode: null,
            lightId: null
        },
        route: {
            active: false,
            isNavigating: false,
            startCoords: null,
            endCoords: null,
            mode: 'driving',
            route: null,
            duration: 0,
            distance: 0,
            lastStopDetection: 0,
            lightsOnRoute: []
        },
        user: {
            isLoggedIn: false,
            firstName: '',
            lastName: '',
            email: '',
            preferredVehicle: 'car'
        },
        state: {
            selectingLocation: false,
            menuActive: false,
            pendingLights: [],
            tileLayer: null
        }
    };

    const dom = {
        map: document.getElementById('map'),
        fabMenuBtn: document.getElementById('fabMenuBtn'),
        menuItems: document.querySelectorAll('.menu-item'),
        modals: {
            route: document.getElementById('routeSearchModal'),
            add: document.getElementById('addLightModal'),
            measure: document.getElementById('measureModal'),
            settings: document.getElementById('settingsModal'),
            account: document.getElementById('accountModal'),
            onboarding: document.getElementById('onboardingModal')
        },
        routePanel: document.querySelector('.route-panel'),
        searchContainer: document.querySelector('.search-container'),
        infoPanel: document.getElementById('infoPanel'),
        infoPanelTitle: document.getElementById('infoPanelTitle'),
        infoPanelNavigate: document.getElementById('infoPanelNavigate'),
        infoPanelMeasure: document.getElementById('infoPanelMeasure'),
        closeInfoPanel: document.getElementById('closeInfoPanel'),
        lightStatusText: document.getElementById('lightStatusText'),
        lightStatusTime: document.getElementById('lightStatusTime'),
        lightStatusBar: document.getElementById('lightStatusBar'),
        overlays: document.querySelectorAll('.modal-overlay'),
        modalContents: document.querySelectorAll('.modal-content'),
        closeButtons: document.querySelectorAll('.close-modal'),
        certificationBanner: document.getElementById('certificationBanner'),
        transportModeChoices: document.querySelectorAll('.transport-mode-choice'),
        transportModeBtns: document.querySelectorAll('.transport-mode-btn'),
        notification: {
            banner: document.getElementById('notificationBanner'),
            title: document.getElementById('notificationTitle'),
            message: document.getElementById('notificationMessage'),
            icon: document.getElementById('notificationIcon'),
            close: document.getElementById('closeNotification')
        },
        route: {
            panel: document.querySelector('.route-panel'),
            estimatedDuration: document.getElementById('estimatedDuration'),
            arrivalTime: document.getElementById('arrivalTime'),
            cancelBtn: document.getElementById('cancelRouteBtn'),
            startNavigationBtn: document.getElementById('startNavigationBtn'),
            drivingModeBtn: document.getElementById('drivingModeBtn'),
            bikingModeBtn: document.getElementById('bikingModeBtn'),
            walkingModeBtn: document.getElementById('walkingModeBtn'),
            calculateBtn: document.getElementById('calculateRouteBtn')
        },
        measure: {
            title: document.getElementById('measureTitle'),
            status: document.getElementById('measureStatus'),
            container: document.getElementById('timerContainer'),
            instructions: document.getElementById('timerInstructions'),
            display: document.getElementById('timerDisplay'),
            result: document.getElementById('measureResult'),
            redBtn: document.getElementById('measureRedBtn'),
            greenBtn: document.getElementById('measureGreenBtn'),
            startBtn: document.getElementById('startTimer'),
            stopBtn: document.getElementById('stopTimer'),
            saveBtn: document.getElementById('saveTimer')
        },
        settings: {
            darkMode: document.getElementById('darkModeToggle'),
            notifications: document.getElementById('notificationsToggle'),
            autoRefresh: document.getElementById('autoRefreshToggle'),
            autoMeasure: document.getElementById('autoMeasureToggle'),
            mapStyle: document.getElementById('mapStyle'),
            defaultTransport: document.getElementById('defaultTransport'),
            resetAppBtn: document.getElementById('resetAppBtn')
        },
        account: {
            name: document.getElementById('accountName'),
            email: document.getElementById('accountEmail'),
            loginButtons: document.getElementById('loginButtons'),
            logoutButton: document.getElementById('logoutButton'),
            loginBtn: document.getElementById('loginBtn'),
            signupBtn: document.getElementById('signupBtn'),
            logoutBtn: document.getElementById('logoutBtn'),
            forms: {
                account: document.getElementById('accountForm'),
                login: document.getElementById('loginForm'),
                signup: document.getElementById('signupForm'),
                onboarding: document.getElementById('onboardingForm')
            }
        },
        form: {
            addLight: document.getElementById('addLightForm')
        }
    };

    init();

    function init() {
        loadSettings();
        setupMapKit();
        loadUserInfo();
        bindEvents();
        setupDraggableModals();
        
        if (!app.user.isLoggedIn && !localStorage.getItem('trafficLightOnboardingCompleted')) {
            showOnboardingModal();
        }
    }

    function loadSettings() {
        try {
            const saved = localStorage.getItem('trafficLightSettings');
            if (saved) {
                const parsed = JSON.parse(saved);
                app.settings = { ...app.settings, ...parsed };
                applySettings();
            }
        } catch (e) {
            console.error('Error loading settings');
        }
    }

    function applySettings() {
        dom.settings.darkMode.checked = app.settings.darkMode;
        dom.settings.notifications.checked = app.settings.notifications;
        dom.settings.autoRefresh.checked = app.settings.autoRefresh;
        dom.settings.autoMeasure.checked = app.settings.autoMeasure;
        dom.settings.mapStyle.value = app.settings.mapStyle;
        dom.settings.defaultTransport.value = app.settings.defaultTransport;
        
        if (app.settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
        }
    }

    function loadUserInfo() {
        const userId = localStorage.getItem('userId') || sessionStorage.getItem('userId');
        if (userId) {
            fetch(`user-info.php?user_id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        app.user.isLoggedIn = true;
                        app.user.firstName = data.user.first_name;
                        app.user.lastName = data.user.last_name;
                        app.user.email = data.user.email;
                        app.user.preferredVehicle = data.user.preferred_vehicle;
                        
                        updateAccountDisplay();
                    } else {
                        app.user.isLoggedIn = false;
                        updateAccountDisplay();
                    }
                })
                .catch(error => {
                    console.error('Error loading user info:', error);
                    app.user.isLoggedIn = false;
                    updateAccountDisplay();
                });
        } else {
            app.user.isLoggedIn = false;
            updateAccountDisplay();
        }
    }

    function updateAccountDisplay() {
        if (app.user.isLoggedIn) {
            dom.account.name.textContent = `${app.user.firstName} ${app.user.lastName}`;
            dom.account.email.textContent = app.user.email;
            dom.account.loginButtons.classList.add('hidden');
            dom.account.logoutButton.classList.remove('hidden');
        } else {
            dom.account.name.textContent = 'Guest User';
            dom.account.email.textContent = 'Not signed in';
            dom.account.loginButtons.classList.remove('hidden');
            dom.account.logoutButton.classList.add('hidden');
        }
    }

    function setupMapKit() {
        fetch('get-apple-map-key.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    mapkit.init({
                        authorizationCallback: function(done) {
                            done(data.token);
                        }
                    });
                    
                    setupMapBox();
                } else {
                    console.error('Error loading MapKit token:', data.message);
                    showNotification('Error', 'Could not initialize maps. Please try again later.', 'error');
                    setupMapBox();
                }
            })
            .catch(error => {
                console.error('Error loading MapKit:', error);
                showNotification('Error', 'Could not initialize maps. Please try again later.', 'error');
                setupMapBox();
            });
    }

    function setupMapBox() {
        fetch('get-mapbox-key.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    mapboxgl.accessToken = data.key;
                    
                    app.map = new mapboxgl.Map({
                        container: 'map',
                        style: `mapbox://styles/mapbox/${app.settings.mapStyle}`,
                        center: [-0.09, 51.505],
                        zoom: 13,
                        logoPosition: 'bottom-left'
                    });
                    
                    app.map.on('load', () => {
                        app.map.addControl(new mapboxgl.NavigationControl({
                            showCompass: false
                        }), 'bottom-right');
                        
                        app.map.addControl(new mapboxgl.GeolocateControl({
                            positionOptions: {
                                enableHighAccuracy: true
                            },
                            trackUserLocation: true,
                            showUserHeading: true
                        }), 'bottom-right');
                        
                        app.directions = new MapboxDirections({
                            accessToken: mapboxgl.accessToken,
                            unit: 'metric',
                            profile: 'mapbox/driving',
                            alternatives: false,
                            geometries: 'geojson',
                            controls: {
                                inputs: false,
                                instructions: false
                            },
                            flyTo: false
                        });
                        
                        app.map.addControl(app.directions, 'top-left');
                        
                        setupSearchBox();
                        loadTrafficLights();
                        startGeolocation();
                    });
                    
                    app.map.on('click', handleMapClick);
                } else {
                    console.error('Error loading MapBox token:', data.message);
                    showNotification('Error', 'Could not initialize maps. Please try again later.', 'error');
                }
            })
            .catch(error => {
                console.error('Error loading MapBox:', error);
                showNotification('Error', 'Could not initialize maps. Please try again later.', 'error');
            });
    }

    function setupSearchBox() {
        const searchBox = new MapboxGeocoder({
            accessToken: mapboxgl.accessToken,
            mapboxgl: mapboxgl,
            marker: false,
            placeholder: 'Search locations or traffic lights...'
        });
        
        document.getElementById('searchBox').appendChild(searchBox.onAdd(app.map));
        
        const startLocationSearch = new MapboxGeocoder({
            accessToken: mapboxgl.accessToken,
            mapboxgl: mapboxgl,
            marker: false,
            placeholder: 'Enter starting location'
        });
        
        const endLocationSearch = new MapboxGeocoder({
            accessToken: mapboxgl.accessToken,
            mapboxgl: mapboxgl,
            marker: false,
            placeholder: 'Enter destination'
        });
        
        document.getElementById('startLocationSearch').appendChild(startLocationSearch.onAdd(app.map));
        document.getElementById('endLocationSearch').appendChild(endLocationSearch.onAdd(app.map));
        
        startLocationSearch.on('result', (e) => {
            app.route.startCoords = [e.result.center[0], e.result.center[1]];
        });
        
        endLocationSearch.on('result', (e) => {
            app.route.endCoords = [e.result.center[0], e.result.center[1]];
        });
        
        searchBox.on('result', (e) => {
            if (app.state.menuActive) {
                toggleMenu();
            }
        });
    }

    function bindEvents() {
        dom.fabMenuBtn.addEventListener('click', toggleMenu);
        
        dom.menuItems.forEach(item => {
            item.querySelector('button').addEventListener('click', () => {
                handleMenuItemClick(item.id);
            });
        });
        
        dom.closeButtons.forEach(btn => {
            btn.addEventListener('click', () => {
                const modal = btn.closest('.modal');
                closeModal(modal);
            });
        });
        
        dom.overlays.forEach(overlay => {
            overlay.addEventListener('click', () => {
                const modal = overlay.closest('.modal');
                closeModal(modal);
            });
        });
        
        dom.closeInfoPanel.addEventListener('click', () => {
            dom.infoPanel.classList.remove('active');
        });
        
        dom.infoPanelNavigate.addEventListener('click', () => {
            if (app.selectedLightId && app.markers[app.selectedLightId]) {
                const light = app.markers[app.selectedLightId].data;
                calculateRoute(null, [light.longitude, light.latitude]);
                dom.infoPanel.classList.remove('active');
            }
        });
        
        dom.infoPanelMeasure.addEventListener('click', () => {
            if (app.selectedLightId) {
                dom.infoPanel.classList.remove('active');
                openMeasureModal(app.selectedLightId);
            }
        });
        
        dom.settings.darkMode.addEventListener('change', toggleDarkMode);
        dom.settings.autoRefresh.addEventListener('change', toggleAutoRefresh);
        dom.settings.notifications.addEventListener('change', saveSettings);
        dom.settings.autoMeasure.addEventListener('change', saveSettings);
        dom.settings.mapStyle.addEventListener('change', changeMapStyle);
        dom.settings.defaultTransport.addEventListener('change', saveSettings);
        
        dom.settings.resetAppBtn.addEventListener('click', resetAppData);
        
        dom.measure.redBtn.addEventListener('click', () => startMeasure('red'));
        dom.measure.greenBtn.addEventListener('click', () => startMeasure('green'));
        dom.measure.startBtn.addEventListener('click', startMeasureTimer);
        dom.measure.stopBtn.addEventListener('click', stopMeasureTimer);
        dom.measure.saveBtn.addEventListener('click', saveMeasureTimer);
        
        dom.account.loginBtn.addEventListener('click', () => {
            showLoginForm();
            openModal(dom.modals.account);
        });
        
        dom.account.signupBtn.addEventListener('click', () => {
            showSignupForm();
            openModal(dom.modals.account);
        });
        
        dom.account.logoutBtn.addEventListener('click', logoutUser);
        
        document.getElementById('switchToSignupBtn').addEventListener('click', (e) => {
            e.preventDefault();
            showSignupForm();
        });
        
        document.getElementById('switchToLoginBtn').addEventListener('click', (e) => {
            e.preventDefault();
            showLoginForm();
        });
        
        dom.account.forms.login.addEventListener('submit', (e) => {
            e.preventDefault();
            loginUser();
        });
        
        dom.account.forms.signup.addEventListener('submit', (e) => {
            e.preventDefault();
            signupUser();
        });
        
        dom.account.forms.onboarding.addEventListener('submit', (e) => {
            e.preventDefault();
            createAccount(true);
        });
        
        dom.form.addLight.addEventListener('submit', handleAddLight);
        
        dom.transportModeChoices.forEach(btn => {
            btn.addEventListener('click', () => {
                dom.transportModeChoices.forEach(b => b.classList.remove('active', 'bg-primary-50', 'border-primary-200', 'text-primary-600'));
                dom.transportModeChoices.forEach(b => b.classList.add('bg-gray-50', 'border-gray-200', 'text-gray-600'));
                
                btn.classList.remove('bg-gray-50', 'border-gray-200', 'text-gray-600');
                btn.classList.add('active', 'bg-primary-50', 'border-primary-200', 'text-primary-600');
                
                app.route.mode = btn.getAttribute('data-mode');
            });
        });
        
        dom.transportModeBtns.forEach(btn => {
            btn.addEventListener('click', () => {
                dom.transportModeBtns.forEach(b => b.classList.remove('active', 'bg-white', 'text-primary-600'));
                dom.transportModeBtns.forEach(b => b.classList.add('text-gray-600'));
                
                btn.classList.remove('text-gray-600');
                btn.classList.add('active', 'bg-white', 'text-primary-600');
                
                let mode = 'driving';
                if (btn.id === 'bikingModeBtn') mode = 'cycling';
                if (btn.id === 'walkingModeBtn') mode = 'walking';
                
                updateRouteProfile(mode);
            });
        });
        
        dom.route.calculateBtn.addEventListener('click', () => {
            if (app.route.startCoords && app.route.endCoords) {
                calculateRoute(app.route.startCoords, app.route.endCoords);
                closeModal(dom.modals.route);
            } else {
                showNotification('Error', 'Please select starting point and destination', 'error');
            }
        });
        
        dom.route.cancelBtn.addEventListener('click', cancelRoute);
        dom.route.startNavigationBtn.addEventListener('click', startNavigation);
        
        dom.notification.close.addEventListener('click', () => {
            dom.notification.banner.style.opacity = '0';
        });
        
        window.addEventListener('online', handleOnline);
        window.addEventListener('offline', handleOffline);
        document.addEventListener('visibilitychange', handleVisibilityChange);
    }

    function setupDraggableModals() {
        const draggableHandles = document.querySelectorAll('.draggable-handle');
        
        draggableHandles.forEach(handle => {
            const modal = handle.closest('.modal-content');
            let startY, currentY, initialY;
            let isDragging = false;
            
            handle.addEventListener('touchstart', e => {
                startY = e.touches[0].clientY;
                initialY = 0;
                isDragging = true;
                modal.style.transition = 'none';
            }, { passive: true });
            
            handle.addEventListener('touchmove', e => {
                if (!isDragging) return;
                currentY = e.touches[0].clientY;
                const diffY = currentY - startY;
                if (diffY < 0) return;
                
                modal.style.transform = `translateY(${diffY}px)`;
            }, { passive: true });
            
            handle.addEventListener('touchend', e => {
                modal.style.transition = 'transform 0.3s ease';
                if (!isDragging) return;
                
                const diffY = currentY - startY;
                if (diffY > 100) {
                    modal.style.transform = 'translateY(100%)';
                    const modalContainer = modal.closest('.modal');
                    setTimeout(() => {
                        modalContainer.style.display = 'none';
                        modal.style.transform = 'translateY(0)';
                    }, 300);
                } else {
                    modal.style.transform = 'translateY(0)';
                }
                
                isDragging = false;
            }, { passive: true });
        });
    }

    function toggleMenu() {
        app.state.menuActive = !app.state.menuActive;
        
        if (app.state.menuActive) {
            dom.fabMenuBtn.innerHTML = '<i class="fas fa-times"></i>';
            
            dom.menuItems.forEach(item => {
                const position = item.getAttribute('data-position');
                setTimeout(() => {
                    item.classList.add('active');
                    item.style.transform = `translateY(-${position * 70}px)`;
                }, position * 50);
            });
        } else {
            dom.fabMenuBtn.innerHTML = '<i class="fas fa-plus"></i>';
            
            dom.menuItems.forEach(item => {
                item.classList.remove('active');
                item.style.transform = '';
            });
        }
    }

    function handleMenuItemClick(id) {
        toggleMenu();
        
        switch (id) {
            case 'accountMenuItem':
                if (app.user.isLoggedIn) {
                    showAccountForm();
                } else {
                    showLoginForm();
                }
                openModal(dom.modals.account);
                break;
                
            case 'addLightMenuItem':
                app.state.selectingLocation = true;
                if (app.userMarker) {
                    const pos = app.userMarker.getLngLat();
                    document.getElementById('latitude').value = pos.lat.toFixed(6);
                    document.getElementById('longitude').value = pos.lng.toFixed(6);
                    
                    if (app.tempMarker) app.tempMarker.remove();
                    app.tempMarker = new mapboxgl.Marker({ draggable: true })
                        .setLngLat([pos.lng, pos.lat])
                        .addTo(app.map);
                    
                    app.tempMarker.on('dragend', updateMarkerPosition);
                }
                openModal(dom.modals.add);
                break;
                
            case 'searchMenuItem':
                openModal(dom.modals.route);
                break;
                
            case 'locateMenuItem':
                locateUser();
                break;
                
            case 'settingsMenuItem':
                openModal(dom.modals.settings);
                break;
        }
    }

    function openModal(modal) {
        modal.style.display = 'flex';
        setTimeout(() => {
            const content = modal.querySelector('.modal-content');
            content.classList.add('active');
        }, 10);
    }

    function closeModal(modal) {
        const content = modal.querySelector('.modal-content');
        content.classList.remove('active');
        
        setTimeout(() => {
            modal.style.display = 'none';
            
            if (modal === dom.modals.add && app.state.selectingLocation) {
                app.state.selectingLocation = false;
                if (app.tempMarker) {
                    app.tempMarker.remove();
                    app.tempMarker = null;
                }
            }
        }, 300);
    }

    function locateUser() {
        if (app.userMarker) {
            const pos = app.userMarker.getLngLat();
            app.map.flyTo({
                center: [pos.lng, pos.lat],
                zoom: 16
            });
        } else {
            app.map.triggerRepaint();
        }
    }

    function handleMapClick(e) {
        if (app.state.selectingLocation) {
            document.getElementById('latitude').value = e.lngLat.lat.toFixed(6);
            document.getElementById('longitude').value = e.lngLat.lng.toFixed(6);
            
            if (app.tempMarker) app.tempMarker.remove();
            app.tempMarker = new mapboxgl.Marker({ draggable: true })
                .setLngLat([e.lngLat.lng, e.lngLat.lat])
                .addTo(app.map);
            
            app.tempMarker.on('dragend', updateMarkerPosition);
        }
    }

    function updateMarkerPosition() {
        const pos = app.tempMarker.getLngLat();
        document.getElementById('latitude').value = pos.lat.toFixed(6);
        document.getElementById('longitude').value = pos.lng.toFixed(6);
    }

    function handleAddLight(e) {
        e.preventDefault();
        
        const submitBtn = dom.form.addLight.querySelector('[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Adding...';
        submitBtn.disabled = true;
        
        const lightData = {
            name: document.getElementById('lightName').value,
            latitude: document.getElementById('latitude').value,
            longitude: document.getElementById('longitude').value,
            direction: document.getElementById('direction').value,
            red_duration: document.getElementById('redDuration').value,
            green_duration: document.getElementById('greenDuration').value,
            user_id: app.user.isLoggedIn ? localStorage.getItem('userId') || sessionStorage.getItem('userId') : null
        };
        
        fetch('api/add-traffic-light.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify(lightData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                app.state.pendingLights.push(data.light.id);
                showCertificationBanner();
                showNotification('Success', 'Traffic light added successfully! Pending certification.', 'success');
                
                closeModal(dom.modals.add);
                dom.form.addLight.reset();
                
                if (app.tempMarker) {
                    app.tempMarker.remove();
                    app.tempMarker = null;
                }
                
                app.state.selectingLocation = false;
                addLightToMap(data.light);
            } else {
                showNotification('Error', 'Error: ' + data.message, 'error');
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Error', 'Connection error', 'error');
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }

    function showCertificationBanner() {
        dom.certificationBanner.classList.remove('hidden');
    }

    function loadTrafficLights() {
        fetch('api/get-traffic-lights.php')
            .then(response => response.json())
            .then(data => {
                if (Array.isArray(data) && data.length > 0) {
                    data.forEach(light => {
                        addLightToMap(light);
                    });
                }
                
                if (app.settings.autoRefresh) {
                    startStatusUpdates();
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error', 'Error loading traffic lights', 'error');
            });
    }

    function addLightToMap(light) {
        const status = getLightStatus(light);
        
        const el = document.createElement('div');
        el.className = 'w-8 h-8 bg-traffic-' + status.color + ' rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white';
        el.innerHTML = '<i class="fas fa-traffic-light"></i>';
        
        const marker = new mapboxgl.Marker(el)
            .setLngLat([light.longitude, light.latitude])
            .addTo(app.map);
        
        marker.getElement().addEventListener('click', () => showLightDetail(light));
        
        app.markers[light.id] = {
            marker: marker,
            data: light,
            element: el
        };
    }

    function getLightStatus(light) {
        const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
        const currentTime = Math.floor(Date.now() / 1000);
        const timeInCycle = currentTime % totalCycle;
        
        if (timeInCycle < light.red_duration) {
            return {
                isRed: true,
                color: 'red',
                label: 'RED',
                timeLeft: light.red_duration - timeInCycle,
                percentage: Math.round(((light.red_duration - timeInCycle) / totalCycle) * 100)
            };
        } else {
            return {
                isRed: false,
                color: 'green',
                label: 'GREEN',
                timeLeft: totalCycle - timeInCycle,
                percentage: Math.round(((totalCycle - timeInCycle) / totalCycle) * 100)
            };
        }
    }

    function updateLightStatus(light) {
        const status = getLightStatus(light);
        
        if (app.markers[light.id]) {
            const el = app.markers[light.id].element;
            if (el) {
                el.className = 'w-8 h-8 bg-traffic-' + status.color + ' rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white';
            }
        }
        
        if (app.selectedLightId === light.id && dom.infoPanel.classList.contains('active')) {
            dom.lightStatusText.textContent = status.label;
            dom.lightStatusTime.textContent = `${status.timeLeft}s until change`;
            dom.lightStatusBar.className = `bg-traffic-${status.color} h-2 rounded-full`;
            dom.lightStatusBar.style.width = `${status.percentage}%`;
        }
        
        if (app.measure.lightId === light.id && dom.modals.measure.style.display === 'flex') {
            dom.measure.status.innerHTML = `
                <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                    ${status.label} (${status.timeLeft}s)
                </span>
            `;
        }
    }

    function updateAllLightStatus() {
        for (const id in app.markers) {
            updateLightStatus(app.markers[id].data);
        }
    }

    function startStatusUpdates() {
        stopStatusUpdates();
        
        function updateStatuses(timestamp) {
            if (!app.lastTimestamp || timestamp - app.lastTimestamp > 1000) {
                app.lastTimestamp = timestamp;
                updateAllLightStatus();
                
                if (app.route.isNavigating && app.settings.autoMeasure) {
                    detectTrafficStops();
                }
            }
            app.rafId = requestAnimationFrame(updateStatuses);
        }
        
        app.rafId = requestAnimationFrame(updateStatuses);
    }

    function stopStatusUpdates() {
        if (app.rafId) {
            cancelAnimationFrame(app.rafId);
            app.rafId = null;
        }
    }

    function showLightDetail(light) {
        app.selectedLightId = light.id;
        
        app.map.flyTo({
            center: [light.longitude, light.latitude],
            zoom: 17
        });
        
        dom.infoPanelTitle.textContent = light.name;
        
        const status = getLightStatus(light);
        
        dom.lightStatusText.textContent = status.label;
        dom.lightStatusTime.textContent = `${status.timeLeft}s until change`;
        dom.lightStatusBar.className = `bg-traffic-${status.color} h-2 rounded-full`;
        dom.lightStatusBar.style.width = `${status.percentage}%`;
        
        dom.infoPanel.classList.add('active');
    }

    function startGeolocation() {
        if ('geolocation' in navigator) {
            navigator.geolocation.getCurrentPosition(
                position => {
                    const { latitude, longitude } = position.coords;
                    
                    if (!app.userMarker) {
                        const el = document.createElement('div');
                        el.className = 'user-marker pulse';
                        
                        app.userMarker = new mapboxgl.Marker(el)
                            .setLngLat([longitude, latitude])
                            .addTo(app.map);
                            
                        app.map.flyTo({
                            center: [longitude, latitude],
                            zoom: 15
                        });
                    } else {
                        app.userMarker.setLngLat([longitude, latitude]);
                    }
                },
                error => {
                    console.error('Geolocation error:', error);
                    showNotification('Error', 'Could not access your location', 'error');
                },
                { enableHighAccuracy: true }
            );
            
            app.userWatchId = navigator.geolocation.watchPosition(
                position => {
                    const { latitude, longitude } = position.coords;
                    
                    if (app.userMarker) {
                        app.userMarker.setLngLat([longitude, latitude]);
                    }
                    
                    if (app.route.isNavigating) {
                        updateNavigationProgress([longitude, latitude]);
                    }
                },
                error => {
                    console.error('Geolocation watch error:', error);
                },
                { enableHighAccuracy: true, maximumAge: 5000 }
            );
        }
    }

    function calculateRoute(start, end) {
        if (!start && app.userMarker) {
            const pos = app.userMarker.getLngLat();
            start = [pos.lng, pos.lat];
        }
        
        if (!start || !end) {
            showNotification('Error', 'Start and end locations are required', 'error');
            return;
        }
        
        if (app.directions) {
            app.directions.setOrigin(start);
            app.directions.setDestination(end);
            app.directions.setProfile(`mapbox/${app.route.mode}`);
            
            app.route.active = true;
            app.route.startCoords = start;
            app.route.endCoords = end;
            
            dom.route.panel.classList.add('active');
            dom.searchContainer.classList.add('route-active');
            
            app.directions.on('route', e => {
                if (e && e.route && e.route[0]) {
                    const route = e.route[0];
                    app.route.duration = route.duration;
                    app.route.distance = route.distance;
                    
                    const minutes = Math.round(route.duration / 60);
                    dom.route.estimatedDuration.textContent = `${minutes} min`;
                    
                    const now = new Date();
                    const arrivalTime = new Date(now.getTime() + route.duration * 1000);
                    dom.route.arrivalTime.textContent = `Arrival at ${arrivalTime.getHours().toString().padStart(2, '0')}:${arrivalTime.getMinutes().toString().padStart(2, '0')}`;
                    
                    findTrafficLightsOnRoute(route.geometry.coordinates);
                }
            });
        }
    }

    function findTrafficLightsOnRoute(coordinates) {
        app.route.lightsOnRoute = [];
        
        for (const id in app.markers) {
            const light = app.markers[id].data;
            const lightCoords = [parseFloat(light.longitude), parseFloat(light.latitude)];
            
            for (let i = 0; i < coordinates.length; i++) {
                const distance = calculateDistance(coordinates[i], lightCoords);
                
                if (distance < 30) {
                    app.route.lightsOnRoute.push({
                        id: id,
                        light: light,
                        routePointIndex: i,
                        distance: distance
                    });
                    break;
                }
            }
        }
    }

    function calculateDistance(point1, point2) {
        const [lon1, lat1] = point1;
        const [lon2, lat2] = point2;
        
        const R = 6371000;
        const φ1 = lat1 * Math.PI / 180;
        const φ2 = lat2 * Math.PI / 180;
        const Δφ = (lat2 - lat1) * Math.PI / 180;
        const Δλ = (lon2 - lon1) * Math.PI / 180;
        
        const a = Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
                  Math.cos(φ1) * Math.cos(φ2) *
                  Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        
        return R * c;
    }

    function updateRouteProfile(mode) {
        app.route.mode = mode;
        
        if (app.route.active && app.directions) {
            app.directions.setProfile(`mapbox/${mode}`);
        }
    }

    function cancelRoute() {
        app.route.active = false;
        app.route.isNavigating = false;
        app.route.lightsOnRoute = [];
        
        dom.route.panel.classList.remove('active');
        dom.searchContainer.classList.remove('route-active');
        
        if (app.directions) {
            app.directions.removeRoutes();
        }
    }

    function startNavigation() {
        if (!app.route.active) return;
        
        app.route.isNavigating = true;
        dom.route.startNavigationBtn.innerHTML = '<i class="fas fa-stop"></i>';
        
        showNotification('Navigation', 'Turn-by-turn navigation started', 'info');
    }

    function updateNavigationProgress(currentPosition) {
        if (!app.route.isNavigating) return;
        
        const routeCoordinates = app.directions.getWaypoints()[0].feature.geometry.coordinates;
        let closestPointIndex = 0;
        let minDistance = Infinity;
        
        for (let i = 0; i < routeCoordinates.length; i++) {
            const distance = calculateDistance(currentPosition, routeCoordinates[i]);
            if (distance < minDistance) {
                minDistance = distance;
                closestPointIndex = i;
            }
        }
        
        const remainingCoordinates = routeCoordinates.slice(closestPointIndex);
        const remainingDistance = calculatePathDistance(remainingCoordinates);
        const remainingTime = (remainingDistance / app.route.distance) * app.route.duration;
        
        const minutes = Math.round(remainingTime / 60);
        dom.route.estimatedDuration.textContent = `${minutes} min`;
        
        const now = new Date();
        const arrivalTime = new Date(now.getTime() + remainingTime * 1000);
        dom.route.arrivalTime.textContent = `Arrival at ${arrivalTime.getHours().toString().padStart(2, '0')}:${arrivalTime.getMinutes().toString().padStart(2, '0')}`;
        
        updateNextTrafficLightInfo(closestPointIndex);
    }

    function calculatePathDistance(coordinates) {
        let totalDistance = 0;
        
        for (let i = 0; i < coordinates.length - 1; i++) {
            totalDistance += calculateDistance(coordinates[i], coordinates[i + 1]);
        }
        
        return totalDistance;
    }

    function updateNextTrafficLightInfo(currentRouteIndex) {
        const upcomingLights = app.route.lightsOnRoute.filter(item => item.routePointIndex > currentRouteIndex)
            .sort((a, b) => a.routePointIndex - b.routePointIndex);
        
        if (upcomingLights.length > 0) {
            const nextLight = upcomingLights[0];
            const lightStatus = getLightStatus(nextLight.light);
            
            const routeCoordinates = app.directions.getWaypoints()[0].feature.geometry.coordinates;
            const distanceToLight = calculatePathDistance(routeCoordinates.slice(currentRouteIndex, nextLight.routePointIndex));
            
            const kmh = app.route.mode === 'driving' ? 50 : app.route.mode === 'cycling' ? 15 : 5;
            const timeToLight = (distanceToLight / 1000) / kmh * 3600;
            
            if (timeToLight < 60) {
                const lightsMessage = `Next light: ${nextLight.light.name} (${Math.round(distanceToLight)}m) - ${lightStatus.label}`;
                const timeMessage = lightStatus.isRed ? 
                    `Red for ${lightStatus.timeLeft}s` : 
                    `Green for ${lightStatus.timeLeft}s`;
                
                showNotification('Traffic Light', `${lightsMessage} - ${timeMessage}`, 'info');
            }
        }
    }

    function detectTrafficStops() {
        if (!app.userMarker || !app.settings.autoMeasure) return;
        
        const now = Date.now();
        const userPosition = app.userMarker.getLngLat();
        
        if (!app.lastPosition) {
            app.lastPosition = userPosition;
            app.lastMovementTime = now;
            return;
        }
        
        const distance = calculateDistance(
            [app.lastPosition.lng, app.lastPosition.lat],
            [userPosition.lng, userPosition.lat]
        );
        
        if (distance < 2) {
            if (!app.stoppedAt) {
                app.stoppedAt = now;
            } else if (now - app.stoppedAt > 2000 && !app.measuringStop) {
                app.measuringStop = true;
                checkForTrafficLightAtStop(userPosition);
            }
        } else {
            if (app.measuringStop && app.measuringLightId) {
                const stopDuration = now - app.stoppedAt;
                
                if (stopDuration > 5000 && stopDuration < 120000) {
                    autoSaveTrafficLightMeasurement(app.measuringLightId, stopDuration / 1000);
                }
                
                app.measuringLightId = null;
            }
            
            app.lastPosition = userPosition;
            app.stoppedAt = null;
            app.measuringStop = false;
        }
    }

    function checkForTrafficLightAtStop(position) {
        let nearestLightId = null;
        let minDistance = Infinity;
        
        for (const id in app.markers) {
            const light = app.markers[id].data;
            const lightPosition = [parseFloat(light.longitude), parseFloat(light.latitude)];
            const distance = calculateDistance([position.lng, position.lat], lightPosition);
            
            if (distance < 50 && distance < minDistance) {
                minDistance = distance;
                nearestLightId = id;
            }
        }
        
        if (nearestLightId) {
            app.measuringLightId = nearestLightId;
            const light = app.markers[nearestLightId].data;
            const status = getLightStatus(light);
            
            if (status.isRed) {
                showNotification('Auto Measurement', `Measuring red light time for "${light.name}"`, 'info');
            }
        }
    }

    function autoSaveTrafficLightMeasurement(lightId, duration) {
        const light = app.markers[lightId].data;
        const status = getLightStatus(light);
        
        if (!status.isRed) return;
        
        fetch('api/update-traffic-light-timing.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                id: lightId,
                duration_type: 'red',
                duration: Math.round(duration),
                auto_measured: true,
                user_id: app.user.isLoggedIn ? localStorage.getItem('userId') || sessionStorage.getItem('userId') : null
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                app.markers[lightId].data = data.light;
                showNotification('Auto Measurement', `Red light time for "${light.name}" updated (${Math.round(duration)}s)`, 'success');
            }
        })
        .catch(error => {
            console.error('Error saving auto measurement:', error);
        });
    }

    function startMeasure(mode) {
        app.measure.mode = mode;
        dom.measure.container.classList.remove('hidden');
        
        if (mode === 'red') {
            dom.measure.instructions.textContent = 'Press "Start" when the light turns red, then "Stop" when it turns green.';
            dom.measure.startBtn.className = 'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors';
            dom.measure.stopBtn.className = 'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors opacity-50';
        } else {
            dom.measure.instructions.textContent = 'Press "Start" when the light turns green, then "Stop" when it turns red.';
            dom.measure.startBtn.className = 'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors';
            dom.measure.stopBtn.className = 'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors opacity-50';
        }
        
        resetMeasureTimer();
    }

    function startMeasureTimer() {
        app.measure.startTime = performance.now();
        app.measure.timer = 0;
        
        if (app.measure.rafId) cancelAnimationFrame(app.measure.rafId);
        
        function updateTimer(timestamp) {
            const elapsed = Math.floor((timestamp - app.measure.startTime) / 1000);
            if (elapsed !== app.measure.timer) {
                app.measure.timer = elapsed;
                const minutes = Math.floor(elapsed / 60);
                const seconds = elapsed % 60;
                dom.measure.display.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            }
            app.measure.rafId = requestAnimationFrame(updateTimer);
        }
        
        app.measure.rafId = requestAnimationFrame(updateTimer);
        dom.measure.display.textContent = '00:00';
        dom.measure.result.textContent = 'Measuring...';
        
        dom.measure.startBtn.disabled = true;
        dom.measure.startBtn.classList.add('opacity-50');
        dom.measure.stopBtn.disabled = false;
        dom.measure.stopBtn.classList.remove('opacity-50');
        dom.measure.saveBtn.disabled = true;
        dom.measure.saveBtn.classList.add('opacity-50');
    }

    function stopMeasureTimer() {
        if (app.measure.rafId) {
            cancelAnimationFrame(app.measure.rafId);
            app.measure.rafId = null;
            
            dom.measure.result.textContent = `Measured duration: ${app.measure.timer} seconds. Click Save to confirm.`;
            
            dom.measure.stopBtn.disabled = true;
            dom.measure.stopBtn.classList.add('opacity-50');
            dom.measure.saveBtn.disabled = false;
            dom.measure.saveBtn.classList.remove('opacity-50');
        }
    }

    function resetMeasureTimer() {
        if (app.measure.rafId) {
            cancelAnimationFrame(app.measure.rafId);
            app.measure.rafId = null;
        }
        app.measure.timer = 0;
        app.measure.startTime = 0;
        dom.measure.display.textContent = '00:00';
        dom.measure.result.textContent = '';
        dom.measure.startBtn.disabled = false;
        dom.measure.startBtn.classList.remove('opacity-50');
        dom.measure.stopBtn.disabled = true;
        dom.measure.stopBtn.classList.add('opacity-50');
        dom.measure.saveBtn.disabled = true;
        dom.measure.saveBtn.classList.add('opacity-50');
    }

    function saveMeasureTimer() {
        const lightId = dom.measure.title.getAttribute('data-id');
        
        if (!lightId || app.measure.timer <= 0 || !app.measure.mode) return;
        
        dom.measure.result.textContent = 'Saving...';
        
        fetch('api/update-traffic-light-timing.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                id: lightId,
                duration_type: app.measure.mode,
                duration: app.measure.timer,
                user_id: app.user.isLoggedIn ? localStorage.getItem('userId') || sessionStorage.getItem('userId') : null
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                dom.measure.result.innerHTML = '<i class="fas fa-check text-green-500 mr-1"></i> Timing updated!';
                
                app.markers[lightId].data = data.light;
                updateLightStatus(data.light);
                
                dom.measure.saveBtn.disabled = true;
                dom.measure.saveBtn.classList.add('opacity-50');
                
                setTimeout(() => {
                    dom.measure.container.classList.add('hidden');
                    app.measure.mode = null;
                }, 1500);
            } else {
                dom.measure.result.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Error: ' + data.message;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            dom.measure.result.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Connection error';
        });
    }

    function openMeasureModal(lightId) {
        const light = app.markers[lightId].data;
        app.measure.lightId = lightId;
        
        dom.measure.title.textContent = light.name;
        dom.measure.title.setAttribute('data-id', lightId);
        
        const status = getLightStatus(light);
        dom.measure.status.innerHTML = `
            <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                ${status.label} (${status.timeLeft}s)
            </span>
        `;
        
        dom.measure.container.classList.add('hidden');
        app.measure.mode = null;
        resetMeasureTimer();
        
        openModal(dom.modals.measure);
    }

    function toggleDarkMode() {
        app.settings.darkMode = dom.settings.darkMode.checked;
        
        if (app.settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
            
            if (app.map && app.settings.mapStyle.includes('day')) {
                const nightStyle = app.settings.mapStyle.replace('day', 'night');
                app.settings.mapStyle = nightStyle;
                dom.settings.mapStyle.value = nightStyle;
                changeMapStyle();
            }
        } else {
            document.documentElement.classList.remove('dark');
            document.body.classList.remove('dark-mode');
            
            if (app.map && app.settings.mapStyle.includes('night')) {
                const dayStyle = app.settings.mapStyle.replace('night', 'day');
                app.settings.mapStyle = dayStyle;
                dom.settings.mapStyle.value = dayStyle;
                changeMapStyle();
            }
        }
        
        saveSettings();
    }

    function changeMapStyle() {
        app.settings.mapStyle = dom.settings.mapStyle.value;
        
        if (app.map) {
            app.map.setStyle(`mapbox://styles/mapbox/${app.settings.mapStyle}`);
        }
        
        saveSettings();
    }

    function toggleAutoRefresh() {
        app.settings.autoRefresh = dom.settings.autoRefresh.checked;
        
        if (app.settings.autoRefresh) {
            startStatusUpdates();
        } else {
            stopStatusUpdates();
        }
        
        saveSettings();
    }

    function saveSettings() {
        app.settings.darkMode = dom.settings.darkMode.checked;
        app.settings.notifications = dom.settings.notifications.checked;
        app.settings.autoRefresh = dom.settings.autoRefresh.checked;
        app.settings.autoMeasure = dom.settings.autoMeasure.checked;
        app.settings.mapStyle = dom.settings.mapStyle.value;
        app.settings.defaultTransport = dom.settings.defaultTransport.value;
        
        try {
            localStorage.setItem('trafficLightSettings', JSON.stringify(app.settings));
        } catch (e) {
            console.error('Error saving settings');
        }
    }

    function resetAppData() {
        if (confirm('Are you sure you want to reset all app data? This will clear all your settings and logout.')) {
            localStorage.removeItem('trafficLightSettings');
            localStorage.removeItem('trafficLightOnboardingCompleted');
            localStorage.removeItem('userId');
            sessionStorage.removeItem('userId');
            
            showNotification('Reset', 'App data reset. Refreshing...', 'info');
            
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        }
    }

    function showNotification(title, message, type) {
        if (!app.settings.notifications && type !== 'error') return;
        
        dom.notification.title.textContent = title;
        dom.notification.message.textContent = message;
        
        let bgColor, iconClass;
        
        switch (type) {
            case 'success':
                bgColor = 'bg-traffic-green';
                iconClass = 'fas fa-check';
                break;
            case 'error':
                bgColor = 'bg-traffic-red';
                iconClass = 'fas fa-exclamation-circle';
                break;
            default:
                bgColor = 'bg-primary-500';
                iconClass = 'fas fa-info-circle';
        }
        
        dom.notification.icon.className = `w-8 h-8 ${bgColor} text-white rounded-full flex items-center justify-center mr-3`;
        dom.notification.icon.innerHTML = `<i class="${iconClass}"></i>`;
        
        dom.notification.banner.style.display = 'block';
        setTimeout(() => {
            dom.notification.banner.style.opacity = '1';
        }, 10);
        
        setTimeout(() => {
            dom.notification.banner.style.opacity = '0';
            setTimeout(() => {
                dom.notification.banner.style.display = 'none';
            }, 300);
        }, 4000);
    }

    function showAccountForm() {
        dom.account.forms.account.classList.remove('hidden');
        dom.account.forms.login.classList.add('hidden');
        dom.account.forms.signup.classList.add('hidden');
        
        if (app.user.isLoggedIn) {
            document.getElementById('firstName').value = app.user.firstName;
            document.getElementById('lastName').value = app.user.lastName;
            document.getElementById('email').value = app.user.email;
            document.getElementById('preferredVehicle').value = app.user.preferredVehicle;
        }
    }

    function showLoginForm() {
        dom.account.forms.account.classList.add('hidden');
        dom.account.forms.login.classList.remove('hidden');
        dom.account.forms.signup.classList.add('hidden');
    }

    function showSignupForm() {
        dom.account.forms.account.classList.add('hidden');
        dom.account.forms.login.classList.add('hidden');
        dom.account.forms.signup.classList.remove('hidden');
    }

    function showOnboardingModal() {
        openModal(dom.modals.onboarding);
    }

    function loginUser() {
        const email = document.getElementById('loginEmail').value;
        const password = document.getElementById('loginPassword').value;
        const rememberMe = document.getElementById('rememberMe').checked;
        
        fetch('api/login.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({ email, password })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                app.user.isLoggedIn = true;
                app.user.firstName = data.user.first_name;
                app.user.lastName = data.user.last_name;
                app.user.email = data.user.email;
                app.user.preferredVehicle = data.user.preferred_vehicle;
                
                if (rememberMe) {
                    localStorage.setItem('userId', data.user.id);
                } else {
                    sessionStorage.setItem('userId', data.user.id);
                }
                
                updateAccountDisplay();
                closeModal(dom.modals.account);
                showNotification('Success', 'You have been logged in successfully', 'success');
            } else {
                showNotification('Error', data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Login error:', error);
            showNotification('Error', 'Connection error', 'error');
        });
    }

    function signupUser() {
        createAccount(false);
    }

    function createAccount(isOnboarding = false) {
        const formPrefix = isOnboarding ? 'onboarding' : 'signup';
        
        const firstName = document.getElementById(`${formPrefix}FirstName`).value;
        const lastName = document.getElementById(`${formPrefix}LastName`).value;
        const email = document.getElementById(`${formPrefix}Email`).value;
        const password = document.getElementById(`${formPrefix}Password`).value;
        const preferredVehicle = document.getElementById(`${formPrefix}Vehicle`).value;
        
        fetch('api/signup.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({ firstName, lastName, email, password, preferredVehicle })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                app.user.isLoggedIn = true;
                app.user.firstName = firstName;
                app.user.lastName = lastName;
                app.user.email = email;
                app.user.preferredVehicle = preferredVehicle;
                
                localStorage.setItem('userId', data.user.id);
                
                if (isOnboarding) {
                    localStorage.setItem('trafficLightOnboardingCompleted', 'true');
                    closeModal(dom.modals.onboarding);
                } else {
                    closeModal(dom.modals.account);
                }
                
                app.settings.defaultTransport = preferredVehicle;
                saveSettings();
                updateAccountDisplay();
                
                showNotification('Success', 'Account created successfully', 'success');
            } else {
                showNotification('Error', data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Signup error:', error);
            showNotification('Error', 'Connection error', 'error');
        });
    }

    function logoutUser() {
        app.user.isLoggedIn = false;
        app.user.firstName = '';
        app.user.lastName = '';
        app.user.email = '';
        
        localStorage.removeItem('userId');
        sessionStorage.removeItem('userId');
        
        updateAccountDisplay();
        closeModal(dom.modals.account);
        
        showNotification('Logout', 'You have been logged out', 'info');
    }

    function handleOnline() {
        showNotification('Connection', 'You are back online', 'success');
        loadTrafficLights();
    }

    function handleOffline() {
        showNotification('Connection', 'You are offline. Some features may be limited', 'error');
    }

    function handleVisibilityChange() {
        if (document.visibilityState === 'visible') {
            updateAllLightStatus();
        }
    }
});