<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);
$error = null;
$success = null;

$travel_id = isset($_GET['id']) ? $_GET['id'] : null;

if ($travel_id) {
    $travel = db_get_one(
        "SELECT t.*, 
            sc.name as start_country_name, 
            ec.name as end_country_name,
            sc.code as start_country_code,
            ec.code as end_country_code
         FROM travels t
         JOIN countries sc ON t.start_country = sc.id
         JOIN countries ec ON t.end_country = ec.id
         WHERE t.id = ? AND t.user_id = ?",
        [$travel_id, $_SESSION['user_id']]
    );
} else {
    $travel = db_get_one(
        "SELECT t.*, 
            sc.name as start_country_name, 
            ec.name as end_country_name,
            sc.code as start_country_code,
            ec.code as end_country_code
         FROM travels t
         JOIN countries sc ON t.start_country = sc.id
         JOIN countries ec ON t.end_country = ec.id
         WHERE t.user_id = ? AND t.status = 'in_progress'
         ORDER BY t.start_time DESC
         LIMIT 1",
        [$_SESSION['user_id']]
    );
}

if (!$travel) {
    header('Location: map.php');
    exit;
}

$now = new DateTime();
$arrival_time = new DateTime($travel['arrival_time']);
$start_time = new DateTime($travel['start_time']);
$total_duration = $start_time->diff($arrival_time);
$remaining_duration = $now->diff($arrival_time);

$is_arrived = $now >= $arrival_time;

if ($is_arrived && $travel['status'] === 'in_progress') {
    // Mettre à jour le statut du voyage
    db_update(
        'travels',
        ['status' => 'completed'],
        'id = ?',
        [$travel['id']]
    );
    
    // Mettre à jour la position de l'utilisateur
    list($end_lat, $end_lng) = explode(',', $travel['end_coordinates']);
    update_user_position($_SESSION['user_id'], $end_lat, $end_lng);
    
    // Ajouter de l'expérience
    add_experience($_SESSION['user_id'], 25);
    
    $success = "Voyage terminé! Tu es arrivé à destination. +25 XP";
    $travel['status'] = 'completed';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'cancel_travel') {
    if ($travel['status'] !== 'in_progress') {
        $error = "Ce voyage ne peut pas être annulé.";
    } else {
        db_update(
            'travels',
            ['status' => 'cancelled'],
            'id = ?',
            [$travel['id']]
        );
        
        $success = "Voyage annulé.";
        $travel['status'] = 'cancelled';
    }
}

// Calculer le pourcentage de progression
$total_seconds = $start_time->getTimestamp() - $arrival_time->getTimestamp();
$elapsed_seconds = $start_time->getTimestamp() - $now->getTimestamp();
$progress_percent = 0;

if ($total_seconds != 0) {
    $progress_percent = min(100, max(0, ($elapsed_seconds / $total_seconds) * 100));
}

if ($is_arrived) {
    $progress_percent = 100;
}

list($start_lat, $start_lng) = explode(',', $travel['start_coordinates']);
list($end_lat, $end_lng) = explode(',', $travel['end_coordinates']);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Voyage en cours | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.7.1/dist/leaflet.css" />
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
        #map {
            height: 400px;
            width: 100%;
            z-index: 1;
        }
        .progress-bar {
            height: 8px;
            border-radius: 4px;
            background-color: #E5E7EB;
            overflow: hidden;
        }
        .progress-value {
            height: 8px;
            border-radius: 4px;
            background-color: #3B82F6;
            transition: width 1s ease;
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <?php if ($user['level'] >= 10): ?>
                        <li><a href="unions.php" class="hover:text-blue-300">Unions</a></li>
                    <?php endif; ?>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Voyage en cours</h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div class="lg:col-span-2">
                <div class="card mb-6">
                    <div id="map"></div>
                </div>
                
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Détails du voyage</h3>
                    
                    <div class="space-y-4">
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <p class="font-medium">Départ</p>
                                <div class="flex items-center mt-1">
                                    <img src="https://flagcdn.com/24x18/<?php echo strtolower($travel['start_country_code']); ?>.png" alt="<?php echo htmlspecialchars($travel['start_country_name']); ?>" class="mr-2" width="24" height="18">
                                    <span><?php echo htmlspecialchars($travel['start_country_name']); ?></span>
                                </div>
                                <p class="text-sm text-gray-500 mt-1">
                                    <?php echo format_datetime($travel['start_time']); ?>
                                </p>
                            </div>
                            
                            <div>
                                <p class="font-medium">Destination</p>
                                <div class="flex items-center mt-1">
                                    <img src="https://flagcdn.com/24x18/<?php echo strtolower($travel['end_country_code']); ?>.png" alt="<?php echo htmlspecialchars($travel['end_country_name']); ?>" class="mr-2" width="24" height="18">
                                    <span><?php echo htmlspecialchars($travel['end_country_name']); ?></span>
                                </div>
                                <p class="text-sm text-gray-500 mt-1">
                                    <?php echo format_datetime($travel['arrival_time']); ?>
                                </p>
                            </div>
                        </div>
                        
                        <div>
                            <p class="font-medium">Mode de transport</p>
                            <p class="mt-1">
                                <?php
                                    $transport_types = [
                                        'car' => 'Voiture',
                                        'train' => 'Train',
                                        'plane' => 'Avion',
                                        'boat' => 'Bateau'
                                    ];
                                    echo $transport_types[$travel['transport_type']] ?? $travel['transport_type'];
                                ?>
                            </p>
                        </div>
                        
                        <div>
                            <p class="font-medium">Statut</p>
                            <p class="mt-1">
                                <?php
                                    $status_text = [
                                        'in_progress' => 'En cours',
                                        'completed' => 'Terminé',
                                        'cancelled' => 'Annulé'
                                    ];
                                    $status_colors = [
                                        'in_progress' => 'text-blue-600',
                                        'completed' => 'text-green-600',
                                        'cancelled' => 'text-red-600'
                                    ];
                                    $status = $status_text[$travel['status']] ?? $travel['status'];
                                    $color = $status_colors[$travel['status']] ?? '';
                                ?>
                                <span class="<?php echo $color; ?> font-medium"><?php echo $status; ?></span>
                            </p>
                        </div>
                        
                        <?php if ($travel['status'] === 'in_progress'): ?>
                            <div>
                                <p class="font-medium">Progression</p>
                                <div class="progress-bar mt-2">
                                    <div class="progress-value" style="width: <?php echo $progress_percent; ?>%"></div>
                                </div>
                                <div class="flex justify-between text-sm mt-1">
                                    <span><?php echo round($progress_percent); ?>%</span>
                                    <?php if (!$is_arrived): ?>
                                        <span id="countdown"></span>
                                    <?php else: ?>
                                        <span>Arrivé</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <?php if (!$is_arrived): ?>
                                <div>
                                    <form method="POST" action="travel.php?id=<?php echo $travel['id']; ?>" onsubmit="return confirm('Es-tu sûr de vouloir annuler ce voyage?')">
                                        <input type="hidden" name="action" value="cancel_travel">
                                        <button type="submit" class="bg-red-600 hover:bg-red-700 text-white py-2 px-4 rounded transition-colors">
                                            Annuler le voyage
                                        </button>
                                    </form>
                                </div>
                            <?php else: ?>
                                <div>
                                    <a href="map.php" class="inline-block bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded transition-colors">
                                        Retour à la carte
                                    </a>
                                </div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div>
                                <a href="map.php" class="inline-block bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded transition-colors">
                                    Retour à la carte