index.php
<?php
$trains = json_decode(file_get_contents('stock.json'), true) ?: [];
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Trains | Départs</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-black text-white">
    <div class="container mx-auto p-4">
        <header class="flex justify-between items-center mb-8">
            <div>
                <h1 class="text-2xl font-light tracking-wider">Départs</h1>
                <p class="text-gray-500 text-sm mt-1" id="last-update"></p>
            </div>
            <div class="text-right">
                <div id="time" class="text-4xl font-light"></div>
                <div id="date" class="text-gray-500 text-sm mt-1"></div>
            </div>
        </header>

        <div id="trains" class="space-y-3"></div>
    </div>

    <div id="announcement" class="fixed bottom-0 left-0 w-full transform translate-y-full transition-transform duration-500">
        <div class="bg-blue-600/10 backdrop-blur-sm">
            <div class="container mx-auto py-6 px-4">
                <p id="announcement-text" class="text-xl font-light"></p>
            </div>
        </div>
    </div>

    <audio id="jingle" preload="auto">
        <source src="sounds/jingle.mp3" type="audio/mpeg">
    </audio>
    <audio id="bell" preload="auto">
        <source src="sounds/bell.mp3" type="audio/mpeg">
    </audio>

    <script>
    // Gestion du temps
    function updateDateTime() {
        const now = new Date();
        document.getElementById('time').textContent = now.toLocaleTimeString('fr-FR', {
            hour: '2-digit',
            minute: '2-digit'
        });
        document.getElementById('date').textContent = now.toLocaleDateString('fr-FR', {
            weekday: 'long',
            day: 'numeric',
            month: 'long'
        });
    }

    // Gestion de la voix
    let voice = null;
    const jingle = document.getElementById('jingle');
    const bell = document.getElementById('bell');
    let isAnnouncing = false;
    let announcementQueue = [];

    window.speechSynthesis.onvoiceschanged = () => {
        const voices = speechSynthesis.getVoices();
        voice = voices.find(v => v.lang === 'fr-FR') || voices[0];
    };

    async function announce(text, useJingle = true) {
        if (isAnnouncing) {
            announcementQueue.push({text, useJingle});
            return;
        }

        isAnnouncing = true;
        const announcement = document.getElementById('announcement');
        const announcementText = document.getElementById('announcement-text');

        if (useJingle) {
            jingle.currentTime = 0;
            try {
                await jingle.play();
                await new Promise(resolve => setTimeout(resolve, 500)); // Attente après le jingle
            } catch (e) {
                console.error('Erreur lors de la lecture du jingle:', e);
            }
        } else {
            bell.currentTime = 0;
            try {
                await bell.play();
            } catch (e) {
                console.error('Erreur lors de la lecture de la cloche:', e);
            }
        }

        announcementText.textContent = text;
        announcement.classList.remove('translate-y-full');

        const utterance = new SpeechSynthesisUtterance(text);
        utterance.voice = voice;
        utterance.rate = 0.85;
        utterance.pitch = 0.95;
        
        utterance.onend = () => {
            setTimeout(() => {
                announcement.classList.add('translate-y-full');
                setTimeout(() => {
                    isAnnouncing = false;
                    if (announcementQueue.length > 0) {
                        const next = announcementQueue.shift();
                        announce(next.text, next.useJingle);
                    }
                }, 500);
            }, 1000);
        };

        speechSynthesis.speak(utterance);
    }

    // Gestion des trains
    function updateTrains() {
        fetch('stock.json?' + new Date().getTime())
        .then(response => response.json())
        .then(data => {
            const trainsDiv = document.getElementById('trains');
            const now = new Date();
            document.getElementById('last-update').textContent = 
                'Dernière mise à jour: ' + now.toLocaleTimeString('fr-FR');

            trainsDiv.innerHTML = '';
            
            data.forEach(train => {
                const arrival = new Date(train.arrival);
                const timeLeft = (arrival - now) / 1000; // en secondes

                // Ne pas afficher les trains partis depuis plus de 20 secondes
                if (timeLeft < -20) return;

                // Annonces
                if (timeLeft <= 30 && timeLeft > 29) {
                    announce(`Votre attention s'il vous plaît. Le train de la ligne ${train.line} à destination de ${train.station} arrivera voie ${train.platform || 'à confirmer'} dans quelques instants.`, true);
                }
                if (timeLeft <= 0 && timeLeft > -1) {
                    announce(`Le train de la ligne ${train.line} à destination de ${train.station} entre en gare voie ${train.platform || 'à confirmer'}.`, false);
                }

                // Formater le temps restant
                let timeDisplay;
                if (timeLeft < 0) {
                    timeDisplay = "À quai";
                } else if (timeLeft < 60) {
                    timeDisplay = Math.ceil(timeLeft) + "s";
                } else if (timeLeft < 3600) {
                    timeDisplay = Math.ceil(timeLeft / 60) + "min";
                } else {
                    const hours = Math.floor(timeLeft / 3600);
                    const mins = Math.ceil((timeLeft % 3600) / 60);
                    timeDisplay = `${hours}h${mins.toString().padStart(2, '0')}`;
                }

                const div = document.createElement('div');
                div.className = `bg-gray-900 p-4 rounded-lg flex items-center justify-between transition-all
                    ${timeLeft <= 30 ? 'border-l-4 border-yellow-500' : ''}
                    ${timeLeft <= 0 ? 'border-l-4 border-green-500' : ''}
                    ${timeLeft < -10 ? 'opacity-50' : ''}`;

                div.innerHTML = `
                    <div class="flex items-center gap-8">
                        <div class="w-24">
                            <div class="text-2xl font-light">${arrival.toLocaleTimeString('fr-FR', {hour: '2-digit', minute: '2-digit'})}</div>
                            <div class="text-sm text-gray-500">${timeDisplay}</div>
                        </div>
                        <div class="flex items-center gap-4">
                            <div class="px-3 py-1 bg-blue-900 rounded text-sm font-light">${train.line}</div>
                            <div>
                                <div class="font-light">${train.station}</div>
                                <div class="text-sm text-gray-500">Voie ${train.platform || '-'}</div>
                            </div>
                        </div>
                    </div>
                    <div class="flex items-center gap-2">
                        <div class="w-2 h-2 rounded-full ${train.status === 'on-time' ? 'bg-green-500' : 'bg-yellow-500'}"></div>
                        <span class="text-sm text-gray-500">${train.status === 'on-time' ? 'À l\'heure' : 'Retardé'}</span>
                    </div>
                `;
                
                trainsDiv.appendChild(div);
            });
        });
    }

    // Initialisation
    updateDateTime();
    updateTrains();
    setInterval(updateDateTime, 1000);
    setInterval(updateTrains, 1000);
    </script>
</body>
</html>