<?php
require_once 'config.php';

if(!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Update profile
if(isset($_POST['update_profile'])) {
    $username = $_POST['username'];
    $email = $_POST['email'];
    $newPassword = $_POST['new_password'];
    
    try {
        if($newPassword) {
            $password = password_hash($newPassword, PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("UPDATE users SET username = ?, email = ?, password = ? WHERE id = ?");
            $stmt->execute([$username, $email, $password, $_SESSION['user_id']]);
        } else {
            $stmt = $pdo->prepare("UPDATE users SET username = ?, email = ? WHERE id = ?");
            $stmt->execute([$username, $email, $_SESSION['user_id']]);
        }
        $_SESSION['username'] = $username;
        $success = "Profile updated successfully!";
    } catch(PDOException $e) {
        $error = "Username or email already exists";
    }
}

// Delete video
if(isset($_POST['delete_video'])) {
    $video_id = $_POST['video_id'];
    $stmt = $pdo->prepare("DELETE FROM content WHERE id = ? AND author_id = ?");
    $stmt->execute([$video_id, $_SESSION['user_id']]);
}

// Get user info
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$_SESSION['user_id']]);
$user = $stmt->fetch();

// Get user videos with stats
$stmt = $pdo->prepare("
    SELECT 
        c.*,
        COUNT(DISTINCT CASE WHEN i.type = 'view' THEN i.id END) as views,
        COUNT(DISTINCT CASE WHEN i.type = 'like' THEN i.id END) as likes,
        COUNT(DISTINCT CASE WHEN i.type = 'dislike' THEN i.id END) as dislikes,
        COUNT(DISTINCT CASE WHEN i.type = 'comment' THEN i.id END) as comments
    FROM content c
    LEFT JOIN interactions i ON c.id = i.content_id
    WHERE c.author_id = ?
    GROUP BY c.id
    ORDER BY c.created_at DESC
");
$stmt->execute([$_SESSION['user_id']]);
$videos = $stmt->fetchAll();

// Calculate total stats
$totalViews = $totalLikes = $totalDislikes = $totalComments = 0;
foreach($videos as $video) {
    $totalViews += $video['views'];
    $totalLikes += $video['likes'];
    $totalDislikes += $video['dislikes'];
    $totalComments += $video['comments'];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile - ForMore</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white p-8">
    <nav class="flex justify-between items-center mb-8">
        <h1 class="text-3xl font-bold">ForMore</h1>
        <div class="flex gap-4">
            <a href="index.php" class="text-blue-400 hover:text-blue-300">Home</a>
            <a href="upload.php" class="text-green-400 hover:text-green-300">Upload Video</a>
        </div>
    </nav>

    <div class="max-w-6xl mx-auto grid grid-cols-1 lg:grid-cols-3 gap-8">
        <!-- Profile Section -->
        <div class="lg:col-span-1">
            <div class="bg-gray-800 rounded-lg p-6">
                <h2 class="text-2xl font-bold mb-6">Profile Settings</h2>
                
                <?php if(isset($success)): ?>
                    <div class="bg-green-500 text-white p-3 rounded mb-4"><?= $success ?></div>
                <?php endif; ?>
                <?php if(isset($error)): ?>
                    <div class="bg-red-500 text-white p-3 rounded mb-4"><?= $error ?></div>
                <?php endif; ?>

                <form method="POST">
                    <div class="mb-4">
                        <label class="block text-gray-300 text-sm font-bold mb-2">Username</label>
                        <input type="text" name="username" value="<?= htmlspecialchars($user['username']) ?>" required 
                            class="shadow appearance-none border rounded w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:shadow-outline">
                    </div>

                    <div class="mb-4">
                        <label class="block text-gray-300 text-sm font-bold mb-2">Email</label>
                        <input type="email" name="email" value="<?= htmlspecialchars($user['email']) ?>" required 
                            class="shadow appearance-none border rounded w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:shadow-outline">
                    </div>

                    <div class="mb-6">
                        <label class="block text-gray-300 text-sm font-bold mb-2">New Password (leave empty to keep current)</label>
                        <input type="password" name="new_password" 
                            class="shadow appearance-none border rounded w-full py-2 px-3 text-gray-700 leading-tight focus:outline-none focus:shadow-outline">
                    </div>

                    <button type="submit" name="update_profile" 
                        class="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded focus:outline-none focus:shadow-outline w-full">
                        Update Profile
                    </button>
                </form>

                <!-- Total Stats -->
                <div class="mt-8">
                    <h3 class="text-xl font-bold mb-4">Channel Statistics</h3>
                    <div class="grid grid-cols-2 gap-4">
                        <div class="bg-gray-700 p-4 rounded">
                            <div class="text-2xl font-bold"><?= number_format($totalViews) ?></div>
                            <div class="text-gray-400">Total Views</div>
                        </div>
                        <div class="bg-gray-700 p-4 rounded">
                            <div class="text-2xl font-bold"><?= number_format($totalLikes) ?></div>
                            <div class="text-gray-400">Total Likes</div>
                        </div>
                        <div class="bg-gray-700 p-4 rounded">
                            <div class="text-2xl font-bold"><?= number_format($totalComments) ?></div>
                            <div class="text-gray-400">Total Comments</div>
                        </div>
                        <div class="bg-gray-700 p-4 rounded">
                            <div class="text-2xl font-bold"><?= count($videos) ?></div>
                            <div class="text-gray-400">Total Videos</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Videos Section -->
        <div class="lg:col-span-2">
            <div class="bg-gray-800 rounded-lg p-6">
                <h2 class="text-2xl font-bold mb-6">My Videos</h2>
                
                <div class="space-y-6">
                    <?php foreach($videos as $video): ?>
                        <div class="bg-gray-700 rounded-lg p-4">
                            <div class="flex items-start gap-4">
                                <img src="<?= htmlspecialchars($video['picture_cover']) ?>" 
                                    class="w-48 h-32 object-cover rounded" alt="Video thumbnail">
                                
                                <div class="flex-grow">
                                    <h3 class="text-xl font-bold"><?= htmlspecialchars($video['name']) ?></h3>
                                    <p class="text-gray-400 mt-2"><?= substr(htmlspecialchars($video['description']), 0, 100) ?>...</p>
                                    
                                    <div class="flex gap-4 mt-4 text-sm">
                                        <span><?= number_format($video['views']) ?> views</span>
                                        <span><?= number_format($video['likes']) ?> likes</span>
                                        <span><?= number_format($video['comments']) ?> comments</span>
                                    </div>
                                    
                                    <div class="flex gap-2 mt-4">
                                        <a href="edit_video.php?id=<?= $video['id'] ?>" 
                                            class="bg-blue-500 hover:bg-blue-700 text-white px-4 py-2 rounded">
                                            Edit
                                        </a>
                                        <form method="POST" class="inline" onsubmit="return confirm('Are you sure you want to delete this video?');">
                                            <input type="hidden" name="video_id" value="<?= $video['id'] ?>">
                                            <button type="submit" name="delete_video" 
                                                class="bg-red-500 hover:bg-red-700 text-white px-4 py-2 rounded">
                                                Delete
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    
                    <?php if(empty($videos)): ?>
                        <div class="text-center text-gray-400 py-8">
                            No videos uploaded yet. 
                            <a href="upload.php" class="text-blue-400 hover:text-blue-300">Upload your first video!</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</body>
</html>