<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Visualisation Espace Lointain - Imators</title>
    <style>
        body {
            font-family: 'Arial', sans-serif;
            background-color: #0a0a1a;
            color: #fff;
            margin: 0;
            padding: 0;
        }
        header {
            background-color: #1a1a2a;
            padding: 20px;
            text-align: center;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.5);
        }
        h1 {
            margin: 0;
            color: #4da6ff;
        }
        .container {
            max-width: 1200px;
            margin: 20px auto;
            padding: 0 20px;
        }
        .search-controls {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        input, select, button {
            padding: 10px;
            border: none;
            border-radius: 4px;
            background-color: #2a2a3a;
            color: #fff;
        }
        button {
            background-color: #4da6ff;
            cursor: pointer;
            transition: background-color 0.3s;
        }
        button:hover {
            background-color: #3a80cc;
        }
        .gallery {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        .image-card {
            background-color: #1a1a2a;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.3);
            transition: transform 0.3s;
        }
        .image-card:hover {
            transform: translateY(-5px);
        }
        .image-container {
            position: relative;
            overflow: hidden;
            height: 200px;
        }
        .image-container img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.5s;
        }
        .image-card:hover .image-container img {
            transform: scale(1.05);
        }
        .image-info {
            padding: 15px;
        }
        .image-title {
            font-size: 18px;
            margin: 0 0 10px 0;
            color: #4da6ff;
        }
        .image-date {
            font-size: 14px;
            color: #aaa;
            margin-bottom: 10px;
        }
        .image-description {
            font-size: 14px;
            line-height: 1.5;
            max-height: 100px;
            overflow: hidden;
        }
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.9);
            z-index: 1000;
            overflow: auto;
            padding: 50px 20px;
        }
        .modal-content {
            max-width: 1000px;
            margin: 0 auto;
            position: relative;
            background-color: #1a1a2a;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
        }
        .modal-close {
            position: absolute;
            top: 10px;
            right: 15px;
            font-size: 24px;
            color: #fff;
            cursor: pointer;
            z-index: 1001;
        }
        .modal-image {
            width: 100%;
            max-height: 70vh;
            object-fit: contain;
        }
        .modal-info {
            padding: 20px;
        }
        .modal-title {
            font-size: 24px;
            margin: 0 0 10px 0;
            color: #4da6ff;
        }
        .modal-date {
            font-size: 16px;
            color: #aaa;
            margin-bottom: 15px;
        }
        .modal-description {
            font-size: 16px;
            line-height: 1.6;
        }
        .loading {
            text-align: center;
            padding: 40px;
            font-size: 20px;
            color: #4da6ff;
            display: none;
        }
        .error {
            background-color: #ff4d4d;
            color: white;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 20px;
            display: none;
        }
    </style>
</head>
<body>
    <header>
        <h1>Visualisation Espace Lointain - Imators</h1>
    </header>
    
    <div class="container">
        <div class="error" id="error-message"></div>
        
        <div class="search-controls">
            <select id="search-type">
                <option value="apod">Image du jour (APOD)</option>
                <option value="search">Recherche par mot-clé</option>
            </select>
            <input type="text" id="search-input" placeholder="Mot-clé (ex: nebula, galaxy...)" style="display: none; flex-grow: 1;">
            <input type="number" id="count-input" placeholder="Nombre d'images (max 100)" value="12" min="1" max="100">
            <button id="search-button">Rechercher</button>
        </div>
        
        <div class="loading" id="loading">Chargement des images...</div>
        
        <div class="gallery" id="gallery"></div>
    </div>
    
    <div class="modal" id="image-modal">
        <div class="modal-content">
            <span class="modal-close" id="modal-close">&times;</span>
            <img class="modal-image" id="modal-image" src="" alt="">
            <div class="modal-info">
                <h2 class="modal-title" id="modal-title"></h2>
                <p class="modal-date" id="modal-date"></p>
                <div class="modal-description" id="modal-description"></div>
            </div>
        </div>
    </div>

    <script>
        const API_KEY = "xxhCuSUbAEFfAzIVhYxiRbvcPLVBWY2nOEpXEVWz";
        const searchTypeSelect = document.getElementById('search-type');
        const searchInput = document.getElementById('search-input');
        const countInput = document.getElementById('count-input');
        const searchButton = document.getElementById('search-button');
        const gallery = document.getElementById('gallery');
        const loading = document.getElementById('loading');
        const errorMessage = document.getElementById('error-message');
        const modal = document.getElementById('image-modal');
        const modalClose = document.getElementById('modal-close');
        const modalImage = document.getElementById('modal-image');
        const modalTitle = document.getElementById('modal-title');
        const modalDate = document.getElementById('modal-date');
        const modalDescription = document.getElementById('modal-description');
        
        searchTypeSelect.addEventListener('change', () => {
            if (searchTypeSelect.value === 'search') {
                searchInput.style.display = 'block';
            } else {
                searchInput.style.display = 'none';
            }
        });
        
        searchButton.addEventListener('click', fetchImages);
        modalClose.addEventListener('click', closeModal);
        
        window.addEventListener('click', (event) => {
            if (event.target === modal) {
                closeModal();
            }
        });
        
        document.addEventListener('keydown', (event) => {
            if (event.key === 'Escape') {
                closeModal();
            }
        });
        
        function fetchImages() {
            const searchType = searchTypeSelect.value;
            const count = Math.min(parseInt(countInput.value) || 12, 100);
            
            let apiUrl;
            
            clearGallery();
            showLoading();
            hideError();
            
            if (searchType === 'apod') {
                apiUrl = `https://api.nasa.gov/planetary/apod?api_key=${API_KEY}&count=${count}`;
            } else {
                const query = searchInput.value.trim();
                if (!query) {
                    showError("Veuillez entrer un mot-clé pour la recherche.");
                    hideLoading();
                    return;
                }
                apiUrl = `https://images-api.nasa.gov/search?q=${encodeURIComponent(query)}&media_type=image&page_size=${count}`;
            }
            
            fetch(apiUrl)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`Erreur HTTP: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    if (searchType === 'apod') {
                        displayApodImages(data);
                    } else {
                        displaySearchImages(data);
                    }
                })
                .catch(error => {
                    showError(`Erreur lors de la récupération des images: ${error.message}`);
                })
                .finally(() => {
                    hideLoading();
                });
        }
        
        function displayApodImages(images) {
            if (!images || images.length === 0) {
                showError("Aucune image trouvée.");
                return;
            }
            
            images.forEach(image => {
                if (image.media_type === 'image') {
                    createImageCard({
                        url: image.url,
                        title: image.title,
                        date: formatDate(image.date),
                        description: image.explanation
                    });
                }
            });
        }
        
        function displaySearchImages(data) {
            const items = data.collection?.items;
            
            if (!items || items.length === 0) {
                showError("Aucune image trouvée.");
                return;
            }
            
            items.forEach(item => {
                const imageData = item.data[0];
                const imageHref = item.links?.find(link => link.rel === 'preview')?.href;
                
                if (imageHref) {
                    createImageCard({
                        url: imageHref.replace('thumb', 'medium'),
                        title: imageData.title,
                        date: formatDate(imageData.date_created),
                        description: imageData.description
                    });
                }
            });
        }
        
        function createImageCard(image) {
            const card = document.createElement('div');
            card.className = 'image-card';
            
            card.innerHTML = `
                <div class="image-container">
                    <img src="${image.url}" alt="${image.title}" loading="lazy">
                </div>
                <div class="image-info">
                    <h3 class="image-title">${image.title}</h3>
                    <p class="image-date">${image.date}</p>
                    <div class="image-description">${truncateText(image.description, 100)}</div>
                </div>
            `;
            
            card.addEventListener('click', () => {
                openModal(image);
            });
            
            gallery.appendChild(card);
        }
        
        function openModal(image) {
            modalImage.src = image.url;
            modalTitle.textContent = image.title;
            modalDate.textContent = image.date;
            modalDescription.textContent = image.description;
            modal.style.display = 'block';
            document.body.style.overflow = 'hidden';
        }
        
        function closeModal() {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }
        
        function formatDate(dateStr) {
            if (!dateStr) return 'Date inconnue';
            
            const date = new Date(dateStr);
            return date.toLocaleDateString('fr-FR', {
                day: 'numeric',
                month: 'long',
                year: 'numeric'
            });
        }
        
        function truncateText(text, maxLength) {
            if (!text) return 'Aucune description';
            
            if (text.length <= maxLength) return text;
            
            return text.substring(0, maxLength) + '...';
        }
        
        function clearGallery() {
            gallery.innerHTML = '';
        }
        
        function showLoading() {
            loading.style.display = 'block';
        }
        
        function hideLoading() {
            loading.style.display = 'none';
        }
        
        function showError(message) {
            errorMessage.textContent = message;
            errorMessage.style.display = 'block';
        }
        
        function hideError() {
            errorMessage.style.display = 'none';
        }
        
        fetchImages();
    </script>
</body>
</html>