const fs = require('fs');
const path = require('path');
const readline = require('readline');

const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout
});

function findAllFiles(directory) {
    let results = [];
    
    const items = fs.readdirSync(directory);
    
    for (const item of items) {
        const fullPath = path.join(directory, item);
        const stat = fs.statSync(fullPath);
        
        if (stat.isDirectory()) {
            results = results.concat(findAllFiles(fullPath));
        } else {
            if (path.extname(fullPath) === '.php') {
                results.push(fullPath);
            }
        }
    }
    
    return results;
}

function replaceFooterTag(filePath, newFooterContent) {
    try {
        let content = fs.readFileSync(filePath, 'utf8');
        
        const footerRegex = /<footer>(.*?)<\/footer>/gs;
        
        if (footerRegex.test(content)) {
            const modifiedContent = content.replace(footerRegex, `<footer>${newFooterContent}</footer>`);
            
            return {
                path: filePath,
                hasFooter: true,
                originalContent: content,
                modifiedContent: modifiedContent
            };
        } else {
            return {
                path: filePath,
                hasFooter: false
            };
        }
    } catch (error) {
        return {
            path: filePath,
            hasFooter: false,
            error: error.message
        };
    }
}

function processFiles() {
    const currentDir = process.cwd();
    const files = findAllFiles(currentDir);
    
    console.log(`Trouvé ${files.length} fichiers PHP dans le répertoire et ses sous-répertoires.`);
    
    let newFooterContent;
    
    try {
        const footerFilePath = path.join(currentDir, 'footer-maj.txt');
        newFooterContent = fs.readFileSync(footerFilePath, 'utf8');
        console.log('Contenu du nouveau footer chargé depuis footer-maj.txt');
        processFilesWithFooter(files, newFooterContent);
    } catch (error) {
        console.log('Erreur lors de la lecture du fichier footer-maj.txt:', error.message);
        console.log('Assure-toi que le fichier footer-maj.txt existe dans le répertoire courant.');
        rl.close();
    }
}

function processFilesWithFooter(files, newFooterContent) {
    const filesWithFooter = [];
    
    for (const file of files) {
        const result = replaceFooterTag(file, newFooterContent);
        
        if (result.hasFooter) {
            filesWithFooter.push(result);
        }
    }
    
    console.log(`Trouvé ${filesWithFooter.length} fichiers PHP contenant des balises footer.`);
    
    if (filesWithFooter.length === 0) {
        console.log('Aucun fichier PHP avec des balises footer trouvé. Fin du programme.');
        rl.close();
        return;
    }
    
    processNextFile(filesWithFooter, 0);
}

function processNextFile(filesWithFooter, index) {
    if (index >= filesWithFooter.length) {
        console.log('Tous les fichiers ont été traités. Fin du programme.');
        rl.close();
        return;
    }
    
    const file = filesWithFooter[index];
    console.log(`\nFichier ${index + 1}/${filesWithFooter.length}: ${file.path}`);
    
    rl.question('Modifier ce fichier ? (yes/no) : ', (answer) => {
        if (answer.toLowerCase() === 'yes') {
            fs.writeFileSync(file.path, file.modifiedContent, 'utf8');
            console.log('Fichier modifié avec succès.');
        } else {
            console.log('Fichier non modifié.');
        }
        
        processNextFile(filesWithFooter, index + 1);
    });
}

processFiles();