<!DOCTYPE html>
<html lang="fr">
<head>
   <meta charset="utf-8">
   <meta name="viewport" content="width=device-width, initial-scale=1">
   <title>IS_Engine</title>
   <script src="https://cdn.tailwindcss.com"></script>
   <link href="https://fonts.googleapis.com/css2?family=Urbanist:wght@300;400;500;600;700&display=swap" rel="stylesheet">
   <link rel="stylesheet" href="https://use.fontawesome.com/releases/v6.4.0/css/all.css">
   <meta name="csrf-token" content="{{ csrf_token() }}">
   <style>
       body { font-family: 'Urbanist', sans-serif; }
       .editor-content { min-height: 85vh; }
       .editor-content:empty:before {
           content: attr(data-placeholder);
           color: #9ca3af;
       }
       .floating-toolbar {
           position: fixed;
           bottom: 30px;
           left: 50%;
           transform: translateX(-50%);
           background: rgba(255, 255, 255, 0.95);
           backdrop-filter: blur(10px);
           border: 1px solid #e5e7eb;
           border-radius: 20px;
           box-shadow: 0 20px 40px rgba(0,0,0,0.1);
           padding: 16px 28px;
           z-index: 50;
           max-width: 90vw;
           overflow-x: auto;
           transition: all 0.3s ease;
       }
       .highlight-correct { background-color: rgba(34, 197, 94, 0.2); border-radius: 4px; }
       .highlight-incorrect { background-color: rgba(239, 68, 68, 0.2); border-radius: 4px; }
       .highlight-uncertain { background-color: rgba(251, 191, 36, 0.2); border-radius: 4px; }
       .highlight-processing { 
           background: linear-gradient(90deg, rgba(59, 130, 246, 0.2) 0%, rgba(147, 51, 234, 0.2) 100%);
           border-radius: 4px;
           animation: pulse 2s infinite;
       }
       .sidebar {
           position: fixed;
           top: 0;
           right: -400px;
           width: 400px;
           height: 100vh;
           background: white;
           border-left: 1px solid #e5e7eb;
           box-shadow: -10px 0 30px rgba(0,0,0,0.1);
           z-index: 60;
           transition: right 0.3s ease;
           overflow-y: auto;
       }
       .sidebar.open {
           right: 0;
       }
       .tool-btn {
           padding: 10px 14px;
           border-radius: 12px;
           border: 1px solid #e5e7eb;
           background: white;
           transition: all 0.2s ease;
           font-size: 14px;
           display: flex;
           align-items: center;
           gap: 6px;
           position: relative;
           overflow: hidden;
       }
       .tool-btn:hover {
           background: #f8fafc;
           border-color: #cbd5e1;
           transform: translateY(-1px);
       }
       .tool-btn.active {
           background: #1f2937;
           color: white;
           border-color: #1f2937;
       }
       .tool-btn.processing {
           background: linear-gradient(90deg, #3b82f6, #8b5cf6);
           color: white;
           border-color: transparent;
       }
       .tool-btn.processing::after {
           content: '';
           position: absolute;
           top: 0;
           left: -100%;
           width: 100%;
           height: 100%;
           background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
           animation: shimmer 1.5s infinite;
       }
       @keyframes shimmer {
           100% { left: 100%; }
       }
       @keyframes pulse {
           0%, 100% { opacity: 1; }
           50% { opacity: 0.7; }
       }
       .modal-overlay {
           position: fixed;
           inset: 0;
           background: rgba(0, 0, 0, 0.5);
           backdrop-filter: blur(4px);
           z-index: 70;
           opacity: 0;
           visibility: hidden;
           transition: all 0.3s ease;
       }
       .modal-overlay.show {
           opacity: 1;
           visibility: visible;
       }
       .modal-content {
           position: absolute;
           top: 50%;
           left: 50%;
           transform: translate(-50%, -50%) scale(0.9);
           background: white;
           border-radius: 16px;
           box-shadow: 0 25px 50px rgba(0,0,0,0.2);
           transition: transform 0.3s ease;
           max-width: 90vw;
           max-height: 90vh;
           overflow-y: auto;
       }
       .modal-overlay.show .modal-content {
           transform: translate(-50%, -50%) scale(1);
       }
       .notification {
           position: fixed;
           top: 20px;
           right: 20px;
           background: white;
           border: 1px solid #e5e7eb;
           border-radius: 12px;
           box-shadow: 0 10px 25px rgba(0,0,0,0.1);
           padding: 16px 20px;
           z-index: 80;
           transform: translateX(400px);
           transition: transform 0.3s ease;
           max-width: 300px;
       }
       .notification.show {
           transform: translateX(0);
       }
       .notification.success {
           border-left: 4px solid #10b981;
       }
       .notification.error {
           border-left: 4px solid #ef4444;
       }
       .notification.warning {
           border-left: 4px solid #f59e0b;
       }
       .notification.info {
           border-left: 4px solid #3b82f6;
       }
       .status-indicator {
           position: fixed;
           bottom: 20px;
           right: 20px;
           background: rgba(255, 255, 255, 0.95);
           backdrop-filter: blur(10px);
           border: 1px solid #e5e7eb;
           border-radius: 12px;
           padding: 12px 16px;
           z-index: 40;
           font-size: 14px;
           display: flex;
           align-items: center;
           gap: 8px;
           opacity: 0;
           visibility: hidden;
           transition: all 0.3s ease;
       }
       .status-indicator.show {
           opacity: 1;
           visibility: visible;
       }
       .progress-bar {
           position: fixed;
           top: 0;
           left: 0;
           width: 0%;
           height: 3px;
           background: linear-gradient(90deg, #3b82f6, #8b5cf6);
           z-index: 90;
           transition: width 0.3s ease;
       }
   </style>
</head>
<body class="bg-white text-black">
   <div class="progress-bar" id="progress-bar"></div>
   
   <header class="border-b border-gray-200 px-6 py-4 bg-white/80 backdrop-blur-sm sticky top-0 z-30">
       <div class="flex justify-between items-center max-w-7xl mx-auto">
           <div class="flex items-center space-x-4">
               <a href="/workspace" class="text-gray-600 hover:text-black text-lg transition-colors">
                   <i class="fas fa-arrow-left"></i>
               </a>
               <h1 class="text-xl font-medium">IS_Engine</h1>
               <div class="w-2 h-2 bg-green-500 rounded-full" id="save-status"></div>
           </div>
           <div class="flex items-center space-x-3">
               <button id="save-btn" class="px-4 py-2 bg-black text-white rounded-lg hover:bg-gray-800 text-sm transition-all duration-200 flex items-center gap-2">
                   <i class="fas fa-save"></i>
                   Save
               </button>
               <button id="share-btn" class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 text-sm transition-all duration-200 flex items-center gap-2">
                   <i class="fas fa-share"></i>
                   Share
               </button>
               <span class="text-sm text-gray-600">Welcome, {{ session('username') }}</span>
           </div>
       </div>
   </header>

   <div class="max-w-4xl mx-auto px-6 py-8">
       <input type="text" id="document-title-input" 
              placeholder="Untitled Document" 
              class="w-full text-3xl font-light border-none outline-none mb-8 placeholder-gray-400 transition-all duration-200 focus:placeholder-gray-300">
       
       <div id="editor" 
            class="editor-content outline-none leading-relaxed text-gray-800 transition-all duration-200" 
            contenteditable="true" 
            data-placeholder="Start writing...">
       </div>
   </div>

   <div class="sidebar" id="results-panel">
       <div class="p-6 border-b border-gray-200 bg-gray-50">
           <div class="flex justify-between items-center">
               <h3 class="font-semibold text-lg">Results</h3>
               <button id="close-results" class="text-gray-400 hover:text-gray-600 p-2 rounded-lg hover:bg-gray-200 transition-colors">
                   <i class="fas fa-times"></i>
               </button>
           </div>
       </div>
       <div id="results-content" class="p-6"></div>
   </div>

   <div class="floating-toolbar">
       <div class="flex items-center space-x-2">
           <button class="tool-btn" data-action="bold" title="Bold">
               <i class="fas fa-bold"></i>
           </button>
           <button class="tool-btn" data-action="italic" title="Italic">
               <i class="fas fa-italic"></i>
           </button>
           <button class="tool-btn" data-action="underline" title="Underline">
               <i class="fas fa-underline"></i>
           </button>
           
           <div class="w-px h-8 bg-gray-300 mx-2"></div>
           
           <button class="tool-btn" data-action="h1" title="Heading 1">H1</button>
           <button class="tool-btn" data-action="h2" title="Heading 2">H2</button>
           <button class="tool-btn" data-action="h3" title="Heading 3">H3</button>
           
           <div class="w-px h-8 bg-gray-300 mx-2"></div>
           
           <button class="tool-btn" data-action="bullet-list" title="Bullet List">
               <i class="fas fa-list-ul"></i>
           </button>
           <button class="tool-btn" data-action="number-list" title="Numbered List">
               <i class="fas fa-list-ol"></i>
           </button>
           
           <div class="w-px h-8 bg-gray-300 mx-2"></div>
           
           <button class="tool-btn" data-claude-action="grammar" title="Grammar Check">
               <i class="fas fa-spell-check"></i>
               Grammar
           </button>
           <button class="tool-btn" data-claude-action="expand" title="Expand Text">
               <i class="fas fa-expand-arrows-alt"></i>
               Expand
           </button>
           <button class="tool-btn" data-claude-action="rephrase" title="Rephrase">
               <i class="fas fa-magic"></i>
               Rephrase
           </button>
           
           <div class="w-px h-8 bg-gray-300 mx-2"></div>
           
           <button class="tool-btn" id="fact-check-btn" title="Fact Check">
               <i class="fas fa-search"></i>
               Fact Check
           </button>
           <button class="tool-btn" id="search-btn" title="Web Search">
               <i class="fas fa-globe"></i>
               Search
           </button>
           
           <div class="w-px h-8 bg-gray-300 mx-2"></div>
           
           <input type="text" id="quick-prompt" placeholder="Ask AI..." 
                  class="px-4 py-2 border border-gray-300 rounded-lg text-sm transition-all duration-200 focus:border-blue-500 focus:outline-none" style="width: 180px;">
           <button class="tool-btn" id="send-prompt" title="Send Prompt">
               <i class="fas fa-paper-plane"></i>
           </button>
       </div>
   </div>

   <div id="save-modal" class="modal-overlay">
       <div class="modal-content w-full max-w-md p-6">
           <h3 class="text-xl font-semibold mb-6 flex items-center gap-3">
               <i class="fas fa-save text-blue-500"></i>
               Save Document
           </h3>
           <input type="text" id="save-title" placeholder="Document name" 
                  class="w-full p-4 border border-gray-300 rounded-lg mb-6 text-sm focus:border-blue-500 focus:outline-none transition-colors">
           <div class="flex justify-end space-x-3">
               <button id="save-cancel" class="px-6 py-2 text-gray-600 hover:bg-gray-100 rounded-lg transition-colors">Cancel</button>
               <button id="save-confirm" class="px-6 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors">Save</button>
           </div>
       </div>
   </div>

   <div id="share-modal" class="modal-overlay">
       <div class="modal-content w-full max-w-md p-6">
           <h3 class="text-xl font-semibold mb-6 flex items-center gap-3">
               <i class="fas fa-share text-green-500"></i>
               Share Document
           </h3>
           <div class="space-y-4 mb-6">
               <label class="flex items-center p-3 border rounded-lg cursor-pointer hover:bg-gray-50 transition-colors">
                   <input type="radio" name="share-type" value="view" class="mr-3" checked>
                   <div>
                       <div class="font-medium">Read only</div>
                       <div class="text-sm text-gray-500">Others can only view</div>
                   </div>
               </label>
               <label class="flex items-center p-3 border rounded-lg cursor-pointer hover:bg-gray-50 transition-colors">
                   <input type="radio" name="share-type" value="edit" class="mr-3">
                   <div>
                       <div class="font-medium">Allow editing</div>
                       <div class="text-sm text-gray-500">Others can edit</div>
                   </div>
               </label>
               <input type="text" id="share-link" readonly 
                      class="w-full p-4 border border-gray-300 rounded-lg bg-gray-50 text-sm" 
                      placeholder="Link will appear here">
           </div>
           <div class="flex justify-end space-x-3">
               <button id="share-cancel" class="px-6 py-2 text-gray-600 hover:bg-gray-100 rounded-lg transition-colors">Cancel</button>
               <button id="share-generate" class="px-6 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors">Generate Link</button>
               <button id="share-copy" class="hidden px-6 py-2 bg-green-500 text-white rounded-lg hover:bg-green-600 transition-colors">Copy Link</button>
           </div>
       </div>
   </div>

   <div id="search-modal" class="modal-overlay">
       <div class="modal-content w-full max-w-lg p-6">
           <h3 class="text-xl font-semibold mb-6 flex items-center gap-3">
               <i class="fas fa-globe text-blue-500"></i>
               Search Information
           </h3>
           <input type="text" id="search-query" placeholder="What do you want to search..." 
                  class="w-full p-4 border border-gray-300 rounded-lg mb-6 text-sm focus:border-blue-500 focus:outline-none transition-colors">
           <div class="flex justify-end space-x-3">
               <button id="search-cancel" class="px-6 py-2 text-gray-600 hover:bg-gray-100 rounded-lg transition-colors">Cancel</button>
               <button id="search-submit" class="px-6 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors">Search</button>
           </div>
       </div>
   </div>

   <div class="status-indicator" id="status-indicator">
       <i class="fas fa-info-circle text-blue-500"></i>
       <span id="status-text">Ready</span>
   </div>

   <div class="fixed inset-0 flex flex-col items-center justify-center bg-white text-center px-6 md:hidden">
       <p class="text-lg text-gray-700">This site is not available on mobile. Please try again from a computer.</p>
   </div>

<script>
let currentDocument = {{ isset($document_id) ? $document_id : 'null' }};
let selectedText = '';
let selectedRange = null;
let editor = document.getElementById('editor');
let autoSaveInterval;
let processingActions = new Set();

if (currentDocument) {
   loadDocument(currentDocument);
}

async function loadDocument(id) {
   showStatus('Loading document...', 'info');
   setProgress(30);
   
   try {
       const response = await fetch(`/api/documents/${id}`);
       if (response.ok) {
           const document = await response.json();
           document.getElementById('document-title-input').value = document.title;
           editor.innerHTML = document.content;
           showNotification('Document loaded successfully', 'success');
           setProgress(100);
           setTimeout(() => setProgress(0), 500);
       }
   } catch (error) {
       console.error('Erreur chargement document:', error);
       showNotification('Error loading document', 'error');
       setProgress(0);
   }
   hideStatus();
}

function startAutoSave() {
   autoSaveInterval = setInterval(async () => {
       const title = document.getElementById('document-title-input').value;
       const content = editor.innerHTML;
       
       if (title.trim() !== '' || content.trim() !== '') {
           await saveDocument(false);
       }
   }, 30000);
}

async function saveDocument(showNotif = true) {
   const title = document.getElementById('document-title-input').value || 'Untitled Document';
   const content = editor.innerHTML;

   if (showNotif) {
       showStatus('Saving document...', 'info');
       setProgress(50);
   }

   try {
       const response = await fetch('/api/documents/save', {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               title: title,
               content: content,
               document_id: currentDocument
           })
       });

       const result = await response.json();
       
       if (result.success) {
           currentDocument = result.document_id;
           updateSaveStatus('saved');
           if (showNotif) {
               showNotification('Document saved successfully!', 'success');
               setProgress(100);
               setTimeout(() => setProgress(0), 500);
           }
       }
   } catch (error) {
       console.error('Erreur sauvegarde:', error);
       updateSaveStatus('error');
       if (showNotif) {
           showNotification('Error saving document', 'error');
           setProgress(0);
       }
   }
   if (showNotif) hideStatus();
}

function updateSaveStatus(status) {
   const indicator = document.getElementById('save-status');
   indicator.className = 'w-2 h-2 rounded-full';
   
   switch(status) {
       case 'saved':
           indicator.classList.add('bg-green-500');
           break;
       case 'saving':
           indicator.classList.add('bg-yellow-500');
           break;
       case 'error':
           indicator.classList.add('bg-red-500');
           break;
       default:
           indicator.classList.add('bg-gray-400');
   }
}

editor.addEventListener('input', () => {
   updateSaveStatus('saving');
   setTimeout(() => updateSaveStatus('saved'), 1000);
});

editor.addEventListener('mouseup', function() {
   selectedText = window.getSelection().toString().trim();
   if (selectedText) {
       selectedRange = window.getSelection().getRangeAt(0).cloneRange();
   }
});

document.querySelectorAll('[data-action]').forEach(button => {
   button.addEventListener('click', function() {
       const action = this.dataset.action;
       executeCommand(action);
       showNotification(`Applied ${action} formatting`, 'info');
   });
});

function executeCommand(action) {
   switch(action) {
       case 'bold':
           document.execCommand('bold');
           break;
       case 'italic':
           document.execCommand('italic');
           break;
       case 'underline':
           document.execCommand('underline');
           break;
       case 'h1':
           document.execCommand('formatBlock', false, 'h1');
           break;
       case 'h2':
           document.execCommand('formatBlock', false, 'h2');
           break;
       case 'h3':
           document.execCommand('formatBlock', false, 'h3');
           break;
       case 'bullet-list':
           document.execCommand('insertUnorderedList');
           break;
       case 'number-list':
           document.execCommand('insertOrderedList');
           break;
   }
   editor.focus();
}

document.querySelectorAll('[data-claude-action]').forEach(button => {
   button.addEventListener('click', function() {
       const action = this.dataset.claudeAction;
       if (selectedText) {
           setButtonProcessing(this, true);
           highlightSelectedText('processing');
           showStatus(`Processing ${action}...`, 'info');
           callClaudeAPI('', action, this);
       } else {
           showNotification('Please select text first', 'warning');
       }
   });
});

document.getElementById('send-prompt').addEventListener('click', () => {
   const prompt = document.getElementById('quick-prompt').value;
   const button = document.getElementById('send-prompt');
   
   if (prompt.trim()) {
       setButtonProcessing(button, true);
       if (selectedText) {
           highlightSelectedText('processing');
       }
       showStatus('Processing custom prompt...', 'info');
       callClaudeAPI(prompt, 'custom', button);
       document.getElementById('quick-prompt').value = '';
   }
});

document.getElementById('quick-prompt').addEventListener('keypress', (e) => {
   if (e.key === 'Enter') {
       document.getElementById('send-prompt').click();
   }
});

function setButtonProcessing(button, processing) {
   if (processing) {
       button.classList.add('processing');
       button.disabled = true;
   } else {
       button.classList.remove('processing');
       button.disabled = false;
   }
}

async function callClaudeAPI(prompt, action, button = null) {
   setProgress(20);
   
   try {
       const response = await fetch('/api/claude', {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               prompt: prompt,
               selected_text: selectedText,
               action: action
           })
       });

       setProgress(60);
       const result = await response.json();
       
       if (result.success) {
           let cleanContent = result.content;
           
           const introPatterns = [
               /^Voici le texte corrigé :\s*/i,
               /^Texte corrigé :\s*/i,
               /^Version développée :\s*/i,
               /^Résumé :\s*/i,
               /^Reformulation :\s*/i,
               /^Voici une reformulation :\s*/i,
               /^Voici un résumé :\s*/i,
               /^Voici le texte développé :\s*/i
           ];
           
           introPatterns.forEach(pattern => {
               cleanContent = cleanContent.replace(pattern, '');
           });
           
           const endPatterns = [
               /\.\s*Il n'y a aucune erreur.*$/i,
               /\.\s*Ce texte ne contient.*$/i,
               /\.\s*La phrase est déjà.*$/i
           ];
           
           endPatterns.forEach(pattern => {
               cleanContent = cleanContent.replace(pattern, '.');
           });
           
           cleanContent = cleanContent.trim();
           
           if (selectedText && selectedRange) {
               highlightSelectedText('correct');
               showResults([{
                   type: 'claude',
                   original: selectedText,
                   suggestion: cleanContent,
                   action: action
               }]);
           } else {
               showResults([{
                   type: 'claude',
                   response: cleanContent
               }]);
           }
           
           showNotification(`${action} completed successfully`, 'success');
           setProgress(100);
       } else {
           if (selectedText) {
               clearHighlights();
           }
           showNotification(`Claude ${action} failed`, 'error');
           setProgress(0);
       }
   } catch (error) {
       if (selectedText) {
           clearHighlights();
       }
       console.error('Erreur Claude API:', error);
       showNotification('Connection error', 'error');
       setProgress(0);
   }
   
   if (button) setButtonProcessing(button, false);
   hideStatus();
   setTimeout(() => setProgress(0), 500);
}

document.getElementById('fact-check-btn').addEventListener('click', () => {
   const documentText = editor.textContent.trim();
   const button = document.getElementById('fact-check-btn');
   
   if (documentText) {
       setButtonProcessing(button, true);
       showStatus('Fact-checking document...', 'info');
       analyzeDocument(documentText, button);
   } else {
       showNotification('Document is empty', 'warning');
   }
});

async function analyzeDocument(text, button) {
   setProgress(30);
   
   try {
       const response = await fetch('/api/fact-check', {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               text: text
           })
       });

       setProgress(70);
       const result = await response.json();
       
       if (result.success) {
           highlightFactCheckResults(result.analysis);
           showResults(result.analysis);
           showNotification('Fact-check completed', 'success');
           setProgress(100);
       } else {
           showNotification('Fact check error', 'error');
           setProgress(0);
       }
   } catch (error) {
       console.error('Erreur fact check:', error);
       showNotification('Connection error', 'error');
       setProgress(0);
   }
   
   setButtonProcessing(button, false);
   hideStatus();
   setTimeout(() => setProgress(0), 500);
}

document.getElementById('search-btn').addEventListener('click', () => {
   showModal('search-modal');
});

document.getElementById('search-submit').addEventListener('click', () => {
   const query = document.getElementById('search-query').value;
   if (query.trim()) {
       hideModal('search-modal');
       showStatus('Searching web...', 'info');
       callPerplexityAPI(query);
   }
});

document.getElementById('search-cancel').addEventListener('click', () => {
   hideModal('search-modal');
});

async function callPerplexityAPI(query) {
   setProgress(40);
   
   try {
       const response = await fetch('/api/search', {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               query: query
           })
       });

       setProgress(80);
       const result = await response.json();
       
       if (result.success) {
           showResults([{
               type: 'search',
               query: query,
               results: result.results,
               summary: result.summary
           }]);
           showNotification('Search completed', 'success');
           setProgress(100);
       } else {
           showNotification('Search error', 'error');
           setProgress(0);
       }
   } catch (error) {
       console.error('Erreur recherche:', error);
       showNotification('Connection error', 'error');
       setProgress(0);
   }
   
   hideStatus();
   setTimeout(() => setProgress(0), 500);
}

function highlightSelectedText(type) {
   if (selectedRange) {
       const span = document.createElement('span');
       span.className = `highlight-${type}`;
       try {
           selectedRange.surroundContents(span);
       } catch (e) {
           console.log('Cannot highlight complex selection');
       }
   }
}

function highlightFactCheckResults(analysis) {
   clearHighlights();
   const editorText = editor.textContent;
   
   analysis.forEach(item => {
       if (item.statement) {
           const regex = new RegExp(item.statement.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi');
           editor.innerHTML = editor.innerHTML.replace(regex, 
               `<span class="highlight-${item.status}">${item.statement}</span>`);
       }
   });
}

function clearHighlights() {
   const highlights = editor.querySelectorAll('[class*="highlight-"]');
   highlights.forEach(highlight => {
       const parent = highlight.parentNode;
       parent.replaceChild(document.createTextNode(highlight.textContent), highlight);
       parent.normalize();
   });
}

function showResults(data) {
   const panel = document.getElementById('results-panel');
   const content = document.getElementById('results-content');
   let html = '';
   
   data.forEach(item => {
       if (item.type === 'claude') {
           if (item.suggestion) {
               html += `
                   <div class="mb-6 p-4 border border-gray-200 rounded-lg bg-white shadow-sm">
                       <div class="flex items-center gap-2 mb-3">
                           <i class="fas fa-robot text-blue-500"></i>
                           <div class="font-semibold">Claude Suggestion</div>
                       </div>
                       <div class="text-sm text-gray-600 mb-3 p-2 bg-gray-50 rounded">
                           <strong>Original:</strong> "${item.original}"
                       </div>
                       <div class="text-sm text-gray-800 mb-4 p-3 bg-blue-50 rounded border-l-4 border-blue-400">
                           ${item.suggestion}
                       </div>
                       <button onclick="applyClaudeEdit('${item.suggestion.replace(/'/g, "\\'")}')" 
                               class="px-4 py-2 bg-green-500 text-white rounded-lg text-sm hover:bg-green-600 transition-colors flex items-center gap-2">
                           <i class="fas fa-check"></i>
                           Apply Changes
                       </button>
                   </div>
               `;
           } else {
               html += `
                   <div class="mb-6 p-4 border border-gray-200 rounded-lg bg-white shadow-sm">
                       <div class="flex items-center gap-2 mb-3">
                           <i class="fas fa-robot text-blue-500"></i>
                           <div class="font-semibold">Claude Response</div>
                       </div>
                       <div class="text-sm text-gray-800 p-3 bg-blue-50 rounded">${item.response}</div>
                   </div>
               `;
           }
       } else if (item.type === 'search') {
           html += `
               <div class="mb-6 p-4 border border-gray-200 rounded-lg bg-white shadow-sm">
                   <div class="flex items-center gap-2 mb-3">
                       <i class="fas fa-globe text-green-500"></i>
                       <div class="font-semibold">Search Results</div>
                   </div>
                   <div class="text-sm text-gray-600 mb-3 p-2 bg-gray-50 rounded">
                       <strong>Query:</strong> "${item.query}"
                   </div>
                   ${item.summary ? `
                       <div class="text-sm text-gray-800 mb-4 p-3 bg-green-50 rounded border-l-4 border-green-400">
                           ${item.summary}
                       </div>
                       <button onclick="insertSearchResult('${item.summary ? item.summary.replace(/'/g, "\\'") : ''}')" 
                               class="px-4 py-2 bg-blue-500 text-white rounded-lg text-sm hover:bg-blue-600 transition-colors flex items-center gap-2">
                           <i class="fas fa-plus"></i>
                           Insert in Document
                       </button>
                   ` : ''}
               </div>
           `;
       } else {
           const statusColors = {
               'correct': { border: 'border-green-400', bg: 'bg-green-50', icon: 'text-green-500', iconName: 'check-circle' },
               'incorrect': { border: 'border-red-400', bg: 'bg-red-50', icon: 'text-red-500', iconName: 'times-circle' },
               'uncertain': { border: 'border-yellow-400', bg: 'bg-yellow-50', icon: 'text-yellow-500', iconName: 'question-circle' }
           };
           const colors = statusColors[item.status] || statusColors['uncertain'];
           
           html += `
               <div class="mb-4 p-4 border-l-4 ${colors.border} ${colors.bg} rounded-r-lg">
                   <div class="flex items-start gap-3">
                       <i class="fas fa-${colors.iconName} ${colors.icon} mt-1"></i>
                       <div class="flex-1">
                           <div class="font-medium text-gray-800 mb-2">${item.statement}</div>
                           <div class="text-sm text-gray-600 mb-2">${item.explanation}</div>
                           ${item.source ? `<div class="text-xs text-blue-600 hover:underline cursor-pointer">Source: ${item.source}</div>` : ''}
                       </div>
                   </div>
               </div>
           `;
       }
   });
   
   content.innerHTML = html;
   panel.classList.add('open');
   showNotification('Results panel opened', 'info');
}

function applyClaudeEdit(newText) {
   if (selectedRange) {
       const selection = window.getSelection();
       selection.removeAllRanges();
       selection.addRange(selectedRange);
       
       if (selection.rangeCount > 0) {
           const range = selection.getRangeAt(0);
           range.deleteContents();
           range.insertNode(document.createTextNode(newText));
       }
       
       clearHighlights();
       selectedText = '';
       selectedRange = null;
       showNotification('Changes applied successfully', 'success');
       updateSaveStatus('saving');
   }
}

function insertSearchResult(text) {
   if (text) {
       const selection = window.getSelection();
       if (selection.rangeCount > 0) {
           const range = selection.getRangeAt(0);
           range.insertNode(document.createTextNode(text + ' '));
       } else {
           editor.appendChild(document.createTextNode(text + ' '));
       }
       editor.focus();
       showNotification('Content inserted', 'success');
       updateSaveStatus('saving');
   }
}

document.getElementById('close-results').addEventListener('click', () => {
   document.getElementById('results-panel').classList.remove('open');
   clearHighlights();
   showNotification('Results panel closed', 'info');
});

document.getElementById('save-btn').addEventListener('click', () => {
   document.getElementById('save-title').value = document.getElementById('document-title-input').value;
   showModal('save-modal');
});

document.getElementById('save-confirm').addEventListener('click', async () => {
   const title = document.getElementById('save-title').value;
   document.getElementById('document-title-input').value = title;
   hideModal('save-modal');
   await saveDocument(true);
});

document.getElementById('save-cancel').addEventListener('click', () => {
   hideModal('save-modal');
});

document.getElementById('share-btn').addEventListener('click', () => {
   if (!currentDocument) {
       showNotification('Please save the document first', 'warning');
       return;
   }
   showModal('share-modal');
});

document.getElementById('share-generate').addEventListener('click', async () => {
   const permission = document.querySelector('input[name="share-type"]:checked').value;
   const button = document.getElementById('share-generate');
   
   button.disabled = true;
   button.textContent = 'Generating...';
   showStatus('Generating share link...', 'info');
   
   try {
       const response = await fetch(`/api/documents/${currentDocument}/share`, {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               permission: permission
           })
       });

       const result = await response.json();
       
       if (result.success) {
           document.getElementById('share-link').value = result.share_url;
           document.getElementById('share-copy').classList.remove('hidden');
           showNotification('Share link generated', 'success');
       }
   } catch (error) {
       console.error('Erreur partage:', error);
       showNotification('Error generating share link', 'error');
   }
   
   button.disabled = false;
   button.textContent = 'Generate Link';
   hideStatus();
});

document.getElementById('share-copy').addEventListener('click', () => {
   const shareLink = document.getElementById('share-link');
   shareLink.select();
   document.execCommand('copy');
   showNotification('Share link copied to clipboard!', 'success');
});

document.getElementById('share-cancel').addEventListener('click', () => {
   hideModal('share-modal');
   document.getElementById('share-link').value = '';
   document.getElementById('share-copy').classList.add('hidden');
});

function showModal(modalId) {
   const modal = document.getElementById(modalId);
   modal.classList.add('show');
}

function hideModal(modalId) {
   const modal = document.getElementById(modalId);
   modal.classList.remove('show');
}

function showNotification(message, type = 'info') {
   const notification = document.createElement('div');
   notification.className = `notification ${type}`;
   
   const iconMap = {
       'success': 'fas fa-check-circle text-green-500',
       'error': 'fas fa-times-circle text-red-500',
       'warning': 'fas fa-exclamation-triangle text-yellow-500',
       'info': 'fas fa-info-circle text-blue-500'
   };
   
   notification.innerHTML = `
       <div class="flex items-center gap-3">
           <i class="${iconMap[type]}"></i>
           <div>
               <div class="font-medium text-sm">${message}</div>
               <div class="text-xs text-gray-500">${new Date().toLocaleTimeString()}</div>
           </div>
       </div>
   `;
   
   document.body.appendChild(notification);
   
   setTimeout(() => notification.classList.add('show'), 100);
   
   setTimeout(() => {
       notification.classList.remove('show');
       setTimeout(() => notification.remove(), 300);
   }, 4000);
}

function showStatus(message, type = 'info') {
   const indicator = document.getElementById('status-indicator');
   const text = document.getElementById('status-text');
   
   const iconMap = {
       'info': 'fas fa-info-circle text-blue-500',
       'success': 'fas fa-check-circle text-green-500',
       'error': 'fas fa-times-circle text-red-500',
       'warning': 'fas fa-exclamation-triangle text-yellow-500'
   };
   
   indicator.innerHTML = `
       <i class="${iconMap[type]}"></i>
       <span>${message}</span>
   `;
   
   indicator.classList.add('show');
}

function hideStatus() {
   const indicator = document.getElementById('status-indicator');
   indicator.classList.remove('show');
}

function setProgress(percentage) {
   const progressBar = document.getElementById('progress-bar');
   progressBar.style.width = percentage + '%';
}

document.addEventListener('click', (e) => {
   if (e.target.classList.contains('modal-overlay')) {
       const modals = document.querySelectorAll('.modal-overlay.show');
       modals.forEach(modal => modal.classList.remove('show'));
   }
});

document.addEventListener('keydown', (e) => {
   if (e.key === 'Escape') {
       const modals = document.querySelectorAll('.modal-overlay.show');
       modals.forEach(modal => modal.classList.remove('show'));
       
       const sidebar = document.getElementById('results-panel');
       if (sidebar.classList.contains('open')) {
           sidebar.classList.remove('open');
           clearHighlights();
       }
   }
});

document.getElementById('document-title-input').addEventListener('input', () => {
   updateSaveStatus('saving');
});

startAutoSave();

window.addEventListener('beforeunload', () => {
   if (autoSaveInterval) {
       clearInterval(autoSaveInterval);
   }
});

showNotification('IS_Engine loaded successfully', 'success');
</script>
</body>
</html>