<?php
require_once 'vendor/autoload.php';

use Dotenv\Dotenv;

// Chargement sécurisé des variables d'environnement
$dotenv = Dotenv::createImmutable(__DIR__);
$dotenv->load();
$dotenv->required(['EVENT_DATE', 'EVENT_ADDRESS', 'EVENT_LAT', 'EVENT_LONG', 'EVENT_TITLE', 'MAPKIT_TOKEN']);

// Validation et formatage de la date
try {
    $event_date = new DateTime($_ENV['EVENT_DATE']);
} catch (Exception $e) {
    die('Date de l\'événement invalide');
}

// Validation des coordonnées
if (!is_numeric($_ENV['EVENT_LAT']) || !is_numeric($_ENV['EVENT_LONG'])) {
    die('Coordonnées géographiques invalides');
}


// Vérification de l'existence des images
foreach ($images as $image) {
    if (!file_exists(ltrim($image['src'], '/'))) {
        die('Image manquante: ' . $image['src']);
    }
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="<?php echo htmlspecialchars($_ENV['EVENT_DESCRIPTION']); ?>">
    <title><?php echo htmlspecialchars($_ENV['EVENT_TITLE']); ?></title>
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <script src="https://cdn.tailwindcss.com" defer></script>
    <script src="https://cdn.apple-mapkit.com/mk/5.x.x/mapkit.js" defer></script>
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Outfit:wght@200;300;400&display=swap">
    
    <style>
        :root {
            --transition-speed: 0.3s;
        }
        
        .smooth-scroll { 
            scroll-behavior: smooth; 
            -webkit-overflow-scrolling: touch;
        }
        
        .countdown { 
            font-variant-numeric: tabular-nums;
            transition: opacity var(--transition-speed);
        }
        
        .image-grid img {
            transition: transform var(--transition-speed);
        }
        
        .image-grid img:hover {
            transform: scale(1.05);
        }

        @media (prefers-reduced-motion: reduce) {
            .smooth-scroll {
                scroll-behavior: auto;
            }
            
            .image-grid img {
                transition: none;
            }
        }

        .loading {
            opacity: 0.5;
        }

        #map {
            transition: opacity var(--transition-speed);
        }
    </style>
</head>
<body class="bg-white font-['Outfit'] text-neutral-900 smooth-scroll">
    <div class="fixed top-0 left-0 right-0 h-[65vh] md:h-screen z-0 image-grid">
        <div class="grid grid-cols-2 md:grid-cols-4 h-full gap-1">
            <?php foreach ($images as $image): ?>
            <div class="relative overflow-hidden">
                <img 
                    src="<?php echo htmlspecialchars($image['src']); ?>" 
                    class="absolute inset-0 w-full h-full object-cover opacity-90"
                    alt="<?php echo htmlspecialchars($image['alt']); ?>"
                    loading="lazy"
                >
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <main class="relative z-10 mt-[60vh] md:mt-[95vh] bg-white min-h-screen">
        <div class="max-w-[90%] md:max-w-[1200px] mx-auto">
            <div class="grid md:grid-cols-2 gap-24 py-24">
                <div class="space-y-20">
                    <div class="space-y-8">
                        <h1 class="text-4xl md:text-5xl font-extralight tracking-tight">
                            <?php echo htmlspecialchars($_ENV['EVENT_TITLE']); ?>
                        </h1>
                        <p class="text-neutral-500 leading-relaxed font-light text-lg">
                            <?php echo nl2br(htmlspecialchars($_ENV['EVENT_DESCRIPTION'])); ?>
                        </p>
                        <div class="text-sm text-neutral-400">
                            <p>Date: <?php echo $event_date->format('d/m/Y H:i'); ?></p>
                            <p>Adresse: <?php echo htmlspecialchars($_ENV['EVENT_ADDRESS']); ?></p>
                        </div>
                    </div>

                    <div class="countdown font-extralight text-5xl md:text-7xl tracking-tighter flex justify-between w-full md:w-3/4" id="countdown">
                        <div class="text-center">
                            <span class="countdown-days">00</span>
                            <div class="text-xs text-neutral-400 mt-2">Jours</div>
                        </div>
                        <div class="text-center">
                            <span class="countdown-hours">00</span>
                            <div class="text-xs text-neutral-400 mt-2">Heures</div>
                        </div>
                        <div class="text-center">
                            <span class="countdown-minutes">00</span>
                            <div class="text-xs text-neutral-400 mt-2">Minutes</div>
                        </div>
                        <div class="text-center">
                            <span class="countdown-seconds">00</span>
                            <div class="text-xs text-neutral-400 mt-2">Secondes</div>
                        </div>
                    </div>
                </div>

                <div class="aspect-[4/5] md:aspect-square relative rounded-lg overflow-hidden shadow-lg" id="map"></div>
            </div>
        </div>
    </main>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const countdown = document.getElementById('countdown');
        const map = document.getElementById('map');
        
        function updateCountdown() {
            const eventDate = new Date('<?php echo $event_date->format('c'); ?>');
            const now = new Date();
            const diff = eventDate - now;

            if (diff < 0) {
                countdown.innerHTML = '<div class="text-center w-full">L\'événement est terminé</div>';
                return;
            }

            const days = Math.floor(diff / (1000 * 60 * 60 * 24));
            const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((diff % (1000 * 60)) / 1000);

            const elements = {
                'days': days,
                'hours': hours,
                'minutes': minutes,
                'seconds': seconds
            };

            for (const [unit, value] of Object.entries(elements)) {
                const element = document.querySelector(`.countdown-${unit}`);
                if (element) {
                    element.textContent = String(value).padStart(2, '0');
                }
            }
        }

        function initMap() {
            try {
                mapkit.init({
                    authorizationCallback: function(done) {
                        done('<?php echo $_ENV['MAPKIT_TOKEN']; ?>');
                    }
                });

                const coordinate = new mapkit.Coordinate(
                    <?php echo floatval($_ENV['EVENT_LAT']); ?>, 
                    <?php echo floatval($_ENV['EVENT_LONG']); ?>
                );

                const map = new mapkit.Map('map', {
                    mapType: mapkit.Map.MapTypes.MutedStandard,
                    showsZoomControl: true,
                    showsMapTypeControl: false,
                    showsCompass: mapkit.FeatureVisibility.Adaptive,
                    showsScale: mapkit.FeatureVisibility.Adaptive
                });

                const annotation = new mapkit.MarkerAnnotation(coordinate, {
                    color: "#000000",
                    title: "<?php echo addslashes($_ENV['EVENT_TITLE']); ?>",
                    subtitle: "<?php echo addslashes($_ENV['EVENT_ADDRESS']); ?>",
                    selected: true,
                    animates: true
                });

                map.colorScheme = mapkit.Map.ColorSchemes.Light;
                map.addAnnotation(annotation);
                map.setCenterAnimated(coordinate, true);
                map.setRegionAnimated(new mapkit.CoordinateRegion(
                    coordinate,
                    new mapkit.CoordinateSpan(0.005, 0.005)
                ));

                document.getElementById('map').classList.remove('loading');
            } catch (error) {
                console.error('Erreur d\'initialisation de la carte:', error);
                document.getElementById('map').innerHTML = '<div class="flex items-center justify-center h-full bg-neutral-100 text-neutral-400">Erreur de chargement de la carte</div>';
            }
        }

        // Initialize
        map.classList.add('loading');
        setInterval(updateCountdown, 1000);
        updateCountdown();
        initMap();
    });
    </script>
</body>
</html>