<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>TrafficLight</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'poppins': ['Poppins', 'sans-serif']
                    }
                }
            }
        }
    </script>
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            font-weight: 300;
        }
        #map {
            height: 75vh;
            width: 100%;
        }
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
        }
        .modal-content {
            background-color: white;
            margin: 10% auto;
            padding: 20px;
            width: 90%;
            max-width: 500px;
            border-radius: 8px;
        }
        .location-dot {
            width: 16px;
            height: 16px;
            background-color: blue;
            border-radius: 50%;
            border: 2px solid white;
            animation: pulse 1.5s infinite;
        }
        @keyframes pulse {
            0% {
                box-shadow: 0 0 0 0 rgba(0, 0, 255, 0.7);
            }
            70% {
                box-shadow: 0 0 0 10px rgba(0, 0, 255, 0);
            }
            100% {
                box-shadow: 0 0 0 0 rgba(0, 0, 255, 0);
            }
        }
    </style>
</head>
<body class="bg-white">
    <header class="py-3 px-4 shadow-sm bg-blue-50 sticky top-0 z-10">
        <h1 class="text-xl text-center font-medium">TrafficLight</h1>
    </header>

    <div id="map"></div>

    <div class="fixed bottom-4 right-4 flex flex-col gap-2 z-20">
        <button id="addLightBtn" class="bg-blue-500 text-white p-3 rounded-full shadow-lg hover:bg-blue-600">
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" class="w-6 h-6">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4" />
            </svg>
        </button>
        <button id="measureBtn" class="bg-green-500 text-white p-3 rounded-full shadow-lg hover:bg-green-600">
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" class="w-6 h-6">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
        </button>
        <button id="locateBtn" class="bg-purple-500 text-white p-3 rounded-full shadow-lg hover:bg-purple-600">
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" class="w-6 h-6">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
            </svg>
        </button>
    </div>

    <!-- Add Traffic Light Modal -->
    <div id="addLightModal" class="modal">
        <div class="modal-content">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-lg font-medium">Add New Traffic Light</h2>
                <button class="close-modal text-gray-500 hover:text-gray-700">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                    </svg>
                </button>
            </div>
            <form id="addTrafficLightForm">
                <div class="mb-3">
                    <label class="block text-sm mb-1">Name</label>
                    <input type="text" id="lightName" class="w-full px-3 py-2 border rounded" required>
                </div>
                <div class="mb-3">
                    <label class="block text-sm mb-1">Location (tap map to set)</label>
                    <div class="flex gap-2">
                        <input type="text" id="latitude" placeholder="Latitude" class="w-full px-3 py-2 border rounded" required readonly>
                        <input type="text" id="longitude" placeholder="Longitude" class="w-full px-3 py-2 border rounded" required readonly>
                    </div>
                </div>
                <div class="mb-3">
                    <label class="block text-sm mb-1">Direction</label>
                    <select id="direction" class="w-full px-3 py-2 border rounded" required>
                        <option value="north">North</option>
                        <option value="east">East</option>
                        <option value="south">South</option>
                        <option value="west">West</option>
                    </select>
                </div>
                <div class="mb-4">
                    <label class="block text-sm mb-1">Cycle Duration (seconds)</label>
                    <div class="flex gap-2">
                        <div class="w-1/2">
                            <label class="block text-xs text-red-500">Red</label>
                            <input type="number" id="redDuration" placeholder="60" class="w-full px-3 py-2 border rounded" required>
                        </div>
                        <div class="w-1/2">
                            <label class="block text-xs text-green-500">Green</label>
                            <input type="number" id="greenDuration" placeholder="30" class="w-full px-3 py-2 border rounded" required>
                        </div>
                    </div>
                </div>
                <button type="submit" class="bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600 w-full">Add Traffic Light</button>
            </form>
        </div>
    </div>

    <!-- Measure Modal -->
    <div id="measureModal" class="modal">
        <div class="modal-content">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-lg font-medium">Measure Traffic Light Timing</h2>
                <button class="close-modal text-gray-500 hover:text-gray-700">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                    </svg>
                </button>
            </div>
            <div class="mb-3">
                <label class="block text-sm mb-1">Select Traffic Light</label>
                <select id="selectLight" class="w-full px-3 py-2 border rounded mb-2">
                    <option value="">-- Select a traffic light --</option>
                </select>
            </div>
            <div class="mb-3">
                <label class="block text-sm mb-1">Current Status</label>
                <div id="lightStatus" class="h-10 flex items-center">
                    <span class="text-gray-500">Select a traffic light to see status</span>
                </div>
            </div>
            <div class="grid grid-cols-3 gap-2 mb-3">
                <button id="startMeasure" class="bg-red-500 text-white px-2 py-2 rounded hover:bg-red-600">Start (Red)</button>
                <button id="stopMeasure" class="bg-green-500 text-white px-2 py-2 rounded hover:bg-green-600">Stop (Green)</button>
                <button id="saveMeasure" class="bg-blue-500 text-white px-2 py-2 rounded hover:bg-blue-600">Save</button>
            </div>
            <div id="measureResult" class="text-sm"></div>
        </div>
    </div>

    <!-- Light Info Modal -->
    <div id="lightInfoModal" class="modal">
        <div class="modal-content">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-lg font-medium" id="lightInfoTitle">Traffic Light Information</h2>
                <button class="close-modal text-gray-500 hover:text-gray-700">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                    </svg>
                </button>
            </div>
            <div id="lightInfoContent" class="mb-4"></div>
            <div id="lightInfoStatus" class="mb-4 py-3 px-4 rounded-lg"></div>
            <div class="flex justify-end">
                <button id="measureThisLight" class="bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600">Measure This Light</button>
            </div>
        </div>
    </div>

    <script>
        // DOM elements
        const addLightBtn = document.getElementById('addLightBtn');
        const measureBtn = document.getElementById('measureBtn');
        const locateBtn = document.getElementById('locateBtn');
        const addLightModal = document.getElementById('addLightModal');
        const measureModal = document.getElementById('measureModal');
        const lightInfoModal = document.getElementById('lightInfoModal');
        const closeModalButtons = document.querySelectorAll('.close-modal');
        
        // Map and location variables
        let map;
        let userMarker;
        let userWatchId;
        let selectingLocation = false;
        let tempMarker;
        let markers = {};
        let selectedLightId = null;
        
        // Initialize map
        initMap();
        
        // Initialize modals
        closeModalButtons.forEach(button => {
            button.addEventListener('click', function() {
                addLightModal.style.display = 'none';
                measureModal.style.display = 'none';
                lightInfoModal.style.display = 'none';
                selectingLocation = false;
                if (tempMarker) {
                    map.removeLayer(tempMarker);
                    tempMarker = null;
                }
            });
        });
        
        // Button click handlers
        addLightBtn.addEventListener('click', function() {
            addLightModal.style.display = 'block';
            selectingLocation = true;
            
            // If user location is available, pre-fill the coordinates
            if (userMarker) {
                const position = userMarker.getLatLng();
                document.getElementById('latitude').value = position.lat.toFixed(6);
                document.getElementById('longitude').value = position.lng.toFixed(6);
                
                // Add temporary marker
                if (tempMarker) map.removeLayer(tempMarker);
                tempMarker = L.marker([position.lat, position.lng], {
                    draggable: true
                }).addTo(map);
                
                tempMarker.on('dragend', function() {
                    const newPos = tempMarker.getLatLng();
                    document.getElementById('latitude').value = newPos.lat.toFixed(6);
                    document.getElementById('longitude').value = newPos.lng.toFixed(6);
                });
            }
        });
        
        measureBtn.addEventListener('click', function() {
            measureModal.style.display = 'block';
        });
        
        locateBtn.addEventListener('click', function() {
            startGeolocation();
        });
        
        // Map initialization
        function initMap() {
            map = L.map('map').setView([51.505, -0.09], 13);
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a>'
            }).addTo(map);
            
            // Map click handler
            map.on('click', function(e) {
                if (selectingLocation) {
                    document.getElementById('latitude').value = e.latlng.lat.toFixed(6);
                    document.getElementById('longitude').value = e.latlng.lng.toFixed(6);
                    
                    // Update or create temporary marker
                    if (tempMarker) map.removeLayer(tempMarker);
                    tempMarker = L.marker([e.latlng.lat, e.latlng.lng], {
                        draggable: true
                    }).addTo(map);
                    
                    tempMarker.on('dragend', function() {
                        const newPos = tempMarker.getLatLng();
                        document.getElementById('latitude').value = newPos.lat.toFixed(6);
                        document.getElementById('longitude').value = newPos.lng.toFixed(6);
                    });
                }
            });
            
            // Start geolocation immediately
            startGeolocation();
            
            // Load traffic lights
            loadTrafficLights();
        }
        
        // Geolocation functions
        function startGeolocation() {
            if (navigator.geolocation) {
                if (userWatchId) navigator.geolocation.clearWatch(userWatchId);
                
                userWatchId = navigator.geolocation.watchPosition(
                    updateUserLocation,
                    handleGeolocationError,
                    { enableHighAccuracy: true, maximumAge: 10000, timeout: 5000 }
                );
                
                // Also get initial position
                navigator.geolocation.getCurrentPosition(
                    updateUserLocation,
                    handleGeolocationError,
                    { enableHighAccuracy: true }
                );
            }
        }
        
        function updateUserLocation(position) {
            const lat = position.coords.latitude;
            const lng = position.coords.longitude;
            
            // Create or update user marker
            if (!userMarker) {
                // Create a custom icon for user location
                const locationIcon = L.divIcon({
                    className: 'location-dot',
                    iconSize: [16, 16]
                });
                
                userMarker = L.marker([lat, lng], {
                    icon: locationIcon,
                    zIndexOffset: 1000
                }).addTo(map);
                
                // Center map on first location update
                map.setView([lat, lng], 16);
            } else {
                userMarker.setLatLng([lat, lng]);
            }
        }
        
        function handleGeolocationError(error) {
            console.error('Geolocation error:', error.message);
            // Silent error, don't alert user
        }
        
        // Traffic light functions
        function loadTrafficLights() {
            fetch('db.php?action=getTrafficLights')
                .then(response => response.json())
                .then(data => {
                    // Vérifier si data est un tableau
                    if (Array.isArray(data)) {
                        for (let i = 0; i < data.length; i++) {
                            addLightToMap(data[i]);
                            addLightToSelect(data[i]);
                        }
                    } else {
                        console.error('Data is not an array:', data);
                    }
                })
                .catch(error => console.error('Error loading traffic lights:', error));
        }
        
        function addLightToMap(light) {
            const marker = L.marker([light.latitude, light.longitude])
                .addTo(map);
                
            marker.on('click', function() {
                showLightInfo(light);
            });
                
            markers[light.id] = {
                marker: marker,
                data: light
            };
            
            updateLightStatus(light);
        }
        
        function addLightToSelect(light) {
            const option = document.createElement('option');
            option.value = light.id;
            option.textContent = light.name;
            document.getElementById('selectLight').appendChild(option);
        }
        
        function updateLightStatus(light) {
            const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
            const currentTime = Math.floor(Date.now() / 1000);
            const timeInCycle = currentTime % totalCycle;
            
            let status, timeLeft, statusClass;
            if (timeInCycle < light.red_duration) {
                status = 'RED';
                timeLeft = light.red_duration - timeInCycle;
                statusClass = 'bg-red-100 text-red-800 border-red-200';
            } else {
                status = 'GREEN';
                timeLeft = totalCycle - timeInCycle;
                statusClass = 'bg-green-100 text-green-800 border-green-200';
            }
            
            // Update marker icon based on status
            if (markers[light.id]) {
                const markerIcon = L.divIcon({
                    className: `marker-icon ${status === 'RED' ? 'red-light' : 'green-light'}`,
                    html: `<div style="width: 24px; height: 24px; background-color: ${status === 'RED' ? 'red' : 'green'}; border-radius: 50%; border: 2px solid white;"></div>`,
                    iconSize: [24, 24],
                    iconAnchor: [12, 12]
                });
                
                markers[light.id].marker.setIcon(markerIcon);
                markers[light.id].data = light;
            }
            
            // Update info modal if open and showing this light
            if (lightInfoModal.style.display === 'block' && selectedLightId == light.id) {
                document.getElementById('lightInfoStatus').className = statusClass;
                document.getElementById('lightInfoStatus').innerHTML = `
                    <div class="font-medium">Current Status: ${status}</div>
                    <div>Changes in: ${timeLeft} seconds</div>
                `;
            }
            
            // Update measure modal if this light is selected
            const selectedLight = document.getElementById('selectLight').value;
            if (selectedLight == light.id) {
                const statusElement = document.getElementById('lightStatus');
                if (status === 'RED') {
                    statusElement.innerHTML = `<span class="px-3 py-1 bg-red-500 text-white rounded">RED (${timeLeft}s)</span>`;
                } else {
                    statusElement.innerHTML = `<span class="px-3 py-1 bg-green-500 text-white rounded">GREEN (${timeLeft}s)</span>`;
                }
            }
        }
        
        function showLightInfo(light) {
            selectedLightId = light.id;
            
            document.getElementById('lightInfoTitle').textContent = light.name;
            document.getElementById('lightInfoContent').innerHTML = `
                <div class="grid grid-cols-2 gap-2">
                    <div class="text-sm text-gray-600">Direction:</div>
                    <div>${light.direction}</div>
                    
                    <div class="text-sm text-gray-600">Red Duration:</div>
                    <div>${light.red_duration} seconds</div>
                    
                    <div class="text-sm text-gray-600">Green Duration:</div>
                    <div>${light.green_duration} seconds</div>
                    
                    <div class="text-sm text-gray-600">Location:</div>
                    <div>${light.latitude}, ${light.longitude}</div>
                </div>
            `;
            
            // Calculate and show current status
            const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
            const currentTime = Math.floor(Date.now() / 1000);
            const timeInCycle = currentTime % totalCycle;
            
            let status, timeLeft, statusClass;
            if (timeInCycle < light.red_duration) {
                status = 'RED';
                timeLeft = light.red_duration - timeInCycle;
                statusClass = 'bg-red-100 text-red-800 border-red-200';
            } else {
                status = 'GREEN';
                timeLeft = totalCycle - timeInCycle;
                statusClass = 'bg-green-100 text-green-800 border-green-200';
            }
            
            document.getElementById('lightInfoStatus').className = statusClass;
            document.getElementById('lightInfoStatus').innerHTML = `
                <div class="font-medium">Current Status: ${status}</div>
                <div>Changes in: ${timeLeft} seconds</div>
            `;
            
            // Show the modal
            lightInfoModal.style.display = 'block';
            
            // Set up measure button
            document.getElementById('measureThisLight').onclick = function() {
                lightInfoModal.style.display = 'none';
                measureModal.style.display = 'block';
                document.getElementById('selectLight').value = light.id;
                
                // Update status display
                const statusElement = document.getElementById('lightStatus');
                if (status === 'RED') {
                    statusElement.innerHTML = `<span class="px-3 py-1 bg-red-500 text-white rounded">RED (${timeLeft}s)</span>`;
                } else {
                    statusElement.innerHTML = `<span class="px-3 py-1 bg-green-500 text-white rounded">GREEN (${timeLeft}s)</span>`;
                }
            };
        }
        
        // Form handlers
        document.getElementById('addTrafficLightForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const lightData = {
                name: document.getElementById('lightName').value,
                latitude: document.getElementById('latitude').value,
                longitude: document.getElementById('longitude').value,
                direction: document.getElementById('direction').value,
                red_duration: document.getElementById('redDuration').value,
                green_duration: document.getElementById('greenDuration').value
            };
            
            fetch('db.php?action=addTrafficLight', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(lightData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    addLightToMap(data.light);
                    addLightToSelect(data.light);
                    document.getElementById('addTrafficLightForm').reset();
                    addLightModal.style.display = 'none';
                    
                    if (tempMarker) {
                        map.removeLayer(tempMarker);
                        tempMarker = null;
                    }
                    
                    selectingLocation = false;
                }
            })
            .catch(error => console.error('Error adding traffic light:', error));
        });
        
        // Measure timing handlers
        let measureStartTime = 0;
        
        document.getElementById('startMeasure').addEventListener('click', function() {
            measureStartTime = Date.now();
            document.getElementById('measureResult').textContent = 'Measuring... Press Green button when light changes';
        });
        
        document.getElementById('stopMeasure').addEventListener('click', function() {
            if (measureStartTime === 0) return;
            
            const duration = Math.floor((Date.now() - measureStartTime) / 1000);
            document.getElementById('measureResult').textContent = `Measured: ${duration} seconds. Click Save to update.`;
        });
        
        document.getElementById('saveMeasure').addEventListener('click', function() {
            const lightId = document.getElementById('selectLight').value;
            if (!lightId) return;
            
            const duration = Math.floor((Date.now() - measureStartTime) / 1000);
            if (duration <= 0) return;
            
            fetch('db.php?action=updateTiming', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    id: lightId,
                    red_duration: duration
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('measureResult').textContent = 'Timing updated successfully!';
                    measureStartTime = 0;
                }
            })
            .catch(error => console.error('Error updating timing:', error));
        });
        
        document.getElementById('selectLight').addEventListener('change', function() {
            const lightId = this.value;
            if (!lightId) return;
            
            fetch(`db.php?action=getTrafficLight&id=${lightId}`)
                .then(response => response.json())
                .then(light => {
                    updateLightStatus(light);
                })
                .catch(error => console.error('Error getting traffic light:', error));
        });
        
        // Real-time updates
        setInterval(() => {
            fetch('db.php?action=getTrafficLights')
                .then(response => response.json())
                .then(data => {
                    // Vérifier si data est un tableau
                    if (Array.isArray(data)) {
                        for (let i = 0; i < data.length; i++) {
                            updateLightStatus(data[i]);
                        }
                    } else {
                        console.error('Data is not an array:', data);
                    }
                })
                .catch(error => console.error('Error updating traffic lights:', error));
        }, 1000);
    </script>
</body>
</html>