document.addEventListener('DOMContentLoaded', () => {
    // Load environment variables
    const env = {
        mapboxToken: process.env.MAPBOX_API_KEY || 'pk.eyJ1IjoiZm9yY2hhbmdlIiwiYSI6ImNtOWhhb2xoMTAzcGkyanF6azlhdHFsanYifQ.tr0iC93JexDq1rU8k8bQVQ',
        apiBaseUrl: '/api'
    };

    const app = {
        map: null,
        markers: {},
        userMarker: null,
        tempMarker: null,
        selectedLightId: null,
        userWatchId: null,
        refreshTimerId: null,
        rafId: null,
        lastTimestamp: 0,
        settings: {
            darkMode: false,
            notifications: true,
            autoRefresh: true,
            predictions: true,
            autoMeasure: true,
            voiceAlert: false,
            defaultZoom: 17,
            mapStyle: 'mapbox://styles/mapbox/streets-v12',
            defaultTransport: 'driving'
        },
        measure: {
            timer: 0,
            startTime: 0,
            rafId: null,
            mode: null,
            lightId: null
        },
        state: {
            selectingLocation: false,
            isLoggedIn: false,
            currentUser: null,
            pendingLights: [],
            tileLayer: null,
            directions: null,
            navigationActive: false,
            routeInfo: null
        }
    };

    const dom = {
        map: document.getElementById('map'),
        sidebar: document.querySelector('.sidebar'),
        lightsList: document.getElementById('lightsList'),
        loadingLights: document.getElementById('loadingLights'),
        noLightsMessage: document.getElementById('noLightsMessage'),
        lightPopup: document.getElementById('lightPopup'),
        popupTitle: document.getElementById('popupTitle'),
        popupContent: document.getElementById('popupContent'),
        popupStatus: document.getElementById('popupStatus'),
        closePopup: document.getElementById('closePopup'),
        popupNavigate: document.getElementById('popupNavigate'),
        popupMeasure: document.getElementById('popupMeasure'),
        searchDirections: document.getElementById('searchDirections'),
        directionsContainer: document.getElementById('directionsContainer'),
        routeModeButtons: document.querySelectorAll('.route-mode-btn'),
        closeDirections: document.getElementById('closeDirections'),
        routeInfoPanel: document.getElementById('routeInfoPanel'),
        routeTitle: document.getElementById('routeTitle'),
        routeModeIcon: document.getElementById('routeModeIcon'),
        routeDistance: document.getElementById('routeDistance'),
        routeDuration: document.getElementById('routeDuration'),
        trafficLightsCount: document.getElementById('trafficLightsCount'),
        arrivalTime: document.getElementById('arrivalTime'),
        arrivalTimeWithTraffic: document.getElementById('arrivalTimeWithTraffic'),
        nextLightStatus: document.getElementById('nextLightStatus'),
        lightsOnRoute: document.getElementById('lightsOnRoute'),
        startNavigation: document.getElementById('startNavigation'),
        closeRouteInfo: document.getElementById('closeRouteInfo'),
        userPanel: document.getElementById('userPanel'),
        userProfileBtn: document.getElementById('userProfileBtn'),
        modeToggle: document.getElementById('modeToggle'),
        routeModeBtn: document.getElementById('routeModeBtn'),
        measureModeBtn: document.getElementById('measureModeBtn'),
        modals: {
            add: document.getElementById('addModal'),
            measure: document.getElementById('measureModal'),
            settings: document.getElementById('settingsModal'),
            login: document.getElementById('loginModal'),
            register: document.getElementById('registerModal')
        },
        overlays: document.querySelectorAll('.modal-overlay'),
        modalBodies: document.querySelectorAll('.modal-body'),
        draggableHandles: document.querySelectorAll('.draggable-handle'),
        closeButtons: document.querySelectorAll('.close-modal'),
        loadingScreen: document.getElementById('loadingScreen'),
        certificationBanner: document.getElementById('certificationBanner'),
        buttons: {
            menuToggle: document.getElementById('menuToggle'),
            headerLocate: document.getElementById('headerLocateBtn'),
            headerNav: document.getElementById('headerNavBtn'),
            addLight: document.getElementById('addLightBtn'),
            showRoute: document.getElementById('showRouteBtn'),
            settings: document.getElementById('sidebarSettingsBtn'),
            nearby: document.getElementById('nearbyBtn'),
            resetApp: document.getElementById('resetAppBtn'),
            measureRed: document.getElementById('measureRedBtn'),
            measureGreen: document.getElementById('measureGreenBtn'),
            startTimer: document.getElementById('startTimer'),
            stopTimer: document.getElementById('stopTimer'),
            saveTimer: document.getElementById('saveTimer'),
            dismissCertBanner: document.getElementById('dismissCertBanner'),
            login: document.getElementById('loginBtn'),
            register: document.getElementById('registerBtn'),
            logout: document.getElementById('logoutBtn'),
            editProfile: document.getElementById('editProfileBtn'),
            goToRegister: document.getElementById('goToRegister'),
            goToLogin: document.getElementById('goToLogin')
        },
        form: {
            addLight: document.getElementById('addLightForm'),
            searchInput: document.getElementById('searchInput'),
            login: document.getElementById('loginForm'),
            register: document.getElementById('registerForm')
        },
        measure: {
            title: document.getElementById('measureTitle'),
            status: document.getElementById('measureStatus'),
            container: document.getElementById('timerContainer'),
            instructions: document.getElementById('timerInstructions'),
            display: document.getElementById('timerDisplay'),
            result: document.getElementById('measureResult')
        },
        settings: {
            darkMode: document.getElementById('darkModeToggle'),
            notifications: document.getElementById('notificationsToggle'),
            autoRefresh: document.getElementById('autoRefreshToggle'),
            predictions: document.getElementById('predictionToggle'),
            autoMeasure: document.getElementById('autoMeasureToggle'),
            voiceAlert: document.getElementById('voiceAlertToggle'),
            defaultZoom: document.getElementById('defaultZoom'),
            mapStyle: document.getElementById('mapStyle'),
            defaultTransport: document.getElementById('defaultTransport')
        },
        user: {
            loggedInSection: document.getElementById('loggedInSection'),
            loggedOutSection: document.getElementById('loggedOutSection'),
            displayName: document.getElementById('userDisplayName'),
            email: document.getElementById('userEmail')
        }
    };

    init();

    function init() {
        loadSettings();
        checkUserSession();
        initMap();
        bindEvents();
        setupDraggableModals();
        
        setTimeout(() => {
            dom.loadingScreen.style.opacity = '0';
            setTimeout(() => {
                dom.loadingScreen.style.display = 'none';
            }, 500);
        }, 800);
    }

    function loadSettings() {
        try {
            const saved = localStorage.getItem('trafficLightSettings');
            if (saved) {
                const parsed = JSON.parse(saved);
                app.settings = { ...app.settings, ...parsed };
                applySettings();
            }
        } catch (e) {
            console.error('Error loading settings');
        }
    }

    function applySettings() {
        // Apply settings to UI elements
        dom.settings.darkMode.checked = app.settings.darkMode;
        dom.settings.notifications.checked = app.settings.notifications;
        dom.settings.autoRefresh.checked = app.settings.autoRefresh;
        dom.settings.predictions.checked = app.settings.predictions;
        dom.settings.autoMeasure.checked = app.settings.autoMeasure;
        dom.settings.voiceAlert.checked = app.settings.voiceAlert;
        dom.settings.defaultZoom.value = app.settings.defaultZoom;
        dom.settings.mapStyle.value = app.settings.mapStyle;
        dom.settings.defaultTransport.value = app.settings.defaultTransport;
        
        if (app.settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
        } else {
            document.documentElement.classList.remove('dark');
            document.body.classList.remove('dark-mode');
        }
    }

    function checkUserSession() {
        // Check if user is logged in
        const userSession = localStorage.getItem('trafficLightUser');
        if (userSession) {
            try {
                const userData = JSON.parse(userSession);
                if (userData && userData.token) {
                    loginUser(userData);
                }
            } catch (e) {
                console.error('Error parsing user session');
            }
        }
    }

    function loginUser(userData) {
        app.state.isLoggedIn = true;
        app.state.currentUser = userData;
        
        // Update UI
        dom.user.loggedOutSection.classList.add('hidden');
        dom.user.loggedInSection.classList.remove('hidden');
        dom.user.displayName.textContent = userData.name;
        dom.user.email.textContent = userData.email;
        
        // Update user avatar
        dom.userProfileBtn.innerHTML = userData.name.charAt(0).toUpperCase();
        
        // Get user-specific data
        fetchUserTrafficLights();
    }

    function logoutUser() {
        app.state.isLoggedIn = false;
        app.state.currentUser = null;
        
        // Update UI
        dom.user.loggedInSection.classList.add('hidden');
        dom.user.loggedOutSection.classList.remove('hidden');
        dom.userProfileBtn.innerHTML = '<i class="fas fa-user"></i>';
        
        // Clear user session
        localStorage.removeItem('trafficLightUser');
    }

    function initMap() {
        mapboxgl.accessToken = env.mapboxToken;
        
        app.map = new mapboxgl.Map({
            container: 'map',
            style: app.settings.mapStyle,
            center: [-0.127, 51.507], // London
            zoom: parseInt(app.settings.defaultZoom),
            attributionControl: false
        });
        
        app.map.addControl(new mapboxgl.AttributionControl(), 'bottom-right');
        app.map.addControl(new mapboxgl.NavigationControl({ showCompass: false }), 'bottom-right');
        
        // Initialize directions control
        app.state.directions = new MapboxDirections({
            accessToken: mapboxgl.accessToken,
            unit: 'metric',
            profile: app.settings.defaultTransport,
            alternatives: true,
            congestion: true,
            controls: {
                inputs: true,
                instructions: false,
                profileSwitcher: false
            },
            placeholderOrigin: 'Choose starting point',
            placeholderDestination: 'Choose destination'
        });
        
        // Append directions control to container instead of map
        dom.directionsContainer.appendChild(app.state.directions.onAdd(app.map));
        
        app.map.on('load', () => {
            loadTrafficLights();
            startGeolocation();
            setupDirectionsEvents();
            checkPendingLights();
        });
        
        app.map.on('click', handleMapClick);
    }

    function setupDirectionsEvents() {
        // Listen to route updates
        app.state.directions.on('route', e => {
            if (e && e.route && e.route[0]) {
                updateRouteInfo(e.route[0]);
            }
        });
        
        app.state.directions.on('clear', () => {
            hideRouteInfoPanel();
        });
    }

    function updateRouteInfo(route) {
        if (!route) return;
        
        // Update route info UI
        const distance = (route.distance / 1000).toFixed(1);
        const duration = Math.round(route.duration / 60);
        
        dom.routeDistance.textContent = `${distance} km`;
        dom.routeDuration.textContent = `${duration} min`;
        
        // Set route mode icon
        const modeIcons = {
            'driving': 'fa-car',
            'walking': 'fa-walking',
            'cycling': 'fa-bicycle'
        };
        const currentMode = app.settings.defaultTransport;
        dom.routeModeIcon.className = `fas ${modeIcons[currentMode] || 'fa-car'}`;
        
        // Calculate estimated arrival time
        const now = new Date();
        const arrivalDate = new Date(now.getTime() + route.duration * 1000);
        const hours = arrivalDate.getHours().toString().padStart(2, '0');
        const mins = arrivalDate.getMinutes().toString().padStart(2, '0');
        dom.arrivalTime.textContent = `${hours}:${mins}`;
        
        // Find traffic lights on route
        findTrafficLightsOnRoute(route.geometry.coordinates);
        
        // Show route info panel
        dom.routeInfoPanel.classList.remove('hidden');
    }

    function findTrafficLightsOnRoute(coordinates) {
        // Buffer distance (in meters) around the route to consider traffic lights
        const bufferDistance = 50;
        
        const lightsOnRoute = [];
        let nextLight = null;
        let minDistance = Infinity;
        
        // Check all traffic lights
        for (const id in app.markers) {
            const light = app.markers[id].data;
            const lightCoords = [parseFloat(light.longitude), parseFloat(light.latitude)];
            
            // Check if light is close to any point on the route
            for (const point of coordinates) {
                const distance = calculateDistance(
                    lightCoords[1], lightCoords[0],
                    point[1], point[0]
                ) * 1000; // Convert to meters
                
                if (distance < bufferDistance) {
                    lightsOnRoute.push({
                        id: id,
                        light: light,
                        distance: distance
                    });
                    
                    // Check if this is the closest upcoming light
                    if (distance < minDistance) {
                        minDistance = distance;
                        nextLight = light;
                    }
                    
                    break;
                }
            }
        }
        
        // Update UI with found traffic lights
        dom.trafficLightsCount.textContent = `${lightsOnRoute.length} lights`;
        updateLightsOnRouteUI(lightsOnRoute);
        
        // Update next light status
        if (nextLight) {
            const status = getLightStatus(nextLight);
            dom.nextLightStatus.textContent = `Next light: ${status.label} (${status.timeLeft}s)`;
            dom.nextLightStatus.className = `text-sm text-traffic-${status.color}`;
        } else {
            dom.nextLightStatus.textContent = 'No traffic lights ahead';
            dom.nextLightStatus.className = 'text-sm text-gray-500';
        }
        
        // Store route info for later use
        app.state.routeInfo = {
            coordinates: coordinates,
            lightsOnRoute: lightsOnRoute,
            nextLight: nextLight
        };
    }

    function updateLightsOnRouteUI(lights) {
        dom.lightsOnRoute.innerHTML = '';
        
        if (lights.length === 0) {
            const emptyEl = document.createElement('p');
            emptyEl.className = 'text-sm text-gray-500 text-center';
            emptyEl.textContent = 'No traffic lights on this route';
            dom.lightsOnRoute.appendChild(emptyEl);
            return;
        }
        
        // Sort lights by distance
        lights.sort((a, b) => a.distance - b.distance);
        
        // Display up to 5 closest lights
        lights.slice(0, 5).forEach(item => {
            const light = item.light;
            const status = getLightStatus(light);
            const lightEl = document.createElement('div');
            lightEl.className = 'flex justify-between items-center p-2 bg-gray-50 rounded-lg';
            lightEl.innerHTML = `
                <div class="flex items-center">
                    <div class="w-3 h-3 rounded-full bg-traffic-${status.color} mr-2"></div>
                    <span class="text-sm font-medium">${light.name}</span>
                </div>
                <span class="text-xs text-gray-500">${status.timeLeft}s</span>
            `;
            dom.lightsOnRoute.appendChild(lightEl);
        });
    }

    function hideRouteInfoPanel() {
        dom.routeInfoPanel.classList.add('hidden');
        app.state.routeInfo = null;
    }

    function startNavigation() {
        if (!app.state.routeInfo) return;
        
        app.state.navigationActive = true;
        dom.startNavigation.innerHTML = '<i class="fas fa-stop"></i> Stop Navigation';
        
        // Enable auto-measure if setting is on
        if (app.settings.autoMeasure) {
            enableAutoMeasure();
        }
        
        // Show notification
        if (app.settings.notifications) {
            showNotification('Navigation started. Drive safely!', 'success');
        }
        
        // Start monitoring position for navigation
        startNavigationMonitoring();
    }

    function stopNavigation() {
        app.state.navigationActive = false;
        dom.startNavigation.innerHTML = '<i class="fas fa-play"></i> Start Navigation';
        
        // Disable auto-measure
        disableAutoMeasure();
        
        // Show notification
        if (app.settings.notifications) {
            showNotification('Navigation stopped', 'info');
        }
    }

    function startNavigationMonitoring() {
        // Logic to monitor user position during navigation
        // and update UI with upcoming traffic lights, etc.
    }

    function enableAutoMeasure() {
        // Set up auto-measurement of traffic lights
        // This will be activated when vehicle stops for 2+ seconds
    }

    function disableAutoMeasure() {
        // Disable auto-measurement of traffic lights
    }

    function saveSettings() {
        app.settings.darkMode = dom.settings.darkMode.checked;
        app.settings.notifications = dom.settings.notifications.checked;
        app.settings.autoRefresh = dom.settings.autoRefresh.checked;
        app.settings.predictions = dom.settings.predictions.checked;
        app.settings.autoMeasure = dom.settings.autoMeasure.checked;
        app.settings.voiceAlert = dom.settings.voiceAlert.checked;
        app.settings.defaultZoom = dom.settings.defaultZoom.value;
        app.settings.mapStyle = dom.settings.mapStyle.value;
        app.settings.defaultTransport = dom.settings.defaultTransport.value;
        
        try {
            localStorage.setItem('trafficLightSettings', JSON.stringify(app.settings));
        } catch (e) {
            console.error('Error saving settings');
        }
    }

    function checkPendingLights() {
        // Check if there are any pending lights
        const pendingLights = localStorage.getItem('trafficLightPendingLights');
        if (pendingLights) {
            try {
                app.state.pendingLights = JSON.parse(pendingLights);
                if (app.state.pendingLights.length > 0) {
                    showCertificationBanner();
                }
            } catch (e) {
                console.error('Error parsing pending lights');
            }
        }
    }

    function bindEvents() {
        dom.buttons.menuToggle.addEventListener('click', toggleSidebar);
        dom.buttons.headerLocate.addEventListener('click', locateUser);
        dom.buttons.headerNav.addEventListener('click', toggleDirections);
        dom.buttons.addLight.addEventListener('click', () => openModal(dom.modals.add));
        dom.buttons.settings.addEventListener('click', () => openModal(dom.modals.settings));
        dom.buttons.showRoute.addEventListener('click', toggleDirections);
        dom.closeDirections.addEventListener('click', hideDirections);
        dom.form.searchInput.addEventListener('input', handleSearch);
        dom.userProfileBtn.addEventListener('click', handleUserProfile);
        
        dom.closePopup.addEventListener('click', () => dom.lightPopup.classList.add('hidden'));
        dom.popupNavigate.addEventListener('click', navigateToSelectedLight);
        dom.popupMeasure.addEventListener('click', () => {
            dom.lightPopup.classList.add('hidden');
            openMeasureModal(app.selectedLightId);
        });
        
        dom.closeButtons.forEach(btn => {
            btn.addEventListener('click', () => {
                const modal = btn.closest('.modal');
                closeModal(modal);
            });
        });
        
        dom.closeRouteInfo.addEventListener('click', hideRouteInfoPanel);
        dom.startNavigation.addEventListener('click', toggleNavigation);
        
        dom.routeModeButtons.forEach(btn => {
            btn.addEventListener('click', () => {
                const mode = btn.getAttribute('data-mode');
                changeRouteMode(mode);
            });
        });
        
        dom.routeModeBtn.addEventListener('click', () => setActiveMode('route'));
        dom.measureModeBtn.addEventListener('click', () => setActiveMode('measure'));
        
        dom.settings.darkMode.addEventListener('change', toggleDarkMode);
        dom.settings.autoRefresh.addEventListener('change', toggleAutoRefresh);
        dom.settings.mapStyle.addEventListener('change', changeMapStyle);
        dom.settings.notifications.addEventListener('change', saveSettings);
        dom.settings.predictions.addEventListener('change', () => {
            saveSettings();
            updateAllLightStatus();
        });
        dom.settings.autoMeasure.addEventListener('change', saveSettings);
        dom.settings.voiceAlert.addEventListener('change', saveSettings);
        dom.settings.defaultTransport.addEventListener('change', () => {
            saveSettings();
            changeRouteMode(app.settings.defaultTransport);
        });
        dom.settings.defaultZoom.addEventListener('change', saveSettings);
        
        dom.buttons.resetApp.addEventListener('click', resetAppData);
        
        dom.buttons.measureRed.addEventListener('click', () => startMeasure('red'));
        dom.buttons.measureGreen.addEventListener('click', () => startMeasure('green'));
        dom.buttons.startTimer.addEventListener('click', startMeasureTimer);
        dom.buttons.stopTimer.addEventListener('click', stopMeasureTimer);
        dom.buttons.saveTimer.addEventListener('click', saveMeasureTimer);
        
        dom.buttons.dismissCertBanner.addEventListener('click', () => {
            dom.certificationBanner.classList.add('hidden');
        });
        
        dom.buttons.login.addEventListener('click', () => openModal(dom.modals.login));
        dom.buttons.register.addEventListener('click', () => openModal(dom.modals.register));
        dom.buttons.logout.addEventListener('click', handleLogout);
        dom.buttons.goToRegister.addEventListener('click', () => {
            closeModal(dom.modals.login);
            openModal(dom.modals.register);
        });
        dom.buttons.goToLogin.addEventListener('click', () => {
            closeModal(dom.modals.register);
            openModal(dom.modals.login);
        });
        
        dom.form.addLight.addEventListener('submit', handleAddLight);
        dom.form.login.addEventListener('submit', handleLogin);
        dom.form.register.addEventListener('submit', handleRegister);
        
        window.addEventListener('online', handleOnline);
        window.addEventListener('offline', handleOffline);
        document.addEventListener('visibilitychange', handleVisibilityChange);
    }

    function toggleSidebar() {
        dom.sidebar.classList.toggle('active');
        dom.buttons.menuToggle.innerHTML = dom.sidebar.classList.contains('active') 
            ? '<i class="fas fa-times"></i>' 
            : '<i class="fas fa-bars"></i>';
    }

    function locateUser() {
        if (app.userMarker) {
            app.map.flyTo({
                center: [app.userMarker.getLngLat().lng, app.userMarker.getLngLat().lat],
                zoom: 16,
                speed: 1.5
            });
        } else {
            startGeolocation();
        }
    }

    function toggleDirections() {
        if (dom.searchDirections.classList.contains('hidden')) {
            showDirections();
        } else {
            hideDirections();
        }
    }

    function showDirections() {
        dom.searchDirections.classList.remove('hidden');
        
        // Set origin to user location if available
        if (app.userMarker) {
            const userPos = app.userMarker.getLngLat();
            app.state.directions.setOrigin([userPos.lng, userPos.lat]);
        }
    }

    function hideDirections() {
        dom.searchDirections.classList.add('hidden');
    }

    function changeRouteMode(mode) {
        // Update UI
        dom.routeModeButtons.forEach(btn => {
            if (btn.getAttribute('data-mode') === mode) {
                btn.classList.add('active');
            } else {
                btn.classList.remove('active');
            }
        });
        
        // Set directions profile
        app.state.directions.setProfile(mode);
        app.settings.defaultTransport = mode;
        saveSettings();
        
        // Recalculate route if we have origin and destination
        app.state.directions.query();
    }

    function toggleNavigation() {
        if (app.state.navigationActive) {
            stopNavigation();
        } else {
            startNavigation();
        }
    }

    function handleUserProfile() {
        if (app.state.isLoggedIn) {
            openModal(dom.modals.settings);
        } else {
            openModal(dom.modals.login);
        }
    }

    function setActiveMode(mode) {
        if (mode === 'route') {
            showDirections();
        } else if (mode === 'measure') {
            // TODO: Implement measure mode
        }
    }

    function setupDraggableModals() {
        dom.draggableHandles.forEach(handle => {
            const modal = handle.closest('.modal-body');
            let startY, currentY, initialY;
            let isDragging = false;
            
            handle.addEventListener('touchstart', e => {
                startY = e.touches[0].clientY;
                initialY = 0;
                isDragging = true;
                modal.style.transition = 'none';
            }, { passive: true });
            
            handle.addEventListener('touchmove', e => {
                if (!isDragging) return;
                currentY = e.touches[0].clientY;
                const diffY = currentY - startY;
                if (diffY < 0) return;
                
                modal.style.transform = `translateY(${diffY}px)`;
            }, { passive: true });
            
            handle.addEventListener('touchend', e => {
                modal.style.transition = 'transform 0.3s ease';
                if (!isDragging) return;
                
                const diffY = currentY - startY;
                if (diffY > 100) {
                    modal.style.transform = 'translateY(100%)';
                    const modalContainer = modal.closest('.modal');
                    setTimeout(() => {
                        modalContainer.style.display = 'none';
                        modal.style.transform = 'translateY(0)';
                    }, 300);
                } else {
                    modal.style.transform = 'translateY(0)';
                }
                
                isDragging = false;
            }, { passive: true });
        });
        
        dom.overlays.forEach(overlay => {
            overlay.addEventListener('click', () => {
                const modal = overlay.closest('.modal');
                closeModal(modal);
            });
        });
    }

    function handleMapClick(e) {
        if (app.state.selectingLocation) {
            document.getElementById('latitude').value = e.lngLat.lat.toFixed(6);
            document.getElementById('longitude').value = e.lngLat.lng.toFixed(6);
            
            if (app.tempMarker) app.tempMarker.remove();
            
            app.tempMarker = new mapboxgl.Marker({
                draggable: true,
                color: '#10b981'
            })
            .setLngLat([e.lngLat.lng, e.lngLat.lat])
            .addTo(app.map);
            
            app.tempMarker.on('dragend', updateMarkerPosition);
        }
    }

    function updateMarkerPosition() {
        const pos = app.tempMarker.getLngLat();
        document.getElementById('latitude').value = pos.lat.toFixed(6);
        document.getElementById('longitude').value = pos.lng.toFixed(6);
    }

    function handleSearch() {
        const searchTerm = dom.form.searchInput.value.toLowerCase();
        const lights = document.querySelectorAll('.light-card');
        
        let hasResults = false;
        
        lights.forEach(light => {
            const name = light.getAttribute('data-name').toLowerCase();
            const visible = name.includes(searchTerm);
            light.style.display = visible ? 'block' : 'none';
            if (visible) hasResults = true;
        });
        
        if (lights.length > 0 && !hasResults) {
            dom.noLightsMessage.classList.remove('hidden');
            dom.noLightsMessage.querySelector('p').textContent = 'No results found';
        } else {
            dom.noLightsMessage.classList.add('hidden');
        }
    }

    function toggleDarkMode() {
        app.settings.darkMode = dom.settings.darkMode.checked;
        
        if (app.settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
            
            // Switch to dark map if not already
            if (!dom.settings.mapStyle.value.includes('night') && !dom.settings.mapStyle.value.includes('dark')) {
                const previousStyle = dom.settings.mapStyle.value;
                dom.settings.mapStyle.value = 'mapbox://styles/mapbox/dark-v11';
                changeMapStyle();
                dom.settings.mapStyle.dataset.previousStyle = previousStyle;
            }
        } else {
            document.documentElement.classList.remove('dark');
            document.body.classList.remove('dark-mode');
            
            // Switch back to previous style
            if ((dom.settings.mapStyle.value.includes('night') || dom.settings.mapStyle.value.includes('dark')) 
                && dom.settings.mapStyle.dataset.previousStyle) {
                dom.settings.mapStyle.value = dom.settings.mapStyle.dataset.previousStyle;
                changeMapStyle();
            } else if (dom.settings.mapStyle.value.includes('dark') || dom.settings.mapStyle.value.includes('night')) {
                dom.settings.mapStyle.value = 'mapbox://styles/mapbox/streets-v12';
                changeMapStyle();
            }
        }
        
        saveSettings();
    }

    function changeMapStyle() {
        const style = dom.settings.mapStyle.value;
        app.settings.mapStyle = style;
        app.map.setStyle(style);
        saveSettings();
    }

    function toggleAutoRefresh() {
        app.settings.autoRefresh = dom.settings.autoRefresh.checked;
        
        if (app.settings.autoRefresh) {
            startStatusUpdates();
        } else {
            stopStatusUpdates();
        }
        
        saveSettings();
    }

    function resetAppData() {
        if (confirm('Are you sure you want to reset all app data? This will clear all your settings and cache.')) {
            localStorage.clear();
            showNotification('App data reset. Refreshing...', 'info');
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        }
    }

    function navigateToSelectedLight() {
        if (app.selectedLightId && app.markers[app.selectedLightId]) {
            const light = app.markers[app.selectedLightId].data;
            
            // Hide popup
            dom.lightPopup.classList.add('hidden');
            
            // Show directions and set destination
            showDirections();
            app.state.directions.setDestination([light.longitude, light.latitude]);
            
            // Set origin to user location if available
            if (app.userMarker) {
                const userPos = app.userMarker.getLngLat();
                app.state.directions.setOrigin([userPos.lng, userPos.lat]);
            }
        }
    }

    function handleLogin(e) {
        e.preventDefault();
        
        const email = document.getElementById('loginEmail').value;
        const password = document.getElementById('loginPassword').value;
        const rememberMe = document.getElementById('rememberMe').checked;
        
        // Show loading state
        const submitBtn = dom.form.login.querySelector('button[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Logging in...';
        submitBtn.disabled = true;
        
        fetch(`${env.apiBaseUrl}/login`, {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({ email, password })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const userData = {
                    token: data.token,
                    name: data.user.name,
                    email: data.user.email,
                    id: data.user.id
                };
                
                // Store user data
                if (rememberMe) {
                    // Store for 30 days
                    localStorage.setItem('trafficLightUser', JSON.stringify(userData));
                } else {
                    // Store for session only
                    sessionStorage.setItem('trafficLightUser', JSON.stringify(userData));
                }
                
                loginUser(userData);
                closeModal(dom.modals.login);
                
                if (app.settings.notifications) {
                    showNotification('Login successful!', 'success');
                }
            } else {
                if (app.settings.notifications) {
                    showNotification('Error: ' + data.message, 'error');
                }
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        })
        .catch(error => {
            console.error('Error:', error);
            if (app.settings.notifications) {
                showNotification('Connection error', 'error');
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }

    function handleRegister(e) {
        e.preventDefault();
        
        const name = document.getElementById('registerName').value;
        const email = document.getElementById('registerEmail').value;
        const password = document.getElementById('registerPassword').value;
        const confirmPassword = document.getElementById('confirmPassword').value;
        
        // Validate passwords match
        if (password !== confirmPassword) {
            showNotification('Passwords do not match', 'error');
            return;
        }
        
        // Show loading state
        const submitBtn = dom.form.register.querySelector('button[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Creating account...';
        submitBtn.disabled = true;
        
        fetch(`${env.apiBaseUrl}/register`, {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({ name, email, password })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const userData = {
                    token: data.token,
                    name: data.user.name,
                    email: data.user.email,
                    id: data.user.id
                };
                
                // Store user data
                localStorage.setItem('trafficLightUser', JSON.stringify(userData));
                
                loginUser(userData);
                closeModal(dom.modals.register);
                
                if (app.settings.notifications) {
                    showNotification('Account created successfully!', 'success');
                }
            } else {
                if (app.settings.notifications) {
                    showNotification('Error: ' + data.message, 'error');
                }
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        })
        .catch(error => {
            console.error('Error:', error);
            if (app.settings.notifications) {
                showNotification('Connection error', 'error');
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }

    function handleLogout() {
        logoutUser();
        
        if (app.settings.notifications) {
            showNotification('Logged out successfully', 'success');
        }
    }

    function openModal(modal) {
        if (modal === dom.modals.add) {
            app.state.selectingLocation = true;
            if (app.userMarker) {
                const pos = app.userMarker.getLngLat();
                document.getElementById('latitude').value = pos.lat.toFixed(6);
                document.getElementById('longitude').value = pos.lng.toFixed(6);
                
                if (app.tempMarker) app.tempMarker.remove();
                
                app.tempMarker = new mapboxgl.Marker({
                    draggable: true,
                    color: '#10b981'
                })
                .setLngLat([pos.lng, pos.lat])
                .addTo(app.map);
                
                app.tempMarker.on('dragend', updateMarkerPosition);
            }
        }
        
        modal.style.display = 'flex';
        const modalBody = modal.querySelector('.modal-body');
        modalBody.style.transform = 'translateY(0)';
    }

    function closeModal(modal) {
        const modalBody = modal.querySelector('.modal-body');
        modalBody.style.transition = 'transform 0.3s ease';
        modalBody.style.transform = 'translateY(100%)';
        
        setTimeout(() => {
            modal.style.display = 'none';
            modalBody.style.transform = 'translateY(0)';
            
            if (modal === dom.modals.add) {
                app.state.selectingLocation = false;
                if (app.tempMarker) {
                    app.tempMarker.remove();
                    app.tempMarker = null;
                }
            }
        }, 300);
    }

    function handleAddLight(e) {
        e.preventDefault();
        
        const submitBtn = dom.form.addLight.querySelector('[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Adding...';
        submitBtn.disabled = true;
        
        const lightData = {
            name: document.getElementById('lightName').value,
            latitude: document.getElementById('latitude').value,
            longitude: document.getElementById('longitude').value,
            direction: document.getElementById('direction').value,
            red_duration: document.getElementById('redDuration').value,
            green_duration: document.getElementById('greenDuration').value
        };
        
        // Add user ID if logged in
        if (app.state.isLoggedIn && app.state.currentUser) {
            lightData.user_id = app.state.currentUser.id;
        }
        
        fetch(`${env.apiBaseUrl}/traffic-lights`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': app.state.isLoggedIn ? `Bearer ${app.state.currentUser.token}` : undefined
            },
            body: JSON.stringify(lightData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Store light ID as pending
                app.state.pendingLights.push(data.light.id);
                localStorage.setItem('trafficLightPendingLights', JSON.stringify(app.state.pendingLights));
                
                showCertificationBanner();
                
                if (app.settings.notifications) {
                    showNotification('Traffic light added successfully! Pending certification.', 'success');
                }
                
                closeModal(dom.modals.add);
                dom.form.addLight.reset();
                
                if (app.tempMarker) {
                    app.tempMarker.remove();
                    app.tempMarker = null;
                }
                
                app.state.selectingLocation = false;
                
                addPendingLightToMap(data.light);
                addPendingLightToSidebar(data.light);
            } else {
                if (app.settings.notifications) {
                    showNotification('Error: ' + data.message, 'error');
                }
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        })
        .catch(error => {
            console.error('Error:', error);
            if (app.settings.notifications) {
                showNotification('Connection error', 'error');
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }

    function showCertificationBanner() {
        dom.certificationBanner.classList.remove('hidden');
        setTimeout(() => {
            dom.certificationBanner.classList.add('fade-in');
        }, 100);
    }

    function addPendingLightToMap(light) {
        const customEl = document.createElement('div');
        customEl.className = 'w-8 h-8 bg-gray-400 rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white';
        customEl.innerHTML = '<i class="fas fa-clock"></i>';
        
        const marker = new mapboxgl.Marker({
            element: customEl
        })
        .setLngLat([light.longitude, light.latitude])
        .addTo(app.map);
        
        light.status = 'pending';
        app.markers[light.id] = {
            marker: marker,
            data: light
        };
    }

    function addPendingLightToSidebar(light) {
        const card = document.createElement('div');
        card.className = 'light-card bg-white shadow-md p-4 mb-3';
        card.setAttribute('data-id', light.id);
        card.setAttribute('data-name', light.name);
        card.setAttribute('data-pending', 'true');
        
        card.innerHTML = `
            <div class="flex justify-between items-center">
                <h3 class="font-medium">${light.name}</h3>
                <span class="px-2.5 py-1 text-xs font-medium rounded-full bg-gray-200 text-gray-600 border border-gray-300">PENDING</span>
            </div>
            <div class="flex items-center text-sm text-gray-500 mt-2">
                <i class="fas fa-location-dot mr-2"></i>
                <span class="capitalize">${light.direction}</span>
                <div class="ml-auto flex items-center">
                    <i class="fas fa-clock mr-1.5"></i> Awaiting review
                </div>
            </div>
        `;
        
        card.addEventListener('click', function() {
            const allCards = document.querySelectorAll('.light-card');
            allCards.forEach(c => c.classList.remove('active'));
            this.classList.add('active');
            
            showLightDetail(light);
            
            if (window.innerWidth <= 768) {
                dom.sidebar.classList.remove('active');
                dom.buttons.menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
            }
        });
        
        dom.lightsList.insertBefore(card, dom.loadingLights);
        dom.noLightsMessage.classList.add('hidden');
    }

    function startMeasure(mode) {
        app.measure.mode = mode;
        dom.measure.container.classList.remove('hidden');
        
        if (mode === 'red') {
            dom.measure.instructions.textContent = 'Press "Start" when the light turns red, then "Stop" when it turns green.';
            dom.buttons.startTimer.className = 'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors';
            dom.buttons.stopTimer.className = 'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors opacity-50';
        } else {
            dom.measure.instructions.textContent = 'Press "Start" when the light turns green, then "Stop" when it turns red.';
            dom.buttons.startTimer.className = 'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors';
            dom.buttons.stopTimer.className = 'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors opacity-50';
        }
        
        resetMeasureTimer();
    }

    function startMeasureTimer() {
        app.measure.startTime = performance.now();
        app.measure.timer = 0;
        
        if (app.measure.rafId) cancelAnimationFrame(app.measure.rafId);
        
        function updateTimer(timestamp) {
            const elapsed = Math.floor((timestamp - app.measure.startTime) / 1000);
            if (elapsed !== app.measure.timer) {
                app.measure.timer = elapsed;
                const minutes = Math.floor(elapsed / 60);
                const seconds = elapsed % 60;
                dom.measure.display.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            }
            app.measure.rafId = requestAnimationFrame(updateTimer);
        }
        
        app.measure.rafId = requestAnimationFrame(updateTimer);
        dom.measure.display.textContent = '00:00';
        dom.measure.result.textContent = 'Measuring...';
        
        dom.buttons.startTimer.disabled = true;
        dom.buttons.startTimer.classList.add('opacity-50');
        dom.buttons.stopTimer.disabled = false;
        dom.buttons.stopTimer.classList.remove('opacity-50');
        dom.buttons.saveTimer.disabled = true;
        dom.buttons.saveTimer.classList.add('opacity-50');
    }

    function stopMeasureTimer() {
        if (app.measure.rafId) {
            cancelAnimationFrame(app.measure.rafId);
            app.measure.rafId = null;
            
            dom.measure.result.textContent = `Measured duration: ${app.measure.timer} seconds. Click Save to confirm.`;
            
            dom.buttons.stopTimer.disabled = true;
            dom.buttons.stopTimer.classList.add('opacity-50');
            dom.buttons.saveTimer.disabled = false;
            dom.buttons.saveTimer.classList.remove('opacity-50');
        }
    }

    function resetMeasureTimer() {
        if (app.measure.rafId) {
            cancelAnimationFrame(app.measure.rafId);
            app.measure.rafId = null;
        }
        app.measure.timer = 0;
        app.measure.startTime = 0;
        dom.measure.display.textContent = '00:00';
        dom.measure.result.textContent = '';
        dom.buttons.startTimer.disabled = false;
        dom.buttons.startTimer.classList.remove('opacity-50');
        dom.buttons.stopTimer.disabled = true;
        dom.buttons.stopTimer.classList.add('opacity-50');
        dom.buttons.saveTimer.disabled = true;
        dom.buttons.saveTimer.classList.add('opacity-50');
    }

    function saveMeasureTimer() {
        const lightId = dom.measure.title.getAttribute('data-id');
        
        if (!lightId || app.measure.timer <= 0 || !app.measure.mode) return;
        
        dom.measure.result.textContent = 'Saving...';
        
        const data = {
            id: lightId,
            duration_type: app.measure.mode,
            duration: app.measure.timer
        };
        
        // Add user ID if logged in
        if (app.state.isLoggedIn && app.state.currentUser) {
            data.user_id = app.state.currentUser.id;
        }
        
        fetch(`${env.apiBaseUrl}/traffic-lights/${lightId}/timing`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': app.state.isLoggedIn ? `Bearer ${app.state.currentUser.token}` : undefined
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                dom.measure.result.innerHTML = '<i class="fas fa-check text-green-500 mr-1"></i> Timing updated!';
                
                app.markers[lightId].data = data.light;
                updateLightStatus(data.light);
                
                dom.buttons.saveTimer.disabled = true;
                dom.buttons.saveTimer.classList.add('opacity-50');
                
                setTimeout(() => {
                    dom.measure.container.classList.add('hidden');
                    app.measure.mode = null;
                }, 1500);
            } else {
                dom.measure.result.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Error: ' + data.message;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            dom.measure.result.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Connection error';
        });
    }

    function openMeasureModal(lightId) {
        const light = app.markers[lightId].data;
        app.measure.lightId = lightId;
        
        dom.measure.title.textContent = light.name;
        dom.measure.title.setAttribute('data-id', lightId);
        
        const status = getLightStatus(light);
        dom.measure.status.innerHTML = `
            <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                ${status.label} (${status.timeLeft}s)
            </span>
        `;
        
        dom.measure.container.classList.add('hidden');
        app.measure.mode = null;
        resetMeasureTimer();
        
        openModal(dom.modals.measure);
    }

    function loadTrafficLights(isRefresh = false) {
        if (!isRefresh) {
            dom.loadingLights.style.display = 'block';
            dom.noLightsMessage.classList.add('hidden');
        }
        
        fetch(`${env.apiBaseUrl}/traffic-lights`)
            .then(response => response.json())
            .then(data => {
                dom.loadingLights.style.display = 'none';
                
                if (isRefresh) {
                    for (let id in app.markers) {
                        app.markers[id].marker.remove();
                        delete app.markers[id];
                    }
                    
                    const lightCards = document.querySelectorAll('.light-card');
                    lightCards.forEach(card => card.remove());
                }
                
                if (Array.isArray(data) && data.length > 0) {
                    data.forEach(light => {
                        addLightToMap(light);
                        addLightToSidebar(light);
                    });
                } else {
                    dom.noLightsMessage.classList.remove('hidden');
                }
                
                if (app.settings.autoRefresh) {
                    startStatusUpdates();
                }
            })
            .catch(error => {
                console.error('Error:', error);
                dom.loadingLights.style.display = 'none';
                dom.noLightsMessage.classList.remove('hidden');
                if (app.settings.notifications) {
                    showNotification('Error loading traffic lights', 'error');
                }
            });
    }

    function fetchUserTrafficLights() {
        if (!app.state.isLoggedIn || !app.state.currentUser) return;
        
        fetch(`${env.apiBaseUrl}/user/traffic-lights`, {
            headers: {
                'Authorization': `Bearer ${app.state.currentUser.token}`
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && Array.isArray(data.lights)) {
                // Store pending lights
                app.state.pendingLights = data.lights
                    .filter(light => light.status === 'pending')
                    .map(light => light.id);
                
                if (app.state.pendingLights.length > 0) {
                    showCertificationBanner();
                    localStorage.setItem('trafficLightPendingLights', JSON.stringify(app.state.pendingLights));
                }
            }
        })
        .catch(error => {
            console.error('Error fetching user traffic lights:', error);
        });
    }

    function addLightToMap(light) {
        const status = getLightStatus(light);
        
        const customEl = document.createElement('div');
        customEl.className = `w-8 h-8 bg-traffic-${status.color} rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white`;
        customEl.innerHTML = '<i class="fas fa-traffic-light"></i>';
        
        const marker = new mapboxgl.Marker({
            element: customEl
        })
        .setLngLat([light.longitude, light.latitude])
        .addTo(app.map);
        
        marker.getElement().addEventListener('click', () => showLightDetail(light));
        
        app.markers[light.id] = {
            marker: marker,
            data: light
        };
    }

    function addLightToSidebar(light) {
        const status = getLightStatus(light);
        
        const card = document.createElement('div');
        card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg';
        card.setAttribute('data-id', light.id);
        card.setAttribute('data-name', light.name);
        
        const showPrediction = app.settings.predictions;
        
        card.innerHTML = `
            <div class="flex justify-between items-center">
                <h3 class="font-medium">${light.name}</h3>
                <span id="status-${light.id}" class="px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white">${status.label}</span>
            </div>
            <div class="flex items-center text-sm text-gray-500 mt-2">
                <i class="fas fa-location-dot mr-2"></i>
                <span class="capitalize">${light.direction}</span>
                <div class="ml-auto flex items-center" id="timer-${light.id}">
                    <i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s
                </div>
            </div>
            ${showPrediction ? `
            <div class="mt-2 text-xs">
                <div class="status-progress">
                    <div class="status-bar bg-traffic-${status.color}" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                </div>
                <div class="flex justify-between mt-1 text-gray-400">
                    <span>0s</span>
                    <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                </div>
            </div>` : ''}
        `;
        
        card.addEventListener('click', function() {
            const allCards = document.querySelectorAll('.light-card');
            allCards.forEach(c => c.classList.remove('active'));
            this.classList.add('active');
            
            showLightDetail(light);
            
            if (window.innerWidth <= 768) {
                dom.sidebar.classList.remove('active');
                dom.buttons.menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
            }
        });
        
        dom.lightsList.insertBefore(card, dom.loadingLights);
    }

    function getLightStatus(light) {
        const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
        const currentTime = Math.floor(Date.now() / 1000);
        const timeInCycle = currentTime % totalCycle;
        
        if (timeInCycle < light.red_duration) {
            return {
                isRed: true,
                color: 'red',
                label: 'RED',
                timeLeft: light.red_duration - timeInCycle
            };
        } else {
            return {
                isRed: false,
                color: 'green',
                label: 'GREEN',
                timeLeft: totalCycle - timeInCycle
            };
        }
    }

    function updateLightStatus(light) {
        const status = getLightStatus(light);
        const showPrediction = app.settings.predictions;
        
        if (app.markers[light.id]) {
            const markerEl = app.markers[light.id].marker.getElement().querySelector('div');
            markerEl.className = `w-8 h-8 bg-traffic-${status.color} rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white`;
        }
        
        const statusElement = document.getElementById(`status-${light.id}`);
        const timerElement = document.getElementById(`timer-${light.id}`);
        
        if (statusElement) {
            statusElement.className = `px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white`;
            statusElement.textContent = status.label;
        }
        
        if (timerElement) {
            timerElement.innerHTML = `<i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s`;
        }
        
        const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
        if (card) {
            let predictionEl = card.querySelector('.mt-2.text-xs');
            
            if (showPrediction) {
                if (predictionEl) {
                    const progressBar = predictionEl.querySelector('.status-bar');
                    if (progressBar) {
                        progressBar.className = `status-bar bg-traffic-${status.color}`;
                        progressBar.style.width = `${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%`;
                    }
                } else {
                    predictionEl = document.createElement('div');
                    predictionEl.className = 'mt-2 text-xs';
                    predictionEl.innerHTML = `
                        <div class="status-progress">
                            <div class="status-bar bg-traffic-${status.color}" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                        </div>
                        <div class="flex justify-between mt-1 text-gray-400">
                            <span>0s</span>
                            <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                        </div>
                    `;
                    card.appendChild(predictionEl);
                }
            } else if (!showPrediction && predictionEl) {
                predictionEl.remove();
            }
        }
        
        if (app.selectedLightId === light.id && !dom.lightPopup.classList.contains('hidden')) {
            const popupStatus = dom.popupStatus;
            popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
            popupStatus.innerHTML = `
                <div class="text-xl font-semibold mb-1">${status.label}</div>
                <div class="flex justify-center items-center gap-2">
                    <i class="fas fa-clock"></i>
                    <span>Changes in ${status.timeLeft} seconds</span>
                </div>
                ${showPrediction ? `
                <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                    <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                </div>
                ` : ''}
            `;
        }
        
        if (app.measure.lightId === light.id && dom.modals.measure.style.display === 'flex') {
            dom.measure.status.innerHTML = `
                <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                    ${status.label} (${status.timeLeft}s)
                </span>
            `;
        }
        
        // Update route info if this light is on the current route
        if (app.state.routeInfo && app.state.routeInfo.lightsOnRoute) {
            const lightOnRoute = app.state.routeInfo.lightsOnRoute.find(item => item.id === light.id);
            if (lightOnRoute) {
                updateLightsOnRouteUI(app.state.routeInfo.lightsOnRoute);
                
                // Update next light status if this is the next light
                if (app.state.routeInfo.nextLight && app.state.routeInfo.nextLight.id === light.id) {
                    dom.nextLightStatus.textContent = `Next light: ${status.label} (${status.timeLeft}s)`;
                    dom.nextLightStatus.className = `text-sm text-traffic-${status.color}`;
                }
            }
        }
    }

    function updateAllLightStatus() {
        for (const id in app.markers) {
            updateLightStatus(app.markers[id].data);
        }
    }

    function startStatusUpdates() {
        stopStatusUpdates();
        
        function updateStatuses(timestamp) {
            if (!app.lastTimestamp || timestamp - app.lastTimestamp > 1000) {
                app.lastTimestamp = timestamp;
                updateAllLightStatus();
            }
            app.rafId = requestAnimationFrame(updateStatuses);
        }
        
        app.rafId = requestAnimationFrame(updateStatuses);
    }

    function stopStatusUpdates() {
        if (app.rafId) {
            cancelAnimationFrame(app.rafId);
            app.rafId = null;
        }
    }

    function showLightDetail(light) {
        app.selectedLightId = light.id;
        
        app.map.flyTo({
            center: [light.longitude, light.latitude],
            zoom: 17,
            speed: 1.5
        });
        
        dom.popupTitle.textContent = light.name;
        dom.popupContent.innerHTML = `
            <div class="grid grid-cols-2 gap-3">
                <div class="text-gray-500">Direction:</div>
                <div class="font-medium capitalize">${light.direction}</div>
                
                <div class="text-gray-500">Red duration:</div>
                <div class="font-medium">${light.red_duration} seconds</div>
                
                <div class="text-gray-500">Green duration:</div>
                <div class="font-medium">${light.green_duration} seconds</div>
                
                <div class="text-gray-500">Total cycle:</div>
                <div class="font-medium">${parseInt(light.red_duration) + parseInt(light.green_duration)} seconds</div>
                
                <div class="text-gray-500">Location:</div>
                <div class="font-medium truncate">${light.latitude.substring(0, 8)}, ${light.longitude.substring(0, 8)}</div>
            </div>
        `;
        
        const status = getLightStatus(light);
        const showPrediction = app.settings.predictions;
        
        dom.popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
        dom.popupStatus.innerHTML = `
            <div class="text-xl font-semibold mb-1">${status.label}</div>
            <div class="flex justify-center items-center gap-2">
                <i class="fas fa-clock"></i>
                <span>Changes in ${status.timeLeft} seconds</span>
            </div>
            ${showPrediction ? `
            <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
            </div>
            ` : ''}
        `;
        
        dom.lightPopup.classList.remove('hidden');
    }

    function startGeolocation() {
        if (navigator.geolocation) {
            if (app.userWatchId) navigator.geolocation.clearWatch(app.userWatchId);
            
            dom.buttons.headerLocate.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>';
            
            app.userWatchId = navigator.geolocation.watchPosition(
                updateUserLocation,
                handleGeolocationError,
                { enableHighAccuracy: true, maximumAge: 10000, timeout: 5000 }
            );
            
            navigator.geolocation.getCurrentPosition(
                updateUserLocation,
                handleGeolocationError,
                { enableHighAccuracy: true }
            );
        }
    }

    function updateUserLocation(position) {
        const lat = position.coords.latitude;
        const lng = position.coords.longitude;
        
        dom.buttons.headerLocate.innerHTML = '<i class="fas fa-location-crosshairs"></i>';
        
        if (!app.userMarker) {
            const locationEl = document.createElement('div');
            locationEl.className = 'location-dot';
            
            app.userMarker = new mapboxgl.Marker({
                element: locationEl
            })
            .setLngLat([lng, lat])
            .addTo(app.map);
            
            app.map.flyTo({
                center: [lng, lat],
                zoom: 16,
                speed: 1.5
            });
        } else {
            app.userMarker.setLngLat([lng, lat]);
        }
        
        // Check for auto-measure during navigation
        if (app.state.navigationActive && app.settings.autoMeasure) {
            checkAutoMeasureTrigger(position);
        }
    }

    function handleGeolocationError(error) {
        console.error('Geolocation error:', error.message);
        
        dom.buttons.headerLocate.innerHTML = '<i class="fas fa-location-crosshairs"></i>';
        
        if (app.settings.notifications) {
            showNotification('Location access denied', 'error');
        }
    }

    function checkAutoMeasureTrigger(position) {
        // TODO: Implement auto-measure logic
        // This would involve checking if user is stopped for 2+ seconds
        // on a road, between intersections, not near start or end of route
    }

    function calculateDistance(lat1, lon1, lat2, lon2) {
        const R = 6371; // Earth radius in km
        const dLat = (lat2 - lat1) * Math.PI / 180;
        const dLon = (lon2 - lon1) * Math.PI / 180;
        const a = 
            Math.sin(dLat / 2) * Math.sin(dLat / 2) +
            Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) * 
            Math.sin(dLon / 2) * Math.sin(dLon / 2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        return R * c; // Distance in km
    }

    function showNotification(message, type) {
        const colors = {
            success: 'bg-traffic-green',
            error: 'bg-traffic-red',
            info: 'bg-primary-500'
        };
        
        const icons = {
            success: 'check-circle',
            error: 'exclamation-circle',
            info: 'info-circle'
        };
        
        const notification = document.createElement('div');
        notification.className = `fixed top-20 left-1/2 transform -translate-x-1/2 ${colors[type]} text-white px-4 py-3 rounded-lg shadow-lg z-50 fade-in flex items-center`;
        notification.innerHTML = `<i class="fas fa-${icons[type]} mr-2"></i>${message}`;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translate(-50%, -10px)';
            notification.style.transition = 'opacity 0.5s, transform 0.5s';
            setTimeout(() => notification.remove(), 500);
        }, 3000);
    }

    function handleOnline() {
        if (app.settings.notifications) {
            showNotification('You are back online', 'success');
        }
        loadTrafficLights(true);
    }

    function handleOffline() {
        if (app.settings.notifications) {
            showNotification('You are offline. Some features may be limited', 'error');
        }
    }

    function handleVisibilityChange() {
        if (document.visibilityState === 'visible') {
            loadTrafficLights(true);
        }
    }
});