<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\DB;

Route::get('/', function () {
    return view('welcome');
});

Route::get('/auth', function() {
    $token = request('token');
    $expires = request('expires');
    $signature = request('signature');
    
    if (!$token || !$expires || !$signature) {
        return redirect('/')->with('error', 'Paramètres manquants');
    }
    
    if (time() > $expires) {
        return redirect('/')->with('error', 'Lien expiré');
    }
    
    $expectedSignature = hash_hmac('sha256', $token . $expires, 'ise_engine_ultra_secure_key_2025_imators_systems');
    
    if (!hash_equals($expectedSignature, $signature)) {
        return redirect('/')->with('error', 'Signature invalide');
    }
    
    $data = json_decode(base64_decode($token), true);
    
    if (!$data || !isset($data['user_id']) || !isset($data['ise_auth'])) {
        return redirect('/')->with('error', 'Token invalide');
    }
    
    session([
        'user_id' => $data['user_id'],
        'email' => $data['email'],
        'username' => $data['username'],
        'authenticated' => true,
        'auth_time' => time()
    ]);
    
    return redirect('/workspace');
});

Route::get('/workspace', function() {
    if (!session('authenticated')) {
        return redirect('/');
    }
    
    if (time() - session('auth_time', 0) > 3600) {
        session()->flush();
        return redirect('/')->with('error', 'Session expirée');
    }
    
    return view('workspace');
})->name('workspace');

Route::get('/editor/{document_id?}', function($document_id = null) {
    if (!session('authenticated')) {
        return redirect('/');
    }
    
    if (time() - session('auth_time', 0) > 3600) {
        session()->flush();
        return redirect('/')->with('error', 'Session expirée');
    }
    
    return view('editor', compact('document_id'));
})->name('editor');

Route::get('/logout', function() {
    session()->flush();
    return redirect('/');
});

// Routes API pour l'éditeur
Route::middleware(['web'])->group(function () {
    
    // Sauvegarde de document
    Route::post('/api/documents/save', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $request = request();
        $title = $request->input('title', 'Document sans titre');
        $content = $request->input('content', '');
        $documentId = $request->input('document_id');

        try {
            if ($documentId) {
                DB::table('documents')
                    ->where('id', $documentId)
                    ->where('user_id', session('user_id'))
                    ->update([
                        'title' => $title,
                        'content' => $content,
                        'updated_at' => now(),
                        'last_accessed_at' => now()
                    ]);
                return response()->json(['success' => true, 'document_id' => $documentId]);
            } else {
                $documentId = DB::table('documents')->insertGetId([
                    'user_id' => session('user_id'),
                    'title' => $title,
                    'content' => $content,
                    'status' => 'draft',
                    'created_at' => now(),
                    'updated_at' => now(),
                    'last_accessed_at' => now()
                ]);
                return response()->json(['success' => true, 'document_id' => $documentId]);
            }
        } catch (Exception $e) {
            return response()->json(['error' => 'Erreur de sauvegarde: ' . $e->getMessage()], 500);
        }
    });

    // Charger un document
    Route::get('/api/documents/{id}', function($id) {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $document = DB::table('documents')
            ->where('id', $id)
            ->where('user_id', session('user_id'))
            ->first();

        if (!$document) {
            return response()->json(['error' => 'Document non trouvé'], 404);
        }

        DB::table('documents')
            ->where('id', $id)
            ->update(['last_accessed_at' => now()]);

        return response()->json($document);
    });

    // Lister les documents
    Route::get('/api/documents', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $documents = DB::table('documents')
            ->where('user_id', session('user_id'))
            ->orderBy('updated_at', 'desc')
            ->select('id', 'title', 'status', 'created_at', 'updated_at', 'last_accessed_at')
            ->get();

        return response()->json($documents);
    });

    // Supprimer un document
    Route::delete('/api/documents/{id}', function($id) {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $deleted = DB::table('documents')
            ->where('id', $id)
            ->where('user_id', session('user_id'))
            ->delete();

        if ($deleted) {
            return response()->json(['success' => true]);
        }
        return response()->json(['error' => 'Document non trouvé'], 404);
    });

    // Partager un document
    Route::post('/api/documents/{id}/share', function($id) {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $request = request();
        $permission = $request->input('permission', 'view');

        $document = DB::table('documents')
            ->where('id', $id)
            ->where('user_id', session('user_id'))
            ->first();

        if (!$document) {
            return response()->json(['error' => 'Document non trouvé'], 404);
        }

        $shareToken = bin2hex(random_bytes(32));

        $shareId = DB::table('document_shares')->insertGetId([
            'document_id' => $id,
            'share_token' => $shareToken,
            'permission' => $permission,
            'is_active' => true,
            'expires_at' => null,
            'created_at' => now(),
            'updated_at' => now()
        ]);

        $shareUrl = url("/shared/{$shareToken}");

        return response()->json([
            'success' => true,
            'share_url' => $shareUrl,
            'permission' => $permission
        ]);
    });

    // Claude API (vraie)
Route::post('/api/claude', function() {
    if (!session('authenticated')) {
        return response()->json(['error' => 'Non authentifié'], 401);
    }

    $request = request();
    $prompt = $request->input('prompt');
    $selectedText = $request->input('selected_text', '');
    $action = $request->input('action', 'custom');

    $systemPrompt = "Tu es un assistant d'écriture expert. Réponds en français de manière concise et professionnelle.";
    
    switch($action) {
        case 'grammar':
            $fullPrompt = "Corrige la grammaire et l'orthographe de ce texte en gardant le même sens et style : {$selectedText}";
            break;
        case 'expand':
            $fullPrompt = "Développe et enrichis ce texte en gardant le même style et en ajoutant des détails pertinents : {$selectedText}";
            break;
        case 'summarize':
            $fullPrompt = "Résume ce texte de manière concise en gardant les points essentiels : {$selectedText}";
            break;
        case 'rephrase':
            $fullPrompt = "Reformule ce texte de manière plus claire et fluide : {$selectedText}";
            break;
        default:
            $fullPrompt = $prompt . ($selectedText ? "\n\nTexte à traiter : {$selectedText}" : '');
    }

    try {
        $apiKey = config('services.claude.api_key');
        
        if (!$apiKey) {
            return response()->json(['error' => 'Clé API Claude manquante'], 500);
        }

        $response = Http::withHeaders([
            'Content-Type' => 'application/json',
            'x-api-key' => $apiKey,
            'anthropic-version' => '2023-06-01'
        ])->post('https://api.anthropic.com/v1/messages', [
            'model' => config('services.claude.model'),
            'max_tokens' => 1000,
            'system' => $systemPrompt,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $fullPrompt
                ]
            ]
        ]);

        if ($response->successful()) {
            $data = $response->json();
            $content = $data['content'][0]['text'] ?? 'Erreur dans la réponse';
            
            return response()->json([
                'success' => true,
                'content' => $content
            ]);
        } else {
            return response()->json(['error' => 'Erreur API Claude: ' . $response->body()], 500);
        }
        
    } catch (Exception $e) {
        return response()->json(['error' => 'Erreur API Claude: ' . $e->getMessage()], 500);
    }
});

    // Perplexity API (vraie)
Route::post('/api/search', function() {
    if (!session('authenticated')) {
        return response()->json(['error' => 'Non authentifié'], 401);
    }

    $request = request();
    $query = $request->input('query');

    try {
        $apiKey = config('services.perplexity.api_key');
        
        if (!$apiKey) {
            return response()->json(['error' => 'Clé API Perplexity manquante'], 500);
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json'
        ])->post('https://api.perplexity.ai/chat/completions', [
            'model' => config('services.perplexity.model'),
            'messages' => [
                [
                    'role' => 'system',
                    'content' => 'Tu es un assistant de recherche expert. Fournis des informations factuelles avec des sources fiables. Réponds en français.'
                ],
                [
                    'role' => 'user',
                    'content' => "Recherche des informations sur : {$query}. Fournis un résumé avec des sources."
                ]
            ],
            'max_tokens' => 1000,
            'temperature' => 0.2,
            'top_p' => 0.9,
            'return_citations' => true,
            'search_domain_filter' => ["perplexity.ai"],
            'return_images' => false,
            'return_related_questions' => false,
            'search_recency_filter' => "month",
            'top_k' => 0,
            'stream' => false,
            'presence_penalty' => 0,
            'frequency_penalty' => 1
        ]);

        if ($response->successful()) {
            $data = $response->json();
            $content = $data['choices'][0]['message']['content'] ?? 'Aucun résultat trouvé';
            $citations = $data['citations'] ?? [];
            
            // Formater les résultats
            $results = [];
            foreach ($citations as $index => $citation) {
                $results[] = [
                    'title' => 'Source ' . ($index + 1),
                    'snippet' => substr($citation, 0, 150) . '...',
                    'url' => 'Source fiable',
                    'source' => 'Perplexity AI'
                ];
            }
            
            return response()->json([
                'success' => true,
                'results' => $results,
                'summary' => $content
            ]);
        } else {
            return response()->json(['error' => 'Erreur API Perplexity: ' . $response->body()], 500);
        }
        
    } catch (Exception $e) {
        return response()->json(['error' => 'Erreur API Perplexity: ' . $e->getMessage()], 500);
    }
});

// Document partagé
Route::get('/shared/{token}', function($token) {
    $share = DB::table('document_shares')
        ->join('documents', 'document_shares.document_id', '=', 'documents.id')
        ->where('document_shares.share_token', $token)
        ->where('document_shares.is_active', true)
        ->select('documents.*', 'document_shares.permission')
        ->first();

    if (!$share) {
        abort(404, 'Document partagé non trouvé');
    }

    return view('shared-document', [
        'document' => $share,
        'permission' => $share->permission
    ]);
});