<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}
require_once 'db.php';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $start_point = $_POST['start_point'] ?? '';
    $end_point = $_POST['end_point'] ?? '';
    $pins = $_POST['pins'] ?? [];
    $status = $_POST['status'] ?? 'ongoing';

    $database = new Database();
    $conn = $database->getConnection();

    $stmt = $conn->prepare("INSERT INTO mesures 
        (user_id, start_point, end_point, pins, status, created_at) 
        VALUES (:user_id, :start_point, :end_point, :pins, :status, NOW())");
    
    $pins_json = json_encode($pins);
    
    $stmt->bindParam(':user_id', $_SESSION['user_id']);
    $stmt->bindParam(':start_point', $start_point);
    $stmt->bindParam(':end_point', $end_point);
    $stmt->bindParam(':pins', $pins_json);
    $stmt->bindParam(':status', $status);
    
    $stmt->execute();

    header("Location: dashboard.php");
    exit();
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Peerkinton - Outil Mesure</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.7.1/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.7.1/dist/leaflet.js"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300&display=swap" rel="stylesheet">
    <style>
        body { 
            font-family: 'Poppins', sans-serif; 
            background-color: black;
            color: white;
        }
        #map { 
            height: 500px; 
            width: 100%;
            background-color: #1a1a1a;
        }
    </style>
</head>
<body class="bg-black text-white min-h-screen">
    <div class="container mx-auto p-6">
        <h1 class="text-2xl mb-6">Outil Mesure</h1>
        
        <div id="map" class="w-full mb-6 rounded-lg"></div>
        
        <form id="mesureForm" method="post" class="bg-gray-900 p-6 rounded">
            <input type="hidden" name="start_point" id="startPointInput">
            <input type="hidden" name="end_point" id="endPointInput">
            <input type="hidden" name="pins" id="pinsInput">
            
            <div class="mb-4">
                <label class="block mb-2">Statut du trajet</label>
                <select name="status" class="w-full bg-black text-white p-3 rounded">
                    <option value="ongoing">En cours</option>
                    <option value="resolved">Résolu</option>
                    <option value="unresolved">Non résolu</option>
                </select>
            </div>
            
            <button type="submit" 
                class="w-full bg-white text-black p-3 rounded hover:bg-gray-200">
                Enregistrer le Trajet
            </button>
        </form>
        
        <div class="mt-4 text-center">
            <a href="dashboard.php" class="text-white underline">Retour</a>
        </div>
    </div>

    <script>
    let map, polyline, startMarker, endMarker, pins = [];
    let isDrawingMode = false;

    function initMap() {
        map = L.map('map').setView([0, 0], 2);

        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors'
        }).addTo(map);

        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(function(position) {
                map.setView([position.coords.latitude, position.coords.longitude], 10);
            });
        }

        map.on('click', function(e) {
            if (isDrawingMode) {
                const pin = {
                    lat: e.latlng.lat,
                    lng: e.latlng.lng,
                    resolved: false
                };
                
                const marker = L.marker([pin.lat, pin.lng], {
                    icon: L.icon({
                        iconUrl: 'https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-red.png',
                        shadowUrl: 'https://cdnjs.cloudflare.com/ajax/libs/leaflet/0.7.7/images/marker-shadow.png'
                    })
                }).addTo(map);

                marker.on('click', function() {
                    const resolved = confirm('Ce point est-il résolu ?');
                    pin.resolved = resolved;
                    
                    marker.setIcon(L.icon({
                        iconUrl: resolved 
                            ? 'https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-green.png'
                            : 'https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-red.png',
                        shadowUrl: 'https://cdnjs.cloudflare.com/ajax/libs/leaflet/0.7.7/images/marker-shadow.png'
                    }));
                });

                pins.push(pin);
            }
        });

        const drawButton = L.Control.extend({
            options: {
                position: 'topright'
            },
            onAdd: function(map) {
                const container = L.DomUtil.create('div', 'leaflet-bar leaflet-control');
                const button = L.DomUtil.create('a', 'leaflet-control-draw', container);
                button.href = '#';
                button.title = 'Mode dessin';
                button.innerHTML = '✏️';
                
                button.onclick = function(e) {
                    e.preventDefault();
                    isDrawingMode = !isDrawingMode;
                    button.style.backgroundColor = isDrawingMode ? 'rgba(255,255,255,0.5)' : '';
                };
                
                return container;
            }
        });
        
        map.addControl(new drawButton());

        const startTrackButton = L.Control.extend({
            options: { position: 'topright' },
            onAdd: function(map) {
                const container = L.DomUtil.create('div', 'leaflet-bar leaflet-control');
                const button = L.DomUtil.create('a', 'leaflet-control-track', container);
                button.href = '#';
                button.title = 'Début du trajet';
                button.innerHTML = '🏁';
                
                button.onclick = function(e) {
                    e.preventDefault();
                    if (startMarker) startMarker.remove();
                    
                    const pos = map.getCenter();
                    startMarker = L.marker([pos.lat, pos.lng], {
                        icon: L.icon({
                            iconUrl: 'https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-green.png',
                            shadowUrl: 'https://cdnjs.cloudflare.com/ajax/libs/leaflet/0.7.7/images/marker-shadow.png'
                        })
                    }).addTo(map);

                    document.getElementById('startPointInput').value = 
                        JSON.stringify({lat: pos.lat, lng: pos.lng});
                };
                
                return container;
            }
        });
        map.addControl(new startTrackButton());

        const endTrackButton = L.Control.extend({
            options: { position: 'topright' },
            onAdd: function(map) {
                const container = L.DomUtil.create('div', 'leaflet-bar leaflet-control');
                const button = L.DomUtil.create('a', 'leaflet-control-track-end', container);
                button.href = '#';
                button.title = 'Fin du trajet';
                button.innerHTML = '🏁';
                
                button.onclick = function(e) {
                    e.preventDefault();
                    if (endMarker) endMarker.remove();
                    
                    const pos = map.getCenter();
                    endMarker = L.marker([pos.lat, pos.lng], {
                        icon: L.icon({
                            iconUrl: 'https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-red.png',
                            shadowUrl: 'https://cdnjs.cloudflare.com/ajax/libs/leaflet/0.7.7/images/marker-shadow.png'
                        })
                    }).addTo(map);

                    document.getElementById('endPointInput').value = 
                        JSON.stringify({lat: pos.lat, lng: pos.lng});
                };
                
                return container;
            }
        });
        map.addControl(new endTrackButton());
    }

    document.getElementById('mesureForm').addEventListener('submit', function(e) {
        document.getElementById('pinsInput').value = JSON.stringify(pins);
    });

    document.addEventListener('DOMContentLoaded', initMap);
    </script>
</body>
</html>