<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);
$inventory = get_user_inventory($_SESSION['user_id']);

$error = null;
$success = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'use_item') {
        $item_id = $_POST['item_id'] ?? '';
        
        if (empty($item_id)) {
            $error = "Objet non spécifié.";
        } else {
            $item = null;
            foreach ($inventory as $inv_item) {
                if ($inv_item['id'] == $item_id) {
                    $item = $inv_item;
                    break;
                }
            }
            
            if (!$item) {
                $error = "Objet non trouvé dans ton inventaire.";
            } else {
                $result = use_item($_SESSION['user_id'], $item_id);
                
                if ($result['success']) {
                    $success = $result['message'];
                    if (isset($result['redirect'])) {
                        header('Location: ' . $result['redirect']);
                        exit;
                    }
                    
                    $inventory = get_user_inventory($_SESSION['user_id']);
                } else {
                    $error = $result['message'];
                }
            }
        }
    }
}

$available_items = db_get_all(
    "SELECT * FROM items WHERE id NOT IN (SELECT item_id FROM inventories WHERE user_id = ?) ORDER BY rarity, name",
    [$_SESSION['user_id']]
);

$currencies = db_get_all(
    "SELECT c.*, cu.name as currency_name, cu.exchange_rate, cu.stability
     FROM countries c
     JOIN currencies cu ON c.id = cu.country_id
     ORDER BY c.name"
);

$user_balances = [];
foreach ($currencies as $curr) {
    $user_balances[$curr['id']] = get_user_currency_balance($_SESSION['user_id'], $curr['id']);
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inventaire | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
        .item-card {
            transition: transform 0.2s, box-shadow 0.2s;
        }
        .item-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 15px rgba(0, 0, 0, 0.1);
        }
        .rarity-common {
            border-color: #9CA3AF;
        }
        .rarity-uncommon {
            border-color: #10B981;
        }
        .rarity-rare {
            border-color: #3B82F6;
        }
        .rarity-epic {
            border-color: #8B5CF6;
        }
        .rarity-legendary {
            border-color: #F59E0B;
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300 font-semibold">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <?php if ($user['level'] >= 10): ?>
                        <li><a href="unions.php" class="hover:text-blue-300">Unions</a></li>
                    <?php endif; ?>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Inventaire</h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div class="lg:col-span-2">
                <div class="card p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Tes objets</h3>
                    
                    <?php if (empty($inventory)): ?>
                        <p class="text-gray-500 italic">Tu n'as aucun objet dans ton inventaire. Complète des missions pour en obtenir!</p>
                    <?php else: ?>
                        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                            <?php foreach ($inventory as $item): ?>
                                <?php
                                    $rarity_colors = [
                                        'common' => 'bg-gray-100 text-gray-800 rarity-common',
                                        'uncommon' => 'bg-green-50 text-green-800 rarity-uncommon',
                                        'rare' => 'bg-blue-50 text-blue-800 rarity-rare',
                                        'epic' => 'bg-purple-50 text-purple-800 rarity-epic',
                                        'legendary' => 'bg-yellow-50 text-yellow-800 rarity-legendary'
                                    ];
                                    $rarity_text = [
                                        'common' => 'Commun',
                                        'uncommon' => 'Peu commun',
                                        'rare' => 'Rare',
                                        'epic' => 'Épique',
                                        'legendary' => 'Légendaire'
                                    ];
                                    $rarity_class = $rarity_colors[$item['rarity']] ?? 'bg-gray-100 text-gray-800';
                                    $rarity_name = $rarity_text[$item['rarity']] ?? $item['rarity'];
                                ?>
                                <div class="item-card border-2 rounded-lg p-4 <?php echo $rarity_class; ?>">
                                    <div class="flex justify-between items-start">
                                        <h4 class="font-medium"><?php echo htmlspecialchars($item['name']); ?></h4>
                                        <span class="text-xs px-1.5 py-0.5 bg-white bg-opacity-50 rounded">
                                            <?php echo $rarity_name; ?>
                                        </span>
                                    </div>
                                    <p class="text-sm mt-2"><?php echo htmlspecialchars($item['description']); ?></p>
                                    <div class="mt-3 text-sm">
                                        <span class="font-medium">Type:</span> 
                                        <?php echo htmlspecialchars($item['type']); ?>
                                    </div>
                                    <div class="mt-3 text-sm">
                                        <span class="font-medium">Quantité:</span> 
                                        <?php echo $item['quantity']; ?>
                                    </div>
                                    <?php if ($item['effect']): ?>
                                        <div class="mt-2 text-sm italic">
                                            <?php echo htmlspecialchars($item['effect']); ?>
                                        </div>
                                    <?php endif; ?>
                                    <form method="POST" action="inventory.php" class="mt-3">
                                        <input type="hidden" name="action" value="use_item">
                                        <input type="hidden" name="item_id" value="<?php echo $item['id']; ?>">
                                        <button type="submit" class="w-full bg-blue-600 hover:bg-blue-700 text-white text-sm py-1 px-2 rounded transition-colors">
                                            Utiliser
                                        </button>
                                    </form>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Objets disponibles</h3>
                    
                    <?php if (empty($available_items)): ?>
                        <p class="text-gray-500 italic">Tu possèdes déjà tous les objets disponibles!</p>
                    <?php else: ?>
                        <div class="overflow-x-auto">
                            <table class="w-full">
                                <thead>
                                    <tr>
                                        <th class="text-left px-2 py-2 border-b">Nom</th>
                                        <th class="text-left px-2 py-2 border-b">Type</th>
                                        <th class="text-left px-2 py-2 border-b">Rareté</th>
                                        <th class="text-left px-2 py-2 border-b">Description</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($available_items as $item): ?>
                                        <tr class="hover:bg-gray-50">
                                            <td class="px-2 py-2 border-b font-medium"><?php echo htmlspecialchars($item['name']); ?></td>
                                            <td class="px-2 py-2 border-b"><?php echo htmlspecialchars($item['type']); ?></td>
                                            <td class="px-2 py-2 border-b">
                                                <?php
                                                    $rarity_colors = [
                                                        'common' => 'bg-gray-100 text-gray-800',
                                                        'uncommon' => 'bg-green-100 text-green-800',
                                                        'rare' => 'bg-blue-100 text-blue-800',
                                                        'epic' => 'bg-purple-100 text-purple-800',
                                                        'legendary' => 'bg-yellow-100 text-yellow-800'
                                                    ];
                                                    $rarity_text = [
                                                        'common' => 'Commun',
                                                        'uncommon' => 'Peu commun',
                                                        'rare' => 'Rare',
                                                        'epic' => 'Épique',
                                                        'legendary' => 'Légendaire'
                                                    ];
                                                    $rarity_class = $rarity_colors[$item['rarity']] ?? 'bg-gray-100 text-gray-800';
                                                    $rarity_name = $rarity_text[$item['rarity']] ?? $item['rarity'];
                                                ?>
                                                <span class="inline-block px-2 py-1 text-xs rounded <?php echo $rarity_class; ?>">
                                                    <?php echo $rarity_name; ?>
                                                </span>
                                            </td>
                                            <td class="px-2 py-2 border-b"><?php echo htmlspecialchars($item['description']); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div>
                <div class="card p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Finances</h3>
                    
                    <?php if (empty($currencies)): ?>
                        <p class="text-gray-500 italic">Aucune devise disponible.</p>
                    <?php else: ?>
                        <div class="space-y-4">
                            <?php foreach ($currencies as $currency): ?>
                                <?php
                                    $stability_colors = [
                                        'unstable' => 'text-red-600',
                                        'stable' => 'text-green-600',
                                        'very_stable' => 'text-blue-600',
                                        'growing' => 'text-purple-600'
                                    ];
                                    $stability_text = [
                                        'unstable' => 'Instable',
                                        'stable' => 'Stable',
                                        'very_stable' => 'Très stable',
                                        'growing' => 'En croissance'
                                    ];
                                    $stability_color = $stability_colors[$currency['stability']] ?? 'text-gray-600';
                                    $stability_name = $stability_text[$currency['stability']] ?? $currency['stability'];
                                    $user_balance = $user_balances[$currency['id']] ?? 0;
                                ?>
                                <div class="border rounded-lg p-4">
                                    <div class="flex justify-between items-center">
                                        <h4 class="font-medium"><?php echo htmlspecialchars($currency['currency_name']); ?></h4>
                                        <div class="flex items-center">
                                            <img src="https://flagcdn.com/24x18/<?php echo strtolower($currency['code']); ?>.png" alt="<?php echo htmlspecialchars($currency['name']); ?>" class="mr-2" width="24" height="18">
                                            <span><?php echo htmlspecialchars($currency['name']); ?></span>
                                        </div>
                                    </div>
                                    <div class="mt-2">
                                        <p class="text-xl font-semibold"><?php echo number_format($user_balance, 2); ?></p>
                                        <p class="text-sm <?php echo $stability_color; ?>">
                                            <?php echo $stability_name; ?> • Taux: <?php echo $currency['exchange_rate']; ?>
                                        </p>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <div class="mt-4">
                            <a href="market.php" class="block w-full bg-blue-600 hover:bg-blue-700 text-white text-center font-medium py-2 px-4 rounded transition-colors">
                                Accéder au marché
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Statistiques d'inventaire</h3>
                    
                    <div class="space-y-3">
                        <div>
                            <p class="font-medium">Objets possédés</p>
                            <p class="text-xl"><?php echo count($inventory); ?> / <?php echo count($inventory) + count($available_items); ?></p>
                        </div>
                        <div>
                            <p class="font-medium">Répartition par rareté</p>
                            <?php
                                $rarity_counts = [
                                    'common' => 0,
                                    'uncommon' => 0,
                                    'rare' => 0,
                                    'epic' => 0,
                                    'legendary' => 0
                                ];
                                
                                foreach ($inventory as $item) {
                                    if (isset($rarity_counts[$item['rarity']])) {
                                        $rarity_counts[$item['rarity']]++;
                                    }
                                }
                            ?>
                            <div class="grid grid-cols-5 gap-1 mt-2">
                                <div class="bg-gray-100 p-2 text-center text-xs rounded">
                                    <span class="block font-medium">Commun</span>
                                    <span><?php echo $rarity_counts['common']; ?></span>
                                </div>
                                <div class="bg-green-50 p-2 text-center text-xs rounded">
                                    <span class="block font-medium">P. commun</span>
                                    <span><?php echo $rarity_counts['uncommon']; ?></span>
                                </div>
                                <div class="bg-blue-50 p-2 text-center text-xs rounded">
                                    <span class="block font-medium">Rare</span>
                                    <span><?php echo $rarity_counts['rare']; ?></span>
                                </div>
                                <div class="bg-purple-50 p-2 text-center text-xs rounded">
                                    <span class="block font-medium">Épique</span>
                                    <span><?php echo $rarity_counts['epic']; ?></span>
                                </div>
                                <div class="bg-yellow-50 p-2 text-center text-xs rounded">
                                    <span class="block font-medium">Légendaire</span>
                                    <span><?php echo $rarity_counts['legendary']; ?></span>
                                </div>
                            </div>
                        </div>
                        <div>
                            <p class="font-medium">Répartition par type</p>
                            <?php
                                $type_counts = [];
                                
                                foreach ($inventory as $item) {
                                    if (!isset($type_counts[$item['type']])) {
                                        $type_counts[$item['type']] = 0;
                                    }
                                    $type_counts[$item['type']]++;
                                }
                            ?>
                            <div class="mt-2 space-y-1">
                                <?php foreach ($type_counts as $type => $count): ?>
                                    <div class="flex justify-between">
                                        <span><?php echo htmlspecialchars($type); ?></span>
                                        <span><?php echo $count; ?></span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script>
        function use_item(item_id, item_name) {
            if (confirm("Utiliser l'objet: " + item_name + "?")) {
                document.getElementById('use_item_form_' + item_id).submit();
            }
        }
    </script>
</body>
</html>