<?php 
require_once 'config.php';

$query = $pdo->query("
    SELECT 
        c.*,
        u.username as author_name,
        COUNT(DISTINCT CASE WHEN i.type = 'view' THEN i.id END) as view_count,
        COUNT(DISTINCT CASE WHEN i.type = 'like' THEN i.id END) as like_count,
        COUNT(DISTINCT CASE WHEN i.type = 'dislike' THEN i.id END) as dislike_count,
        COUNT(DISTINCT CASE WHEN i.type = 'comment' THEN i.id END) as comment_count
    FROM content c
    LEFT JOIN users u ON c.author_id = u.id
    LEFT JOIN interactions i ON c.id = i.content_id
    GROUP BY c.id
    ORDER BY c.created_at DESC
");

$videos = $query->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ForMore</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Titillium+Web:wght@200;300;400;600;700;900&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Titillium Web', sans-serif;
        }
        .formore {
            font-weight: 900;
            background: linear-gradient(to right, #3b82f6, #1d4ed8);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        .video-card {
            transition: all 0.3s ease;
        }
        .video-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
        }
    </style>
</head>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white p-8">
    <nav class="max-w-7xl mx-auto flex justify-between items-center mb-12">
        <h1 class="text-5xl formore">ForMore</h1>
        <div class="flex items-center gap-6">
            <?php if(isset($_SESSION['user_id'])): ?>
                <a href="profile.php" class="text-gray-300 hover:text-white"><?= htmlspecialchars($_SESSION['username']) ?></a>
                <a href="upload.php" class="bg-blue-600 hover:bg-blue-700 px-6 py-2 rounded-xl transition-all">Upload</a>
                <a href="logout.php" class="text-red-400 hover:text-red-300">Logout</a>
            <?php else: ?>
                <a href="login.php" class="text-blue-400 hover:text-blue-300">Login</a>
                <a href="signup.php" class="bg-blue-600 hover:bg-blue-700 px-6 py-2 rounded-xl transition-all">Sign Up</a>
            <?php endif; ?>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
        <?php foreach($videos as $video): ?>
            <div class="video-card bg-gray-800/50 backdrop-blur-sm rounded-2xl overflow-hidden cursor-pointer" onclick="openModal(<?= $video['id'] ?>)">
                <div class="relative aspect-video">
                    <img src="<?= htmlspecialchars($video['picture_cover']) ?>" alt="<?= htmlspecialchars($video['name']) ?>" class="w-full h-full object-cover">
                    <div class="absolute bottom-2 right-2 bg-black/70 backdrop-blur-sm px-3 py-1 rounded-lg text-sm">
                        <?= htmlspecialchars($video['time']) ?>
                    </div>
                </div>
                <div class="p-4">
                    <h2 class="text-lg font-bold line-clamp-1"><?= htmlspecialchars($video['name']) ?></h2>
                    <p class="text-gray-400 text-sm mt-2 line-clamp-2"><?= htmlspecialchars($video['description']) ?></p>
                    <div class="flex justify-between items-center mt-4">
                        <span class="text-sm text-gray-300"><?= htmlspecialchars($video['author_name']) ?></span>
                        <div class="flex items-center gap-2 text-sm text-gray-400">
                            <span><?= number_format($video['view_count']) ?> views</span>
                        </div>
                    </div>
                    <div class="flex flex-wrap gap-2 mt-3">
                        <?php foreach(explode(',', $video['tags']) as $tag): ?>
                            <span class="bg-blue-900/50 backdrop-blur-sm px-3 py-1 rounded-full text-xs">
                                <?= htmlspecialchars(trim($tag)) ?>
                            </span>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>

    <div id="videoModal" class="hidden fixed inset-0 bg-black/95 backdrop-blur-sm z-50 overflow-y-auto">
        <div class="min-h-screen flex items-center justify-center p-4">
            <div class="bg-gray-800/50 backdrop-blur-sm rounded-2xl w-full max-w-5xl">
                <div id="modalContent" class="p-6">
                    <div class="flex justify-between items-center mb-4">
                        <h3 class="text-2xl font-bold" id="videoTitle"></h3>
                        <button onclick="closeModal()" class="text-gray-400 hover:text-white transition-colors">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                        </button>
                    </div>
                    
                    <div class="relative bg-gray-900 rounded-xl overflow-hidden aspect-video mb-6">
                        <div id="videoContainer" class="w-full h-full">
                            <div class="flex items-center justify-center h-full">
                                <div class="animate-pulse">
                                    <svg class="w-12 h-12 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 3l14 9-14 9V3z"></path>
                                    </svg>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="space-y-6">
                        <div id="videoInfo" class="space-y-4">
                            <div class="flex items-center gap-4">
                                <a href="#" id="authorLink" class="flex items-center gap-3 text-blue-400 hover:text-blue-300">
                                    <div class="w-10 h-10 rounded-full bg-gray-700"></div>
                                    <span id="authorName" class="font-semibold"></span>
                                </a>
                                <div id="videoStats" class="flex gap-4 text-gray-400 text-sm"></div>
                            </div>
                            
                            <div id="videoDescription" class="text-gray-300 whitespace-pre-wrap"></div>
                            
                            <div id="videoTags" class="flex flex-wrap gap-2"></div>
                        </div>

                        <?php if(isset($_SESSION['user_id'])): ?>
                        <div class="flex items-center gap-4">
                            <button onclick="handleInteraction('like')" class="px-4 py-2 rounded-xl bg-gray-700 hover:bg-gray-600 transition-colors">
                                <span id="likeCount">0</span> Likes
                            </button>
                            <button onclick="handleInteraction('dislike')" class="px-4 py-2 rounded-xl bg-gray-700 hover:bg-gray-600 transition-colors">
                                <span id="dislikeCount">0</span> Dislikes
                            </button>
                        </div>

                        <div class="border-t border-gray-700 pt-6">
                            <form onsubmit="handleComment(event)" class="space-y-4">
                                <textarea id="commentText" rows="3" 
                                    class="w-full bg-gray-900/50 backdrop-blur-sm border-0 rounded-xl p-4 text-white focus:ring-2 focus:ring-blue-500"
                                    placeholder="Add a comment..."></textarea>
                                <button type="submit" 
                                    class="px-6 py-2 bg-blue-600 hover:bg-blue-700 rounded-xl transition-colors">
                                    Comment
                                </button>
                            </form>
                        </div>
                        <?php endif; ?>

                        <div id="comments" class="space-y-4 border-t border-gray-700 pt-6">
                            <h4 class="text-xl font-bold">Comments</h4>
                            <div id="commentsList" class="space-y-4"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        let currentVideoId = null;

        function openModal(videoId) {
            currentVideoId = videoId;
            document.getElementById('videoModal').classList.remove('hidden');
            document.body.style.overflow = 'hidden';
            
            fetch('get_video.php?id=' + videoId)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('videoTitle').textContent = data.name;
                    document.getElementById('authorName').textContent = data.author_name;
                    document.getElementById('authorLink').href = 'profile.php?id=' + data.author_id;
                    document.getElementById('videoStats').innerHTML = `
                        <span>${data.view_count} views</span>
                        <span>${data.like_count} likes</span>
                        <span>${data.dislike_count} dislikes</span>
                    `;
                    
                    document.getElementById('videoContainer').innerHTML = `
                        <video src="${data.url_video}" controls class="w-full h-full" autoplay>
                            Your browser does not support the video tag.
                        </video>
                    `;
                    
                    document.getElementById('videoDescription').textContent = data.description;
                    
                    const tagsHtml = data.tags.split(',').map(tag => 
                        `<span class="bg-blue-900/50 backdrop-blur-sm px-3 py-1 rounded-full text-sm">${tag.trim()}</span>`
                    ).join('');
                    document.getElementById('videoTags').innerHTML = tagsHtml;

                    loadComments(videoId);
                });
        }

        function closeModal() {
            document.getElementById('videoModal').classList.add('hidden');
            document.body.style.overflow = 'auto';
            currentVideoId = null;
        }

        function handleInteraction(type) {
            if (!currentVideoId) return;
            
            fetch('get_video.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=${type}&video_id=${currentVideoId}`
            }).then(() => {
                openModal(currentVideoId);
            });
        }

        function loadComments(videoId) {
            fetch('get_comments.php?video_id=' + videoId)
                .then(response => response.json())
                .then(comments => {
                    const commentsHtml = comments.map(comment => `
                        <div class="bg-gray-900/50 backdrop-blur-sm rounded-xl p-4">
                            <div class="flex justify-between mb-2">
                                <a href="profile.php?id=${comment.user_id}" class="text-blue-400 hover:text-blue-300 font-semibold">
                                    ${comment.username}
                                </a>
                                <span class="text-gray-400 text-sm">${comment.created_at}</span>
                            </div>
                            <p class="text-gray-300">${comment.comment_text}</p>
                        </div>
                    `).join('');
                    
                    document.getElementById('commentsList').innerHTML = commentsHtml;
                });
        }

        function handleComment(event) {
            event.preventDefault();
            if (!currentVideoId) return;
            
            const comment = document.getElementById('commentText').value;
            if (!comment.trim()) return;

            fetch('get_video.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `comment=${encodeURIComponent(comment)}&video_id=${currentVideoId}`
            }).then(() => {
                document.getElementById('commentText').value = '';
                loadComments(currentVideoId);
            });
        }

        document.addEventListener('keydown', e => {
            if (e.key === 'Escape') closeModal();
        });
    </script>
</body>
</html>