<?php
require_once 'auth_check.php';
require_once 'db.php';
session_start();

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="imators-verification" content="verified">
    <title>Imators Systems</title>
    <link href="https://cdn.imators.com/logo.png" rel="icon" type="image/png" />
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="finance.js"></script>
    <script src="workspace.js"></script>
    <script src="update_roleinimators.js"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {
                    fontFamily: {
                        'poppins': ['Poppins', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <link rel="stylesheet" href="dashboard.css">
    <script src="meeting.js"></script>
</head>
<body class="bg-white dark:bg-black text-black dark:text-white min-h-screen transition-colors duration-200 p-4 md:p-6 lg:p-8">
    <div class="loading-overlay hidden">
        <div class="loading-spinner"></div>
    </div>

    <div class="max-w-[1400px] mx-auto">
        <?php if(isset($_GET['success'])): ?>
        <div class="success-notification fixed top-4 right-4 bg-green-500 dark:bg-green-600 text-white px-6 py-3 rounded-lg shadow-lg z-50">
            <div class="flex items-center space-x-2">
                <i class="fas fa-check-circle"></i>
                <span>Operation completed successfully!</span>
            </div>
        </div>
        <?php endif; ?>

        <?php if(isset($_GET['error'])): ?>
        <div class="error-notification fixed top-4 right-4 bg-red-500 dark:bg-red-600 text-white px-6 py-3 rounded-lg shadow-lg z-50">
            <div class="flex items-center space-x-2">
                <i class="fas fa-exclamation-circle"></i>
                <span>An error occurred during the operation.</span>
            </div>
        </div>
        <?php endif; ?>

        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            <div class="flex flex-col lg:flex-row items-center justify-between mb-8 space-y-4 lg:space-y-0">
                <div class="flex items-center justify-between w-full lg:w-auto">
                    <div class="flex items-center space-x-4">
                        <img src="https://cdn.imators.com/logo.png" alt="Logo" class="h-10 lg:h-12 w-auto">
                        <div>
                            <h1 class="text-2xl lg:text-3xl font-light">Imators Guest</h1>
                            <p class="text-sm text-gray-600 dark:text-gray-400">Welcome to the dashboard</p>
                        </div>
                    </div>
                    <?php if(isset($_SESSION['user_id'])): ?>
                        <div class="flex lg:hidden relative" id="profile-menu">
                            <button class="flex items-center space-x-3 p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors">
                                <div class="w-10 h-10 rounded-full overflow-hidden bg-gray-200 dark:bg-gray-700 flex items-center justify-center">
                                    <?php if (!empty($_SESSION['profile_picture'])): ?>
                                        <img src="<?= htmlspecialchars($_SESSION['profile_picture']) ?>" 
                                             alt="Profile" 
                                             class="w-full h-full object-cover"
                                             onerror="this.onerror=null; this.src='https://cdn.imators.com/default-avatar.png';">
                                    <?php else: ?>
                                        <i class="fas fa-user text-gray-400 text-xl"></i>
                                    <?php endif; ?>
                                </div>
                                <i class="fas fa-chevron-down text-xs text-gray-400"></i>
                            </button>

                            <div class="absolute right-0 mt-2 w-64 bg-white dark:bg-gray-800 rounded-lg shadow-lg py-2 hidden z-50" 
                                 id="profile-dropdown">
                                <div class="px-4 py-3 border-b border-gray-200 dark:border-gray-700">
                                    <p class="text-sm font-medium"><?= htmlspecialchars($_SESSION['username']) ?></p>
                                    <p class="text-xs text-gray-500 dark:text-gray-400"><?= htmlspecialchars($_SESSION['email']) ?></p>
                                </div>
                                
                                <div class="py-2">
                                    <a href="#" id="view-profile" class="flex items-center px-4 py-2 text-sm text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700">
                                        <i class="fas fa-user-circle w-5"></i>
                                        <span>View Profile</span>
                                    </a>
                                    <a href="https://idsma.imators.com" class="flex items-center px-4 py-2 text-sm text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700">
                                        <i class="fas fa-cog w-5"></i>
                                        <span>Settings</span>
                                    </a>
                                </div>

                                <div class="border-t border-gray-200 dark:border-gray-700"></div>

                                <div class="py-2">
                                    <a href="logout.php" class="flex items-center px-4 py-2 text-sm text-red-600 dark:text-red-400 hover:bg-gray-100 dark:hover:bg-gray-700">
                                        <i class="fas fa-sign-out-alt w-5"></i>
                                        <span>Sign Out</span>
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>

                <div class="flex flex-col lg:flex-row items-center space-y-4 lg:space-y-0 lg:space-x-4 w-full lg:w-auto">
                    <div class="flex items-center space-x-4 w-full lg:w-auto">
                        <button id="theme-toggle" class="p-2 rounded-lg bg-gray-100 dark:bg-gray-800 hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors">
                            <i class="fas fa-sun dark:hidden"></i>
                            <i class="fas fa-moon hidden dark:block"></i>
                        </button>

                        <div class="relative w-full lg:w-96">
                            <input type="text" id="global-search" placeholder="Search users or documents..." 
                                class="w-full px-4 py-2 pl-10 bg-gray-100 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg
                                       focus:outline-none focus:ring-2 focus:ring-gray-400 dark:focus:ring-gray-500">
                            <i class="fas fa-search absolute left-3 top-3 text-gray-400"></i>
                        </div>
                    </div>

                    <?php if(isset($_SESSION['user_id'])): ?>
                        <div class="hidden lg:flex relative" id="profile-menu-desktop">
                            <button class="flex items-center space-x-3 p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors">
                                <div class="w-10 h-10 rounded-full overflow-hidden bg-gray-200 dark:bg-gray-700 flex items-center justify-center">
                                    <?php if (!empty($_SESSION['profile_picture'])): ?>
                                        <img src="<?= htmlspecialchars($_SESSION['profile_picture']) ?>" 
                                             alt="Profile" 
                                             class="w-full h-full object-cover"
                                             onerror="this.onerror=null; this.src='https://cdn.imators.com/default-avatar.png';">
                                    <?php else: ?>
                                        <i class="fas fa-user text-gray-400 text-xl"></i>
                                    <?php endif; ?>
                                </div>
                                <div class="text-left">
                                    <p class="text-sm font-medium"><?= htmlspecialchars($_SESSION['username']) ?></p>
                                    <p class="mt-1 text-xs">You are <?= htmlspecialchars($_SESSION['roleinimators'] ?? 'N/A') ?></p>
                                </div>
                                <i class="fas fa-chevron-down text-xs text-gray-400"></i>
                            </button>

                            <div class="absolute right-0 mt-2 w-64 bg-white dark:bg-gray-800 rounded-lg shadow-lg py-2 hidden z-50" 
                                 id="profile-dropdown-desktop">
                                <div class="px-4 py-3 border-b border-gray-200 dark:border-gray-700">
                                    <p class="text-sm font-medium"><?= htmlspecialchars($_SESSION['username']) ?></p>
                                    <p class="text-xs text-gray-500 dark:text-gray-400"><?= htmlspecialchars($_SESSION['email']) ?></p>
                                </div>
                                
                                <div class="py-2">
                                    <a href="#" id="view-profile-desktop" class="flex items-center px-4 py-2 text-sm text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700">
                                        <i class="fas fa-user-circle w-5"></i>
                                        <span>View Profile</span>
                                    </a>
                                    <a href="https://idsma.imators.com" class="flex items-center px-4 py-2 text-sm text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700">
                                        <i class="fas fa-cog w-5"></i>
                                        <span>Settings</span>
                                    </a>
                                </div>

                                <div class="border-t border-gray-200 dark:border-gray-700"></div>

                                <div class="py-2">
                                    <a href="logout.php" class="flex items-center px-4 py-2 text-sm text-red-600 dark:text-red-400 hover:bg-gray-100 dark:hover:bg-gray-700">
                                        <i class="fas fa-sign-out-alt w-5"></i>
                                        <span>Sign Out</span>
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <div id="profile-modal" class="fixed inset-0 bg-black/50 backdrop-blur-sm hidden items-center justify-center z-50">
            <div class="bg-white dark:bg-gray-800 rounded-lg w-full max-w-2xl mx-4 shadow-xl">
                <div class="p-6">
                    <div class="space-y-4">
                        <div class="flex justify-between items-start">
                            <h2 class="text-xl font-bold flex items-center space-x-2">
                                <i class="fas fa-user-circle text-gray-600 dark:text-gray-400"></i>
                                <span>My Imators Profile</span>
                            </h2>
                            <button class="close-profile-modal text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors ml-4">
                                <i class="fas fa-times text-xl"></i>
                            </button>
                        </div>
                        
                        <p class="text-sm text-gray-600 dark:text-gray-400 max-w-3xl mt-4">
                            This is the information related to your Imators account. For the modified render you on idsma.imators.com. 
                            It is strongly not recommended to send you test tickets between your Dashboard account and professional.
                        </p>
                    </div>

                    <div class="mt-4 grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div class="md:col-span-1">
                            <div class="flex flex-col items-center space-y-4">
                                <div class="w-32 h-32 rounded-full overflow-hidden bg-gray-200 dark:bg-gray-700">
                                    <?php if (!empty($_SESSION['profile_picture'])): ?>
                                        <img src="<?= htmlspecialchars($_SESSION['profile_picture']) ?>" 
                                             alt="Profile Picture" 
                                             class="w-full h-full object-cover"
                                             onerror="this.onerror=null; this.src='https://cdn.imators.com/default-avatar.png';">
                                    <?php else: ?>
                                        <div class="w-full h-full flex items-center justify-center">
                                            <i class="fas fa-user text-gray-400 text-4xl"></i>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <div class="md:col-span-2">
                            <div class="space-y-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">Username</label>
                                    <p class="mt-1 text-lg"><?= htmlspecialchars($_SESSION['username']) ?></p>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">Email</label>
                                    <p class="mt-1 text-lg"><?= htmlspecialchars($_SESSION['email']) ?></p>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">Role</label>
                                    <p class="mt-1 text-lg"><?= htmlspecialchars($_SESSION['role'] ? 'Employee' : 'User') ?></p>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">Imators Role</label>
                                    <p class="mt-1 text-lg"><?= htmlspecialchars($_SESSION['roleinimators'] ?? 'N/A') ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="mb-8 overflow-x-auto">
            <nav class="border-b border-gray-200 dark:border-gray-700 min-w-max">
                <div class="flex space-x-8">
                    <button class="user-tab px-4 py-3 font-medium border-b-2 border-black dark:border-white flex items-center space-x-2" data-tab="users">
                        <i class="fas fa-users"></i>
                        <span>Users</span>
                    </button>
                    <button class="doc-tab px-4 py-3 font-medium border-b-2 border-transparent hover:text-gray-600 dark:hover:text-gray-300 flex items-center space-x-2" data-tab="documents">
                        <i class="fas fa-file-alt"></i>
                        <span>Documents</span>
                    </button>
                    <button class="sites-tab px-4 py-3 font-medium border-b-2 border-transparent hover:text-gray-600 dark:hover:text-gray-300 flex items-center space-x-2" data-tab="sites">
                        <i class="fas fa-globe"></i>
                        <span>Websites</span>
                    </button>
                    <a href="tickets_page.php" class="tickets-tab px-4 py-3 font-medium border-b-2 border-transparent hover:text-gray-600 dark:hover:text-gray-300 flex items-center space-x-2" data-tab="tickets">
                        <i class="fas fa-ticket-alt"></i>
                        <span>Tickets</span>
                    </a>
                    <button class="email-tab px-4 py-3 font-medium border-b-2 border-transparent hover:text-gray-600 dark:hover:text-gray-300 flex items-center space-x-2" data-tab="email">
                        <i class="fas fa-envelope"></i>
                        <span>Email</span>
                    </button>
                    <button class="stats-tab px-4 py-3 font-medium border-b-2 border-transparent hover:text-gray-600 dark:hover:text-gray-300 flex items-center space-x-2" data-tab="stats">
                        <i class="fas fa-chart-bar"></i>
                        <span>Stats</span>
                    </button>
                    <button class="identity-tab px-4 py-3 font-medium border-b-2 border-transparent hover:text-gray-600 dark:hover:text-gray-300 flex items-center space-x-2" data-tab="identity">
                        <i class="fas fa-id-card"></i>
                        <span>Identity</span>
                    </button>
                    <button class="meeting-tab px-4 py-3 font-medium border-b-2 border-transparent hover:text-gray-600 dark:hover:text-gray-300 flex items-center space-x-2" data-tab="meeting">
    <i class="fas fa-video"></i>
    <span>Meeting</span>
</button>
                    
<?php if(isset($_SESSION['roleinimators']) && $_SESSION['roleinimators'] === 'CEO'): ?>
<button class="secure-settings-tab px-4 py-3 font-medium border-b-2 border-transparent hover:text-gray-600 dark:hover:text-gray-300 flex items-center space-x-2" data-tab="secure-settings">
    <i class="fas fa-shield-alt"></i>
    <span>Secure Settings</span>
</button>
<?php endif; ?>

                </div>
            </nav>
        </div>
        
<div id="meeting-content" class="tab-content hidden">
    <div id="mobile-block" class="hidden fixed inset-0 bg-black z-[9999] flex items-center justify-center p-4">
        <div class="text-center text-white">
            <i class="fas fa-mobile-alt text-6xl mb-4"></i>
            <h2 class="text-2xl font-bold mb-2">Desktop Only</h2>
            <p class="text-gray-300">Imators Meeting is only available on desktop devices for the best experience.</p>
        </div>
    </div>

    <div id="desktop-meeting" class="grid grid-cols-1 gap-6">
        <div class="bg-gray-100 dark:bg-gray-800 p-4 md:p-6 rounded-lg">
            <div class="flex flex-col md:flex-row justify-between items-start md:items-center gap-4 mb-6">
                <div class="flex flex-col">
                    <h2 class="text-xl md:text-2xl font-medium flex items-center space-x-2">
                        <i class="fas fa-video text-gray-600 dark:text-gray-400"></i>
                        <span>Imators Meeting</span>
                    </h2>
                    <p class="text-sm text-gray-500 dark:text-gray-400 mt-1">
                        Join or create secure video meetings with your team
                    </p>
                </div>
                <?php if(isset($_SESSION['roleinimators']) && $_SESSION['roleinimators'] === 'CEO'): ?>
                <button id="create-meeting-btn" class="w-full md:w-auto flex items-center justify-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                    <i class="fas fa-plus"></i>
                    <span>Create Meeting</span>
                </button>
                <?php endif; ?>
            </div>

            <div id="meeting-lobby" class="space-y-6">
                <div class="bg-white dark:bg-gray-900 p-4 md:p-6 rounded-lg">
                    <h3 class="text-lg font-medium mb-4 flex items-center space-x-2">
                        <i class="fas fa-sign-in-alt text-blue-500"></i>
                        <span>Join a Meeting</span>
                    </h3>
                    <div class="flex gap-3">
                        <input type="text" id="meeting-code-input" placeholder="Enter meeting code" maxlength="12"
                               class="flex-1 px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 border border-gray-200 dark:border-gray-600 focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white uppercase font-mono">
                        <button id="join-meeting-btn" class="px-6 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg transition-colors flex items-center space-x-2">
                            <i class="fas fa-arrow-right"></i>
                            <span>Join</span>
                        </button>
                    </div>
                </div>

                <div class="bg-white dark:bg-gray-900 p-4 md:p-6 rounded-lg">
                    <h3 class="text-lg font-medium mb-4 flex items-center space-x-2">
                        <i class="fas fa-clock text-green-500"></i>
                        <span>Active Meetings</span>
                    </h3>
                    <div id="active-meetings-list" class="space-y-3">
                        <div class="text-center text-gray-500 py-4">Loading meetings...</div>
                    </div>
                </div>

                <?php if(isset($_SESSION['roleinimators']) && $_SESSION['roleinimators'] === 'CEO'): ?>
                <div class="bg-white dark:bg-gray-900 p-4 md:p-6 rounded-lg">
                    <h3 class="text-lg font-medium mb-4 flex items-center space-x-2">
                        <i class="fas fa-history text-purple-500"></i>
                        <span>Meeting History</span>
                    </h3>
                    <div id="meeting-history" class="space-y-2 max-h-60 overflow-y-auto">
                        <div class="text-center text-gray-500 py-4">Loading history...</div>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <div id="meeting-room" class="hidden">
                <div class="bg-white dark:bg-gray-900 rounded-lg overflow-hidden">
                    <div class="p-4 border-b border-gray-200 dark:border-gray-700 flex justify-between items-center">
                        <div class="flex items-center gap-4">
                            <h3 class="text-xl font-medium" id="meeting-room-title">Meeting in progress</h3>
                            <span class="text-sm text-gray-500 font-mono" id="meeting-room-code"></span>
                            <button id="copy-meeting-link" class="text-xs px-3 py-1 bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                                <i class="fas fa-link mr-1"></i>Copy link
                            </button>
                        </div>
                        <div class="flex items-center space-x-2">
                            <span class="text-sm text-gray-500">
                                <i class="fas fa-users mr-1"></i>
                                <span id="participants-count">1</span>
                            </span>
                            <?php if(isset($_SESSION['roleinimators']) && $_SESSION['roleinimators'] === 'CEO'): ?>
                            <button id="end-meeting-btn" class="text-xs px-3 py-1 bg-red-600 hover:bg-red-700 text-white rounded-lg transition-colors">
                                <i class="fas fa-stop mr-1"></i>End for all
                            </button>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="grid grid-cols-4 gap-4 p-4">
                        <div class="col-span-3 space-y-4">
                            <div id="videos-grid" class="grid gap-3 min-h-[400px]"></div>
                            
                            <div class="bg-gray-100 dark:bg-gray-800 rounded-lg p-4">
                                <div class="flex justify-center items-center gap-4">
                                    <button id="toggle-audio" class="w-12 h-12 rounded-full bg-gray-600 hover:bg-gray-700 text-white transition-colors flex items-center justify-center" title="Microphone">
                                        <i class="fas fa-microphone"></i>
                                    </button>
                                    <button id="toggle-video" class="w-12 h-12 rounded-full bg-gray-600 hover:bg-gray-700 text-white transition-colors flex items-center justify-center" title="Camera">
                                        <i class="fas fa-video"></i>
                                    </button>
                                    <button id="share-screen" class="w-12 h-12 rounded-full bg-gray-600 hover:bg-gray-700 text-white transition-colors flex items-center justify-center" title="Share screen">
                                        <i class="fas fa-desktop"></i>
                                    </button>
                                    <button id="leave-meeting" class="w-12 h-12 rounded-full bg-red-600 hover:bg-red-700 text-white transition-colors flex items-center justify-center" title="Leave meeting">
                                        <i class="fas fa-phone-slash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>

                        <div class="col-span-1">
                            <div class="bg-gray-100 dark:bg-gray-800 rounded-lg h-full flex flex-col">
                                <div class="p-4 border-b border-gray-200 dark:border-gray-700">
                                    <div class="flex space-x-2">
                                        <button id="tab-chat" class="flex-1 py-2 text-sm font-medium border-b-2 border-black dark:border-white">
                                            Chat
                                        </button>
                                        <button id="tab-participants" class="flex-1 py-2 text-sm font-medium border-b-2 border-transparent text-gray-500">
                                            Participants
                                        </button>
                                    </div>
                                </div>

                                <div id="chat-panel" class="flex-1 flex flex-col min-h-0">
                                    <div id="chat-messages" class="flex-1 p-4 overflow-y-auto space-y-2"></div>
                                    <div class="p-4 border-t border-gray-200 dark:border-gray-700">
                                        <div class="flex space-x-2">
                                            <input type="text" id="chat-input" placeholder="Message..." 
                                                   class="flex-1 px-3 py-2 text-sm rounded-lg bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                                            <button id="send-message" class="px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                                                <i class="fas fa-paper-plane"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>

                                <div id="participants-panel" class="hidden flex-1 p-4 overflow-y-auto">
                                    <div id="participants-list" class="space-y-2"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div id="create-meeting-modal" class="hidden fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
    <div class="bg-white dark:bg-gray-800 rounded-lg w-full max-w-md shadow-xl">
        <div class="p-6">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-xl font-bold flex items-center space-x-2">
                    <i class="fas fa-video text-green-500"></i>
                    <span>Create Meeting</span>
                </h2>
                <button class="close-create-meeting text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>

            <form id="create-meeting-form" class="space-y-4">
                <div>
                    <label class="block text-sm font-medium mb-2">Meeting Title</label>
                    <input type="text" id="meeting-title" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 border border-gray-200 dark:border-gray-600 focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Description (optional)</label>
                    <textarea id="meeting-description" rows="3"
                              class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 border border-gray-200 dark:border-gray-600 focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white"></textarea>
                </div>

                <div class="flex justify-end space-x-3 pt-4">
                    <button type="button" class="flex items-center space-x-2 px-4 py-2 border border-gray-200 dark:border-gray-600 rounded-lg text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors close-create-meeting">
                        <i class="fas fa-times"></i>
                        <span>Cancel</span>
                    </button>
                    <button type="submit" class="flex items-center space-x-2 px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg transition-colors">
                        <i class="fas fa-plus"></i>
                        <span>Create</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>
        
        <div id="secure-settings-content" class="tab-content hidden">
    <div class="bg-gray-100 dark:bg-gray-800 p-6 rounded-lg">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-xl font-medium flex items-center space-x-2">
                <i class="fas fa-shield-alt text-red-600 dark:text-red-400"></i>
                <span>Security Password Management</span>
            </h2>
            <div class="flex space-x-2">
                <button id="generate-all-passwords" class="px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg transition-colors flex items-center space-x-2">
                    <i class="fas fa-magic"></i>
                    <span>Generate 20 Passwords</span>
                </button>
                <button id="refresh-password-list" class="px-4 py-2 bg-gray-600 hover:bg-gray-700 text-white rounded-lg transition-colors">
                    <i class="fas fa-sync-alt"></i>
                </button>
            </div>
        </div>

        <div class="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-700 rounded-lg p-4 mb-6">
            <div class="flex items-start space-x-2">
                <i class="fas fa-exclamation-triangle text-red-500 mt-0.5"></i>
                <div class="text-sm text-red-700 dark:text-red-300">
                    <p class="font-medium mb-1">CEO Access Only</p>
                    <p>This section manages the 20 security passwords for accessing restricted tabs. Handle with extreme care.</p>
                </div>
            </div>
        </div>

        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <div class="bg-white dark:bg-gray-900 p-4 rounded-lg">
                <h3 class="text-lg font-medium mb-4 flex items-center space-x-2">
                    <i class="fas fa-plus text-green-500"></i>
                    <span>Create New Password</span>
                </h3>
                
                <form id="create-password-form" class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium mb-2">Password (12 characters)</label>
                        <div class="flex space-x-2">
                            <input type="text" id="new-password" maxlength="12" 
                                   class="flex-1 px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                          border border-gray-200 dark:border-gray-600
                                          focus:outline-none focus:ring-2 focus:ring-red-500
                                          font-mono"
                                   placeholder="Enter 12 chars">
                            <button type="button" id="generate-random" class="px-3 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                                <i class="fas fa-dice"></i>
                            </button>
                        </div>
                        <div class="text-xs text-gray-500 mt-1">
                            Length: <span id="password-length">0</span>/12
                        </div>
                    </div>
                    
                    <button type="submit" class="w-full px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg transition-colors flex items-center justify-center space-x-2">
                        <i class="fas fa-plus"></i>
                        <span>Create Password</span>
                    </button>
                </form>
            </div>

            <div class="bg-white dark:bg-gray-900 p-4 rounded-lg">
                <h3 class="text-lg font-medium mb-4 flex items-center space-x-2">
                    <i class="fas fa-list text-blue-500"></i>
                    <span>Password Statistics</span>
                </h3>
                
                <div class="space-y-3">
                    <div class="flex justify-between">
                        <span class="text-sm text-gray-600 dark:text-gray-400">Total Passwords:</span>
                        <span class="text-sm font-medium" id="total-passwords">0</span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-sm text-gray-600 dark:text-gray-400">Active:</span>
                        <span class="text-sm font-medium text-green-600" id="active-passwords">0</span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-sm text-gray-600 dark:text-gray-400">Inactive:</span>
                        <span class="text-sm font-medium text-red-600" id="inactive-passwords">0</span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-sm text-gray-600 dark:text-gray-400">Most Used:</span>
                        <span class="text-sm font-medium" id="max-usage">0</span>
                    </div>
                </div>
            </div>
        </div>

        <div class="bg-white dark:bg-gray-900 p-4 rounded-lg mt-6">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-medium flex items-center space-x-2">
                    <i class="fas fa-key text-yellow-500"></i>
                    <span>Password List</span>
                </h3>
                <div class="text-sm text-gray-500">
                    Click password to reveal • Right-click to copy
                </div>
            </div>

            <div class="overflow-x-auto">
                <table class="min-w-full">
                    <thead class="bg-gray-50 dark:bg-gray-800">
                        <tr>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">ID</th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Password</th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Status</th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Usage</th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Last Used</th>
                            <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="password-list" class="divide-y divide-gray-200 dark:divide-gray-700">
                        <tr>
                            <td colspan="6" class="px-4 py-8 text-center text-gray-500">
                                <i class="fas fa-spinner fa-spin mr-2"></i>
                                Loading passwords...
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <div id="generated-passwords-modal" class="hidden fixed inset-0 bg-black/70 backdrop-blur-sm flex items-center justify-center z-50">
            <div class="bg-white dark:bg-gray-800 rounded-lg w-full max-w-2xl mx-4 shadow-xl">
                <div class="p-6">
                    <div class="flex items-center space-x-3 mb-6">
                        <div class="w-12 h-12 bg-green-100 dark:bg-green-900 rounded-full flex items-center justify-center">
                            <i class="fas fa-check text-green-600 dark:text-green-400 text-xl"></i>
                        </div>
                        <div>
                            <h2 class="text-xl font-bold text-green-600 dark:text-green-400">Passwords Generated</h2>
                            <p class="text-sm text-gray-600 dark:text-gray-400">20 new security passwords created</p>
                        </div>
                    </div>
                    
                    <div class="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-700 rounded-lg p-4 mb-6">
                        <div class="flex items-start space-x-2">
                            <i class="fas fa-exclamation-triangle text-yellow-500 mt-0.5"></i>
                            <div class="text-sm text-yellow-700 dark:text-yellow-300">
                                <p class="font-medium mb-1">Important</p>
                                <p>Save these passwords securely! They will not be displayed again.</p>
                            </div>
                        </div>
                    </div>

                    <div class="bg-gray-100 dark:bg-gray-700 rounded-lg p-4 mb-6">
                        <div class="grid grid-cols-2 gap-2 text-sm font-mono" id="generated-passwords-list">
                        </div>
                    </div>

                    <div class="flex justify-end space-x-3">
                        <button id="copy-all-passwords" class="px-4 py-2 border border-gray-200 dark:border-gray-600 rounded-lg text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors">
                            <i class="fas fa-copy mr-2"></i>Copy All
                        </button>
                        <button id="close-passwords-modal" class="px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-lg transition-colors">
                            I've Saved Them
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const newPasswordInput = document.getElementById('new-password');
    const passwordLengthSpan = document.getElementById('password-length');
    
    if (newPasswordInput) {
        newPasswordInput.addEventListener('input', function() {
            passwordLengthSpan.textContent = this.value.length;
        });

        document.getElementById('generate-random').addEventListener('click', function() {
            const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*';
            let password = '';
            for (let i = 0; i < 12; i++) {
                password += chars.charAt(Math.floor(Math.random() * chars.length));
            }
            newPasswordInput.value = password;
            passwordLengthSpan.textContent = '12';
        });

        document.getElementById('create-password-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            const password = newPasswordInput.value;
            
            if (password.length !== 12) {
                alert('Password must be exactly 12 characters');
                return;
            }

            try {
                const response = await fetch('secure_auth.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        action: 'create_password',
                        password: password
                    })
                });

                const data = await response.json();
                
                if (data.success) {
                    newPasswordInput.value = '';
                    passwordLengthSpan.textContent = '0';
                    loadPasswordList();
                    alert('Password created successfully');
                } else {
                    alert(data.message);
                }
            } catch (error) {
                alert('Error creating password');
            }
        });

        document.getElementById('generate-all-passwords').addEventListener('click', async function() {
            if (!confirm('This will delete all existing passwords and generate 20 new ones. Continue?')) {
                return;
            }

            try {
                const response = await fetch('secure_auth.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'generate_passwords' })
                });

                const data = await response.json();
                
                if (data.success) {
                    showGeneratedPasswords(data.passwords);
                    loadPasswordList();
                } else {
                    alert(data.message);
                }
            } catch (error) {
                alert('Error generating passwords');
            }
        });

        document.getElementById('refresh-password-list').addEventListener('click', loadPasswordList);

        document.querySelector('.secure-settings-tab').addEventListener('click', function() {
            loadPasswordList();
        });
    }

    function showGeneratedPasswords(passwords) {
        const modal = document.getElementById('generated-passwords-modal');
        const list = document.getElementById('generated-passwords-list');
        
        list.innerHTML = '';
        passwords.forEach((password, index) => {
            const div = document.createElement('div');
            div.className = 'p-2 bg-white dark:bg-gray-800 rounded border';
            div.innerHTML = `<span class="text-xs text-gray-500">${index + 1}:</span> ${password}`;
            list.appendChild(div);
        });

        modal.classList.remove('hidden');

        document.getElementById('copy-all-passwords').onclick = function() {
            navigator.clipboard.writeText(passwords.join('\n'));
            this.innerHTML = '<i class="fas fa-check mr-2"></i>Copied!';
            setTimeout(() => {
                this.innerHTML = '<i class="fas fa-copy mr-2"></i>Copy All';
            }, 2000);
        };

        document.getElementById('close-passwords-modal').onclick = function() {
            modal.classList.add('hidden');
        };
    }

    async function loadPasswordList() {
        try {
            const response = await fetch('secure_auth.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action: 'get_passwords' })
            });

            const data = await response.json();
            
            if (data.success) {
                updatePasswordList(data.passwords);
                updateStats(data.passwords);
            }
        } catch (error) {
            console.error('Error loading passwords:', error);
        }
    }

    function updatePasswordList(passwords) {
        const tbody = document.getElementById('password-list');
        tbody.innerHTML = '';

        if (passwords.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="px-4 py-8 text-center text-gray-500">No passwords found</td></tr>';
            return;
        }

        passwords.forEach(password => {
            const row = document.createElement('tr');
            row.className = 'hover:bg-gray-50 dark:hover:bg-gray-800';
            row.innerHTML = `
                <td class="px-4 py-3 text-sm">${password.id}</td>
                <td class="px-4 py-3 text-sm font-mono">
                    <span class="password-hidden cursor-pointer text-gray-500">••••••••••••</span>
                </td>
                <td class="px-4 py-3 text-sm">
                    <span class="px-2 py-1 text-xs rounded-full ${password.is_active ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200'}">
                        ${password.is_active ? 'Active' : 'Inactive'}
                    </span>
                </td>
                <td class="px-4 py-3 text-sm">${password.usage_count}</td>
                <td class="px-4 py-3 text-sm">${password.last_used || 'Never'}</td>
                <td class="px-4 py-3 text-sm space-x-2">
                    <button class="toggle-password text-blue-600 hover:text-blue-800" data-id="${password.id}">
                        <i class="fas fa-toggle-${password.is_active ? 'on' : 'off'}"></i>
                    </button>
                    <button class="delete-password text-red-600 hover:text-red-800" data-id="${password.id}">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            `;
            tbody.appendChild(row);
        });

        tbody.querySelectorAll('.toggle-password').forEach(btn => {
            btn.addEventListener('click', async function() {
                const id = this.dataset.id;
                await togglePassword(id);
            });
        });

        tbody.querySelectorAll('.delete-password').forEach(btn => {
            btn.addEventListener('click', async function() {
                const id = this.dataset.id;
                if (confirm('Delete this password?')) {
                    await deletePassword(id);
                }
            });
        });
    }

    function updateStats(passwords) {
        const total = passwords.length;
        const active = passwords.filter(p => p.is_active).length;
        const inactive = total - active;
        const maxUsage = Math.max(...passwords.map(p => p.usage_count), 0);

        document.getElementById('total-passwords').textContent = total;
        document.getElementById('active-passwords').textContent = active;
        document.getElementById('inactive-passwords').textContent = inactive;
        document.getElementById('max-usage').textContent = maxUsage;
    }

    async function togglePassword(id) {
        try {
            const response = await fetch('secure_auth.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'toggle_password',
                    passwordId: parseInt(id)
                })
            });

            const data = await response.json();
            if (data.success) {
                loadPasswordList();
            }
        } catch (error) {
            console.error('Error toggling password:', error);
        }
    }

    async function deletePassword(id) {
        try {
            const response = await fetch('secure_auth.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'delete_password',
                    passwordId: parseInt(id)
                })
            });

            const data = await response.json();
            if (data.success) {
                loadPasswordList();
            }
        } catch (error) {
            console.error('Error deleting password:', error);
        }
    }
});
</script>
        
        <div id="identity-content" class="tab-content hidden">
    <div class="bg-gray-100 dark:bg-gray-800 p-6 rounded-lg">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-xl font-medium flex items-center space-x-2">
                <i class="fas fa-id-card text-gray-600 dark:text-gray-400"></i>
                <span>Identity Verifications</span>
            </h2>
            <button id="refresh-identity"
                    class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                <i class="fas fa-sync-alt"></i>
                <span>Refresh</span>
            </button>
        </div>

        <div class="bg-blue-50 dark:bg-blue-900 p-4 rounded-lg mb-6 border border-blue-200 dark:border-blue-700">
            <h3 class="font-medium mb-2 flex items-center space-x-2">
                <i class="fas fa-info-circle text-blue-500 dark:text-blue-300"></i>
                <span>Warning!</span>
            </h3>
            <p class="text-sm text-blue-700 dark:text-blue-300">
            If the <strong>service</strong> is not <strong>Imators</strong> or <strong>service</strong>,
            approval must be done via the associated database:
            <br>
            Example: <strong>VertChasseur</strong> → <strong>VertChasseur</strong> DB (third-party DB).
        </p>
        </div>

        <div class="overflow-x-auto">
            <table class="min-w-full bg-white dark:bg-gray-900 rounded-lg shadow-sm">
                <thead class="bg-gray-50 dark:bg-gray-800">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">ID</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Profile ID</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Service</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Country</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Status</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody id="identity-verifications-list">
                </tbody>
            </table>
        </div>
    </div>
</div>


<script>
    
    document.addEventListener('DOMContentLoaded', function() {
    // Gestion de l'onglet Identity
    document.querySelector('.identity-tab').addEventListener('click', function() {
        loadIdentityVerifications();
    });

    // Gestion du bouton Refresh
    document.getElementById('refresh-identity').addEventListener('click', function() {
        loadIdentityVerifications();
    });

    // Fonction pour charger les vérifications d'identité
    function loadIdentityVerifications() {
        const listElement = document.getElementById('identity-verifications-list');
        listElement.innerHTML = '<tr><td colspan="6" class="text-center py-4"><i class="fas fa-spinner fa-spin"></i> Loading...</td></tr>';

        fetch('get_identity_verifications.php')
            .then(response => response.json())
            .then(data => {
                listElement.innerHTML = '';
                if (data.length === 0) {
                    listElement.innerHTML = '<tr><td colspan="6" class="text-center py-4 text-gray-500 dark:text-gray-400">No identity verifications found.</td></tr>';
                    return;
                }

                data.forEach(verification => {
                    const row = document.createElement('tr');
                    row.className = 'border-b border-gray-200 dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors';

                    // Déterminer la couleur du statut
                    let statusColor = '';
                    switch(verification.status) {
                        case 'approved':
                            statusColor = 'text-green-500';
                            break;
                        case 'pending':
                            statusColor = 'text-yellow-500';
                            break;
                        case 'rejected':
                            statusColor = 'text-red-500';
                            break;
                        default:
                            statusColor = 'text-gray-500 dark:text-gray-400';
                    }

                    row.innerHTML = `
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 dark:text-white">${verification.id}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">${verification.profile_id}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">${verification.service}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">${verification.country}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm ${statusColor}">${verification.status}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                            <button class="text-blue-600 dark:text-blue-400 hover:text-blue-900 dark:hover:text-blue-300">
                                <i class="fas fa-eye"></i> View
                            </button>
                        </td>
                    `;
                    listElement.appendChild(row);
                });
            })
            .catch(error => {
                console.error('Error loading identity verifications:', error);
                listElement.innerHTML = '<tr><td colspan="6" class="text-center py-4 text-red-500 dark:text-red-400">Error loading data.</td></tr>';
            });
    }
});

    
</script>
        
<div id="stats-content" class="tab-content hidden">
    <div class="bg-gray-100 dark:bg-gray-800 p-4 rounded-lg">
        <div class="flex justify-between items-center mb-4">
            <h2 class="text-lg font-medium flex items-center space-x-2">
                <i class="fas fa-chart-bar text-gray-600 dark:text-gray-400"></i>
                <span>Visitor Statistics (imators.com)</span>
            </h2>
            <div class="flex items-center space-x-3">
                <span class="text-xs text-gray-500 dark:text-gray-400">Auto-refresh: <span id="refresh-countdown">10</span>s</span>
                <button id="refresh-stats" class="text-xs px-3 py-1 bg-black dark:bg-white text-white dark:text-black rounded hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                    <i class="fas fa-sync-alt"></i> Refresh
                </button>
            </div>
        </div>

        <div class="grid grid-cols-2 lg:grid-cols-6 gap-3 mb-4">
            <div class="bg-white dark:bg-gray-900 p-3 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <p class="text-xs text-gray-500 dark:text-gray-400 mb-1">Total Visits</p>
                <h3 class="text-xl font-bold text-black dark:text-white transition-all duration-500" id="total-visits">--</h3>
            </div>
            <div class="bg-white dark:bg-gray-900 p-3 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <p class="text-xs text-gray-500 dark:text-gray-400 mb-1">Today</p>
                <h3 class="text-xl font-bold text-black dark:text-white transition-all duration-500" id="today-visits">--</h3>
            </div>
            <div class="bg-white dark:bg-gray-900 p-3 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <p class="text-xs text-gray-500 dark:text-gray-400 mb-1">This Week</p>
                <h3 class="text-xl font-bold text-black dark:text-white transition-all duration-500" id="week-visits">--</h3>
            </div>
            <div class="bg-white dark:bg-gray-900 p-3 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <p class="text-xs text-gray-500 dark:text-gray-400 mb-1">Countries</p>
                <h3 class="text-xl font-bold text-black dark:text-white transition-all duration-500" id="unique-countries">--</h3>
            </div>
            <div class="bg-white dark:bg-gray-900 p-3 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <p class="text-xs text-gray-500 dark:text-gray-400 mb-1">Languages</p>
                <h3 class="text-xl font-bold text-black dark:text-white transition-all duration-500" id="unique-languages">--</h3>
            </div>
            <div class="bg-white dark:bg-gray-900 p-3 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <p class="text-xs text-gray-500 dark:text-gray-400 mb-1">Avg/Day</p>
                <h3 class="text-xl font-bold text-black dark:text-white transition-all duration-500" id="avg-daily">--</h3>
            </div>
        </div>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-4 mb-4">
            <div class="bg-white dark:bg-gray-900 p-4 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <h3 class="text-sm font-medium mb-3 flex items-center space-x-2">
                    <i class="fas fa-flag text-gray-600 dark:text-gray-400"></i>
                    <span>Countries</span>
                    <span class="text-xs text-gray-500 transition-all duration-300" id="countries-total">(0)</span>
                </h3>
                <div id="countries-list" class="space-y-2 max-h-48 overflow-y-auto"></div>
            </div>
            
            <div class="bg-white dark:bg-gray-900 p-4 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <h3 class="text-sm font-medium mb-3 flex items-center space-x-2">
                    <i class="fas fa-language text-gray-600 dark:text-gray-400"></i>
                    <span>Languages</span>
                    <span class="text-xs text-gray-500 transition-all duration-300" id="languages-total">(0)</span>
                </h3>
                <div id="languages-list" class="space-y-2 max-h-48 overflow-y-auto"></div>
            </div>

            <div class="bg-white dark:bg-gray-900 p-4 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <h3 class="text-sm font-medium mb-3 flex items-center space-x-2">
                    <i class="fas fa-clock text-gray-600 dark:text-gray-400"></i>
                    <span>Recent Activity</span>
                </h3>
                <div id="recent-visits" class="space-y-2 max-h-48 overflow-y-auto text-xs"></div>
            </div>
        </div>

        <div class="grid grid-cols-1 lg:grid-cols-2 gap-4">
            <div class="bg-white dark:bg-gray-900 p-4 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <h3 class="text-sm font-medium mb-3 flex items-center space-x-2">
                    <i class="fas fa-globe text-gray-600 dark:text-gray-400"></i>
                    <span>IP Addresses</span>
                    <span class="text-xs text-gray-500 transition-all duration-300" id="ips-total">(0 unique)</span>
                </h3>
                <div id="ips-list" class="space-y-2 max-h-32 overflow-y-auto text-xs"></div>
            </div>

            <div class="bg-white dark:bg-gray-900 p-4 rounded border border-gray-200 dark:border-gray-700 transition-all duration-300 hover:shadow-md">
                <h3 class="text-sm font-medium mb-3 flex items-center space-x-2">
                    <i class="fas fa-calendar text-gray-600 dark:text-gray-400"></i>
                    <span>Daily Breakdown</span>
                </h3>
                <div id="daily-breakdown" class="space-y-1 max-h-32 overflow-y-auto text-xs"></div>
            </div>
        </div>
    </div>
</div>

<script>
let statsRefreshInterval;
let countdownInterval;
let countdownValue = 10;

function animateNumber(elementId, newValue) {
    const element = document.getElementById(elementId);
    const currentValue = parseInt(element.textContent) || 0;
    
    if (currentValue === newValue) return;
    
    const duration = 800;
    const startTime = performance.now();
    const difference = newValue - currentValue;
    
    function updateNumber(currentTime) {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);
        
        const easeOutQuart = 1 - Math.pow(1 - progress, 4);
        const currentNumber = Math.round(currentValue + (difference * easeOutQuart));
        
        element.textContent = currentNumber;
        
        if (progress < 1) {
            requestAnimationFrame(updateNumber);
        } else {
            element.textContent = newValue;
        }
    }
    
    requestAnimationFrame(updateNumber);
}

function loadStats() {
    fetch('get_stats.php')
        .then(response => response.json())
        .then(data => {
            animateNumber('total-visits', data.total_visits);
            animateNumber('today-visits', data.today_visits);
            animateNumber('week-visits', data.week_visits);
            animateNumber('unique-countries', data.unique_countries);
            animateNumber('unique-languages', data.unique_languages);
            animateNumber('avg-daily', data.avg_daily);
            
            updateCountriesList(data.top_countries);
            updateLanguagesList(data.top_languages);
            updateRecentVisits(data.recent_visits);
            updateIPsList(data.top_ips);
            updateDailyBreakdown(data.daily_breakdown);
            
            resetCountdown();
        })
        .catch(error => console.error('Stats error:', error));
}

function updateCountriesList(countries) {
    const container = document.getElementById('countries-list');
    const totalElement = document.getElementById('countries-total');
    
    container.style.opacity = '0.5';
    setTimeout(() => {
        container.innerHTML = '';
        totalElement.textContent = `(${countries.length})`;
        
        countries.forEach((country, index) => {
            const item = document.createElement('div');
            item.className = 'flex justify-between items-center py-1 border-b border-gray-100 dark:border-gray-700 opacity-0 transform translate-y-2 transition-all duration-300';
            item.innerHTML = `
                <span class="text-sm">${country.country || 'Unknown'}</span>
                <span class="text-sm font-medium">${country.count}</span>
            `;
            container.appendChild(item);
            
            setTimeout(() => {
                item.classList.remove('opacity-0', 'translate-y-2');
            }, index * 50);
        });
        
        container.style.opacity = '1';
    }, 200);
}

function updateLanguagesList(languages) {
    const container = document.getElementById('languages-list');
    const totalElement = document.getElementById('languages-total');
    
    container.style.opacity = '0.5';
    setTimeout(() => {
        container.innerHTML = '';
        totalElement.textContent = `(${languages.length})`;
        
        languages.forEach((lang, index) => {
            const item = document.createElement('div');
            item.className = 'flex justify-between items-center py-1 border-b border-gray-100 dark:border-gray-700 opacity-0 transform translate-y-2 transition-all duration-300';
            item.innerHTML = `
                <span class="text-sm">${lang.language || 'Unknown'}</span>
                <span class="text-sm font-medium">${lang.count}</span>
            `;
            container.appendChild(item);
            
            setTimeout(() => {
                item.classList.remove('opacity-0', 'translate-y-2');
            }, index * 50);
        });
        
        container.style.opacity = '1';
    }, 200);
}

function updateRecentVisits(visits) {
    const container = document.getElementById('recent-visits');
    
    container.style.opacity = '0.5';
    setTimeout(() => {
        container.innerHTML = '';
        
        visits.forEach((visit, index) => {
            const item = document.createElement('div');
            item.className = 'p-2 bg-gray-50 dark:bg-gray-800 rounded opacity-0 transform translate-y-2 transition-all duration-300';
            item.innerHTML = `
                <div class="flex justify-between">
                    <span class="font-medium">${visit.country || 'Unknown'}</span>
                    <span class="text-gray-500">${visit.time_ago}</span>
                </div>
                <div class="text-gray-600 dark:text-gray-400">${visit.ip_address} • ${visit.language || 'Unknown'}</div>
            `;
            container.appendChild(item);
            
            setTimeout(() => {
                item.classList.remove('opacity-0', 'translate-y-2');
            }, index * 100);
        });
        
        container.style.opacity = '1';
    }, 200);
}

function updateIPsList(ips) {
    const container = document.getElementById('ips-list');
    const totalElement = document.getElementById('ips-total');
    
    container.style.opacity = '0.5';
    setTimeout(() => {
        container.innerHTML = '';
        totalElement.textContent = `(${ips.length} unique)`;
        
        ips.forEach((ip, index) => {
            const item = document.createElement('div');
            item.className = 'flex justify-between items-center py-1 opacity-0 transform translate-y-2 transition-all duration-300';
            item.innerHTML = `
                <span>${ip.ip_address}</span>
                <span class="font-medium">${ip.count}</span>
            `;
            container.appendChild(item);
            
            setTimeout(() => {
                item.classList.remove('opacity-0', 'translate-y-2');
            }, index * 75);
        });
        
        container.style.opacity = '1';
    }, 200);
}

function updateDailyBreakdown(daily) {
    const container = document.getElementById('daily-breakdown');
    
    container.style.opacity = '0.5';
    setTimeout(() => {
        container.innerHTML = '';
        
        daily.forEach((day, index) => {
            const item = document.createElement('div');
            item.className = 'flex justify-between items-center py-1 opacity-0 transform translate-y-2 transition-all duration-300';
            item.innerHTML = `
                <span>${day.date}</span>
                <span class="font-medium">${day.count} visits</span>
            `;
            container.appendChild(item);
            
            setTimeout(() => {
                item.classList.remove('opacity-0', 'translate-y-2');
            }, index * 75);
        });
        
        container.style.opacity = '1';
    }, 200);
}

function startAutoRefresh() {
    if (statsRefreshInterval) clearInterval(statsRefreshInterval);
    if (countdownInterval) clearInterval(countdownInterval);
    
    statsRefreshInterval = setInterval(loadStats, 10000);
    startCountdown();
}

function stopAutoRefresh() {
    if (statsRefreshInterval) clearInterval(statsRefreshInterval);
    if (countdownInterval) clearInterval(countdownInterval);
}

function startCountdown() {
    countdownValue = 10;
    document.getElementById('refresh-countdown').textContent = countdownValue;
    
    countdownInterval = setInterval(() => {
        countdownValue--;
        document.getElementById('refresh-countdown').textContent = countdownValue;
        if (countdownValue <= 0) {
            countdownValue = 10;
        }
    }, 1000);
}

function resetCountdown() {
    if (countdownInterval) clearInterval(countdownInterval);
    startCountdown();
}

document.querySelector('.stats-tab').addEventListener('click', function() {
    loadStats();
    startAutoRefresh();
});

document.getElementById('refresh-stats').addEventListener('click', function() {
    loadStats();
    resetCountdown();
});

document.querySelectorAll('.tab-content').forEach(tab => {
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.attributeName === 'class') {
                const target = mutation.target;
                if (target.id === 'stats-content' && target.classList.contains('hidden')) {
                    stopAutoRefresh();
                }
            }
        });
    });
    observer.observe(tab, { attributes: true });
});
</script>

        <div id="visits-content" class="tab-content hidden">
    <div class="grid grid-cols-1 gap-6">
        <div class="bg-gray-100 dark:bg-gray-800 p-6 rounded-lg">
            <div class="flex justify-between items-center mb-6">
                <div class="flex flex-col">
                    <h2 class="text-xl font-medium flex items-center space-x-2">
                        <i class="fas fa-chart-line text-gray-600 dark:text-gray-400"></i>
                        <span>Analytics Dashboard</span>
                    </h2>
                    <p class="text-sm text-gray-500 dark:text-gray-400 mt-1">
                        View traffic statistics for your websites via Umami API
                    </p>
                </div>
                <div class="flex space-x-2">
                    <select id="timeframe-selector" class="text-sm p-2 bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg">
                        <option value="24h">Last 24 hours</option>
                        <option value="7d" selected>Last 7 days</option>
                        <option value="30d">Last 30 days</option>
                        <option value="6m">Last 6 months</option>
                    </select>
                    <button id="refresh-analytics" class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                        <i class="fas fa-sync-alt"></i>
                        <span>Refresh</span>
                    </button>
                </div>
            </div>

            <!-- API Key Config -->
            <div id="api-key-config" class="mb-6 p-4 bg-white dark:bg-gray-900 rounded-lg">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-medium">Umami API Configuration</h3>
                    <button id="save-api-config" class="text-xs p-2 bg-black text-white dark:bg-white dark:text-black rounded-lg">
                        <i class="fas fa-save"></i> Save
                    </button>
                </div>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium mb-2" for="umami-api-url">API URL</label>
                        <input type="text" id="umami-api-url" placeholder="https://analytics.example.com/api" 
                               class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 border border-gray-200 dark:border-gray-600 focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                    </div>
                    <div>
                        <label class="block text-sm font-medium mb-2" for="umami-api-key">API Key</label>
                        <input type="password" id="umami-api-key" placeholder="Your Umami API key" 
                               class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 border border-gray-200 dark:border-gray-600 focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                    </div>
                </div>
            </div>

            <!-- Website selector -->
            <div id="website-selector" class="mb-6">
                <label class="block text-sm font-medium mb-2" for="website-select">Select Website</label>
                <select id="website-select" class="w-full px-4 py-2 rounded-lg bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                    <option value="">Loading websites...</option>
                </select>
            </div>

            <!-- Stats Overview -->
            <div id="stats-overview" class="mb-6 grid grid-cols-1 md:grid-cols-4 gap-4">
                <div class="bg-white dark:bg-gray-900 p-4 rounded-lg shadow">
                    <div class="flex justify-between items-start">
                        <div>
                            <p class="text-gray-500 dark:text-gray-400 text-sm font-medium">Page Views</p>
                            <h3 class="text-2xl font-bold" id="page-views-count">--</h3>
                        </div>
                        <div class="bg-blue-100 dark:bg-blue-900 p-2 rounded-lg">
                            <i class="fas fa-eye text-blue-500"></i>
                        </div>
                    </div>
                    <p class="text-xs text-gray-500 mt-2" id="pageviews-trend">--</p>
                </div>
                
                <div class="bg-white dark:bg-gray-900 p-4 rounded-lg shadow">
                    <div class="flex justify-between items-start">
                        <div>
                            <p class="text-gray-500 dark:text-gray-400 text-sm font-medium">Unique Visitors</p>
                            <h3 class="text-2xl font-bold" id="visitors-count">--</h3>
                        </div>
                        <div class="bg-green-100 dark:bg-green-900 p-2 rounded-lg">
                            <i class="fas fa-users text-green-500"></i>
                        </div>
                    </div>
                    <p class="text-xs text-gray-500 mt-2" id="visitors-trend">--</p>
                </div>
                
                <div class="bg-white dark:bg-gray-900 p-4 rounded-lg shadow">
                    <div class="flex justify-between items-start">
                        <div>
                            <p class="text-gray-500 dark:text-gray-400 text-sm font-medium">Bounce Rate</p>
                            <h3 class="text-2xl font-bold" id="bounce-rate">--</h3>
                        </div>
                        <div class="bg-yellow-100 dark:bg-yellow-900 p-2 rounded-lg">
                            <i class="fas fa-sign-out-alt text-yellow-500"></i>
                        </div>
                    </div>
                    <p class="text-xs text-gray-500 mt-2" id="bounce-trend">--</p>
                </div>
                
                <div class="bg-white dark:bg-gray-900 p-4 rounded-lg shadow">
                    <div class="flex justify-between items-start">
                        <div>
                            <p class="text-gray-500 dark:text-gray-400 text-sm font-medium">Avg. Duration</p>
                            <h3 class="text-2xl font-bold" id="avg-duration">--</h3>
                        </div>
                        <div class="bg-purple-100 dark:bg-purple-900 p-2 rounded-lg">
                            <i class="fas fa-clock text-purple-500"></i>
                        </div>
                    </div>
                    <p class="text-xs text-gray-500 mt-2" id="duration-trend">--</p>
                </div>
            </div>

            <!-- Charts -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                <div class="bg-white dark:bg-gray-900 p-4 rounded-lg shadow">
                    <h3 class="text-lg font-medium mb-4">Visitors Over Time</h3>
                    <div id="visitors-chart" class="h-64"></div>
                </div>
                
                <div class="bg-white dark:bg-gray-900 p-4 rounded-lg shadow">
                    <h3 class="text-lg font-medium mb-4">Top Pages</h3>
                    <div id="pages-list" class="space-y-2 max-h-64 overflow-y-auto">
                        <div class="text-center text-gray-500 p-4">Select a website to view top pages</div>
                    </div>
                </div>
                
                <div class="bg-white dark:bg-gray-900 p-4 rounded-lg shadow">
                    <h3 class="text-lg font-medium mb-4">Referrers</h3>
                    <div id="referrers-list" class="space-y-2 max-h-64 overflow-y-auto">
                        <div class="text-center text-gray-500 p-4">Select a website to view referrers</div>
                    </div>
                </div>
                
                <div class="bg-white dark:bg-gray-900 p-4 rounded-lg shadow">
                    <h3 class="text-lg font-medium mb-4">Browsers & Devices</h3>
                    <div id="devices-chart" class="h-64"></div>
                </div>
            </div>
        </div>
    </div>
</div>

        <div id="workspace-content" class="tab-content hidden">
    <div class="grid grid-cols-1 lg:grid-cols-[300px,1fr] gap-6">
        <!-- Sidebar with document list and projects -->
        <div class="bg-gray-100 dark:bg-gray-800 p-4 rounded-lg">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-lg font-medium">Documents & Projects</h2>
                <div class="flex space-x-2">
                    <button id="new-document-btn" class="p-2 bg-black text-white dark:bg-white dark:text-black rounded-lg text-xs flex items-center space-x-1">
                        <i class="fas fa-plus"></i>
                        <span>Document</span>
                    </button>
                    <button id="new-project-btn" class="p-2 bg-gray-700 text-white dark:bg-gray-300 dark:text-black rounded-lg text-xs flex items-center space-x-1">
                        <i class="fas fa-folder-plus"></i>
                        <span>Project</span>
                    </button>
                </div>
            </div>
            
            <div class="mb-4">
                <input type="text" id="search-workspace" placeholder="Search..." 
                       class="w-full px-3 py-2 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg">
            </div>
            
            <div class="space-y-4">
                <div>
                    <h3 class="font-medium mb-2 flex items-center space-x-2">
                        <i class="fas fa-file text-blue-500"></i>
                        <span>My Documents</span>
                    </h3>
                    <div id="personal-documents-list" class="space-y-2 max-h-60 overflow-y-auto">
                        <div class="text-center text-gray-500 p-4">Loading...</div>
                    </div>
                </div>
                
                <div>
                    <h3 class="font-medium mb-2 flex items-center space-x-2">
                        <i class="fas fa-project-diagram text-purple-500"></i>
                        <span>Projects & Logs</span>
                    </h3>
                    <div id="projects-list" class="space-y-2 max-h-60 overflow-y-auto">
                        <div class="text-center text-gray-500 p-4">Loading...</div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Document content -->
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-lg overflow-hidden">
            <div class="p-4 border-b border-gray-200 dark:border-gray-700 flex justify-between items-center">
                <div class="flex items-center space-x-4">
                    <div id="document-icon-container">
                        <i class="fas fa-file-alt text-blue-500 text-xl"></i>
                    </div>
                    <div>
                        <input type="text" id="document-title" placeholder="Document title" 
                               class="text-lg font-bold bg-transparent border-0 outline-none focus:ring-0 w-full">
                        <div class="flex items-center space-x-2 text-sm text-gray-500 dark:text-gray-400">
                            <span id="document-type">Personal document</span>
                            <span id="document-date">Last modified: --/--/----</span>
                        </div>
                    </div>
                </div>
                <div class="flex space-x-2">
                    <button id="share-document-btn" class="p-2 rounded-lg bg-gray-100 dark:bg-gray-700 text-xs flex items-center space-x-1">
                        <i class="fas fa-share-alt"></i>
                        <span>Share</span>
                    </button>
                    <div class="relative" id="document-actions">
                        <button class="p-2 rounded-lg bg-gray-100 dark:bg-gray-700">
                            <i class="fas fa-ellipsis-v"></i>
                        </button>
                        <div class="hidden absolute right-0 mt-2 w-48 bg-white dark:bg-gray-700 rounded-lg shadow-lg py-2 z-10" id="document-actions-dropdown">
                            <button class="w-full text-left px-4 py-2 text-sm hover:bg-gray-100 dark:hover:bg-gray-600 flex items-center space-x-2">
                                <i class="fas fa-trash text-red-500"></i>
                                <span>Delete</span>
                            </button>
                            <button id="convert-to-project-btn" class="w-full text-left px-4 py-2 text-sm hover:bg-gray-100 dark:hover:bg-gray-600 flex items-center space-x-2">
                                <i class="fas fa-project-diagram text-purple-500"></i>
                                <span>Convert to project log</span>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="p-4">
                <div class="bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg overflow-hidden mb-4">
                    <div class="bg-gray-100 dark:bg-gray-800 border-b border-gray-200 dark:border-gray-600 p-2 flex space-x-2">
                        <button type="button" data-format="bold" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded">
                            <i class="fas fa-bold"></i>
                        </button>
                        <button type="button" data-format="italic" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded">
                            <i class="fas fa-italic"></i>
                        </button>
                        <button type="button" data-format="underline" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded">
                            <i class="fas fa-underline"></i>
                        </button>
                        <div class="w-px bg-gray-300 dark:bg-gray-500"></div>
                        <button type="button" data-format="h1" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded">
                            <i class="fas fa-heading"></i>
                        </button>
                        <button type="button" data-format="h2" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded text-sm">
                            H2
                        </button>
                        <button type="button" data-format="h3" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded text-xs">
                            H3
                        </button>
                        <div class="w-px bg-gray-300 dark:bg-gray-500"></div>
                        <button type="button" data-format="link" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded">
                            <i class="fas fa-link"></i>
                        </button>
                        <button type="button" data-format="list-ul" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded">
                            <i class="fas fa-list-ul"></i>
                        </button>
                        <button type="button" data-format="list-ol" class="p-2 hover:bg-gray-200 dark:hover:bg-gray-600 rounded">
                            <i class="fas fa-list-ol"></i>
                        </button>
                    </div>
                    <div id="document-editor" contenteditable="true" 
                         class="min-h-[400px] p-4 focus:outline-none"></div>
                </div>
                
                <div class="flex justify-end">
                    <button id="save-document-btn" class="px-4 py-2 bg-black text-white dark:bg-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors flex items-center space-x-2">
                        <i class="fas fa-save"></i>
                        <span>Save</span>
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Project creation modal -->
<div id="project-modal" class="hidden fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50">
    <div class="bg-white dark:bg-gray-800 rounded-lg w-full max-w-md mx-4 shadow-xl">
        <div class="p-6">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-xl font-bold flex items-center space-x-2" id="project-modal-title">
                    <i class="fas fa-folder-plus text-purple-500"></i>
                    <span>New Project</span>
                </h2>
                <button class="close-project-modal text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>

            <form id="project-form" class="space-y-4">
                <input type="hidden" id="project-id" name="project-id">
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="project-name">Project name</label>
                    <input type="text" id="project-name" name="project-name" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="project-description">Description</label>
                    <textarea id="project-description" name="project-description" rows="3"
                              class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                     border border-gray-200 dark:border-gray-600
                                     focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white"></textarea>
                </div>
                
                <div class="flex justify-end space-x-3 pt-4">
                    <button type="button" 
                            class="flex items-center space-x-2 px-4 py-2 border border-gray-200 dark:border-gray-600 
                                   rounded-lg text-gray-700 dark:text-gray-300 
                                   hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors close-project-modal">
                        <i class="fas fa-times"></i>
                        <span>Cancel</span>
                    </button>
                    <button type="submit" 
                            class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black 
                                   rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                        <i class="fas fa-save"></i>
                        <span>Create</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Associate with project modal -->
<div id="associate-project-modal" class="hidden fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50">
    <div class="bg-white dark:bg-gray-800 rounded-lg w-full max-w-md mx-4 shadow-xl">
        <div class="p-6">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-xl font-bold flex items-center space-x-2">
                    <i class="fas fa-project-diagram text-purple-500"></i>
                    <span>Associate with project</span>
                </h2>
                <button class="close-associate-modal text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>

            <form id="associate-project-form" class="space-y-4">
                <input type="hidden" id="document-to-associate" name="document-to-associate">
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="project-select">Select a project</label>
                    <select id="project-select" name="project-select" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                        <option value="">-- Select a project --</option>
                    </select>
                </div>
                
                <div class="flex justify-end space-x-3 pt-4">
                    <button type="button" 
                            class="flex items-center space-x-2 px-4 py-2 border border-gray-200 dark:border-gray-600 
                                   rounded-lg text-gray-700 dark:text-gray-300 
                                   hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors close-associate-modal">
                        <i class="fas fa-times"></i>
                        <span>Cancel</span>
                    </button>
                    <button type="submit" 
                            class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black 
                                   rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                        <i class="fas fa-link"></i>
                        <span>Associate</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

        <div id="finance-content" class="tab-content hidden">
    <div class="grid grid-cols-1 lg:grid-cols-12 gap-6">
        <!-- Résumé financier -->
        <div class="lg:col-span-12 grid grid-cols-1 md:grid-cols-4 gap-4">
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <div class="flex justify-between items-start">
                    <div>
                        <p class="text-gray-500 dark:text-gray-400 text-sm font-medium">Revenus (30j)</p>
                        <h3 class="text-2xl font-bold text-green-500" id="total-revenue">0 €</h3>
                    </div>
                    <div class="bg-green-100 dark:bg-green-900 p-2 rounded-lg">
                        <i class="fas fa-arrow-up text-green-500"></i>
                    </div>
                </div>
                <p class="text-xs text-gray-500 mt-2" id="revenue-trend">Chargement...</p>
            </div>
            
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <div class="flex justify-between items-start">
                    <div>
                        <p class="text-gray-500 dark:text-gray-400 text-sm font-medium">Dépenses (30j)</p>
                        <h3 class="text-2xl font-bold text-red-500" id="total-expenses">0 €</h3>
                    </div>
                    <div class="bg-red-100 dark:bg-red-900 p-2 rounded-lg">
                        <i class="fas fa-arrow-down text-red-500"></i>
                    </div>
                </div>
                <p class="text-xs text-gray-500 mt-2" id="expenses-trend">Chargement...</p>
            </div>
            
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <div class="flex justify-between items-start">
                    <div>
                        <p class="text-gray-500 dark:text-gray-400 text-sm font-medium">Bénéfice (30j)</p>
                        <h3 class="text-2xl font-bold" id="total-profit">0 €</h3>
                    </div>
                    <div class="bg-blue-100 dark:bg-blue-900 p-2 rounded-lg">
                        <i class="fas fa-chart-line text-blue-500"></i>
                    </div>
                </div>
                <p class="text-xs text-gray-500 mt-2" id="profit-ratio">Chargement...</p>
            </div>
            
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <div class="flex justify-between items-start">
                    <div>
                        <p class="text-gray-500 dark:text-gray-400 text-sm font-medium">Objectifs</p>
                        <h3 class="text-2xl font-bold text-purple-500" id="goals-completed">0/0</h3>
                    </div>
                    <div class="bg-purple-100 dark:bg-purple-900 p-2 rounded-lg">
                        <i class="fas fa-bullseye text-purple-500"></i>
                    </div>
                </div>
                <p class="text-xs text-gray-500 mt-2" id="goal-progress">Chargement...</p>
            </div>
        </div>
        
        <!-- Graphique principal -->
        <div class="lg:col-span-8">
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-medium">Aperçu financier</h2>
                    <div class="flex space-x-2">
                        <select id="chart-period" class="text-xs p-2 border rounded-lg bg-gray-100 dark:bg-gray-700">
                            <option value="30">30 jours</option>
                            <option value="90">90 jours</option>
                            <option value="180">6 mois</option>
                            <option value="365">1 an</option>
                        </select>
                        <select id="chart-type" class="text-xs p-2 border rounded-lg bg-gray-100 dark:bg-gray-700">
                            <option value="line">Ligne</option>
                            <option value="bar">Barres</option>
                            <option value="stacked">Barres empilées</option>
                        </select>
                    </div>
                </div>
                <div id="finance-chart" class="w-full h-80"></div>
            </div>
        </div>
        
        <!-- Objectifs financiers et dernières transactions -->
        <div class="lg:col-span-4">
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow mb-6">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-medium">Objectifs financiers</h2>
                    <button id="add-goal-btn" class="text-xs p-2 bg-black text-white dark:bg-white dark:text-black rounded-lg">
                        <i class="fas fa-plus"></i> Ajouter
                    </button>
                </div>
                <div id="financial-goals" class="space-y-3 max-h-48 overflow-y-auto">
                    <!-- Les objectifs seront chargés dynamiquement ici -->
                    <div class="text-center text-gray-500 py-4">Chargement des objectifs...</div>
                </div>
            </div>
            
            <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-medium">Dernières transactions</h2>
                    <button id="add-transaction-btn" class="text-xs p-2 bg-black text-white dark:bg-white dark:text-black rounded-lg">
                        <i class="fas fa-plus"></i> Ajouter
                    </button>
                </div>
                <div id="latest-transactions" class="space-y-3 max-h-56 overflow-y-auto">
                    <!-- Les transactions seront chargées dynamiquement ici -->
                    <div class="text-center text-gray-500 py-4">Chargement des transactions...</div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal d'ajout de transaction -->
<div id="transaction-modal" class="hidden fixed inset-0 bg-black/50 z-50 flex items-center justify-center">
    <div class="bg-white dark:bg-gray-800 rounded-lg w-full max-w-md mx-4 shadow-xl">
        <div class="p-6">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-xl font-bold flex items-center space-x-2" id="transaction-modal-title">
                    <i class="fas fa-exchange-alt text-gray-600 dark:text-gray-400"></i>
                    <span>Nouvelle transaction</span>
                </h2>
                <button class="close-transaction-modal text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>

            <form id="transaction-form" class="space-y-4">
                <input type="hidden" id="transaction-id" name="transaction-id">
                
                <div>
                    <label class="block text-sm font-medium mb-2">Type</label>
                    <div class="flex space-x-2">
                        <label class="flex items-center p-3 border rounded-lg cursor-pointer flex-1 bg-gray-100 dark:bg-gray-700">
                            <input type="radio" name="transaction-type" value="revenue" checked class="mr-2">
                            <span class="text-green-500"><i class="fas fa-arrow-up mr-2"></i> Revenu</span>
                        </label>
                        <label class="flex items-center p-3 border rounded-lg cursor-pointer flex-1 bg-gray-100 dark:bg-gray-700">
                            <input type="radio" name="transaction-type" value="expense" class="mr-2">
                            <span class="text-red-500"><i class="fas fa-arrow-down mr-2"></i> Dépense</span>
                        </label>
                    </div>
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="transaction-amount">Montant (€)</label>
                    <input type="number" id="transaction-amount" name="amount" step="0.01" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="transaction-category">Catégorie</label>
                    <select id="transaction-category" name="category" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                        <!-- Options de revenus -->
                        <optgroup label="Revenus" id="revenue-categories">
                            <option value="sales">Ventes</option>
                            <option value="services">Services</option>
                            <option value="subscriptions">Abonnements</option>
                            <option value="investments">Investissements</option>
                            <option value="other_revenue">Autre revenu</option>
                        </optgroup>
                        <!-- Options de dépenses -->
                        <optgroup label="Dépenses" id="expense-categories">
                            <option value="salaries">Salaires</option>
                            <option value="rent">Loyer</option>
                            <option value="equipment">Équipement</option>
                            <option value="marketing">Marketing</option>
                            <option value="software">Logiciels</option>
                            <option value="taxes">Taxes</option>
                            <option value="other_expense">Autre dépense</option>
                        </optgroup>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="transaction-date">Date</label>
                    <input type="date" id="transaction-date" name="date" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="transaction-description">Description</label>
                    <textarea id="transaction-description" name="description" rows="2"
                              class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                     border border-gray-200 dark:border-gray-600
                                     focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white"></textarea>
                </div>
                
                <div class="flex justify-end space-x-3 pt-4">
                    <button type="button" 
                            class="flex items-center space-x-2 px-4 py-2 border border-gray-200 dark:border-gray-600 
                                   rounded-lg text-gray-700 dark:text-gray-300 
                                   hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors close-transaction-modal">
                        <i class="fas fa-times"></i>
                        <span>Annuler</span>
                    </button>
                    <button type="submit" 
                            class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black 
                                   rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                        <i class="fas fa-save"></i>
                        <span>Enregistrer</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal d'ajout d'objectif financier -->
<div id="goal-modal" class="hidden fixed inset-0 bg-black/50 z-50 flex items-center justify-center">
    <div class="bg-white dark:bg-gray-800 rounded-lg w-full max-w-md mx-4 shadow-xl">
        <div class="p-6">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-xl font-bold flex items-center space-x-2" id="goal-modal-title">
                    <i class="fas fa-bullseye text-gray-600 dark:text-gray-400"></i>
                    <span>Nouvel objectif</span>
                </h2>
                <button class="close-goal-modal text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>

            <form id="goal-form" class="space-y-4">
                <input type="hidden" id="goal-id" name="goal-id">
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="goal-title">Titre</label>
                    <input type="text" id="goal-title" name="title" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white"
                           placeholder="Titre de l'objectif">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="goal-type">Type</label>
                    <select id="goal-type" name="type" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                        <option value="revenue">Revenu cible</option>
                        <option value="expense">Limiter dépenses</option>
                        <option value="profit">Objectif de bénéfice</option>
                        <option value="savings">Épargne</option>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="goal-target">Montant cible (€)</label>
                    <input type="number" id="goal-target" name="target" step="0.01" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="goal-start-date">Date de début</label>
                    <input type="date" id="goal-start-date" name="start_date" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="goal-end-date">Date d'échéance</label>
                    <input type="date" id="goal-end-date" name="end_date" required
                           class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                  border border-gray-200 dark:border-gray-600
                                  focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2" for="goal-notes">Notes</label>
                    <textarea id="goal-notes" name="notes" rows="2"
                              class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                     border border-gray-200 dark:border-gray-600
                                     focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white"
                              placeholder="Notes ou détails (optionnel)"></textarea>
                </div>
                
                <div class="flex justify-end space-x-3 pt-4">
                    <button type="button" 
                            class="flex items-center space-x-2 px-4 py-2 border border-gray-200 dark:border-gray-600 
                                   rounded-lg text-gray-700 dark:text-gray-300 
                                   hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors close-goal-modal">
                        <i class="fas fa-times"></i>
                        <span>Annuler</span>
                    </button>
                    <button type="submit" 
                            class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black 
                                   rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                        <i class="fas fa-save"></i>
                        <span>Enregistrer</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>
        
<div id="tickets-content" class="tab-content hidden">
    <div class="grid grid-cols-1 lg:grid-cols-[350px,1fr] gap-4">
        <div class="bg-gray-100 dark:bg-gray-800 rounded-lg overflow-hidden">
            <div class="p-4 border-b border-gray-200 dark:border-gray-700">
                <h2 class="text-xl font-medium flex items-center space-x-2">
                    <i class="fas fa-ticket-alt text-gray-600 dark:text-gray-400"></i>
                    <span>Support Tickets</span>
                </h2>
            </div>
            <div class="h-[calc(100vh-300px)] overflow-y-auto" id="tickets-list">
            </div>
        </div>

        <div class="bg-gray-100 dark:bg-gray-800 rounded-lg overflow-hidden" id="conversation-view">
            <div class="p-4 border-b border-gray-200 dark:border-gray-700 flex justify-between items-center">
                <div>
                    <h3 class="text-xl font-medium" id="selected-ticket-subject">Select a ticket</h3>
                    <p class="text-sm text-gray-600 dark:text-gray-400" id="selected-ticket-id"></p>
                </div>
                <select id="ticket-status" class="px-3 py-1 rounded-full text-sm border bg-white dark:bg-gray-800 hidden">
                    <option value="open">Open</option>
                    <option value="in_progress">In Progress</option>
                    <option value="resolved">Resolved</option>
                    <option value="closed">Closed</option>
                </select>
            </div>
            <div class="h-[calc(100vh-400px)] overflow-y-auto p-4 space-y-4" id="conversation-messages">
                <p class="text-center text-gray-500 dark:text-gray-400">Select a ticket to view the conversation. You can select preconceived phrases from the conversation bar. Please do not under any circumstances leave or delete the employee protocol, as this may result in legal action and a license.</p>
            </div>
            <div class="p-4 border-t border-gray-200 dark:border-gray-700" id="reply-form">
                <div class="flex flex-col space-y-3">
                    <div class="flex flex-wrap gap-2 mb-2" id="preset-phrases">
                        <button class="preset-phrase px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                            Thank you for contacting Imators Support
                        </button>
                        <button class="preset-phrase px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                            I'm looking into this issue for you
                        </button>
                        <button class="preset-phrase px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                            Could you provide more details?
                        </button>
                        <button class="preset-phrase px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                            This issue has been resolved
                        </button>
                        <button class="preset-phrase px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                            Please try clearing your cache and reloading
                        </button>
                        <button class="preset-phrase px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                            I'll escalate this to our development team
                        </button>
                        <button class="preset-phrase px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                            Is there anything else I can help with?
                        </button>
                        <button class="preset-phrase px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors">
                            We'll keep you updated on this issue
                        </button>
                    </div>
                    <div class="flex space-x-2">
                        <input type="text" id="reply-message" 
                               class="flex-1 px-4 py-2 rounded-lg border bg-white dark:bg-gray-800 
                                      focus:outline-none focus:ring-2 focus:ring-gray-400 dark:focus:ring-gray-500"
                               placeholder="Type your response..." disabled>
                        <button id="send-reply" disabled
                                class="px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg 
                                       hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors 
                                       disabled:opacity-50 disabled:cursor-not-allowed
                                       flex items-center space-x-2">
                            <i class="fas fa-paper-plane"></i>
                            <span>Send</span>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

        <script>

                // Ajouter ce code à ton fichier JavaScript existant

document.addEventListener('DOMContentLoaded', function() {
    // Gestion des phrases préconçues
    const presetPhrases = document.querySelectorAll('.preset-phrase');
    const replyInput = document.getElementById('reply-message');
    
    presetPhrases.forEach(phrase => {
        phrase.addEventListener('click', function() {
            // Si l'entrée est désactivée, ne faites rien
            if (replyInput.disabled) return;
            
            // Insérer la phrase préconçue dans le champ de réponse
            replyInput.value = this.textContent.trim();
            
            // Mettre le focus sur l'entrée pour permettre à l'utilisateur de continuer à écrire
            replyInput.focus();
        });
    });
    
    // Fonction pour initialiser le chargement des tickets
    function loadTickets() {
        const ticketsList = document.getElementById('tickets-list');
        
        // Simulation de chargement de tickets (à remplacer par un appel API réel)
        fetch('get_tickets.php')
            .then(response => response.json())
            .then(tickets => {
                ticketsList.innerHTML = '';
                
                if (tickets.length === 0) {
                    ticketsList.innerHTML = '<div class="p-4 text-center text-gray-500">Aucun ticket trouvé</div>';
                    return;
                }
                
                tickets.forEach(ticket => {
                    const ticketItem = document.createElement('div');
                    ticketItem.classList.add('ticket-item', 'p-4', 'border-b', 'border-gray-200', 'dark:border-gray-700', 
                                            'hover:bg-gray-200', 'dark:hover:bg-gray-700', 'cursor-pointer', 'transition-colors');
                    
                    // Définir la couleur en fonction du statut
                    let statusColor = '';
                    switch(ticket.status) {
                        case 'open':
                            statusColor = 'bg-red-500';
                            break;
                        case 'in_progress':
                            statusColor = 'bg-yellow-500';
                            break;
                        case 'resolved':
                            statusColor = 'bg-green-500';
                            break;
                        case 'closed':
                            statusColor = 'bg-gray-500';
                            break;
                    }
                    
                    ticketItem.innerHTML = `
                        <div class="flex items-start space-x-3">
                            <div class="w-3 h-3 rounded-full mt-1.5 ${statusColor}"></div>
                            <div>
                                <h3 class="font-medium">${ticket.subject}</h3>
                                <p class="text-sm text-gray-600 dark:text-gray-400">${ticket.user_email}</p>
                                <div class="flex items-center space-x-2 mt-1">
                                    <span class="text-xs text-gray-500">#${ticket.id}</span>
                                    <span class="text-xs text-gray-500">${ticket.created_at}</span>
                                </div>
                            </div>
                        </div>
                    `;
                    
                    ticketItem.addEventListener('click', () => showTicketConversation(ticket.id));
                    ticketsList.appendChild(ticketItem);
                });
            })
            .catch(error => {
                console.error('Erreur lors du chargement des tickets:', error);
                ticketsList.innerHTML = '<div class="p-4 text-center text-red-500">Erreur lors du chargement des tickets</div>';
            });
    }
    
    // Fonction pour afficher la conversation d'un ticket
    function showTicketConversation(ticketId) {
        const subjectElement = document.getElementById('selected-ticket-subject');
        const idElement = document.getElementById('selected-ticket-id');
        const messagesContainer = document.getElementById('conversation-messages');
        const replyInput = document.getElementById('reply-message');
        const sendButton = document.getElementById('send-reply');
        const statusSelect = document.getElementById('ticket-status');
        
        subjectElement.textContent = 'Loading...';
        idElement.textContent = `Ticket #${ticketId}`;
        messagesContainer.innerHTML = '<div class="flex justify-center"><div class="animate-spin rounded-full h-8 w-8 border-2 border-gray-300 border-t-black dark:border-t-white"></div></div>';
        
        // Activer le formulaire de réponse
        replyInput.disabled = false;
        sendButton.disabled = false;
        
        // Afficher le sélecteur de statut
        statusSelect.classList.remove('hidden');
        
        // Simulation de chargement de la conversation (à remplacer par un appel API réel)
        fetch(`get_ticket_conversation.php?id=${ticketId}`)
            .then(response => response.json())
            .then(data => {
                subjectElement.textContent = data.ticket.subject;
                
                // Mettre à jour le sélecteur de statut
                statusSelect.value = data.ticket.status;
                
                messagesContainer.innerHTML = '';
                
                // Ajouter le message initial
                const initialMessage = document.createElement('div');
                initialMessage.classList.add('bg-white', 'dark:bg-gray-900', 'p-4', 'rounded-lg', 'shadow-sm');
                initialMessage.innerHTML = `
                    <div class="flex justify-between items-start">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 rounded-full bg-gray-200 dark:bg-gray-700 flex items-center justify-center">
                                <i class="fas fa-user text-gray-400"></i>
                            </div>
                            <div>
                                <p class="font-medium">${data.ticket.user_name}</p>
                                <p class="text-xs text-gray-500">${data.ticket.created_at}</p>
                            </div>
                        </div>
                    </div>
                    <div class="mt-3 text-gray-700 dark:text-gray-300">
                        ${data.ticket.message}
                    </div>
                `;
                messagesContainer.appendChild(initialMessage);
                
                // Ajouter les réponses
                data.responses.forEach(response => {
                    const messageElement = document.createElement('div');
                    const isEmployee = response.is_employee;
                    
                    messageElement.classList.add('p-4', 'rounded-lg', 'shadow-sm');
                    if (isEmployee) {
                        messageElement.classList.add('bg-blue-50', 'dark:bg-blue-900', 'ml-6');
                    } else {
                        messageElement.classList.add('bg-white', 'dark:bg-gray-900');
                    }
                    
                    messageElement.innerHTML = `
                        <div class="flex justify-between items-start">
                            <div class="flex items-center space-x-3">
                                <div class="w-10 h-10 rounded-full ${isEmployee ? 'bg-blue-100 dark:bg-blue-800' : 'bg-gray-200 dark:bg-gray-700'} flex items-center justify-center">
                                    <i class="fas ${isEmployee ? 'fa-headset' : 'fa-user'} ${isEmployee ? 'text-blue-500' : 'text-gray-400'}"></i>
                                </div>
                                <div>
                                    <p class="font-medium">${response.user_name} ${isEmployee ? '<span class="text-xs bg-blue-100 dark:bg-blue-800 text-blue-800 dark:text-blue-200 px-2 py-0.5 rounded-full">Staff</span>' : ''}</p>
                                    <p class="text-xs text-gray-500">${response.created_at}</p>
                                </div>
                            </div>
                        </div>
                        <div class="mt-3 text-gray-700 dark:text-gray-300">
                            ${response.message}
                        </div>
                    `;
                    messagesContainer.appendChild(messageElement);
                });
                
                // Faire défiler vers le bas pour voir le dernier message
                messagesContainer.scrollTop = messagesContainer.scrollHeight;
            })
            .catch(error => {
                console.error('Erreur lors du chargement de la conversation:', error);
                messagesContainer.innerHTML = '<div class="p-4 text-center text-red-500">Erreur lors du chargement de la conversation</div>';
            });
    }
    
    // Gestionnaire pour l'envoi de réponse
    const sendReplyButton = document.getElementById('send-reply');
    const replyMessageInput = document.getElementById('reply-message');
    
    sendReplyButton.addEventListener('click', function() {
        const message = replyMessageInput.value.trim();
        const ticketId = document.getElementById('selected-ticket-id').textContent.replace('Ticket #', '');
        
        if (!message) return;
        
        // Désactiver le bouton pendant l'envoi
        sendReplyButton.disabled = true;
        sendReplyButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> <span>Sending...</span>';
        
        // Simulation d'envoi (à remplacer par un appel API réel)
        fetch('send_reply.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                ticket_id: ticketId,
                message: message
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Vider le champ de réponse
                replyMessageInput.value = '';
                
                // Recharger la conversation pour afficher la nouvelle réponse
                showTicketConversation(ticketId);
                
                // Réactiver le bouton
                sendReplyButton.disabled = false;
                sendReplyButton.innerHTML = '<i class="fas fa-paper-plane"></i> <span>Send</span>';
            } else {
                throw new Error(data.message || 'Error sending reply');
            }
        })
        .catch(error => {
            console.error('Erreur lors de l\'envoi de la réponse:', error);
            alert('Erreur lors de l\'envoi de la réponse. Veuillez réessayer.');
            
            // Réactiver le bouton
            sendReplyButton.disabled = false;
            sendReplyButton.innerHTML = '<i class="fas fa-paper-plane"></i> <span>Send</span>';
        });
    });
    
    // Gestionnaire pour le changement de statut
    const statusSelect = document.getElementById('ticket-status');
    
    statusSelect.addEventListener('change', function() {
        const ticketId = document.getElementById('selected-ticket-id').textContent.replace('Ticket #', '');
        const newStatus = this.value;
        
        // Simulation de mise à jour du statut (à remplacer par un appel API réel)
        fetch('update_ticket_status.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                ticket_id: ticketId,
                status: newStatus
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Notification pour l'utilisateur
                alert(`Le statut du ticket a été mis à jour à "${newStatus}"`);
                
                // Recharger la liste des tickets pour refléter le nouveau statut
                loadTickets();
            } else {
                throw new Error(data.message || 'Error updating ticket status');
            }
        })
        .catch(error => {
            console.error('Erreur lors de la mise à jour du statut:', error);
            alert('Erreur lors de la mise à jour du statut. Veuillez réessayer.');
            
            // Réinitialiser le sélecteur au statut précédent
            this.value = data.ticket.status;
        });
    });
    
    // Initialiser le chargement des tickets lorsque l'onglet est affiché
    document.querySelector('.tickets-tab').addEventListener('click', function() {
        loadTickets();
    });
});
            
        </script>
        
        <div id="email-content" class="tab-content hidden">
            <div class="grid grid-cols-1 gap-6">
                <div class="bg-gray-100 dark:bg-gray-800 p-6 rounded-lg">
                    <h2 class="text-xl font-medium mb-6 flex items-center space-x-2">
                        <i class="fas fa-paper-plane text-gray-600 dark:text-gray-400"></i>
                        <span>Send Email</span>
                    </h2>

                    <div id="email-server-status" class="mb-6">
                        <div class="flex items-center space-x-2">
                            <div class="animate-spin rounded-full h-4 w-4 border-2 border-gray-500 border-t-black dark:border-t-white"></div>
                            <span>Checking email server status...</span>
                        </div>
                    </div>

                    <form id="send-email-form" class="space-y-6">
                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-user text-gray-500 dark:text-gray-400"></i>
                                <span>Recipient</span>
                            </label>
                            <select id="email-recipient" name="recipient" required
                                    class="w-full px-4 py-2 rounded-lg bg-white dark:bg-gray-700 
                                           border border-gray-200 dark:border-gray-600
                                           focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                                <option value="">Select a recipient</option>
                                <?php
                                $db1 = getDB1Connection();
                                $query = $db1->query("SELECT id, username, email FROM utilisateurs ORDER BY username");
                                while($user = $query->fetch(PDO::FETCH_ASSOC)) {
                                    echo "<option value='" . htmlspecialchars($user['id']) . "' 
                                                data-email='" . htmlspecialchars($user['email']) . "'
                                                data-username='" . htmlspecialchars($user['username']) . "'>
                                            " . htmlspecialchars($user['username']) . " (" . htmlspecialchars($user['email']) . ")
                                          </option>";
                                }
                                ?>
                            </select>
                        </div>

                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-heading text-gray-500 dark:text-gray-400"></i>
                                <span>Subject</span>
                            </label>
                            <input type="text" id="email-subject" name="subject" required
                                   placeholder="Enter email subject"
                                   class="w-full px-4 py-2 rounded-lg bg-white dark:bg-gray-700 
                                          border border-gray-200 dark:border-gray-600
                                          focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                        </div>

                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-pen-fancy text-gray-500 dark:text-gray-400"></i>
                                <span>Content</span>
                            </label>
                            <div class="border border-gray-200 dark:border-gray-600 rounded-lg overflow-hidden">
                                <div class="bg-white dark:bg-gray-700 border-b border-gray-200 dark:border-gray-600 p-2 flex space-x-2">
                                    <button type="button" data-format="bold" class="p-2 hover:bg-gray-100 dark:hover:bg-gray-600 rounded">
                                        <i class="fas fa-bold"></i>
                                    </button>
                                    <button type="button" data-format="italic" class="p-2 hover:bg-gray-100 dark:hover:bg-gray-600 rounded">
                                        <i class="fas fa-italic"></i>
                                    </button>
                                    <button type="button" data-format="underline" class="p-2 hover:bg-gray-100 dark:hover:bg-gray-600 rounded">
                                        <i class="fas fa-underline"></i>
                                    </button>
                                    <div class="w-px bg-gray-200 dark:bg-gray-600"></div>
                                    <button type="button" data-format="link" class="p-2 hover:bg-gray-100 dark:hover:bg-gray-600 rounded">
                                        <i class="fas fa-link"></i>
                                    </button>
                                </div>
                                <div id="email-editor" contenteditable="true"
                                     class="min-h-[200px] p-4 bg-white dark:bg-gray-700 focus:outline-none"
                                     placeholder="Compose your email here... Use {username} to automatically insert recipient's name"></div>
                            </div>
                        </div>

                        <div class="border border-gray-200 dark:border-gray-600 rounded-lg p-4">
                            <h3 class="text-sm font-medium mb-4 flex items-center space-x-2">
                                <i class="fas fa-eye text-gray-500 dark:text-gray-400"></i>
                                <span>Preview</span>
                            </h3>
                            <div id="email-preview" class="bg-white dark:bg-gray-900 rounded-lg p-6">
                            </div>
                        </div>

                        <div class="flex justify-end space-x-3">
                            <button type="button" id="refresh-preview"
                                    class="flex items-center space-x-2 px-4 py-2 border border-gray-200 dark:border-gray-600 
                                           rounded-lg text-gray-700 dark:text-gray-300 
                                           hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors">
                                <i class="fas fa-sync-alt"></i>
                                <span>Refresh Preview</span>
                            </button>
                            <button type="submit"
                                    class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black 
                                           rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                                <i class="fas fa-paper-plane"></i>
                                <span>Send Email</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <div id="users-content" class="tab-content">
            <div class="mb-6 flex justify-between items-center">
                <p class="text-sm text-gray-600 dark:text-gray-400" id="users-count">Loading users...</p>
                <button id="add-user-btn" class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                    <i class="fas fa-user-plus"></i>
                    <span>Add User</span>
                </button>
            </div>
            <div id="users-grid" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                <?php
                $db1 = getDB1Connection();
                $query = $db1->query("SELECT * FROM utilisateurs ORDER BY username");
                while($user = $query->fetch(PDO::FETCH_ASSOC)) {
                ?>
                    <div class="user-card bg-gray-100 dark:bg-gray-800 p-6 rounded-lg cursor-pointer hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors"
                         data-user-id="<?= $user['id'] ? htmlspecialchars($user['id']) : '' ?>"
                         data-email="<?= $user['email'] ? htmlspecialchars($user['email']) : '' ?>"
                         data-username="<?= $user['username'] ? htmlspecialchars($user['username']) : '' ?>"
                         data-note="<?= $user['note'] ? htmlspecialchars($user['note']) : '' ?>"
                         data-role="<?= isset($user['role']) ? htmlspecialchars($user['role']) : '0' ?>"
                         data-roleinimators="<?= isset($user['roleinimators']) ? htmlspecialchars($user['roleinimators']) : '' ?>"
                         data-searchable="<?= htmlspecialchars(
                             ($user['username'] ? $user['username'] : '') . ' ' . 
                             ($user['email'] ? $user['email'] : '') . ' ' . 
                             ($user['note'] ? $user['note'] : '') . ' ' .
                             ($user['roleinimators'] ? $user['roleinimators'] : '')
                         ) ?>">
                        <div class="flex justify-between items-start">
                            <div>
                                <div class="flex items-center space-x-2">
                                    <i class="fas fa-user text-gray-500 dark:text-gray-400"></i>
                                    <h3 class="font-medium text-lg username"><?= $user['username'] ? htmlspecialchars($user['username']) : '' ?></h3>
                                    <?php if (isset($user['role']) && $user['role'] == 1): ?>
                                        <span class="bg-purple-100 text-purple-800 text-xs px-2 py-1 rounded-full dark:bg-purple-900 dark:text-purple-200">
                                            Employee
                                        </span>
                                    <?php endif; ?>
                                </div>
                                <div class="flex items-center space-x-2 mt-1">
                                    <i class="fas fa-envelope text-gray-500 dark:text-gray-400"></i>
                                    <p class="text-sm text-gray-600 dark:text-gray-400 email"><?= $user['email'] ? htmlspecialchars($user['email']) : '' ?></p>
                                </div>
                                <?php if (!empty($user['roleinimators'])): ?>
                                <div class="flex items-center space-x-2 mt-1">
                                    <i class="fas fa-id-badge text-gray-500 dark:text-gray-400"></i>
                                    <p class="text-sm text-blue-600 dark:text-blue-400 roleinimators"><?= htmlspecialchars($user['roleinimators']) ?></p>
                                </div>
                                <?php endif; ?>
                            </div>
                            <button class="text-xs px-3 py-1 bg-gray-200 dark:bg-gray-700 rounded-full hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors flex items-center space-x-1">
                                <i class="fas fa-edit"></i>
                                <span>Edit</span>
                            </button>
                        </div>
                        <div class="mt-4">
                            <div class="flex items-start space-x-2">
                                <i class="fas fa-sticky-note text-gray-500 dark:text-gray-400 mt-1"></i>
                                <p class="text-sm text-gray-600 dark:text-gray-400 note">
                                    <?= $user['note'] ? htmlspecialchars($user['note']) : 'No note' ?>
                                </p>
                            </div>
                        </div>
                    </div>
                <?php
                }
                ?>
            </div>
        </div>

        <div id="documents-content" class="tab-content hidden">
            <div class="grid grid-cols-1 gap-6">
                <div class="bg-gray-100 dark:bg-gray-800 p-6 rounded-lg">
                    <h2 class="text-xl font-medium mb-4 flex items-center space-x-2">
                        <i class="fas fa-cloud-upload-alt text-gray-600 dark:text-gray-400"></i>
                        <span>Upload Document</span>
                    </h2>
                    <form action="upload_document.php" method="post" enctype="multipart/form-data" class="space-y-4">
                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-file-pdf text-gray-600 dark:text-gray-400"></i>
                                <span>Select PDF File</span>
                            </label>
                            <input type="file" name="pdf_file" accept=".pdf" 
                                   class="block w-full text-sm text-gray-600 dark:text-gray-400
                                          file:mr-4 file:py-2 file:px-4 file:rounded-full 
                                          file:border-0 file:text-sm file:font-medium
                                          file:bg-gray-200 file:dark:bg-gray-700 
                                          file:text-gray-700 file:dark:text-gray-300
                                          hover:file:bg-gray-300 hover:file:dark:bg-gray-600
                                          transition-all">
                        </div>
                        <button type="submit" 
                                class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg
                                       hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                            <i class="fas fa-upload"></i>
                            <span>Upload</span>
                        </button>
                    </form>
                </div>

                <div class="bg-gray-100 dark:bg-gray-800 p-6 rounded-lg">
                    <h2 class="text-xl font-medium mb-4 flex items-center space-x-2">
                        <i class="fas fa-file-alt text-gray-600 dark:text-gray-400"></i>
                        <span>Document List</span>
                    </h2>
                    <p class="text-sm text-gray-600 dark:text-gray-400 mb-4" id="documents-count">Loading documents...</p>
                    <div id="documents-list" class="space-y-2">
                        <?php
                        $db2 = getDB2Connection();
                        $query = $db2->query("SELECT * FROM documents ORDER BY created_at DESC");
                        while($doc = $query->fetch(PDO::FETCH_ASSOC)) {
                        ?>
                            <div class="document-item flex items-center justify-between p-4 bg-white dark:bg-gray-900 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                                 data-searchable="<?= htmlspecialchars(basename($doc['pdf-link'])) ?>">
                                <div class="flex items-center space-x-3">
                                    <i class="fas fa-file-pdf text-red-500"></i>
                                    <a href="<?= htmlspecialchars($doc['pdf-link']) ?>" target="_blank"
                                       class="text-blue-600 dark:text-blue-400 hover:underline document-name">
                                        <?= htmlspecialchars(basename($doc['pdf-link'])) ?>
                                    </a>
                                </div>
                                <div class="flex items-center space-x-2 text-gray-500 dark:text-gray-400">
                                    <i class="far fa-clock"></i>
                                    <span class="text-xs">
                                        <?= date('Y-m-d H:i', strtotime($doc['created_at'])) ?>
                                    </span>
                                </div>
                            </div>
                        <?php
                        }
                        ?>
                    </div>
                </div>
            </div>
        </div>

        <div id="sites-content" class="tab-content hidden">
            <div class="grid grid-cols-1 gap-6">
                <div class="bg-gray-100 dark:bg-gray-800 p-6 rounded-lg">
                    <div class="flex justify-between items-center mb-6">
                        <div class="flex flex-col">
                            <h2 class="text-xl font-medium flex items-center space-x-2">
                                <i class="fas fa-globe text-gray-600 dark:text-gray-400"></i>
                                <span>Imators Website Monitoring</span>
                            </h2>
                            <p class="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                A site marked as an error means that it is not connected to the Imators Systems service.
                            </p>
                        </div>
                        <button id="refresh-sites" 
                                class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg
                                        hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                            <i class="fas fa-sync-alt"></i>
                            <span>Refresh</span>
                        </button>
                    </div>
                    <div id="sites-list" class="space-y-4"></div>
                </div>
            </div>
        </div>

        <div id="user-modal" class="hidden fixed inset-0 bg-black/50 dark:bg-black/70 backdrop-blur-sm flex items-center justify-center z-50">
            <div class="bg-white dark:bg-gray-800 rounded-lg w-full max-w-md mx-4 shadow-xl">
                <div class="p-6">
                    <div class="flex justify-between items-center mb-6">
                        <h2 class="text-xl font-bold flex items-center space-x-2" id="user-modal-title">
                            <i class="fas fa-user-edit text-gray-600 dark:text-gray-400"></i>
                            <span>Edit User</span>
                        </h2>
                        <button id="close-modal" class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors">
                            <i class="fas fa-times text-xl"></i>
                        </button>
                    </div>

                    <form id="edit-user-form" class="space-y-4">
                        <input type="hidden" id="user-id" name="user-id">
                        
                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-envelope text-gray-500 dark:text-gray-400"></i>
                                <span>Email</span>
                            </label>
                            <input type="email" id="email" name="email" required
                                   class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                          border border-gray-200 dark:border-gray-600
                                          focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                        </div>

                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-user text-gray-500 dark:text-gray-400"></i>
                                <span>Username</span>
                            </label>
                            <input type="text" id="username" name="username" required
                                   class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                          border border-gray-200 dark:border-gray-600
                                          focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                        </div>

                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-key text-gray-500 dark:text-gray-400"></i>
                                <span>Password</span>
                            </label>
                            <input type="password" id="password" name="password"
                                   placeholder="Leave empty to keep current password" 
                                   class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                          border border-gray-200 dark:border-gray-600
                                          focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                        </div>

                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-user-tag text-gray-500 dark:text-gray-400"></i>
                                <span>Role</span>
                            </label>
                            <select id="role" name="role" required
                                   class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                          border border-gray-200 dark:border-gray-600
                                          focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                                <option value="0">User</option>
                                <option value="1">Employee</option>
                            </select>
                        </div>

                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-id-badge text-gray-500 dark:text-gray-400"></i>
                                <span>Imators Role</span>
                            </label>
                            <input type="text" id="roleinimators" name="roleinimators"
                                   placeholder="e.g. CEO, Developer, Designer..." 
                                   class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                          border border-gray-200 dark:border-gray-600
                                          focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white">
                        </div>

                        <div>
                            <label class="block text-sm font-medium mb-2 flex items-center space-x-2">
                                <i class="fas fa-sticky-note text-gray-500 dark:text-gray-400"></i>
                                <span>Note</span>
                            </label>
                            <textarea id="note" name="note" rows="3" 
                                      class="w-full px-4 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 
                                             border border-gray-200 dark:border-gray-600
                                             focus:outline-none focus:ring-2 focus:ring-black dark:focus:ring-white"></textarea>
                        </div>

                        <div class="flex justify-between space-x-3 pt-4">
                            <button type="button" id="delete-user-btn"
                                    class="flex items-center space-x-2 px-4 py-2 bg-red-500 hover:bg-red-600 text-white 
                                           rounded-lg transition-colors">
                                <i class="fas fa-trash"></i>
                                <span>Delete User</span>
                            </button>
                            <div class="flex space-x-3">
                                <button type="button" 
                                        class="flex items-center space-x-2 px-4 py-2 border border-gray-200 dark:border-gray-600 
                                               rounded-lg text-gray-700 dark:text-gray-300 
                                               hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors close-modal">
                                    <i class="fas fa-times"></i>
                                    <span>Cancel</span>
                                </button>
                                <button type="submit" 
                                        class="flex items-center space-x-2 px-4 py-2 bg-black dark:bg-white text-white dark:text-black 
                                               rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                                    <i class="fas fa-save"></i>
                                    <span>Save</span>
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <footer class="bg-white text-black dark:bg-black dark:text-white mt-8">
        <div class="text-center py-4 border-t border-gray-200 dark:border-gray-800">
            <p class="text-sm mb-2">&copy; 2025 Imators LLC. All rights reserved.</p>
            <p class="text-ms mb-2">Actual version running on the 1.7 version of the Imators Systems.</p>
            <p class="text-xs">Imators is a registered LLC (Limited Liability Company). All our products, whether technologically physical or simply online, are subject to our rules, which you can access.</p>
        </div>
    </footer>

    <script src="dashboard.js"></script>
</body>
</html>