<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Suivi des Astéroïdes - Imators</title>
    <style>
        body {
            font-family: 'Arial', sans-serif;
            background-color: #0a0a1a;
            color: #fff;
            margin: 0;
            padding: 0;
        }
        header {
            background-color: #1a1a2a;
            padding: 20px;
            text-align: center;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.5);
        }
        h1 {
            margin: 0;
            color: #ff6b4a;
        }
        .container {
            max-width: 1200px;
            margin: 20px auto;
            padding: 0 20px;
        }
        .search-controls {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            flex-wrap: wrap;
            align-items: center;
        }
        .date-range {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        input, select, button {
            padding: 10px;
            border: none;
            border-radius: 4px;
            background-color: #2a2a3a;
            color: #fff;
        }
        button {
            background-color: #ff6b4a;
            cursor: pointer;
            transition: background-color 0.3s;
        }
        button:hover {
            background-color: #e55a3a;
        }
        .asteroid-list {
            display: grid;
            grid-template-columns: 1fr;
            gap: 15px;
        }
        .asteroid-card {
            background-color: #1a1a2a;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.3);
            transition: transform 0.3s;
            padding: 20px;
            position: relative;
        }
        .asteroid-card:hover {
            transform: translateY(-5px);
        }
        .hazardous {
            border-left: 5px solid #ff4d4d;
        }
        .safe {
            border-left: 5px solid #4dff4d;
        }
        .hazard-indicator {
            position: absolute;
            top: 10px;
            right: 10px;
            border-radius: 50%;
            width: 20px;
            height: 20px;
        }
        .hazard-indicator.danger {
            background-color: #ff4d4d;
        }
        .hazard-indicator.safe {
            background-color: #4dff4d;
        }
        .asteroid-name {
            font-size: 20px;
            margin: 0 0 15px 0;
            color: #ff6b4a;
        }
        .asteroid-details {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 15px;
        }
        .detail-item {
            background-color: #2a2a3a;
            padding: 10px;
            border-radius: 4px;
        }
        .detail-label {
            color: #aaa;
            font-size: 12px;
            margin-bottom: 5px;
        }
        .detail-value {
            font-weight: bold;
            font-size: 16px;
        }
        .stats-container {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            margin-bottom: 20px;
        }
        .stat-card {
            background-color: #1a1a2a;
            padding: 15px;
            border-radius: 8px;
            flex: 1;
            min-width: 200px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.3);
            text-align: center;
        }
        .stat-value {
            font-size: 28px;
            font-weight: bold;
            margin: 10px 0;
        }
        .stat-label {
            color: #aaa;
            font-size: 14px;
        }
        .danger-stat {
            color: #ff4d4d;
        }
        .safe-stat {
            color: #4dff4d;
        }
        .size-stat {
            color: #4da6ff;
        }
        .distance-stat {
            color: #ffcc4d;
        }
        .loading {
            text-align: center;
            padding: 40px;
            font-size: 20px;
            color: #ff6b4a;
            display: none;
        }
        .error {
            background-color: #ff4d4d;
            color: white;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 20px;
            display: none;
        }
        .sort-filter {
            display: flex;
            justify-content: space-between;
            margin-bottom: 20px;
            flex-wrap: wrap;
            gap: 10px;
        }
        .filter-group, .sort-group {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        .no-results {
            text-align: center;
            padding: 40px;
            font-size: 18px;
            color: #aaa;
            display: none;
        }
        .pagination {
            display: flex;
            justify-content: center;
            margin-top: 20px;
            gap: 10px;
        }
        .pagination button {
            padding: 8px 15px;
        }
        .pagination button:disabled {
            background-color: #555;
            cursor: not-allowed;
        }
        .page-info {
            display: flex;
            align-items: center;
            color: #aaa;
        }
        @media (max-width: 768px) {
            .search-controls, .date-range {
                flex-direction: column;
                align-items: stretch;
            }
            .sort-filter {
                flex-direction: column;
            }
            .filter-group, .sort-group {
                flex-wrap: wrap;
            }
        }
    </style>
</head>
<body>
    <header>
        <h1>Suivi des Astéroïdes - Imators</h1>
    </header>
    
    <div class="container">
        <div class="error" id="error-message"></div>
        
        <div class="search-controls">
            <div class="date-range">
                <label>Date début:</label>
                <input type="date" id="start-date">
            </div>
            <div class="date-range">
                <label>Date fin (max 7 jours):</label>
                <input type="date" id="end-date">
            </div>
            <button id="search-button">Rechercher</button>
        </div>
        
        <div class="loading" id="loading">Chargement des données...</div>
        <div class="no-results" id="no-results">Aucun astéroïde trouvé pour cette période.</div>
        
        <div class="stats-container" id="stats-container">
            <div class="stat-card">
                <div class="stat-value" id="total-count">0</div>
                <div class="stat-label">Astéroïdes détectés</div>
            </div>
            <div class="stat-card">
                <div class="stat-value danger-stat" id="hazardous-count">0</div>
                <div class="stat-label">Potentiellement dangereux</div>
            </div>
            <div class="stat-card">
                <div class="stat-value size-stat" id="max-diameter">0</div>
                <div class="stat-label">Diamètre max (m)</div>
            </div>
            <div class="stat-card">
                <div class="stat-value distance-stat" id="closest-approach">0</div>
                <div class="stat-label">Approche min. (km)</div>
            </div>
        </div>
        
        <div class="sort-filter">
            <div class="filter-group">
                <label>Filtrer:</label>
                <select id="hazard-filter">
                    <option value="all">Tous</option>
                    <option value="hazardous">Potentiellement dangereux</option>
                    <option value="safe">Sans danger</option>
                </select>
                <select id="size-filter">
                    <option value="all">Toute taille</option>
                    <option value="small">Petit (< 100m)</option>
                    <option value="medium">Moyen (100m - 500m)</option>
                    <option value="large">Grand (> 500m)</option>
                </select>
            </div>
            <div class="sort-group">
                <label>Trier par:</label>
                <select id="sort-by">
                    <option value="date">Date d'approche</option>
                    <option value="name">Nom</option>
                    <option value="diameter">Diamètre</option>
                    <option value="distance">Distance</option>
                    <option value="velocity">Vitesse</option>
                </select>
                <select id="sort-order">
                    <option value="asc">Croissant</option>
                    <option value="desc">Décroissant</option>
                </select>
            </div>
        </div>
        
        <div class="asteroid-list" id="asteroid-list"></div>
        
        <div class="pagination">
            <button id="prev-page" disabled>Précédent</button>
            <div class="page-info">Page <span id="current-page">1</span></div>
            <button id="next-page">Suivant</button>
        </div>
    </div>

    <script>
        const API_KEY = "xxhCuSUbAEFfAzIVhYxiRbvcPLVBWY2nOEpXEVWz";
        const startDateInput = document.getElementById('start-date');
        const endDateInput = document.getElementById('end-date');
        const searchButton = document.getElementById('search-button');
        const asteroidList = document.getElementById('asteroid-list');
        const loading = document.getElementById('loading');
        const errorMessage = document.getElementById('error-message');
        const noResults = document.getElementById('no-results');
        const statsContainer = document.getElementById('stats-container');
        
        const totalCountElement = document.getElementById('total-count');
        const hazardousCountElement = document.getElementById('hazardous-count');
        const maxDiameterElement = document.getElementById('max-diameter');
        const closestApproachElement = document.getElementById('closest-approach');
        
        const hazardFilter = document.getElementById('hazard-filter');
        const sizeFilter = document.getElementById('size-filter');
        const sortBy = document.getElementById('sort-by');
        const sortOrder = document.getElementById('sort-order');
        
        const prevPageButton = document.getElementById('prev-page');
        const nextPageButton = document.getElementById('next-page');
        const currentPageElement = document.getElementById('current-page');
        
        let allAsteroids = [];
        let filteredAsteroids = [];
        let currentPage = 1;
        const itemsPerPage = 10;
        
        function initDateInputs() {
            const today = new Date();
            const todayStr = formatDateForInput(today);
            
            const maxEndDate = new Date();
            maxEndDate.setDate(today.getDate() + 7);
            const maxEndDateStr = formatDateForInput(maxEndDate);
            
            startDateInput.value = todayStr;
            startDateInput.max = maxEndDateStr;
            
            endDateInput.value = todayStr;
            endDateInput.max = maxEndDateStr;
            
            startDateInput.addEventListener('change', updateEndDateMin);
            endDateInput.addEventListener('change', updateStartDateMax);
        }
        
        function updateEndDateMin() {
            endDateInput.min = startDateInput.value;
            
            const startDate = new Date(startDateInput.value);
            const maxEndDate = new Date(startDate);
            maxEndDate.setDate(startDate.getDate() + 7);
            
            const today = new Date();
            const absoluteMaxEndDate = new Date();
            absoluteMaxEndDate.setDate(today.getDate() + 7);
            
            const finalMaxDate = new Date(Math.min(maxEndDate.getTime(), absoluteMaxEndDate.getTime()));
            endDateInput.max = formatDateForInput(finalMaxDate);
            
            if (new Date(endDateInput.value) > finalMaxDate) {
                endDateInput.value = formatDateForInput(finalMaxDate);
            }
        }
        
        function updateStartDateMax() {
            const endDate = new Date(endDateInput.value);
            const minStartDate = new Date(endDate);
            minStartDate.setDate(endDate.getDate() - 7);
            
            const finalMinStartDate = formatDateForInput(minStartDate);
            if (new Date(startDateInput.value) < minStartDate) {
                startDateInput.value = finalMinStartDate;
            }
            
            startDateInput.min = finalMinStartDate;
        }
        
        function formatDateForInput(date) {
            return date.toISOString().split('T')[0];
        }
        
        searchButton.addEventListener('click', fetchAsteroids);
        hazardFilter.addEventListener('change', applyFiltersAndSort);
        sizeFilter.addEventListener('change', applyFiltersAndSort);
        sortBy.addEventListener('change', applyFiltersAndSort);
        sortOrder.addEventListener('change', applyFiltersAndSort);
        prevPageButton.addEventListener('click', goToPrevPage);
        nextPageButton.addEventListener('click', goToNextPage);
        
        function fetchAsteroids() {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                showError("Veuillez sélectionner des dates de début et de fin.");
                return;
            }
            
            const start = new Date(startDate);
            const end = new Date(endDate);
            const diffDays = Math.round((end - start) / (1000 * 60 * 60 * 24)) + 1;
            
            if (diffDays > 7) {
                showError("La période ne peut pas dépasser 7 jours.");
                return;
            }
            
            clearAsteroids();
            showLoading();
            hideError();
            hideNoResults();
            
            const apiUrl = `https://api.nasa.gov/neo/rest/v1/feed?start_date=${startDate}&end_date=${endDate}&api_key=${API_KEY}`;
            
            fetch(apiUrl)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`Erreur HTTP: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    processAsteroids(data);
                })
                .catch(error => {
                    showError(`Erreur lors de la récupération des données: ${error.message}`);
                })
                .finally(() => {
                    hideLoading();
                });
        }
        
        function processAsteroids(data) {
            allAsteroids = [];
            
            const nearEarthObjects = data.near_earth_objects;
            
            Object.keys(nearEarthObjects).forEach(date => {
                nearEarthObjects[date].forEach(asteroid => {
                    const closeApproach = asteroid.close_approach_data[0];
                    
                    allAsteroids.push({
                        id: asteroid.id,
                        name: asteroid.name.replace('(', '').replace(')', ''),
                        hazardous: asteroid.is_potentially_hazardous_asteroid,
                        diameterMin: Math.round(asteroid.estimated_diameter.meters.estimated_diameter_min),
                        diameterMax: Math.round(asteroid.estimated_diameter.meters.estimated_diameter_max),
                        approachDate: closeApproach.close_approach_date,
                        distance: parseInt(closeApproach.miss_distance.kilometers),
                        velocity: parseFloat(closeApproach.relative_velocity.kilometers_per_second).toFixed(2),
                        orbitingBody: closeApproach.orbiting_body
                    });
                });
            });
            
            if (allAsteroids.length === 0) {
                showNoResults();
                hideStatsContainer();
                return;
            }
            
            updateStats();
            currentPage = 1;
            applyFiltersAndSort();
        }
        
        function applyFiltersAndSort() {
            const hazardValue = hazardFilter.value;
            const sizeValue = sizeFilter.value;
            const sortByValue = sortBy.value;
            const sortOrderValue = sortOrder.value;
            
            filteredAsteroids = allAsteroids.filter(asteroid => {
                let passesHazardFilter = true;
                if (hazardValue === 'hazardous') passesHazardFilter = asteroid.hazardous;
                if (hazardValue === 'safe') passesHazardFilter = !asteroid.hazardous;
                
                let passesSizeFilter = true;
                const avgDiameter = (asteroid.diameterMin + asteroid.diameterMax) / 2;
                if (sizeValue === 'small') passesSizeFilter = avgDiameter < 100;
                if (sizeValue === 'medium') passesSizeFilter = avgDiameter >= 100 && avgDiameter <= 500;
                if (sizeValue === 'large') passesSizeFilter = avgDiameter > 500;
                
                return passesHazardFilter && passesSizeFilter;
            });
            
            filteredAsteroids.sort((a, b) => {
                let comparison = 0;
                
                switch (sortByValue) {
                    case 'name':
                        comparison = a.name.localeCompare(b.name);
                        break;
                    case 'diameter':
                        comparison = ((a.diameterMin + a.diameterMax) / 2) - ((b.diameterMin + b.diameterMax) / 2);
                        break;
                    case 'distance':
                        comparison = a.distance - b.distance;
                        break;
                    case 'velocity':
                        comparison = parseFloat(a.velocity) - parseFloat(b.velocity);
                        break;
                    default: // date
                        comparison = new Date(a.approachDate) - new Date(b.approachDate);
                }
                
                return sortOrderValue === 'asc' ? comparison : -comparison;
            });
            
            if (filteredAsteroids.length === 0) {
                showNoResults();
            } else {
                hideNoResults();
            }
            
            displayAsteroids();
        }
        
        function displayAsteroids() {
            clearAsteroids();
            
            const startIndex = (currentPage - 1) * itemsPerPage;
            const endIndex = Math.min(startIndex + itemsPerPage, filteredAsteroids.length);
            const pageAsteroids = filteredAsteroids.slice(startIndex, endIndex);
            
            pageAsteroids.forEach(asteroid => {
                createAsteroidCard(asteroid);
            });
            
            updatePagination();
        }
        
        function createAsteroidCard(asteroid) {
            const card = document.createElement('div');
            card.className = `asteroid-card ${asteroid.hazardous ? 'hazardous' : 'safe'}`;
            
            const avgDiameter = (asteroid.diameterMin + asteroid.diameterMax) / 2;
            const formattedDate = formatDate(asteroid.approachDate);
            
            card.innerHTML = `
                <div class="hazard-indicator ${asteroid.hazardous ? 'danger' : 'safe'}"></div>
                <h3 class="asteroid-name">${asteroid.name}</h3>
                <div class="asteroid-details">
                    <div class="detail-item">
                        <div class="detail-label">Date d'approche</div>
                        <div class="detail-value">${formattedDate}</div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Distance (km)</div>
                        <div class="detail-value">${formatNumber(asteroid.distance)}</div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Vitesse (km/s)</div>
                        <div class="detail-value">${asteroid.velocity}</div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Diamètre estimé (m)</div>
                        <div class="detail-value">${asteroid.diameterMin} - ${asteroid.diameterMax}</div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Corps céleste orbité</div>
                        <div class="detail-value">${asteroid.orbitingBody}</div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Potentiellement dangereux</div>
                        <div class="detail-value">${asteroid.hazardous ? 'Oui' : 'Non'}</div>
                    </div>
                </div>
            `;
            
            asteroidList.appendChild(card);
        }
        
        function updateStats() {
            const hazardousCount = allAsteroids.filter(a => a.hazardous).length;
            
            let maxDiameter = 0;
            let closestDistance = Infinity;
            
            allAsteroids.forEach(asteroid => {
                const avgDiameter = (asteroid.diameterMin + asteroid.diameterMax) / 2;
                if (avgDiameter > maxDiameter) maxDiameter = avgDiameter;
                if (asteroid.distance < closestDistance) closestDistance = asteroid.distance;
            });
            
            totalCountElement.textContent = allAsteroids.length;
            hazardousCountElement.textContent = hazardousCount;
            maxDiameterElement.textContent = formatNumber(Math.round(maxDiameter));
            closestApproachElement.textContent = formatNumber(Math.round(closestDistance));
            
            showStatsContainer();
        }
        
        function updatePagination() {
            const totalPages = Math.ceil(filteredAsteroids.length / itemsPerPage);
            currentPageElement.textContent = currentPage;
            
            prevPageButton.disabled = currentPage === 1;
            nextPageButton.disabled = currentPage === totalPages || totalPages === 0;
        }
        
        function goToPrevPage() {
            if (currentPage > 1) {
                currentPage--;
                displayAsteroids();
                window.scrollTo({ top: 0, behavior: 'smooth' });
            }
        }
        
        function goToNextPage() {
            const totalPages = Math.ceil(filteredAsteroids.length / itemsPerPage);
            if (currentPage < totalPages) {
                currentPage++;
                displayAsteroids();
                window.scrollTo({ top: 0, behavior: 'smooth' });
            }
        }
        
        function formatDate(dateStr) {
            const date = new Date(dateStr);
            return date.toLocaleDateString('fr-FR', {
                day: 'numeric',
                month: 'long',
                year: 'numeric'
            });
        }
        
        function formatNumber(num) {
            return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, " ");
        }
        
        function clearAsteroids() {
            asteroidList.innerHTML = '';
        }
        
        function showLoading() {
            loading.style.display = 'block';
        }
        
        function hideLoading() {
            loading.style.display = 'none';
        }
        
        function showError(message) {
            errorMessage.textContent = message;
            errorMessage.style.display = 'block';
        }
        
        function hideError() {
            errorMessage.style.display = 'none';
        }
        
        function showNoResults() {
            noResults.style.display = 'block';
        }
        
        function hideNoResults() {
            noResults.style.display = 'none';
        }
        
        function showStatsContainer() {
            statsContainer.style.display = 'flex';
        }
        
        function hideStatsContainer() {
            statsContainer.style.display = 'none';
        }
        
        initDateInputs();
    </script>
</body>
</html>