document.addEventListener('DOMContentLoaded', () => {
    const app = {
        map: null,
        mapboxKey: '',
        mapboxDirections: null,
        markers: {},
        userMarker: null,
        tempMarker: null,
        selectedLightId: null,
        userWatchId: null,
        refreshTimerId: null,
        rafId: null,
        lastTimestamp: 0,
        currentRoute: null,
        isNavigating: false,
        settings: {
            darkMode: false,
            notifications: true,
            autoRefresh: true,
            predictions: true,
            autoMeasure: true,
            defaultZoom: 15,
            mapStyle: 'streets'
        },
        measure: {
            timer: 0,
            startTime: 0,
            rafId: null,
            mode: null,
            lightId: null
        },
        navigation: {
            startPosition: null,
            endPosition: null,
            coordinates: [],
            lightsOnRoute: [],
            travelMode: 'driving',
            distance: null,
            duration: null,
            startTime: null,
            estimatedArrival: null,
            currentStep: 0,
            lastStop: null,
            stopDuration: 0
        },
        state: {
            selectingLocation: false,
            currentSlide: 1,
            routePanelOpen: false
        }
    };

    const dom = {
        map: document.getElementById('map'),
        sidebar: document.querySelector('.sidebar'),
        lightsList: document.getElementById('lightsList'),
        loadingLights: document.getElementById('loadingLights'),
        noLightsMessage: document.getElementById('noLightsMessage'),
        lightPopup: document.getElementById('lightPopup'),
        popupTitle: document.getElementById('popupTitle'),
        popupContent: document.getElementById('popupContent'),
        popupStatus: document.getElementById('popupStatus'),
        closePopup: document.getElementById('closePopup'),
        popupNavigate: document.getElementById('popupNavigate'),
        popupMeasure: document.getElementById('popupMeasure'),
        routePlannerPanel: document.getElementById('routePlannerPanel'),
        routeDetails: document.getElementById('routeDetails'),
        routeTime: document.getElementById('routeTime'),
        routeDistance: document.getElementById('routeDistance'),
        routeLights: document.getElementById('routeLights'),
        arrivalTime: document.getElementById('arrivalTime'),
        closeRoutePanel: document.getElementById('closeRoutePanel'),
        calculateRoute: document.getElementById('calculateRoute'),
        startNavigation: document.getElementById('startNavigation'),
        navigationPanel: document.getElementById('navigationPanel'),
        modals: {
            add: document.getElementById('addModal'),
            measure: document.getElementById('measureModal'),
            settings: document.getElementById('settingsModal'),
            login: document.getElementById('loginModal'),
            welcome: document.getElementById('welcomeModal')
        },
        overlays: document.querySelectorAll('.modal-overlay'),
        modalBodies: document.querySelectorAll('.modal-body'),
        draggableHandles: document.querySelectorAll('.draggable-handle'),
        closeButtons: document.querySelectorAll('.close-modal'),
        loadingScreen: document.getElementById('loadingScreen'),
        buttons: {
            menuToggle: document.getElementById('menuToggle'),
            headerLocate: document.getElementById('headerLocateBtn'),
            headerAdd: document.getElementById('headerAddBtn'),
            route: document.getElementById('routeBtn'),
            refresh: document.getElementById('refreshBtn'),
            settings: document.getElementById('settingsBtn'),
            nearby: document.getElementById('nearbyBtn'),
            routes: document.getElementById('routesBtn'),
            skipWelcome: document.getElementById('skipWelcome'),
            nextSlide: document.getElementById('nextSlide'),
            resetApp: document.getElementById('resetAppBtn'),
            measureRed: document.getElementById('measureRedBtn'),
            measureGreen: document.getElementById('measureGreenBtn'),
            startTimer: document.getElementById('startTimer'),
            stopTimer: document.getElementById('stopTimer'),
            saveTimer: document.getElementById('saveTimer'),
            routeTypeCar: document.getElementById('routeTypeCar'),
            routeTypeBike: document.getElementById('routeTypeBike'),
            routeTypeWalk: document.getElementById('routeTypeWalk'),
            endNavigation: document.getElementById('endNavigation')
        },
        form: {
            addLight: document.getElementById('addLightForm'),
            searchInput: document.getElementById('searchInput'),
            startLocation: document.getElementById('startLocation'),
            endLocation: document.getElementById('endLocation'),
            login: document.getElementById('loginForm'),
            register: document.getElementById('registerForm')
        },
        measure: {
            title: document.getElementById('measureTitle'),
            status: document.getElementById('measureStatus'),
            container: document.getElementById('timerContainer'),
            instructions: document.getElementById('timerInstructions'),
            display: document.getElementById('timerDisplay'),
            result: document.getElementById('measureResult')
        },
        welcome: {
            slides: document.querySelectorAll('.onboarding-slide'),
            dots: document.querySelectorAll('.onboarding-dot')
        },
        settings: {
            darkMode: document.getElementById('darkModeToggle'),
            notifications: document.getElementById('notificationsToggle'),
            autoRefresh: document.getElementById('autoRefreshToggle'),
            predictions: document.getElementById('predictionToggle'),
            autoMeasure: document.getElementById('autoMeasureToggle'),
            defaultZoom: document.getElementById('defaultZoom'),
            mapStyle: document.getElementById('mapStyle')
        },
        auth: {
            showLogin: document.getElementById('showLoginForm'),
            showRegister: document.getElementById('showRegisterForm'),
            authTitle: document.getElementById('authTitle'),
            login: document.getElementById('loginBtn')
        },
        navigation: {
            nextDirection: document.getElementById('nextDirection'),
            nextStreet: document.getElementById('nextStreet'),
            nextDistance: document.getElementById('nextDistance'),
            etaText: document.getElementById('etaText'),
            currentSpeed: document.getElementById('currentSpeed'),
            totalDistance: document.getElementById('totalDistance'),
            totalTime: document.getElementById('totalTime')
        }
    };

    init();

    function init() {
        fetchMapboxKey()
            .then(() => {
                loadSettings();
                initMap();
                bindEvents();
                setupDraggableModals();
                checkFirstVisit();
                checkLoggedInUser();
                
                setTimeout(() => {
                    dom.loadingScreen.style.opacity = '0';
                    setTimeout(() => {
                        dom.loadingScreen.style.display = 'none';
                    }, 500);
                }, 800);
            })
            .catch(error => {
                console.error('Failed to initialize app:', error);
                showNotification('Failed to initialize app. Please refresh.', 'error');
                dom.loadingScreen.innerHTML = '<div class="text-center"><p class="text-xl text-red-500 mb-2">Failed to load app</p><p>Please refresh the page</p></div>';
            });
    }
    
    function fetchMapboxKey() {
        return fetch('get-mapbox-key.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    app.mapboxKey = data.key;
                } else {
                    throw new Error('Failed to get Mapbox API key');
                }
            });
    }

    function loadSettings() {
        try {
            const saved = localStorage.getItem('trafficLightSettings');
            if (saved) {
                const parsed = JSON.parse(saved);
                app.settings = { ...app.settings, ...parsed };
                applySettings();
            }
        } catch (e) {
            console.error('Error loading settings');
        }
    }

    function applySettings() {
        if (dom.settings.darkMode) dom.settings.darkMode.checked = app.settings.darkMode;
        if (dom.settings.notifications) dom.settings.notifications.checked = app.settings.notifications;
        if (dom.settings.autoRefresh) dom.settings.autoRefresh.checked = app.settings.autoRefresh;
        if (dom.settings.predictions) dom.settings.predictions.checked = app.settings.predictions;
        if (dom.settings.autoMeasure) dom.settings.autoMeasure.checked = app.settings.autoMeasure;
        if (dom.settings.defaultZoom) dom.settings.defaultZoom.value = app.settings.defaultZoom;
        if (dom.settings.mapStyle) dom.settings.mapStyle.value = app.settings.mapStyle;
        
        if (app.settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
        }
    }

    function saveSettings() {
        if (dom.settings.darkMode) app.settings.darkMode = dom.settings.darkMode.checked;
        if (dom.settings.notifications) app.settings.notifications = dom.settings.notifications.checked;
        if (dom.settings.autoRefresh) app.settings.autoRefresh = dom.settings.autoRefresh.checked;
        if (dom.settings.predictions) app.settings.predictions = dom.settings.predictions.checked;
        if (dom.settings.autoMeasure) app.settings.autoMeasure = dom.settings.autoMeasure.checked;
        if (dom.settings.defaultZoom) app.settings.defaultZoom = dom.settings.defaultZoom.value;
        if (dom.settings.mapStyle) app.settings.mapStyle = dom.settings.mapStyle.value;
        
        try {
            localStorage.setItem('trafficLightSettings', JSON.stringify(app.settings));
        } catch (e) {
            console.error('Error saving settings');
        }
    }

    function checkFirstVisit() {
        if (!localStorage.getItem('trafficLightAppVisited')) {
            dom.modals.welcome.style.display = 'flex';
            localStorage.setItem('trafficLightAppVisited', 'true');
        } else {
            dom.modals.welcome.style.display = 'none';
        }
        
        // Hide all modals by default
        Object.values(dom.modals).forEach(modal => {
            if (modal !== dom.modals.welcome) modal.style.display = 'none';
        });
    }
    
    function checkLoggedInUser() {
        // Check cookies
        const userId = getCookie('user_id');
        const username = getCookie('username');
        
        if (userId && username) {
            // Auto login
            fetch('db.php?action=autoLogin', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    user_id: userId,
                    username: username
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    updateUIForLoggedInUser(data.user);
                }
            })
            .catch(error => console.error('Auto login error:', error));
        }
    }
    
    function getCookie(name) {
        const value = `; ${document.cookie}`;
        const parts = value.split(`; ${name}=`);
        if (parts.length === 2) return parts.pop().split(';').shift();
        return null;
    }

    function initMap() {
        mapboxgl.accessToken = app.mapboxKey;
        
        app.map = new mapboxgl.Map({
            container: 'map',
            style: getMapStyle(),
            center: [0, 0],
            zoom: parseInt(app.settings.defaultZoom)
        });
        
        app.map.addControl(new mapboxgl.NavigationControl({
            showCompass: true,
            showZoom: true,
            visualizePitch: true
        }), 'bottom-right');
        
        app.map.on('load', () => {
            app.map.on('click', handleMapClick);
            loadTrafficLights();
            startGeolocation();
            
            // Initialize directions
            app.mapboxDirections = new MapboxDirections({
                accessToken: mapboxgl.accessToken,
                unit: 'metric',
                profile: 'mapbox/driving',
                alternatives: false,
                controls: false,
                interactive: false
            });
        });
    }
    
    function getMapStyle() {
        switch (app.settings.mapStyle) {
            case 'satellite':
                return 'mapbox://styles/mapbox/satellite-streets-v12';
            case 'navigation':
                return 'mapbox://styles/mapbox/navigation-day-v2';
            case 'dark':
                return 'mapbox://styles/mapbox/dark-v11';
            default:
                return 'mapbox://styles/mapbox/streets-v12';
        }
    }

    function updateMapStyle() {
        app.map.setStyle(getMapStyle());
    }

    function setupDraggableModals() {
        dom.draggableHandles.forEach(handle => {
            const modal = handle.closest('.modal-body');
            let startY, currentY, initialY;
            let isDragging = false;
            
            handle.addEventListener('touchstart', e => {
                startY = e.touches[0].clientY;
                initialY = 0;
                isDragging = true;
                modal.style.transition = 'none';
            }, { passive: true });
            
            handle.addEventListener('touchmove', e => {
                if (!isDragging) return;
                currentY = e.touches[0].clientY;
                const diffY = currentY - startY;
                if (diffY < 0) return;
                
                modal.style.transform = `translateY(${diffY}px)`;
            }, { passive: true });
            
            handle.addEventListener('touchend', e => {
                modal.style.transition = 'transform 0.3s ease';
                if (!isDragging) return;
                
                const diffY = currentY - startY;
                if (diffY > 100) {
                    modal.style.transform = 'translateY(100%)';
                    const modalContainer = modal.closest('.modal');
                    setTimeout(() => {
                        modalContainer.style.display = 'none';
                        modal.style.transform = 'translateY(0)';
                    }, 300);
                } else {
                    modal.style.transform = 'translateY(0)';
                }
                
                isDragging = false;
            }, { passive: true });
        });
        
        dom.overlays.forEach(overlay => {
            overlay.addEventListener('click', () => {
                const modal = overlay.closest('.modal');
                closeModal(modal);
            });
        });
    }

    function bindEvents() {
        dom.buttons.menuToggle.addEventListener('click', toggleSidebar);
        dom.buttons.headerLocate.addEventListener('click', locateUser);
        dom.buttons.headerAdd.addEventListener('click', () => openModal(dom.modals.add));
        dom.buttons.settings.addEventListener('click', () => openModal(dom.modals.settings));
        dom.buttons.refresh.addEventListener('click', refreshLights);
        dom.buttons.nearby.addEventListener('click', findNearbyLights);
        dom.buttons.route.addEventListener('click', toggleRoutePanel);
        dom.buttons.routes.addEventListener('click', showSavedRoutes);
        dom.form.searchInput.addEventListener('input', handleSearch);
        
        dom.closePopup.addEventListener('click', () => dom.lightPopup.classList.add('hidden'));
        dom.popupNavigate.addEventListener('click', navigateToSelectedLight);
        dom.popupMeasure.addEventListener('click', () => {
            dom.lightPopup.classList.add('hidden');
            openMeasureModal(app.selectedLightId);
        });
        
        dom.closeButtons.forEach(btn => {
            btn.addEventListener('click', () => {
                const modal = btn.closest('.modal');
                closeModal(modal);
            });
        });
        
        dom.closeRoutePanel.addEventListener('click', () => {
            dom.routePlannerPanel.classList.add('hidden');
            app.state.routePanelOpen = false;
        });
        
        // Settings event listeners
        if (dom.settings.darkMode) {
            dom.settings.darkMode.addEventListener('change', toggleDarkMode);
        }
        if (dom.settings.autoRefresh) {
            dom.settings.autoRefresh.addEventListener('change', toggleAutoRefresh);
        }
        if (dom.settings.notifications) {
            dom.settings.notifications.addEventListener('change', saveSettings);
        }
        if (dom.settings.predictions) {
            dom.settings.predictions.addEventListener('change', () => {
                saveSettings();
                updateAllLightStatus();
            });
        }
        if (dom.settings.autoMeasure) {
            dom.settings.autoMeasure.addEventListener('change', saveSettings);
        }
        if (dom.settings.defaultZoom) {
            dom.settings.defaultZoom.addEventListener('change', saveSettings);
        }
        if (dom.settings.mapStyle) {
            dom.settings.mapStyle.addEventListener('change', () => {
                saveSettings();
                updateMapStyle();
            });
        }
        
        // Route panel events
        dom.buttons.routeTypeCar.addEventListener('click', () => setTravelMode('driving'));
        dom.buttons.routeTypeBike.addEventListener('click', () => setTravelMode('cycling'));
        dom.buttons.routeTypeWalk.addEventListener('click', () => setTravelMode('walking'));
        dom.calculateRoute.addEventListener('click', calculateRoute);
        dom.startNavigation.addEventListener('click', startRouteNavigation);
        dom.buttons.endNavigation.addEventListener('click', endNavigation);
        
        // Measure events
        dom.buttons.resetApp.addEventListener('click', resetAppData);
        dom.buttons.measureRed.addEventListener('click', () => startMeasure('red'));
        dom.buttons.measureGreen.addEventListener('click', () => startMeasure('green'));
        dom.buttons.startTimer.addEventListener('click', startMeasureTimer);
        dom.buttons.stopTimer.addEventListener('click', stopMeasureTimer);
        dom.buttons.saveTimer.addEventListener('click', saveMeasureTimer);
        
        // Authentication events
        if (dom.auth.showLogin) dom.auth.showLogin.addEventListener('click', showLoginForm);
        if (dom.auth.showRegister) dom.auth.showRegister.addEventListener('click', showRegisterForm);
        if (dom.auth.login) dom.auth.login.addEventListener('click', () => openModal(dom.modals.login));
        if (dom.form.login) dom.form.login.addEventListener('submit', handleLogin);
        if (dom.form.register) dom.form.register.addEventListener('submit', handleRegister);
        
        // Welcome/Onboarding
        dom.buttons.skipWelcome.addEventListener('click', () => closeModal(dom.modals.welcome));
        dom.buttons.nextSlide.addEventListener('click', nextOnboardingSlide);
        dom.welcome.dots.forEach(dot => {
            dot.addEventListener('click', () => {
                const slide = parseInt(dot.getAttribute('data-dot'));
                goToSlide(slide);
            });
        });
        
        // Add light form
        dom.form.addLight.addEventListener('submit', handleAddLight);
        
        // Window/document events
        window.addEventListener('online', handleOnline);
        window.addEventListener('offline', handleOffline);
        document.addEventListener('visibilitychange', handleVisibilityChange);
    }

    function toggleSidebar() {
        dom.sidebar.classList.toggle('active');
        dom.buttons.menuToggle.innerHTML = dom.sidebar.classList.contains('active') 
            ? '<i class="fas fa-times"></i>' 
            : '<i class="fas fa-bars"></i>';
    }

    function locateUser() {
        if (app.userMarker) {
            app.map.flyTo({
                center: app.userMarker.getLngLat(),
                zoom: 16,
                essential: true
            });
        } else {
            startGeolocation();
        }
    }

    function refreshLights() {
        dom.buttons.refresh.innerHTML = '<i class="fas fa-sync-alt fa-spin"></i>';
        loadTrafficLights(true);
        setTimeout(() => {
            dom.buttons.refresh.innerHTML = '<i class="fas fa-sync-alt"></i>';
            if (app.settings.notifications) {
                showNotification('Traffic lights refreshed', 'success');
            }
        }, 800);
    }

    function findNearbyLights() {
        if (app.userMarker) {
            app.map.flyTo({
                center: app.userMarker.getLngLat(),
                zoom: 16,
                essential: true
            });
            highlightNearbyLights();
        } else {
            startGeolocation();
        }
    }
    
    function toggleRoutePanel() {
        if (app.state.routePanelOpen) {
            dom.routePlannerPanel.classList.add('hidden');
            app.state.routePanelOpen = false;
        } else {
            if (app.userMarker) {
                const position = app.userMarker.getLngLat();
                reverseGeocode(position.lng, position.lat)
                    .then(placeName => {
                        dom.form.startLocation.value = placeName || `${position.lat.toFixed(6)}, ${position.lng.toFixed(6)}`;
                    });
            }
            
            dom.routePlannerPanel.classList.remove('hidden');
            app.state.routePanelOpen = true;
        }
    }
    
    function showSavedRoutes() {
        fetch('db.php?action=getUserRoutes')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show routes in sidebar or in a modal
                    showRoutesListInSidebar(data.routes);
                } else {
                    if (app.settings.notifications) {
                        showNotification(data.message, 'error');
                    }
                }
            })
            .catch(error => {
                console.error('Error fetching routes:', error);
                if (app.settings.notifications) {
                    showNotification('Error loading saved routes', 'error');
                }
            });
    }
    
    function showRoutesListInSidebar(routes) {
        dom.lightsList.innerHTML = '';
        
        if (routes.length === 0) {
            dom.noLightsMessage.querySelector('p').textContent = 'No saved routes';
            dom.noLightsMessage.classList.remove('hidden');
            return;
        }
        
        dom.noLightsMessage.classList.add('hidden');
        
        routes.forEach(route => {
            const card = document.createElement('div');
            card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg';
            
            const startName = route.route_data.start_point.place_name || 'Start';
            const endName = route.route_data.end_point.place_name || 'Destination';
            const distance = route.route_data.distance ? `${(route.route_data.distance / 1000).toFixed(1)} km` : '';
            const duration = route.route_data.duration ? `${Math.round(route.route_data.duration / 60)} min` : '';
            
            card.innerHTML = `
                <div class="flex justify-between items-center">
                    <h3 class="font-medium">${route.name}</h3>
                    <span class="px-2.5 py-1 text-xs font-medium rounded-full bg-primary-100 text-primary-800">
                        <i class="fas fa-route mr-1"></i>${distance}
                    </span>
                </div>
                <div class="flex items-center text-sm text-gray-500 mt-2">
                    <i class="fas fa-map-marker-alt text-primary-500 mr-2"></i>
                    <span class="truncate">${startName} → ${endName}</span>
                </div>
                <div class="mt-2 flex justify-between text-xs">
                    <span class="text-gray-500">${duration}</span>
                    <span class="text-gray-500">${formatDate(route.created_at)}</span>
                </div>
            `;
            
            card.addEventListener('click', () => {
                loadSavedRoute(route);
            });
            
            dom.lightsList.appendChild(card);
        });
    }
    
    function formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString();
    }
    
    function loadSavedRoute(route) {
        // Set the route form data
        dom.form.startLocation.value = route.route_data.start_point.place_name || '';
        dom.form.endLocation.value = route.route_data.end_point.place_name || '';
        
        app.navigation.startPosition = route.route_data.start_point;
        app.navigation.endPosition = route.route_data.end_point;
        app.navigation.travelMode = route.route_data.travel_mode || 'driving';
        
        // Set travel mode button active state
        setTravelMode(app.navigation.travelMode, false);
        
        // Show the route panel
        dom.routePlannerPanel.classList.remove('hidden');
        app.state.routePanelOpen = true;
        
        // Close sidebar on mobile
        if (window.innerWidth <= 768) {
            dom.sidebar.classList.remove('active');
            dom.buttons.menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
        }
        
        // Display route details
        displayRouteDetails(route.route_data);
        
        // Draw the route on the map
        drawRoute(route.route_data.coordinates);
    }

    function openModal(modal) {
        if (modal === dom.modals.add) {
            app.state.selectingLocation = true;
            if (app.userMarker) {
                const pos = app.userMarker.getLngLat();
                document.getElementById('latitude').value = pos.lat.toFixed(6);
                document.getElementById('longitude').value = pos.lng.toFixed(6);
                
                if (app.tempMarker) app.tempMarker.remove();
                app.tempMarker = new mapboxgl.Marker({ draggable: true })
                    .setLngLat([pos.lng, pos.lat])
                    .addTo(app.map);
                
                app.tempMarker.on('dragend', updateMarkerPosition);
            }
        }
        
        modal.style.display = 'flex';
        const modalBody = modal.querySelector('.modal-body');
        modalBody.style.transform = 'translateY(0)';
    }

    function closeModal(modal) {
        const modalBody = modal.querySelector('.modal-body');
        modalBody.style.transition = 'transform 0.3s ease';
        modalBody.style.transform = 'translateY(100%)';
        
        setTimeout(() => {
            modal.style.display = 'none';
            modalBody.style.transform = 'translateY(0)';
            
            if (modal === dom.modals.add) {
                app.state.selectingLocation = false;
                if (app.tempMarker) {
                    app.tempMarker.remove();
                    app.tempMarker = null;
                }
            }
        }, 300);
    }

    function handleMapClick(e) {
        if (app.state.selectingLocation) {
            document.getElementById('latitude').value = e.lngLat.lat.toFixed(6);
            document.getElementById('longitude').value = e.lngLat.lng.toFixed(6);
            
            if (app.tempMarker) app.tempMarker.remove();
            app.tempMarker = new mapboxgl.Marker({ draggable: true })
                .setLngLat([e.lngLat.lng, e.lngLat.lat])
                .addTo(app.map);
            
            app.tempMarker.on('dragend', updateMarkerPosition);
        }
    }

    function updateMarkerPosition() {
        const pos = app.tempMarker.getLngLat();
        document.getElementById('latitude').value = pos.lat.toFixed(6);
        document.getElementById('longitude').value = pos.lng.toFixed(6);
    }

    function handleSearch() {
        const searchTerm = dom.form.searchInput.value.toLowerCase();
        const lights = document.querySelectorAll('.light-card');
        
        let hasResults = false;
        
        lights.forEach(light => {
            const name = light.getAttribute('data-name')?.toLowerCase() || '';
            const visible = name.includes(searchTerm);
            light.style.display = visible ? 'block' : 'none';
            if (visible) hasResults = true;
        });
        
        if (lights.length > 0 && !hasResults) {
            dom.noLightsMessage.classList.remove('hidden');
            dom.noLightsMessage.querySelector('p').textContent = 'No results found';
        } else {
            dom.noLightsMessage.classList.add('hidden');
        }
    }

    function toggleDarkMode() {
        app.settings.darkMode = dom.settings.darkMode.checked;
        
        if (app.settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
            
            if (dom.settings.mapStyle.value !== 'dark') {
                const previousStyle = dom.settings.mapStyle.value;
                dom.settings.mapStyle.value = 'dark';
                app.settings.mapStyle = 'dark';
                updateMapStyle();
                dom.settings.mapStyle.dataset.previousStyle = previousStyle;
            }
        } else {
            document.documentElement.classList.remove('dark');
            document.body.classList.remove('dark-mode');
            
            if (dom.settings.mapStyle.value === 'dark' && dom.settings.mapStyle.dataset.previousStyle) {
                dom.settings.mapStyle.value = dom.settings.mapStyle.dataset.previousStyle;
                app.settings.mapStyle = dom.settings.mapStyle.value;
                updateMapStyle();
            } else if (dom.settings.mapStyle.value === 'dark') {
                dom.settings.mapStyle.value = 'streets';
                app.settings.mapStyle = 'streets';
                updateMapStyle();
            }
        }
        
        saveSettings();
    }

    function toggleAutoRefresh() {
        app.settings.autoRefresh = dom.settings.autoRefresh.checked;
        
        if (app.settings.autoRefresh) {
            startStatusUpdates();
        } else {
            stopStatusUpdates();
        }
        
        saveSettings();
    }

    function resetAppData() {
        if (confirm('Are you sure you want to reset all app data? This will clear all your settings and cache.')) {
            localStorage.clear();
            showNotification('App data reset. Refreshing...', 'info');
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        }
    }

    function navigateToSelectedLight() {
        if (app.selectedLightId && app.markers[app.selectedLightId]) {
            const light = app.markers[app.selectedLightId].data;
            
            dom.form.startLocation.value = '';
            dom.form.endLocation.value = light.name;
            app.navigation.endPosition = {
                center: [parseFloat(light.longitude), parseFloat(light.latitude)],
                place_name: light.name
            };
            
            if (app.userMarker) {
                const userPosition = app.userMarker.getLngLat();
                reverseGeocode(userPosition.lng, userPosition.lat)
                    .then(placeName => {
                        dom.form.startLocation.value = placeName || `${userPosition.lat.toFixed(6)}, ${userPosition.lng.toFixed(6)}`;
                        app.navigation.startPosition = {
                            center: [userPosition.lng, userPosition.lat],
                            place_name: placeName || `${userPosition.lat.toFixed(6)}, ${userPosition.lng.toFixed(6)}`
                        };
                        
                        // Show route panel and hide light popup
                        dom.lightPopup.classList.add('hidden');
                        dom.routePlannerPanel.classList.remove('hidden');
                        app.state.routePanelOpen = true;
                    });
            } else {
                // Show route panel and hide light popup
                dom.lightPopup.classList.add('hidden');
                dom.routePlannerPanel.classList.remove('hidden');
                app.state.routePanelOpen = true;
                
                // Request user location
                startGeolocation();
            }
        }
    }
    
    function reverseGeocode(lng, lat) {
        return fetch(`https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?access_token=${app.mapboxKey}`)
            .then(response => response.json())
            .then(data => {
                if (data.features && data.features.length > 0) {
                    return data.features[0].place_name;
                }
                return null;
            })
            .catch(error => {
                console.error('Reverse geocoding error:', error);
                return null;
            });
    }
    
    function setTravelMode(mode, calculate = true) {
        app.navigation.travelMode = mode;
        
        // Update UI
        document.querySelectorAll('.travel-type-btn').forEach(btn => {
            btn.classList.remove('active');
        });
        
        let activeBtn;
        switch (mode) {
            case 'cycling':
                activeBtn = dom.buttons.routeTypeBike;
                break;
            case 'walking':
                activeBtn = dom.buttons.routeTypeWalk;
                break;
            default:
                activeBtn = dom.buttons.routeTypeCar;
        }
        
        if (activeBtn) activeBtn.classList.add('active');
        
        // Recalculate route if requested
        if (calculate && app.navigation.startPosition && app.navigation.endPosition) {
            calculateRoute();
        }
    }
    
    function calculateRoute() {
        if (!dom.form.startLocation.value || !dom.form.endLocation.value) {
            if (app.settings.notifications) {
                showNotification('Please enter start and end locations', 'error');
            }
            return;
        }
        
        // If we don't have coordinates for start/end, geocode them
        const promises = [];
        
        if (!app.navigation.startPosition || !app.navigation.startPosition.center) {
            promises.push(
                geocodeLocation(dom.form.startLocation.value)
                    .then(result => {
                        if (result) app.navigation.startPosition = result;
                    })
            );
        }
        
        if (!app.navigation.endPosition || !app.navigation.endPosition.center) {
            promises.push(
                geocodeLocation(dom.form.endLocation.value)
                    .then(result => {
                        if (result) app.navigation.endPosition = result;
                    })
            );
        }
        
        // Update button state
        dom.calculateRoute.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Calculating...';
        dom.calculateRoute.disabled = true;
        
        Promise.all(promises)
            .then(() => {
                if (!app.navigation.startPosition || !app.navigation.endPosition) {
                    throw new Error('Could not geocode locations');
                }
                
                // Get route from Mapbox Directions API
                return fetch(`https://api.mapbox.com/directions/v5/mapbox/${app.navigation.travelMode}/${app.navigation.startPosition.center[0]},${app.navigation.startPosition.center[1]};${app.navigation.endPosition.center[0]},${app.navigation.endPosition.center[1]}?geometries=geojson&overview=full&steps=true&access_token=${app.mapboxKey}`);
            })
            .then(response => response.json())
            .then(data => {
                if (data.code !== 'Ok' || !data.routes || data.routes.length === 0) {
                    throw new Error('No route found');
                }
                
                const route = data.routes[0];
                app.navigation.coordinates = route.geometry.coordinates;
                app.navigation.distance = route.distance;
                app.navigation.duration = route.duration;
                app.navigation.steps = route.legs[0].steps;
                
                // Draw route on map
                drawRoute(app.navigation.coordinates);
                
                // Find traffic lights on route
                return fetch('db.php?action=getLightsOnRoute', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        coordinates: app.navigation.coordinates
                    })
                });
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    app.navigation.lightsOnRoute = data.lights;
                    
                    // Display the route details in the panel
                    displayRouteDetails();
                    
                    // Show "Start Navigation" button
                    dom.startNavigation.classList.remove('hidden');
                    
                    // Fit the map to the route
                    fitMapToRoute(app.navigation.coordinates);
                } else {
                    throw new Error(data.message || 'Error fetching traffic lights on route');
                }
            })
            .catch(error => {
                console.error('Route calculation error:', error);
                if (app.settings.notifications) {
                    showNotification(error.message || 'Error calculating route', 'error');
                }
            })
            .finally(() => {
                // Reset button state
                dom.calculateRoute.innerHTML = '<i class="fas fa-route"></i> Calculate Route';
                dom.calculateRoute.disabled = false;
            });
    }
    
    function geocodeLocation(query) {
        return fetch(`https://api.mapbox.com/geocoding/v5/mapbox.places/${encodeURIComponent(query)}.json?access_token=${app.mapboxKey}`)
            .then(response => response.json())
            .then(data => {
                if (data.features && data.features.length > 0) {
                    return {
                        center: data.features[0].center,
                        place_name: data.features[0].place_name
                    };
                }
                return null;
            });
    }
    
    function drawRoute(coordinates) {
        // Check if route layer already exists
        if (app.map.getSource('route')) {
            // Update existing source
            app.map.getSource('route').setData({
                type: 'Feature',
                properties: {},
                geometry: {
                    type: 'LineString',
                    coordinates: coordinates
                }
            });
        } else {
            // Add new source and layer
            app.map.addSource('route', {
                type: 'geojson',
                data: {
                    type: 'Feature',
                    properties: {},
                    geometry: {
                        type: 'LineString',
                        coordinates: coordinates
                    }
                }
            });
            
            app.map.addLayer({
                id: 'route',
                type: 'line',
                source: 'route',
                layout: {
                    'line-join': 'round',
                    'line-cap': 'round'
                },
                paint: {
                    'line-color': '#0ea5e9',
                    'line-width': 6,
                    'line-opacity': 0.8
                }
            });
        }
    }
    
    function fitMapToRoute(coordinates) {
        if (!coordinates || coordinates.length === 0) return;
        
        const bounds = coordinates.reduce((bounds, coord) => {
            return bounds.extend(coord);
        }, new mapboxgl.LngLatBounds(coordinates[0], coordinates[0]));
        
        app.map.fitBounds(bounds, {
            padding: 50,
            maxZoom: 15,
            duration: 1000
        });
    }
    
    function displayRouteDetails() {
        // Show route details container
        dom.routeDetails.classList.remove('hidden');
        
        // Format distance and duration
        const distance = app.navigation.distance ? (app.navigation.distance / 1000).toFixed(1) : 0;
        const duration = app.navigation.duration ? Math.round(app.navigation.duration / 60) : 0;
        
        dom.routeDistance.textContent = `${distance} km`;
        dom.routeTime.textContent = `${duration} min`;
        
        // Calculate traffic lights status
        let redLights = 0;
        let greenLights = 0;
        
        app.navigation.lightsOnRoute.forEach(light => {
            const status = getLightStatus(light);
            if (status.isRed) {
                redLights++;
            } else {
                greenLights++;
            }
        });
        
        dom.routeLights.innerHTML = `
            <span class="inline-block px-2 py-1 bg-traffic-red/20 text-traffic-red rounded text-xs font-medium">${redLights} red</span>
            <span class="inline-block px-2 py-1 bg-traffic-green/20 text-traffic-green rounded text-xs font-medium">${greenLights} green</span>
        `;
        
        // Calculate estimated arrival time
        const now = new Date();
        const arrivalTime = new Date(now.getTime() + (app.navigation.duration * 1000));
        const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        
        dom.arrivalTime.textContent = formattedArrival;
    }
    
    function startRouteNavigation() {
        if (!app.navigation.coordinates || app.navigation.coordinates.length === 0) {
            if (app.settings.notifications) {
                showNotification('Please calculate a route first', 'error');
            }
            return;
        }
        
        // Hide route panel
        dom.routePlannerPanel.classList.add('hidden');
        
        // Show navigation panel
        dom.navigationPanel.classList.remove('hidden');
        
        // Set navigation state
        app.isNavigating = true;
        app.navigation.startTime = new Date();
        app.navigation.currentStep = 0;
        
        // Update navigation instructions
        updateNavigationInstructions();
        
        // Start tracking user location more frequently
        if (app.userWatchId) {
            navigator.geolocation.clearWatch(app.userWatchId);
        }
        
        app.userWatchId = navigator.geolocation.watchPosition(
            updateNavigationPosition,
            handleGeolocationError,
            { enableHighAccuracy: true, maximumAge: 2000, timeout: 5000 }
        );
        
        // Enable auto-measure if setting is on
        if (app.settings.autoMeasure) {
            startAutoDetectStops();
        }
    }
    
    function updateNavigationInstructions() {
        if (!app.navigation.steps || app.navigation.steps.length === 0 || 
            app.navigation.currentStep >= app.navigation.steps.length) {
            return;
        }
        
        const step = app.navigation.steps[app.navigation.currentStep];
        
        // Update instruction text
        dom.navigation.nextDirection.textContent = step.maneuver.instruction || 'Continue';
        
        // Update street name if available
        if (step.name) {
            dom.navigation.nextStreet.textContent = `on ${step.name}`;
        } else {
            dom.navigation.nextStreet.textContent = '';
        }
        
        // Update distance
        const distance = step.distance;
        let formattedDistance;
        
        if (distance < 100) {
            formattedDistance = `${Math.round(distance)} m`;
        } else if (distance < 1000) {
            formattedDistance = `${Math.round(distance / 10) * 10} m`;
        } else {
            formattedDistance = `${(distance / 1000).toFixed(1)} km`;
        }
        
        dom.navigation.nextDistance.textContent = formattedDistance;
        
        // Update ETA
        const now = new Date();
        const remainingDuration = app.navigation.duration - 
            ((now.getTime() - app.navigation.startTime.getTime()) / 1000);
        
        if (remainingDuration > 0) {
            const arrivalTime = new Date(now.getTime() + (remainingDuration * 1000));
            const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
            dom.navigation.etaText.textContent = `ETA ${formattedArrival}`;
        } else {
            dom.navigation.etaText.textContent = 'Arriving';
        }
        
        // Update progress info
        const totalDistance = app.navigation.distance;
        const remainingDistance = app.navigation.steps.slice(app.navigation.currentStep)
            .reduce((total, step) => total + step.distance, 0);
        
        const coveredDistance = totalDistance - remainingDistance;
        dom.navigation.totalDistance.textContent = `${(coveredDistance / 1000).toFixed(1)} / ${(totalDistance / 1000).toFixed(1)} km`;
        
        const totalMinutes = Math.round(app.navigation.duration / 60);
        const elapsedMinutes = Math.round((now.getTime() - app.navigation.startTime.getTime()) / 60000);
        dom.navigation.totalTime.textContent = `${elapsedMinutes} / ${totalMinutes} min`;
    }
    
    function updateNavigationPosition(position) {
        const lat = position.coords.latitude;
        const lng = position.coords.longitude;
        const speed = position.coords.speed || 0;
        
        // Update user marker
        if (!app.userMarker) {
            const el = document.createElement('div');
            el.className = 'location-dot';
            
            app.userMarker = new mapboxgl.Marker(el)
                .setLngLat([lng, lat])
                .addTo(app.map);
        } else {
            app.userMarker.setLngLat([lng, lat]);
        }
        
        // If navigating, center map on user and update navigation info
        if (app.isNavigating) {
            // Center map on user with forward facing direction
            app.map.flyTo({
                center: [lng, lat],
                zoom: 16,
                bearing: position.coords.heading || 0,
                pitch: 60,
                essential: true
            });
            
            // Update speed display
            const speedKmh = speed ? Math.round(speed * 3.6) : 0;
            dom.navigation.currentSpeed.textContent = `${speedKmh} km/h`;
            
            // Check if we've reached the next step
            checkNavigationProgress([lng, lat]);
            
            // Auto-detect stops if enabled
            if (app.settings.autoMeasure && speed < 0.5) {
                handleNavigationStop([lng, lat]);
            } else if (speed >= 0.5 && app.navigation.lastStop) {
                handleNavigationResume();
            }
        }
    }
    
    function checkNavigationProgress(userPosition) {
        if (!app.navigation.steps || app.navigation.currentStep >= app.navigation.steps.length) {
            return;
        }
        
        const currentStep = app.navigation.steps[app.navigation.currentStep];
        const nextStep = app.navigation.steps[app.navigation.currentStep + 1];
        
        if (nextStep) {
            // Check if user is close to the next maneuver point
            const maneuverPoint = nextStep.maneuver.location;
            const distanceToNextStep = getDistance(userPosition, maneuverPoint);
            
            if (distanceToNextStep < 30) { // Within 30 meters
                app.navigation.currentStep++;
                updateNavigationInstructions();
            }
        }
        
        // Check if we've reached destination (last step)
        if (app.navigation.currentStep === app.navigation.steps.length - 1) {
            const finalPoint = app.navigation.coordinates[app.navigation.coordinates.length - 1];
            const distanceToEnd = getDistance(userPosition, finalPoint);
            
            if (distanceToEnd < 50) { // Within 50 meters of destination
                handleNavigationComplete();
            }
        }
    }
    
    function getDistance(point1, point2) {
        // Convert to radians
        const lat1 = point1[1] * Math.PI / 180;
        const lon1 = point1[0] * Math.PI / 180;
        const lat2 = point2[1] * Math.PI / 180;
        const lon2 = point2[0] * Math.PI / 180;
        
        // Haversine formula
        const dlon = lon2 - lon1;
        const dlat = lat2 - lat1;
        const a = Math.sin(dlat/2)**2 + Math.cos(lat1) * Math.cos(lat2) * Math.sin(dlon/2)**2;
        const c = 2 * Math.asin(Math.sqrt(a));
        const r = 6371000; // Earth radius in meters
        
        return c * r;
    }
    
    function handleNavigationComplete() {
        if (app.settings.notifications) {
            showNotification('You have reached your destination!', 'success');
        }
        
        // End navigation
        endNavigation();
        
        // Celebrate animation or feedback could be added here
    }
    
    function startAutoDetectStops() {
        // This will be handled by the updateNavigationPosition function
        console.log('Auto-detect stops enabled');
    }
    
    function handleNavigationStop(position) {
        // If this is a new stop (not already tracking)
        if (!app.navigation.lastStop) {
            app.navigation.lastStop = {
                position: position,
                time: new Date(),
                duration: 0
            };
        } else {
            // Update stop duration
            const now = new Date();
            app.navigation.stopDuration = (now - app.navigation.lastStop.time) / 1000;
            
            // If stopped for more than 2 seconds at a new location
            if (app.navigation.stopDuration > 2) {
                // Check if we're on a road between intersections
                checkForTrafficLight(position);
            }
        }
    }
    
    function handleNavigationResume() {
        // If we were stopped for at least 2 seconds
        if (app.navigation.lastStop && app.navigation.stopDuration > 2) {
            // Reset stop tracking
            app.navigation.lastStop = null;
            app.navigation.stopDuration = 0;
        }
    }
    
    function checkForTrafficLight(position) {
        // First check if we're on a road (not at beginning or end of route)
        const routeLength = app.navigation.coordinates.length;
        
        if (routeLength <= 2) return;
        
        const startPoint = app.navigation.coordinates[0];
        const endPoint = app.navigation.coordinates[routeLength - 1];
        
        // Skip if we're too close to start or end
        const distanceFromStart = getDistance(position, startPoint);
        const distanceFromEnd = getDistance(position, endPoint);
        
        if (distanceFromStart < 1000 || distanceFromEnd < 1000) {
            return;
        }
        
        // Check if we're near an intersection
        let isNearIntersection = false;
        
        // Simplified intersection detection - better algorithms would be used in production
        for (let i = 1; i < routeLength - 1; i++) {
            const prevPoint = app.navigation.coordinates[i - 1];
            const currPoint = app.navigation.coordinates[i];
            const nextPoint = app.navigation.coordinates[i + 1];
            
            // Calculate angles between segments
            const angle1 = calculateBearing(prevPoint, currPoint);
            const angle2 = calculateBearing(currPoint, nextPoint);
            const angleDiff = Math.abs(angle1 - angle2);
            
            // If there's a significant turn and we're close to this point
            if ((angleDiff > 30 && angleDiff < 150) && getDistance(position, currPoint) < 50) {
                isNearIntersection = true;
                break;
            }
        }
        
        if (!isNearIntersection) return;
        
        // Likely at a traffic light, send data to server
        autoDetectTrafficLight(position, app.navigation.stopDuration);
    }
    
    function calculateBearing(point1, point2) {
        const lat1 = point1[1] * Math.PI / 180;
        const lon1 = point1[0] * Math.PI / 180;
        const lat2 = point2[1] * Math.PI / 180;
        const lon2 = point2[0] * Math.PI / 180;
        
        const y = Math.sin(lon2 - lon1) * Math.cos(lat2);
        const x = Math.cos(lat1) * Math.sin(lat2) - Math.sin(lat1) * Math.cos(lat2) * Math.cos(lon2 - lon1);
        const bearing = Math.atan2(y, x) * 180 / Math.PI;
        
        return (bearing + 360) % 360;
    }
    
    function autoDetectTrafficLight(position, duration) {
        // Determine likely direction based on route
        let direction = 'unknown';
        
        if (position.coords && position.coords.heading) {
            const heading = position.coords.heading;
            
            if (heading >= 315 || heading < 45) direction = 'north';
            else if (heading >= 45 && heading < 135) direction = 'east';
            else if (heading >= 135 && heading < 225) direction = 'south';
            else direction = 'west';
        }
        
        // Send to server
        fetch('db.php?action=autoDetectLight', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                latitude: position[1],
                longitude: position[0],
                duration: duration,
                duration_type: 'red', // Assuming red when stopped
                direction: direction
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && app.settings.notifications) {
                showNotification(data.message, 'success');
            }
        })
        .catch(error => console.error('Auto-detect light error:', error));
    }
    
    function endNavigation() {
        app.isNavigating = false;
        
        // Hide navigation panel
        dom.navigationPanel.classList.add('hidden');
        
        // Reset navigation data
        app.navigation.startTime = null;
        app.navigation.currentStep = 0;
        app.navigation.lastStop = null;
        app.navigation.stopDuration = 0;
        
        // Return map to normal view
        if (app.userMarker) {
            app.map.flyTo({
                center: app.userMarker.getLngLat(),
                zoom: parseInt(app.settings.defaultZoom),
                pitch: 0,
                bearing: 0,
                essential: true
            });
        }
        
        // Remove route from map
        if (app.map.getLayer('route')) {
            app.map.removeLayer('route');
            app.map.removeSource('route');
        }
        
        // Restore normal user tracking
        if (app.userWatchId) {
            navigator.geolocation.clearWatch(app.userWatchId);
        }
        startGeolocation();
    }

    function handleAddLight(e) {
        e.preventDefault();
        
        const submitBtn = dom.form.addLight.querySelector('[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Adding...';
        submitBtn.disabled = true;
        
        const lightData = {
            name: document.getElementById('lightName').value,
            latitude: document.getElementById('latitude').value,
            longitude: document.getElementById('longitude').value,
            direction: document.getElementById('direction').value,
            red_duration: document.getElementById('redDuration').value,
            green_duration: document.getElementById('greenDuration').value
        };
        
        fetch('db.php?action=addTrafficLight', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify(lightData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                addLightToMap(data.light);
                addLightToSidebar(data.light);
                
                if (app.settings.notifications) {
                    showNotification(data.message || 'Traffic light added successfully!', 'success');
                }
                
                closeModal(dom.modals.add);
                dom.form.addLight.reset();
                
                if (app.tempMarker) {
                    app.tempMarker.remove();
                    app.tempMarker = null;
                }
                
                app.state.selectingLocation = false;
                dom.noLightsMessage.classList.add('hidden');
                
                app.map.flyTo({
                    center: [parseFloat(data.light.longitude), parseFloat(data.light.latitude)],
                    zoom: 16,
                    essential: true
                });
            } else {
                if (app.settings.notifications) {
                    showNotification('Error: ' + data.message, 'error');
                }
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        })
        .catch(error => {
            console.error('Error:', error);
            if (app.settings.notifications) {
                showNotification('Connection error', 'error');
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }
    
    function showLoginForm() {
        dom.form.login.classList.remove('hidden');
        dom.form.register.classList.add('hidden');
        dom.auth.showLogin.classList.add('border-primary-500', 'text-primary-500');
        dom.auth.showLogin.classList.remove('border-transparent', 'text-gray-500');
        dom.auth.showRegister.classList.add('border-transparent', 'text-gray-500');
        dom.auth.showRegister.classList.remove('border-primary-500', 'text-primary-500');
        dom.auth.authTitle.textContent = 'Login';
    }
    
    function showRegisterForm() {
        dom.form.login.classList.add('hidden');
        dom.form.register.classList.remove('hidden');
        dom.auth.showRegister.classList.add('border-primary-500', 'text-primary-500');
        dom.auth.showRegister.classList.remove('border-transparent', 'text-gray-500');
        dom.auth.showLogin.classList.add('border-transparent', 'text-gray-500');
        dom.auth.showLogin.classList.remove('border-primary-500', 'text-primary-500');
        dom.auth.authTitle.textContent = 'Register';
    }
    
    function handleLogin(e) {
        e.preventDefault();
        
        const formData = new FormData(dom.form.login);
        const submitBtn = dom.form.login.querySelector('[type="submit"]');
        const originalText = submitBtn.textContent;
        
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Logging in...';
        submitBtn.disabled = true;
        
        fetch('db.php?action=login', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                email: formData.get('email'),
                password: formData.get('password'),
                remember: formData.get('remember') === 'on'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (app.settings.notifications) {
                    showNotification('Login successful', 'success');
                }
                
                updateUIForLoggedInUser(data.user);
                closeModal(dom.modals.login);
            } else {
                if (app.settings.notifications) {
                    showNotification('Error: ' + data.message, 'error');
                }
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        })
        .catch(error => {
            console.error('Login error:', error);
            if (app.settings.notifications) {
                showNotification('Connection error', 'error');
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }
    
    function handleRegister(e) {
        e.preventDefault();
        
        const formData = new FormData(dom.form.register);
        const submitBtn = dom.form.register.querySelector('[type="submit"]');
        const originalText = submitBtn.textContent;
        
        // Validate passwords match
        if (formData.get('password') !== formData.get('password_confirm')) {
            if (app.settings.notifications) {
                showNotification('Passwords do not match', 'error');
            }
            return;
        }
        
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Registering...';
        submitBtn.disabled = true;
        
        fetch('db.php?action=register', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                username: formData.get('username'),
                email: formData.get('email'),
                password: formData.get('password')
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (app.settings.notifications) {
                    showNotification('Registration successful', 'success');
                }
                
                updateUIForLoggedInUser(data.user);
                closeModal(dom.modals.login);
            } else {
                if (app.settings.notifications) {
                    showNotification('Error: ' + data.message, 'error');
                }
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        })
        .catch(error => {
            console.error('Registration error:', error);
            if (app.settings.notifications) {
                showNotification('Connection error', 'error');
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }
    
    function updateUIForLoggedInUser(user) {
        // Update sidebar
        const loginSection = document.querySelector('.sidebar > .p-4.border-t');
        if (loginSection) {
            loginSection.innerHTML = `
                <div class="flex items-center mb-3">
                    <div class="w-10 h-10 rounded-full bg-primary-500 text-white flex items-center justify-center mr-3">
                        <i class="fas fa-user"></i>
                    </div>
                    <div>
                        <p class="font-medium text-sm">${user.username}</p>
                        <p class="text-xs text-gray-500">Logged in</p>
                    </div>
                    <button id="logoutBtn" class="ml-auto text-gray-500 hover:text-gray-700">
                        <i class="fas fa-sign-out-alt"></i>
                    </button>
                </div>
            `;
            
            // Add logout button event
            document.getElementById('logoutBtn').addEventListener('click', handleLogout);
        }
    }
    
    function handleLogout() {
        fetch('db.php?action=logout')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (app.settings.notifications) {
                        showNotification('Logout successful', 'success');
                    }
                    
                    // Update UI for logged out state
                    const loginSection = document.querySelector('.sidebar > .p-4.border-t');
                    if (loginSection) {
                        loginSection.innerHTML = `
                            <button id="loginBtn" class="w-full bg-primary-500 hover:bg-primary-600 text-white font-medium px-4 py-2.5 rounded-lg transition-colors flex items-center justify-center gap-2">
                                <i class="fas fa-sign-in-alt"></i> Login / Sign Up
                            </button>
                        `;
                        
                        document.getElementById('loginBtn').addEventListener('click', () => openModal(dom.modals.login));
                    }
                } else {
                    if (app.settings.notifications) {
                        showNotification('Error: ' + data.message, 'error');
                    }
                }
            })
            .catch(error => {
                console.error('Logout error:', error);
                if (app.settings.notifications) {
                    showNotification('Connection error', 'error');
                }
            });
    }

    function nextOnboardingSlide() {
        if (app.state.currentSlide < dom.welcome.slides.length) {
            goToSlide(app.state.currentSlide + 1);
        } else {
            closeModal(dom.modals.welcome);
        }
    }

    function goToSlide(slideNumber) {
        app.state.currentSlide = slideNumber;
        
        dom.welcome.slides.forEach(slide => {
            slide.classList.add('hidden');
        });
        
        dom.welcome.dots.forEach(dot => {
            dot.classList.remove('active');
        });
        
        document.querySelector(`.onboarding-slide[data-slide="${slideNumber}"]`)?.classList.remove('hidden');
        document.querySelector(`.onboarding-dot[data-dot="${slideNumber}"]`)?.classList.add('active');
        
        dom.buttons.nextSlide.textContent = slideNumber === dom.welcome.slides.length ? 'Get Started' : 'Next';
    }

    function startMeasure(mode) {
        app.measure.mode = mode;
        dom.measure.container.classList.remove('hidden');
        
        if (mode === 'red') {
            dom.measure.instructions.textContent = 'Press "Start" when the light turns red, then "Stop" when it turns green.';
            dom.buttons.startTimer.className = 'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors';
            dom.buttons.stopTimer.className = 'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors opacity-50';
        } else {
            dom.measure.instructions.textContent = 'Press "Start" when the light turns green, then "Stop" when it turns red.';
            dom.buttons.startTimer.className = 'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors';
            dom.buttons.stopTimer.className = 'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors opacity-50';
        }
        
        resetMeasureTimer();
    }

    function startMeasureTimer() {
        app.measure.startTime = performance.now();
        app.measure.timer = 0;
        
        if (app.measure.rafId) cancelAnimationFrame(app.measure.rafId);
        
        function updateTimer(timestamp) {
            const elapsed = Math.floor((timestamp - app.measure.startTime) / 1000);
            if (elapsed !== app.measure.timer) {
                app.measure.timer = elapsed;
                const minutes = Math.floor(elapsed / 60);
                const seconds = elapsed % 60;
                dom.measure.display.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            }
            app.measure.rafId = requestAnimationFrame(updateTimer);
        }
        
        app.measure.rafId = requestAnimationFrame(updateTimer);
        dom.measure.display.textContent = '00:00';
        dom.measure.result.textContent = 'Measuring...';
        
        dom.buttons.startTimer.disabled = true;
        dom.buttons.startTimer.classList.add('opacity-50');
        dom.buttons.stopTimer.disabled = false;
        dom.buttons.stopTimer.classList.remove('opacity-50');
        dom.buttons.saveTimer.disabled = true;
        dom.buttons.saveTimer.classList.add('opacity-50');
    }

    function stopMeasureTimer() {
        if (app.measure.rafId) {
            cancelAnimationFrame(app.measure.rafId);
            app.measure.rafId = null;
            
            dom.measure.result.textContent = `Measured duration: ${app.measure.timer} seconds. Click Save to confirm.`;
            
            dom.buttons.stopTimer.disabled = true;
            dom.buttons.stopTimer.classList.add('opacity-50');
            dom.buttons.saveTimer.disabled = false;
            dom.buttons.saveTimer.classList.remove('opacity-50');
        }
    }

    function resetMeasureTimer() {
        if (app.measure.rafId) {
            cancelAnimationFrame(app.measure.rafId);
            app.measure.rafId = null;
        }
        app.measure.timer = 0;
        app.measure.startTime = 0;
        dom.measure.display.textContent = '00:00';
        dom.measure.result.textContent = '';
        dom.buttons.startTimer.disabled = false;
        dom.buttons.startTimer.classList.remove('opacity-50');
        dom.buttons.stopTimer.disabled = true;
        dom.buttons.stopTimer.classList.add('opacity-50');
        dom.buttons.saveTimer.disabled = true;
        dom.buttons.saveTimer.classList.add('opacity-50');
    }

    function saveMeasureTimer() {
        const lightId = dom.measure.title.getAttribute('data-id');
        
        if (!lightId || app.measure.timer <= 0 || !app.measure.mode) return;
        
        dom.measure.result.textContent = 'Saving...';
        
        fetch('db.php?action=updateTiming', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({
                id: lightId,
                duration_type: app.measure.mode,
                duration: app.measure.timer
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                dom.measure.result.innerHTML = '<i class="fas fa-check text-green-500 mr-1"></i> Timing updated!';
                
                app.markers[lightId].data = data.light;
                updateLightStatus(data.light);
                
                dom.buttons.saveTimer.disabled = true;
                dom.buttons.saveTimer.classList.add('opacity-50');
                
                setTimeout(() => {
                    dom.measure.container.classList.add('hidden');
                    app.measure.mode = null;
                }, 1500);
            } else {
                dom.measure.result.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Error: ' + data.message;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            dom.measure.result.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Connection error';
        });
    }

    function openMeasureModal(lightId) {
        const light = app.markers[lightId].data;
        app.measure.lightId = lightId;
        
        dom.measure.title.textContent = light.name;
        dom.measure.title.setAttribute('data-id', lightId);
        
        const status = getLightStatus(light);
        dom.measure.status.innerHTML = `
            <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                ${status.label} (${status.timeLeft}s)
            </span>
        `;
        
        dom.measure.container.classList.add('hidden');
        app.measure.mode = null;
        resetMeasureTimer();
        
        openModal(dom.modals.measure);
    }

    function loadTrafficLights(isRefresh = false) {
        if (!isRefresh) {
            dom.loadingLights.style.display = 'block';
            dom.noLightsMessage.classList.add('hidden');
        }
        
        fetch('db.php?action=getTrafficLights')
            .then(response => response.json())
            .then(data => {
                dom.loadingLights.style.display = 'none';
                
                if (isRefresh) {
                    for (let id in app.markers) {
                        app.markers[id].marker.remove();
                        delete app.markers[id];
                    }
                    
                    const lightCards = document.querySelectorAll('.light-card');
                    lightCards.forEach(card => card.remove());
                }
                
                if (Array.isArray(data) && data.length > 0) {
                    data.forEach(light => {
                        addLightToMap(light);
                        addLightToSidebar(light);
                    });
                } else {
                    dom.noLightsMessage.classList.remove('hidden');
                }
                
                if (app.settings.autoRefresh) {
                    startStatusUpdates();
                }
            })
            .catch(error => {
                console.error('Error:', error);
                dom.loadingLights.style.display = 'none';
                dom.noLightsMessage.classList.remove('hidden');
                if (app.settings.notifications) {
                    showNotification('Error loading traffic lights', 'error');
                }
            });
    }

    function addLightToMap(light) {
        const status = getLightStatus(light);
        
        const el = document.createElement('div');
        el.className = 'w-8 h-8 bg-traffic-' + status.color + ' rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white';
        el.innerHTML = '<i class="fas fa-traffic-light"></i>';
        
        const marker = new mapboxgl.Marker(el)
            .setLngLat([parseFloat(light.longitude), parseFloat(light.latitude)])
            .addTo(app.map);
        
        marker.getElement().addEventListener('click', () => showLightDetail(light));
        
        app.markers[light.id] = {
            marker: marker,
            data: light
        };
    }

    function addLightToSidebar(light) {
        const status = getLightStatus(light);
        
        const card = document.createElement('div');
        card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg';
        card.setAttribute('data-id', light.id);
        card.setAttribute('data-name', light.name);
        
        const showPrediction = app.settings.predictions;
        
        card.innerHTML = `
            <div class="flex justify-between items-center">
                <h3 class="font-medium">${light.name}</h3>
                <span id="status-${light.id}" class="px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white">${status.label}</span>
            </div>
            <div class="flex items-center text-sm text-gray-500 mt-2">
                <i class="fas fa-location-dot mr-2"></i>
                <span class="capitalize">${light.direction}</span>
                <div class="ml-auto flex items-center" id="timer-${light.id}">
                    <i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s
                </div>
            </div>
            ${showPrediction ? `
            <div class="mt-2 text-xs">
                <div class="w-full bg-gray-100 rounded-full h-1.5">
                    <div class="bg-traffic-${status.color} h-1.5 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                </div>
                <div class="flex justify-between mt-1 text-gray-400">
                    <span>0s</span>
                    <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                </div>
            </div>` : ''}
        `;
        
        card.addEventListener('click', function() {
            const allCards = document.querySelectorAll('.light-card');
            allCards.forEach(c => c.classList.remove('active'));
            this.classList.add('active');
            
            showLightDetail(light);
            
            if (window.innerWidth <= 768) {
                dom.sidebar.classList.remove('active');
                dom.buttons.menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
            }
        });
        
        dom.lightsList.insertBefore(card, dom.loadingLights);
    }

    function getLightStatus(light) {
        const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
        const currentTime = Math.floor(Date.now() / 1000);
        const timeInCycle = currentTime % totalCycle;
        
        if (timeInCycle < light.red_duration) {
            return {
                isRed: true,
                color: 'red',
                label: 'RED',
                timeLeft: light.red_duration - timeInCycle
            };
        } else {
            return {
                isRed: false,
                color: 'green',
                label: 'GREEN',
                timeLeft: totalCycle - timeInCycle
            };
        }
    }

    function updateLightStatus(light) {
        const status = getLightStatus(light);
        const showPrediction = app.settings.predictions;
        
        if (app.markers[light.id]) {
            const el = app.markers[light.id].marker.getElement().firstChild;
            
            // Update marker color
            el.className = el.className.replace(/bg-traffic-\w+/, `bg-traffic-${status.color}`);
        }
        
        const statusElement = document.getElementById(`status-${light.id}`);
        const timerElement = document.getElementById(`timer-${light.id}`);
        
        if (statusElement) {
            statusElement.className = `px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white`;
            statusElement.textContent = status.label;
        }
        
        if (timerElement) {
            timerElement.innerHTML = `<i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s`;
        }
        
        const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
        if (card) {
            let predictionEl = card.querySelector('.mt-2.text-xs');
            
            if (showPrediction) {
                if (predictionEl) {
                    const progressBar = predictionEl.querySelector('.bg-traffic-red, .bg-traffic-green');
                    if (progressBar) {
                        progressBar.className = `bg-traffic-${status.color} h-1.5 rounded-full`;
                        progressBar.style.width = `${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%`;
                    }
                } else {
                    predictionEl = document.createElement('div');
                    predictionEl.className = 'mt-2 text-xs';
                    predictionEl.innerHTML = `
                        <div class="w-full bg-gray-100 rounded-full h-1.5">
                            <div class="bg-traffic-${status.color} h-1.5 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                        </div>
                        <div class="flex justify-between mt-1 text-gray-400">
                            <span>0s</span>
                            <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                        </div>
                    `;
                    card.appendChild(predictionEl);
                }
            } else if (!showPrediction && predictionEl) {
                predictionEl.remove();
            }
        }
        
        if (app.selectedLightId === light.id && !dom.lightPopup.classList.contains('hidden')) {
            const popupStatus = dom.popupStatus;
            popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
            popupStatus.innerHTML = `
                <div class="text-xl font-semibold mb-1">${status.label}</div>
                <div class="flex justify-center items-center gap-2">
                    <i class="fas fa-clock"></i>
                    <span>Changes in ${status.timeLeft} seconds</span>
                </div>
                ${showPrediction ? `
                <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                    <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                </div>
                ` : ''}
            `;
        }
        
        if (app.measure.lightId === light.id && dom.modals.measure.style.display === 'flex') {
            dom.measure.status.innerHTML = `
                <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                    ${status.label} (${status.timeLeft}s)
                </span>
            `;
        }
    }

    function updateAllLightStatus() {
        for (const id in app.markers) {
            updateLightStatus(app.markers[id].data);
        }
    }

    function startStatusUpdates() {
        stopStatusUpdates();
        
        function updateStatuses(timestamp) {
            if (!app.lastTimestamp || timestamp - app.lastTimestamp > 1000) {
                app.lastTimestamp = timestamp;
                updateAllLightStatus();
            }
            app.rafId = requestAnimationFrame(updateStatuses);
        }
        
        app.rafId = requestAnimationFrame(updateStatuses);
    }

    function stopStatusUpdates() {
        if (app.rafId) {
            cancelAnimationFrame(app.rafId);
            app.rafId = null;
        }
    }

    function showLightDetail(light) {
        app.selectedLightId = light.id;
        
        app.map.flyTo({
            center: [parseFloat(light.longitude), parseFloat(light.latitude)],
            zoom: 17,
            essential: true
        });
        
        dom.popupTitle.textContent = light.name;
        dom.popupContent.innerHTML = `
            <div class="grid grid-cols-2 gap-3">
                <div class="text-gray-500">Direction:</div>
                <div class="font-medium capitalize">${light.direction}</div>
                
                <div class="text-gray-500">Red duration:</div>
                <div class="font-medium">${light.red_duration} seconds</div>
                
                <div class="text-gray-500">Green duration:</div>
                <div class="font-medium">${light.green_duration} seconds</div>
                
                <div class="text-gray-500">Total cycle:</div>
                <div class="font-medium">${parseInt(light.red_duration) + parseInt(light.green_duration)} seconds</div>
                
                <div class="text-gray-500">Location:</div>
                <div class="font-medium truncate">${light.latitude.substring(0, 8)}, ${light.longitude.substring(0, 8)}</div>
            </div>
        `;
        
        const status = getLightStatus(light);
        const showPrediction = app.settings.predictions;
        
        dom.popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
        dom.popupStatus.innerHTML = `
            <div class="text-xl font-semibold mb-1">${status.label}</div>
            <div class="flex justify-center items-center gap-2">
                <i class="fas fa-clock"></i>
                <span>Changes in ${status.timeLeft} seconds</span>
            </div>
            ${showPrediction ? `
            <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
            </div>
            ` : ''}
        `;
        
        dom.lightPopup.classList.remove('hidden');
    }

    function startGeolocation() {
        if (navigator.geolocation) {
            if (app.userWatchId) navigator.geolocation.clearWatch(app.userWatchId);
            
            dom.buttons.headerLocate.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>';
            
            app.userWatchId = navigator.geolocation.watchPosition(
                updateUserLocation,
                handleGeolocationError,
                { enableHighAccuracy: true, maximumAge: 10000, timeout: 5000 }
            );
            
            navigator.geolocation.getCurrentPosition(
                updateUserLocation,
                handleGeolocationError,
                { enableHighAccuracy: true }
            );
        }
    }

    function updateUserLocation(position) {
        const lat = position.coords.latitude;
        const lng = position.coords.longitude;
        
        dom.buttons.headerLocate.innerHTML = '<i class="fas fa-location-crosshairs"></i>';
        
        if (!app.userMarker) {
            const el = document.createElement('div');
            el.className = 'location-dot';
            
            app.userMarker = new mapboxgl.Marker(el)
                .setLngLat([lng, lat])
                .addTo(app.map);
            
            app.map.flyTo({
                center: [lng, lat],
                zoom: 16,
                essential: true
            });
        } else {
            app.userMarker.setLngLat([lng, lat]);
        }
    }

    function handleGeolocationError(error) {
        console.error('Geolocation error:', error.message);
        
        dom.buttons.headerLocate.innerHTML = '<i class="fas fa-location-crosshairs"></i>';
        
        if (app.settings.notifications) {
            showNotification('Location access denied', 'error');
        }
    }

    function highlightNearbyLights() {
        if (!app.userMarker) return;
        
        const userPos = app.userMarker.getLngLat();
        const nearbyLights = [];
        
        for (const id in app.markers) {
            const markerPos = app.markers[id].marker.getLngLat();
            const distance = getDistance(
                [userPos.lng, userPos.lat],
                [markerPos.lng, markerPos.lat]
            );
            
            if (distance < 1000) {
                nearbyLights.push({
                    id: id,
                    distance: distance
                });
            }
        }
        
        nearbyLights.sort((a, b) => a.distance - b.distance);
        
        document.querySelectorAll('.light-card').forEach(card => {
            card.classList.remove('active');
        });
        
        if (nearbyLights.length > 0) {
            if (app.settings.notifications) {
                showNotification(`${nearbyLights.length} traffic lights found within 1km`, 'info');
            }
            
            nearbyLights.slice(0, 5).forEach(light => {
                const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
                if (card) {
                    card.classList.add('active');
                    card.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                }
            });
        } else {
            if (app.settings.notifications) {
                showNotification('No traffic lights nearby', 'info');
            }
        }
    }

    function showNotification(message, type) {
        const colors = {
            success: 'bg-traffic-green',
            error: 'bg-traffic-red',
            info: 'bg-primary-500'
        };
        
        const icons = {
            success: 'check-circle',
            error: 'exclamation-circle',
            info: 'info-circle'
        };
        
        const notification = document.createElement('div');
        notification.className = `fixed top-20 left-1/2 transform -translate-x-1/2 ${colors[type]} text-white px-4 py-3 rounded-lg shadow-lg z-50 fade-in flex items-center`;
        notification.innerHTML = `<i class="fas fa-${icons[type]} mr-2"></i>${message}`;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translate(-50%, -10px)';
            notification.style.transition = 'opacity 0.5s, transform 0.5s';
            setTimeout(() => notification.remove(), 500);
        }, 3000);
    }

    function handleOnline() {
        if (app.settings.notifications) {
            showNotification('You are back online', 'success');
        }
        loadTrafficLights(true);
    }

    function handleOffline() {
        if (app.settings.notifications) {
            showNotification('You are offline. Some features may be limited', 'error');
        }
    }

    function handleVisibilityChange() {
        if (document.visibilityState === 'visible') {
            loadTrafficLights(true);
        }
    }

    // Corrigé pour éviter les problèmes avec service-worker.js
    function handleInstallPrompt() {
        let deferredPrompt;
        
        window.addEventListener('beforeinstallprompt', (e) => {
            e.preventDefault();
            deferredPrompt = e;
            
            const installBanner = document.createElement('div');
            installBanner.className = 'fixed bottom-0 left-0 right-0 bg-primary-500 text-white py-3 px-4 shadow-lg z-40 fade-in';
            installBanner.innerHTML = `
                <div class="flex items-center justify-between max-w-md mx-auto">
                    <div>
                        <p class="font-medium">Add TrafficLight to Home Screen</p>
                        <p class="text-sm">Get quick access to traffic lights data</p>
                    </div>
                    <div class="flex space-x-2">
                        <button id="dismissInstall" class="px-3 py-1.5 bg-primary-600 hover:bg-primary-700 rounded font-medium">Later</button>
                        <button id="installApp" class="px-3 py-1.5 bg-white text-primary-600 hover:bg-gray-100 rounded font-medium">Install</button>
                    </div>
                </div>
            `;
            
           document.body.appendChild(installBanner);
            
            document.getElementById('dismissInstall').addEventListener('click', () => {
                installBanner.style.opacity = '0';
                installBanner.style.transform = 'translateY(100%)';
                installBanner.style.transition = 'opacity 0.5s, transform 0.5s';
                setTimeout(() => installBanner.remove(), 500);
            });
            
            document.getElementById('installApp').addEventListener('click', () => {
                deferredPrompt.prompt();
                deferredPrompt.userChoice.then((choiceResult) => {
                    if (choiceResult.outcome === 'accepted') {
                        console.log('User accepted the install prompt');
                        installBanner.style.opacity = '0';
                        installBanner.style.transform = 'translateY(100%)';
                        installBanner.style.transition = 'opacity 0.5s, transform 0.5s';
                        setTimeout(() => installBanner.remove(), 500);
                    }
                    deferredPrompt = null;
                });
            });
        });
    }
    
    handleInstallPrompt();
});