<?php
require_once 'db.php';

// Pagination settings
$articles_per_page = 12;
$current_page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($current_page - 1) * $articles_per_page;

// Search functionality
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$search_condition = '';
$params = [];

if (!empty($search)) {
    $search_condition = "WHERE title LIKE :search OR content LIKE :search";
    $params[':search'] = "%$search%";
}

// Get total articles for pagination
$count_query = "SELECT COUNT(*) FROM data_article $search_condition";
$stmt = $db->prepare($count_query);
if (!empty($params)) {
    $stmt->execute($params);
} else {
    $stmt->execute();
}
$total_articles = $stmt->fetchColumn();
$total_pages = ceil($total_articles / $articles_per_page);

// Get articles for current page
$query = "SELECT id, title, SUBSTRING(content, 1, 300) as preview, created_at, author, read_time 
          FROM data_article 
          $search_condition 
          ORDER BY created_at DESC 
          LIMIT :offset, :limit";

$stmt = $db->prepare($query);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':limit', $articles_per_page, PDO::PARAM_INT);
if (!empty($params)) {
    $stmt->bindValue(':search', $params[':search'], PDO::PARAM_STR);
}
$stmt->execute();
$articles = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en" class="h-full">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Data Articles - Peerkinton</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://api.fontshare.com/v2/css?f[]=satoshi@400,500,700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Satoshi', sans-serif;
            background: #fafafa;
        }

        .article-card {
            transition: all 0.3s ease;
        }

        .article-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }

        .search-input {
            transition: all 0.3s ease;
        }

        .search-input:focus {
            box-shadow: 0 0 0 2px rgba(0, 0, 0, 0.1);
        }
    </style>
</head>
<body class="h-full">
    <?php include '../header.php'; ?>

    <main class="pt-2 px-6">
        <div class="max-w-6xl mx-auto">
            <p>
            This page brings together the data accumulated by our institute. Our students have the role of editor and are responsible for writing articles on a wide range of subjects. You can consult them below.
            </p>
        </div>
    </main>

    

    <main class="pt-32 pb-16 px-6">
        <div class="max-w-6xl mx-auto">
            <!-- Search Section -->
            <div class="mb-12">
                <form method="GET" class="max-w-2xl mx-auto">
                    <div class="relative">
                        <input 
                            type="text" 
                            name="search" 
                            value="<?php echo htmlspecialchars($search); ?>"
                            placeholder="Search articles..." 
                            class="search-input w-full px-6 py-4 text-gray-900 placeholder-gray-500 bg-white rounded-full border border-gray-200 focus:outline-none"
                        >
                        <button type="submit" class="absolute right-4 top-1/2 transform -translate-y-1/2">
                            <svg class="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                            </svg>
                        </button>
                    </div>
                </form>
            </div>

            <!-- Articles Grid -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                <?php foreach ($articles as $article): ?>
                    <a href="article.php?id=<?php echo $article['id']; ?>" class="article-card bg-white rounded-xl p-6 border border-gray-100">
                        <article>
                            <header class="mb-4">
                                <h2 class="text-xl font-medium text-gray-900 mb-2">
                                    <?php echo htmlspecialchars($article['title']); ?>
                                </h2>
                                <div class="flex items-center text-sm text-gray-500 space-x-4">
                                    <span><?php echo htmlspecialchars($article['author']); ?></span>
                                    <span>·</span>
                                    <span><?php echo $article['read_time']; ?> min read</span>
                                </div>
                            </header>
                            <p class="text-gray-600 mb-4 line-clamp-3">
                                <?php echo htmlspecialchars($article['preview']); ?>...
                            </p>
                            <div class="text-sm text-gray-500">
                                <?php echo date('M j, Y', strtotime($article['created_at'])); ?>
                            </div>
                        </article>
                    </a>
                <?php endforeach; ?>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <div class="mt-12 flex justify-center space-x-2">
                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <a 
                            href="?page=<?php echo $i; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>"
                            class="<?php echo $current_page === $i ? 'bg-gray-900 text-white' : 'bg-white text-gray-900 hover:bg-gray-100'; ?> px-4 py-2 rounded-lg text-sm"
                        >
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>
</body>
</html>