<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);
$country = get_user_country($_SESSION['user_id']);

$error = null;
$success = null;
$command_output = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['command'])) {
    $command = trim($_POST['command']);
    
    if (empty($command)) {
        $error = "Commande vide";
    } else {
        $command_parts = explode(' ', $command);
        $action = strtolower($command_parts[0]);
        
        switch ($action) {
            case 'help':
                $command_output = [
                    'type' => 'info',
                    'content' => [
                        "Commandes disponibles:",
                        "- help : affiche cette aide",
                        "- status : affiche l'état du système",
                        "- list : liste les combinaisons enregistrées",
                        "- add [combinaison] : ajoute une combinaison",
                        "- scan [pays] : scan des vulnérabilités d'un pays",
                        "- attack [pays] [combinaison] : attaque un système ennemi",
                        "- clear : efface l'écran"
                    ]
                ];
                break;
                
            case 'status':
                $command_output = [
                    'type' => 'info',
                    'content' => [
                        "Statut du système national:",
                        "- Pays: " . $country['name'],
                        "- Niveau de sécurité: " . $country['system_security_level'] . "/5",
                        "- État économique: " . $country['economy_status'],
                        "- Combinaisons enregistrées: " . db_get_value("SELECT COUNT(*) FROM system_combinations WHERE country_id = ? AND active = 1", [$country['id']])
                    ]
                ];
                break;
                
            case 'list':
                $limit = 10;
                if (isset($command_parts[1]) && is_numeric($command_parts[1])) {
                    $limit = intval($command_parts[1]);
                }
                
                $combinations = db_get_all(
                    "SELECT sc.*, u.username 
                     FROM system_combinations sc
                     JOIN users u ON sc.added_by = u.id
                     WHERE sc.country_id = ? AND sc.active = 1
                     ORDER BY sc.created_at DESC
                     LIMIT ?",
                    [$country['id'], $limit]
                );
                
                $content = ["Liste des dernières combinaisons enregistrées:"];
                
                if (empty($combinations)) {
                    $content[] = "Aucune combinaison enregistrée.";
                } else {
                    foreach ($combinations as $comb) {
                        $content[] = "- " . $comb['combination'] . " (ajoutée par " . $comb['username'] . " le " . format_datetime($comb['created_at']) . ")";
                    }
                }
                
                $command_output = [
                    'type' => 'info',
                    'content' => $content
                ];
                break;
                
            case 'add':
                if (!isset($command_parts[1])) {
                    $command_output = [
                        'type' => 'error',
                        'content' => ["Erreur: Combinaison manquante. Usage: add [combinaison]"]
                    ];
                } else {
                    $combination = $command_parts[1];
                    
                    if (strlen($combination) !== 3 || !ctype_digit($combination)) {
                        $command_output = [
                            'type' => 'error',
                            'content' => ["Erreur: La combinaison doit être un nombre à 3 chiffres."]
                        ];
                    } else {
                        $exists = db_get_value(
                            "SELECT COUNT(*) FROM system_combinations WHERE country_id = ? AND combination = ? AND active = 1",
                            [$country['id'], $combination]
                        );
                        
                        if ($exists > 0) {
                            $command_output = [
                                'type' => 'error',
                                'content' => ["Erreur: Cette combinaison est déjà enregistrée dans le système."]
                            ];
                        } else {
                            add_system_combination($country['id'], $combination, $_SESSION['user_id']);
                            add_experience($_SESSION['user_id'], 50);
                            
                            $command_output = [
                                'type' => 'success',
                                'content' => [
                                    "Combinaison ajoutée avec succès!",
                                    "+50 XP",
                                    "Nouvelle combinaison: " . $combination
                                ]
                            ];
                        }
                    }
                }
                break;
                
            case 'scan':
                if (!isset($command_parts[1])) {
                    $command_output = [
                        'type' => 'error',
                        'content' => ["Erreur: Pays cible manquant. Usage: scan [pays]"]
                    ];
                } else {
                    $target_country_name = implode(' ', array_slice($command_parts, 1));
                    
                    $target_country = db_get_one(
                        "SELECT * FROM countries WHERE name LIKE ?",
                        ['%' . $target_country_name . '%']
                    );
                    
                    if (!$target_country) {
                        $command_output = [
                            'type' => 'error',
                            'content' => ["Erreur: Pays non trouvé. Vérifiez le nom du pays."]
                        ];
                    } else if ($target_country['id'] == $country['id']) {
                        $command_output = [
                            'type' => 'error',
                            'content' => ["Erreur: Vous ne pouvez pas scanner votre propre pays."]
                        ];
                    } else {
                        $combinations_count = db_get_value(
                            "SELECT COUNT(*) FROM system_combinations WHERE country_id = ? AND active = 1",
                            [$target_country['id']]
                        );
                        
                        $vulnerability_score = 5 - $target_country['system_security_level'];
                        $chance_of_success = min(100, max(5, (100 - ($combinations_count / 10)) * (1 + $vulnerability_score * 0.2)));
                        
                        $command_output = [
                            'type' => 'info',
                            'content' => [
                                "Scan du pays: " . $target_country['name'],
                                "Niveau de sécurité: " . $target_country['system_security_level'] . "/5",
                                "Combinaisons estimées: " . ($combinations_count > 0 ? $combinations_count : "Inconnues"),
                                "Vulnérabilité estimée: " . number_format($chance_of_success, 1) . "%",
                                "",
                                "Conseil: Essayez d'attaquer avec une combinaison aléatoire à 3 chiffres."
                            ]
                        ];
                        
                        add_experience($_SESSION['user_id'], 25);
                        $command_output['content'][] = "+25 XP pour l'analyse";
                    }
                }
                break;
                
            case 'attack':
                if (!isset($command_parts[1]) || !isset($command_parts[2])) {
                    $command_output = [
                        'type' => 'error',
                        'content' => ["Erreur: Informations manquantes. Usage: attack [pays] [combinaison]"]
                    ];
                } else {
                    $combination = $command_parts[count($command_parts) - 1];
                    $target_country_name = implode(' ', array_slice($command_parts, 1, -1));
                    
                    if (strlen($combination) !== 3 || !ctype_digit($combination)) {
                        $command_output = [
                            'type' => 'error',
                            'content' => ["Erreur: La combinaison doit être un nombre à 3 chiffres."]
                        ];
                    } else {
                        $target_country = db_get_one(
                            "SELECT * FROM countries WHERE name LIKE ?",
                            ['%' . $target_country_name . '%']
                        );
                        
                        if (!$target_country) {
                            $command_output = [
                                'type' => 'error',
                                'content' => ["Erreur: Pays non trouvé. Vérifiez le nom du pays."]
                            ];
                        } else if ($target_country['id'] == $country['id']) {
                            $command_output = [
                                'type' => 'error',
                                'content' => ["Erreur: Vous ne pouvez pas attaquer votre propre pays."]
                            ];
                        } else {
                            $result = attack_country_system($_SESSION['user_id'], $target_country['id'], $combination);
                            
                            if ($result['success']) {
                                $command_output = [
                                    'type' => 'success',
                                    'content' => [
                                        "ATTAQUE RÉUSSIE!",
                                        "Cible: " . $target_country['name'],
                                        "Combinaison vulnérable: " . $combination,
                                        "Résultat: " . $result['message'],
                                        "+500 XP pour l'attaque réussie!"
                                    ]
                                ];
                            } else {
                                $command_output = [
                                    'type' => 'error',
                                    'content' => [
                                        "ATTAQUE ÉCHOUÉE",
                                        "Cible: " . $target_country['name'],
                                        "Combinaison tentée: " . $combination,
                                        "Résultat: " . $result['message'],
                                        "",
                                        "Conseil: Essayez une autre combinaison ou effectuez un scan pour évaluer les vulnérabilités."
                                    ]
                                ];
                            }
                        }
                    }
                }
                break;
                
            case 'clear':
                $command_output = [
                    'type' => 'clear',
                    'content' => []
                ];
                break;
                
            default:
                $command_output = [
                    'type' => 'error',
                    'content' => ["Commande non reconnue: " . $action . ". Tapez 'help' pour voir la liste des commandes."]
                ];
        }
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Terminal | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .terminal {
            background-color: #0D1117;
            color: #E6EDF3;
            font-family: 'Courier New', monospace;
            border-radius: 8px;
            overflow: hidden;
            height: 70vh;
            display: flex;
            flex-direction: column;
        }
        .terminal-header {
            background-color: #161B22;
            padding: 8px 16px;
            display: flex;
            align-items: center;
        }
        .terminal-circle {
            height: 12px;
            width: 12px;
            border-radius: 50%;
            margin-right: 6px;
        }
        .terminal-title {
            margin-left: 8px;
            font-size: 0.85rem;
            color: #8B949E;
        }
        .terminal-body {
            padding: 16px;
            flex-grow: 1;
            overflow-y: auto;
        }
        .terminal-line {
            line-height: 1.5;
            white-space: pre-wrap;
            word-break: break-word;
        }
        .terminal-prompt {
            color: #7EE787;
        }
        .terminal-error {
            color: #F85149;
        }
        .terminal-success {
            color: #56D364;
        }
        .terminal-input-container {
            display: flex;
            padding: 8px 16px;
            background-color: #0D1117;
            border-top: 1px solid #30363D;
        }
        .terminal-input {
            background-color: transparent;
            border: none;
            color: #E6EDF3;
            flex-grow: 1;
            font-family: 'Courier New', monospace;
            outline: none;
        }
        .blink {
            animation: blink-animation 1s steps(2, start) infinite;
        }
        @keyframes blink-animation {
            to {
                visibility: hidden;
            }
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="terminal.php" class="hover:text-blue-300 font-semibold">Terminal</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <?php if ($user['level'] >= 10): ?>
                        <li><a href="unions.php" class="hover:text-blue-300">Unions</a></li>
                    <?php endif; ?>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Terminal système</h2>
        
        <div class="terminal">
            <div class="terminal-header">
                <div class="terminal-circle bg-red-500"></div>
                <div class="terminal-circle bg-yellow-500"></div>
                <div class="terminal-circle bg-green-500"></div>
                <div class="terminal-title">Terminal de sécurité - <?php echo htmlspecialchars($country['name']); ?></div>
            </div>
            <div class="terminal-body" id="terminal-output">
                <div class="terminal-line">Système Cergies v3.7.2 - Terminal de sécurité national</div>
                <div class="terminal-line">Connexion établie: Agent <?php echo htmlspecialchars($user['username']); ?></div>
                <div class="terminal-line">Niveau d'autorisation: <?php echo $user['level']; ?></div>
                <div class="terminal-line">Date: <?php echo date('Y-m-d H:i:s'); ?></div>
                <div class="terminal-line">--------------------------------------------</div>
                <div class="terminal-line">Tapez 'help' pour afficher la liste des commandes.</div>
                <div class="terminal-line">--------------------------------------------</div>
                
                <?php if ($command_output): ?>
                    <div class="terminal-line"><span class="terminal-prompt">$</span> <?php echo htmlspecialchars($_POST['command']); ?></div>
                    <?php foreach ($command_output['content'] as $line): ?>
                        <div class="terminal-line <?php echo $command_output['type'] === 'error' ? 'terminal-error' : ($command_output['type'] === 'success' ? 'terminal-success' : ''); ?>">
                            <?php echo htmlspecialchars($line); ?>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            <form method="POST" action="terminal.php" id="terminal-form" class="terminal-input-container">
                <span class="terminal-prompt mr-2">$</span>
                <input type="text" name="command" id="command-input" class="terminal-input" autocomplete="off" autofocus>
            </form>
        </div>
    </main>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const terminalOutput = document.getElementById('terminal-output');
            const commandInput = document.getElementById('command-input');
            
            terminalOutput.scrollTop = terminalOutput.scrollHeight;
            commandInput.focus();
            
            <?php if ($command_output && $command_output['type'] === 'clear'): ?>
                terminalOutput.innerHTML = `
                    <div class="terminal-line">Système Cergies v3.7.2 - Terminal de sécurité national</div>
                    <div class="terminal-line">Connexion établie: Agent <?php echo htmlspecialchars($user['username']); ?></div>
                    <div class="terminal-line">Niveau d'autorisation: <?php echo $user['level']; ?></div>
                    <div class="terminal-line">Date: <?php echo date('Y-m-d H:i:s'); ?></div>
                    <div class="terminal-line">--------------------------------------------</div>
                    <div class="terminal-line">Tapez 'help' pour afficher la liste des commandes.</div>
                    <div class="terminal-line">--------------------------------------------</div>
                `;
            <?php endif; ?>
            
            const commandHistory = [];
            let historyIndex = -1;
            
            document.getElementById('terminal-form').addEventListener('submit', function() {
                const command = commandInput.value.trim();
                if (command) {
                    commandHistory.unshift(command);
                    historyIndex = -1;
                }
            });
            
            commandInput.addEventListener('keydown', function(e) {
                if (e.key === 'ArrowUp') {
                    e.preventDefault();
                    historyIndex = Math.min(historyIndex + 1, commandHistory.length - 1);
                    if (historyIndex >= 0) {
                        commandInput.value = commandHistory[historyIndex];
                    }
                } else if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    historyIndex = Math.max(historyIndex - 1, -1);
                    if (historyIndex >= 0) {
                        commandInput.value = commandHistory[historyIndex];
                    } else {
                        commandInput.value = '';
                    }
                }
            });
        });
    </script>
</body>
</html>