<!DOCTYPE html>
<html lang="fr">
<head>
   <meta charset="utf-8">
   <meta name="viewport" content="width=device-width, initial-scale=1">
   <title>IS_Engine</title>
   <script src="https://cdn.tailwindcss.com"></script>
   <link href="https://fonts.googleapis.com/css2?family=Urbanist:wght@300;400;500;600;700&display=swap" rel="stylesheet">
   <link rel="stylesheet" href="https://use.fontawesome.com/releases/v6.4.0/css/all.css">
   <meta name="csrf-token" content="{{ csrf_token() }}">
   <style>
       body { font-family: 'Urbanist', sans-serif; }
       .editor-content { min-height: 85vh; }
       .editor-content:empty:before {
           content: attr(data-placeholder);
           color: #9ca3af;
       }
       .floating-toolbar {
           position: fixed;
           bottom: 30px;
           left: 50%;
           transform: translateX(-50%);
           background: white;
           border: 1px solid #e5e7eb;
           border-radius: 16px;
           box-shadow: 0 10px 30px rgba(0,0,0,0.15);
           padding: 12px 24px;
           z-index: 50;
           max-width: 90vw;
           overflow-x: auto;
       }
       .highlight-correct { background-color: rgba(34, 197, 94, 0.2); }
       .highlight-incorrect { background-color: rgba(239, 68, 68, 0.2); }
       .highlight-uncertain { background-color: rgba(251, 191, 36, 0.2); }
       .highlight-processing { background-color: rgba(59, 130, 246, 0.2); }
       .sidebar {
           position: fixed;
           top: 50%;
           right: 20px;
           transform: translateY(-50%);
           background: white;
           border: 1px solid #e5e7eb;
           border-radius: 12px;
           box-shadow: 0 10px 25px rgba(0,0,0,0.1);
           padding: 16px;
           width: 300px;
           max-height: 80vh;
           overflow-y: auto;
           z-index: 40;
       }
       .tool-btn {
           padding: 8px 12px;
           border-radius: 8px;
           border: 1px solid #e5e7eb;
           background: white;
           transition: all 0.2s;
           font-size: 14px;
       }
       .tool-btn:hover {
           background: #f3f4f6;
           border-color: #d1d5db;
       }
       .tool-btn.active {
           background: #1f2937;
           color: white;
           border-color: #1f2937;
       }
   </style>
</head>
<body class="bg-white text-black">
   <header class="border-b border-gray-200 px-6 py-4">
       <div class="flex justify-between items-center max-w-7xl mx-auto">
           <div class="flex items-center space-x-4">
               <a href="/workspace" class="text-gray-600 hover:text-black text-lg">
                   <i class="fas fa-arrow-left"></i>
               </a>
               <h1 class="text-xl font-medium">IS_Engine</h1>
           </div>
           <div class="flex items-center space-x-3">
               <button id="save-btn" class="px-4 py-2 bg-black text-white rounded-md hover:bg-gray-800 text-sm">
                   Save
               </button>
               <button id="share-btn" class="px-4 py-2 border border-gray-300 rounded-md hover:bg-gray-50 text-sm">
                   Share
               </button>
               <span class="text-sm text-gray-600">Welcome, {{ session('username') }}</span>
           </div>
       </div>
   </header>

   <div class="max-w-4xl mx-auto px-6 py-8">
       <input type="text" id="document-title-input" 
              placeholder="Untitled Document" 
              class="w-full text-3xl font-light border-none outline-none mb-8 placeholder-gray-400">
       
       <div id="editor" 
            class="editor-content outline-none leading-relaxed text-gray-800" 
            contenteditable="true" 
            data-placeholder="Start writing...">
       </div>
   </div>

   <div class="sidebar hidden" id="results-panel">
       <div class="flex justify-between items-center mb-4">
           <h3 class="font-medium">Results</h3>
           <button id="close-results" class="text-gray-400 hover:text-gray-600">
               <i class="fas fa-times"></i>
           </button>
       </div>
       <div id="results-content"></div>
   </div>

   <div class="floating-toolbar">
       <div class="flex items-center space-x-3">
           <button class="tool-btn" data-action="bold">
               <i class="fas fa-bold"></i>
           </button>
           <button class="tool-btn" data-action="italic">
               <i class="fas fa-italic"></i>
           </button>
           <button class="tool-btn" data-action="underline">
               <i class="fas fa-underline"></i>
           </button>
           
           <div class="w-px h-6 bg-gray-300 mx-2"></div>
           
           <button class="tool-btn" data-action="h1">H1</button>
           <button class="tool-btn" data-action="h2">H2</button>
           <button class="tool-btn" data-action="h3">H3</button>
           
           <div class="w-px h-6 bg-gray-300 mx-2"></div>
           
           <button class="tool-btn" data-action="bullet-list">
               <i class="fas fa-list-ul"></i>
           </button>
           <button class="tool-btn" data-action="number-list">
               <i class="fas fa-list-ol"></i>
           </button>
           
           <div class="w-px h-6 bg-gray-300 mx-2"></div>
           
           <button class="tool-btn" data-claude-action="grammar">
               <i class="fas fa-spell-check"></i> Fix
           </button>
           <button class="tool-btn" data-claude-action="expand">
               <i class="fas fa-expand-arrows-alt"></i> Expand
           </button>
           <button class="tool-btn" data-claude-action="rephrase">
               <i class="fas fa-magic"></i> Rephrase
           </button>
           
           <div class="w-px h-6 bg-gray-300 mx-2"></div>
           
           <button class="tool-btn" id="fact-check-btn">
               <i class="fas fa-search"></i> Fact Check
           </button>
           <button class="tool-btn" id="search-btn">
               <i class="fas fa-globe"></i> Search
           </button>
           
           <div class="w-px h-6 bg-gray-300 mx-2"></div>
           
           <input type="text" id="quick-prompt" placeholder="Ask AI..." 
                  class="px-3 py-2 border border-gray-300 rounded text-sm" style="width: 150px;">
           <button class="tool-btn" id="send-prompt">
               <i class="fas fa-paper-plane"></i>
           </button>
       </div>
   </div>

   <div id="save-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4">
       <div class="bg-white rounded-lg p-6 w-full max-w-md">
           <h3 class="text-lg font-medium mb-4">Save Document</h3>
           <input type="text" id="save-title" placeholder="Document name" 
                  class="w-full p-3 border rounded mb-4">
           <div class="flex justify-end space-x-3">
               <button id="save-cancel" class="px-4 py-2 text-gray-600">Cancel</button>
               <button id="save-confirm" class="px-4 py-2 bg-black text-white rounded">Save</button>
           </div>
       </div>
   </div>

   <div id="share-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4">
       <div class="bg-white rounded-lg p-6 w-full max-w-md">
           <h3 class="text-lg font-medium mb-4">Share Document</h3>
           <div class="space-y-3 mb-4">
               <label class="flex items-center">
                   <input type="radio" name="share-type" value="view" class="mr-2" checked>
                   <span class="text-sm">Read only</span>
               </label>
               <label class="flex items-center">
                   <input type="radio" name="share-type" value="edit" class="mr-2">
                   <span class="text-sm">Allow editing</span>
               </label>
               <input type="text" id="share-link" readonly 
                      class="w-full p-3 border rounded bg-gray-50" 
                      placeholder="Link will appear here">
           </div>
           <div class="flex justify-end space-x-3">
               <button id="share-cancel" class="px-4 py-2 text-gray-600">Cancel</button>
               <button id="share-generate" class="px-4 py-2 bg-black text-white rounded">Generate</button>
               <button id="share-copy" class="hidden px-4 py-2 bg-black text-white rounded">Copy</button>
           </div>
       </div>
   </div>

   <div id="search-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4">
       <div class="bg-white rounded-lg p-6 w-full max-w-lg">
           <h3 class="text-lg font-medium mb-4">Search Information</h3>
           <input type="text" id="search-query" placeholder="What do you want to search..." 
                  class="w-full p-3 border rounded mb-4">
           <div class="flex justify-end space-x-3">
               <button id="search-cancel" class="px-4 py-2 text-gray-600">Cancel</button>
               <button id="search-submit" class="px-4 py-2 bg-black text-white rounded">Search</button>
           </div>
       </div>
   </div>

   <div class="fixed inset-0 flex flex-col items-center justify-center bg-white text-center px-6 md:hidden">
       <p class="text-lg text-gray-700">This site is not available on mobile. Please try again from a computer.</p>
   </div>

<script>
let currentDocument = {{ isset($document_id) ? $document_id : 'null' }};
let selectedText = '';
let selectedRange = null;
let editor = document.getElementById('editor');
let autoSaveInterval;

if (currentDocument) {
   loadDocument(currentDocument);
}

async function loadDocument(id) {
   try {
       const response = await fetch(`/api/documents/${id}`);
       if (response.ok) {
           const document = await response.json();
           document.getElementById('document-title-input').value = document.title;
           editor.innerHTML = document.content;
       }
   } catch (error) {
       console.error('Erreur chargement document:', error);
   }
}

function startAutoSave() {
   autoSaveInterval = setInterval(async () => {
       const title = document.getElementById('document-title-input').value;
       const content = editor.innerHTML;
       
       if (title.trim() !== '' || content.trim() !== '') {
           await saveDocument(false);
       }
   }, 30000);
}

async function saveDocument(showNotif = true) {
   const title = document.getElementById('document-title-input').value || 'Untitled Document';
   const content = editor.innerHTML;

   try {
       const response = await fetch('/api/documents/save', {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               title: title,
               content: content,
               document_id: currentDocument
           })
       });

       const result = await response.json();
       
       if (result.success) {
           currentDocument = result.document_id;
           if (showNotif) {
               showNotification('Document saved!');
           }
       }
   } catch (error) {
       console.error('Erreur sauvegarde:', error);
       if (showNotif) {
           showNotification('Save error');
       }
   }
}

editor.addEventListener('mouseup', function() {
   selectedText = window.getSelection().toString().trim();
   if (selectedText) {
       selectedRange = window.getSelection().getRangeAt(0).cloneRange();
   }
});

document.querySelectorAll('[data-action]').forEach(button => {
   button.addEventListener('click', function() {
       const action = this.dataset.action;
       executeCommand(action);
   });
});

function executeCommand(action) {
   switch(action) {
       case 'bold':
           document.execCommand('bold');
           break;
       case 'italic':
           document.execCommand('italic');
           break;
       case 'underline':
           document.execCommand('underline');
           break;
       case 'h1':
           document.execCommand('formatBlock', false, 'h1');
           break;
       case 'h2':
           document.execCommand('formatBlock', false, 'h2');
           break;
       case 'h3':
           document.execCommand('formatBlock', false, 'h3');
           break;
       case 'bullet-list':
           document.execCommand('insertUnorderedList');
           break;
       case 'number-list':
           document.execCommand('insertOrderedList');
           break;
   }
   editor.focus();
}

document.querySelectorAll('[data-claude-action]').forEach(button => {
   button.addEventListener('click', function() {
       const action = this.dataset.claudeAction;
       if (selectedText) {
           highlightSelectedText('processing');
           callClaudeAPI('', action);
       } else {
           showNotification('Select text first');
       }
   });
});

document.getElementById('send-prompt').addEventListener('click', () => {
   const prompt = document.getElementById('quick-prompt').value;
   if (prompt.trim()) {
       if (selectedText) {
           highlightSelectedText('processing');
       }
       callClaudeAPI(prompt, 'custom');
       document.getElementById('quick-prompt').value = '';
   }
});

document.getElementById('quick-prompt').addEventListener('keypress', (e) => {
   if (e.key === 'Enter') {
       document.getElementById('send-prompt').click();
   }
});

async function callClaudeAPI(prompt, action) {
   try {
       const response = await fetch('/api/claude', {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               prompt: prompt,
               selected_text: selectedText,
               action: action
           })
       });

       const result = await response.json();
       
       if (result.success) {
           let cleanContent = result.content;
           
           const introPatterns = [
               /^Voici le texte corrigé :\s*/i,
               /^Texte corrigé :\s*/i,
               /^Version développée :\s*/i,
               /^Résumé :\s*/i,
               /^Reformulation :\s*/i,
               /^Voici une reformulation :\s*/i,
               /^Voici un résumé :\s*/i,
               /^Voici le texte développé :\s*/i
           ];
           
           introPatterns.forEach(pattern => {
               cleanContent = cleanContent.replace(pattern, '');
           });
           
           const endPatterns = [
               /\.\s*Il n'y a aucune erreur.*$/i,
               /\.\s*Ce texte ne contient.*$/i,
               /\.\s*La phrase est déjà.*$/i
           ];
           
           endPatterns.forEach(pattern => {
               cleanContent = cleanContent.replace(pattern, '.');
           });
           
           cleanContent = cleanContent.trim();
           
           if (selectedText && selectedRange) {
               highlightSelectedText('correct');
               showResults([{
                   type: 'claude',
                   original: selectedText,
                   suggestion: cleanContent,
                   action: action
               }]);
           } else {
               showResults([{
                   type: 'claude',
                   response: cleanContent
               }]);
           }
       } else {
           if (selectedText) {
               clearHighlights();
           }
           showNotification('Claude error');
       }
   } catch (error) {
       if (selectedText) {
           clearHighlights();
       }
       console.error('Erreur Claude API:', error);
       showNotification('Connection error');
   }
}

document.getElementById('fact-check-btn').addEventListener('click', () => {
   const documentText = editor.textContent.trim();
   if (documentText) {
       analyzeDocument(documentText);
   } else {
       showNotification('Document is empty');
   }
});

async function analyzeDocument(text) {
   const sentences = text.match(/[^\.!?]+[\.!?]+/g) || [];
   
   try {
       const response = await fetch('/api/fact-check', {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               text: text
           })
       });

       const result = await response.json();
       
       if (result.success) {
           highlightFactCheckResults(result.analysis);
           showResults(result.analysis);
       } else {
           showNotification('Fact check error');
       }
   } catch (error) {
       console.error('Erreur fact check:', error);
       showNotification('Connection error');
   }
}

document.getElementById('search-btn').addEventListener('click', () => {
   document.getElementById('search-modal').classList.remove('hidden');
});

document.getElementById('search-submit').addEventListener('click', () => {
   const query = document.getElementById('search-query').value;
   if (query.trim()) {
       callPerplexityAPI(query);
       document.getElementById('search-modal').classList.add('hidden');
   }
});

document.getElementById('search-cancel').addEventListener('click', () => {
   document.getElementById('search-modal').classList.add('hidden');
});

async function callPerplexityAPI(query) {
   try {
       const response = await fetch('/api/search', {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               query: query
           })
       });

       const result = await response.json();
       
       if (result.success) {
           showResults([{
               type: 'search',
               query: query,
               results: result.results,
               summary: result.summary
           }]);
       } else {
           showNotification('Search error');
       }
   } catch (error) {
       console.error('Erreur recherche:', error);
       showNotification('Connection error');
   }
}

function highlightSelectedText(type) {
   if (selectedRange) {
       const span = document.createElement('span');
       span.className = `highlight-${type}`;
       try {
           selectedRange.surroundContents(span);
       } catch (e) {
           console.log('Cannot highlight complex selection');
       }
   }
}

function highlightFactCheckResults(analysis) {
   clearHighlights();
   const editorText = editor.textContent;
   
   analysis.forEach(item => {
       if (item.statement) {
           const regex = new RegExp(item.statement.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi');
           editor.innerHTML = editor.innerHTML.replace(regex, 
               `<span class="highlight-${item.status}">${item.statement}</span>`);
       }
   });
}

function clearHighlights() {
   const highlights = editor.querySelectorAll('[class*="highlight-"]');
   highlights.forEach(highlight => {
       const parent = highlight.parentNode;
       parent.replaceChild(document.createTextNode(highlight.textContent), highlight);
       parent.normalize();
   });
}

function showResults(data) {
   const panel = document.getElementById('results-panel');
   const content = document.getElementById('results-content');
   let html = '';
   
   data.forEach(item => {
       if (item.type === 'claude') {
           if (item.suggestion) {
               html += `
                   <div class="mb-4 p-3 border rounded">
                       <div class="text-sm font-medium mb-2">Claude Suggestion</div>
                       <div class="text-xs text-gray-600 mb-2">Original: "${item.original}"</div>
                       <div class="text-sm text-gray-800 mb-3">${item.suggestion}</div>
                       <button onclick="applyClaudeEdit('${item.suggestion.replace(/'/g, "\\'")}')" 
                               class="px-3 py-1 bg-green-500 text-white rounded text-xs">Apply</button>
                   </div>
               `;
           } else {
               html += `
                   <div class="mb-4 p-3 border rounded">
                       <div class="text-sm font-medium mb-2">Claude Response</div>
                       <div class="text-sm text-gray-800">${item.response}</div>
                   </div>
               `;
           }
       } else if (item.type === 'search') {
           html += `
               <div class="mb-4 p-3 border rounded">
                   <div class="text-sm font-medium mb-2">Search Results</div>
                   <div class="text-xs text-gray-600 mb-2">Query: "${item.query}"</div>
                   ${item.summary ? `<div class="text-sm text-gray-800 mb-2">${item.summary}</div>` : ''}
                   <button onclick="insertSearchResult('${item.summary ? item.summary.replace(/'/g, "\\'") : ''}')" 
                           class="px-3 py-1 bg-blue-500 text-white rounded text-xs">Insert</button>
               </div>
           `;
       } else {
           const statusColor = item.status === 'correct' ? 'green' : 
                              item.status === 'incorrect' ? 'red' : 'yellow';
           html += `
               <div class="mb-3 p-3 border-l-4 border-${statusColor}-400">
                   <div class="text-sm font-medium">${item.statement}</div>
                   <div class="text-xs text-gray-600 mt-1">${item.explanation}</div>
                   ${item.source ? `<div class="text-xs text-blue-600 mt-1">Source: ${item.source}</div>` : ''}
               </div>
           `;
       }
   });
   
   content.innerHTML = html;
   panel.classList.remove('hidden');
}

function applyClaudeEdit(newText) {
   if (selectedRange) {
       const selection = window.getSelection();
       selection.removeAllRanges();
       selection.addRange(selectedRange);
       
       if (selection.rangeCount > 0) {
           const range = selection.getRangeAt(0);
           range.deleteContents();
           range.insertNode(document.createTextNode(newText));
       }
       
       clearHighlights();
       selectedText = '';
       selectedRange = null;
   }
}

function insertSearchResult(text) {
   if (text) {
       const selection = window.getSelection();
       if (selection.rangeCount > 0) {
           const range = selection.getRangeAt(0);
           range.insertNode(document.createTextNode(text + ' '));
       } else {
           editor.appendChild(document.createTextNode(text + ' '));
       }
       editor.focus();
   }
}

document.getElementById('close-results').addEventListener('click', () => {
   document.getElementById('results-panel').classList.add('hidden');
   clearHighlights();
});

document.getElementById('save-btn').addEventListener('click', () => {
   document.getElementById('save-title').value = document.getElementById('document-title-input').value;
   document.getElementById('save-modal').classList.remove('hidden');
});

document.getElementById('save-confirm').addEventListener('click', async () => {
   const title = document.getElementById('save-title').value;
   document.getElementById('document-title-input').value = title;
   await saveDocument(true);
   document.getElementById('save-modal').classList.add('hidden');
});

document.getElementById('save-cancel').addEventListener('click', () => {
   document.getElementById('save-modal').classList.add('hidden');
});

document.getElementById('share-btn').addEventListener('click', () => {
   if (!currentDocument) {
       showNotification('Save document first');
       return;
   }
   document.getElementById('share-modal').classList.remove('hidden');
});

document.getElementById('share-generate').addEventListener('click', async () => {
   const permission = document.querySelector('input[name="share-type"]:checked').value;
   
   try {
       const response = await fetch(`/api/documents/${currentDocument}/share`, {
           method: 'POST',
           headers: {
               'Content-Type': 'application/json',
               'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
           },
           body: JSON.stringify({
               permission: permission
           })
       });

       const result = await response.json();
       
       if (result.success) {
           document.getElementById('share-link').value = result.share_url;
           document.getElementById('share-copy').classList.remove('hidden');
       }
   } catch (error) {
       console.error('Erreur partage:', error);
       showNotification('Share error');
   }
});

document.getElementById('share-copy').addEventListener('click', () => {
   const shareLink = document.getElementById('share-link');
   shareLink.select();
   document.execCommand('copy');
   showNotification('Link copied!');
});

document.getElementById('share-cancel').addEventListener('click', () => {
   document.getElementById('share-modal').classList.add('hidden');
   document.getElementById('share-link').value = '';
   document.getElementById('share-copy').classList.add('hidden');
});

function showNotification(message) {
   const notification = document.createElement('div');
   notification.className = 'fixed top-4 right-4 bg-black text-white px-4 py-2 rounded shadow-lg z-50';
   notification.textContent = message;
   document.body.appendChild(notification);
   setTimeout(() => notification.remove(), 3000);
}

startAutoSave();

window.addEventListener('beforeunload', () => {
   if (autoSaveInterval) {
       clearInterval(autoSaveInterval);
   }
});
</script>
</body>
</html>