<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

$errors = [];
$success = false;

// Récupération des pays depuis la base de données pour le formulaire
$countries = db_get_all("SELECT id, name, code FROM countries ORDER BY name");

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Récupération des données du formulaire
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $password_confirm = $_POST['password_confirm'] ?? '';
    $country_id = $_POST['country_id'] ?? '';
    $city = trim($_POST['city'] ?? '');
    $language = $_POST['language'] ?? '';
    $bio = trim($_POST['bio'] ?? '');
    
    // Validation des données
    if (empty($username)) {
        $errors[] = "Le nom d'utilisateur est requis";
    } elseif (strlen($username) < 3 || strlen($username) > 20) {
        $errors[] = "Le nom d'utilisateur doit contenir entre 3 et 20 caractères";
    } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        $errors[] = "Le nom d'utilisateur ne peut contenir que des lettres, chiffres et underscores";
    }
    
    if (empty($email)) {
        $errors[] = "L'adresse email est requise";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "L'adresse email n'est pas valide";
    }
    
    if (empty($password)) {
        $errors[] = "Le mot de passe est requis";
    } elseif (strlen($password) < 8) {
        $errors[] = "Le mot de passe doit contenir au moins 8 caractères";
    }
    
    if ($password !== $password_confirm) {
        $errors[] = "Les mots de passe ne correspondent pas";
    }
    
    if (empty($country_id)) {
        $errors[] = "Le pays est requis";
    }
    
    if (empty($city)) {
        $errors[] = "La ville est requise";
    }
    
    if (empty($language)) {
        $errors[] = "La langue est requise";
    }
    
    // Vérification si le nom d'utilisateur existe déjà
    $existing_user = db_get_one("SELECT id FROM users WHERE username = ?", [$username]);
    if ($existing_user) {
        $errors[] = "Ce nom d'utilisateur est déjà utilisé";
    }
    
    // Vérification si l'email existe déjà
    $existing_email = db_get_one("SELECT id FROM users WHERE email = ?", [$email]);
    if ($existing_email) {
        $errors[] = "Cette adresse email est déjà utilisée";
    }
    
    // Si aucune erreur, on enregistre l'utilisateur
    if (empty($errors)) {
        $password_hash = password_hash($password, PASSWORD_DEFAULT);
        
        $user_data = [
            'username' => $username,
            'email' => $email,
            'password_hash' => $password_hash,
            'country_id' => $country_id,
            'city' => $city,
            'language' => $language,
            'bio' => $bio,
            'level' => 1,
            'experience' => 0,
            'reputation' => 0,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $user_id = db_insert('users', $user_data);
        
        if ($user_id) {
            $success = true;
        } else {
            $errors[] = "Une erreur est survenue lors de l'inscription";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inscription | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f8f9fa;
        }
        .form-card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <div class="min-h-screen flex items-center justify-center px-4 py-12">
        <div class="form-card w-full max-w-md p-8">
            <h1 class="text-2xl font-semibold text-center mb-6">Créer votre compte Cergies</h1>
            
            <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    <p>Inscription réussie! Tu peux maintenant <a href="login.php" class="underline">te connecter</a>.</p>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($errors)): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                    <ul class="list-disc pl-5">
                        <?php foreach ($errors as $error): ?>
                            <li><?php echo $error; ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <?php if (!$success): ?>
                <form method="POST" action="register.php" class="space-y-4">
                    <div>
                        <label for="username" class="block text-sm font-medium mb-1">Nom d'utilisateur</label>
                        <input type="text" id="username" name="username" value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                    </div>
                    
                    <div>
                        <label for="email" class="block text-sm font-medium mb-1">Adresse email</label>
                        <input type="email" id="email" name="email" value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                    </div>
                    
                    <div>
                        <label for="password" class="block text-sm font-medium mb-1">Mot de passe</label>
                        <input type="password" id="password" name="password" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                    </div>
                    
                    <div>
                        <label for="password_confirm" class="block text-sm font-medium mb-1">Confirmer le mot de passe</label>
                        <input type="password" id="password_confirm" name="password_confirm" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                    </div>
                    
                    <div>
                        <label for="country_id" class="block text-sm font-medium mb-1">Pays</label>
                        <select id="country_id" name="country_id" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                            <option value="">Sélectionne un pays</option>
                            <?php foreach ($countries as $country): ?>
                                <option value="<?php echo $country['id']; ?>" <?php echo (isset($_POST['country_id']) && $_POST['country_id'] == $country['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($country['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label for="city" class="block text-sm font-medium mb-1">Ville</label>
                        <input type="text" id="city" name="city" value="<?php echo isset($_POST['city']) ? htmlspecialchars($_POST['city']) : ''; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                    </div>
                    
                    <div>
                        <label for="language" class="block text-sm font-medium mb-1">Langue</label>
                        <select id="language" name="language" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                            <option value="">Sélectionne une langue</option>
                            <option value="fr" <?php echo (isset($_POST['language']) && $_POST['language'] == 'fr') ? 'selected' : ''; ?>>Français</option>
                            <option value="en" <?php echo (isset($_POST['language']) && $_POST['language'] == 'en') ? 'selected' : ''; ?>>Anglais</option>
                            <option value="es" <?php echo (isset($_POST['language']) && $_POST['language'] == 'es') ? 'selected' : ''; ?>>Espagnol</option>
                            <option value="de" <?php echo (isset($_POST['language']) && $_POST['language'] == 'de') ? 'selected' : ''; ?>>Allemand</option>
                            <option value="it" <?php echo (isset($_POST['language']) && $_POST['language'] == 'it') ? 'selected' : ''; ?>>Italien</option>
                        </select>
                    </div>
                    
                    <div>
                        <label for="bio" class="block text-sm font-medium mb-1">Histoire de ton personnage</label>
                        <textarea id="bio" name="bio" rows="4" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"><?php echo isset($_POST['bio']) ? htmlspecialchars($_POST['bio']) : ''; ?></textarea>
                    </div>
                    
                    <div>
                        <button type="submit" class="w-full bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-colors">
                            Créer mon compte
                        </button>
                    </div>
                    
                    <div class="text-center text-sm">
                        Déjà un compte? <a href="login.php" class="text-blue-600 hover:underline">Connecte-toi ici</a>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>