<?php
require_once 'config.php';

// Get profile info (own profile or other user)
$profile_id = isset($_GET['id']) ? (int)$_GET['id'] : ($_SESSION['user_id'] ?? null);

if(!$profile_id) {
    header('Location: login.php');
    exit;
}

// Get user info
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$profile_id]);
$user = $stmt->fetch();

if(!$user) {
    header('Location: index.php');
    exit;
}

// Check if it's own profile
$isOwnProfile = isset($_SESSION['user_id']) && $_SESSION['user_id'] == $profile_id;

// Handle avatar upload
if($isOwnProfile && isset($_FILES['avatar'])) {
    $uploadDir = 'uploads/avatars/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }

    $fileName = uniqid() . '_' . basename($_FILES['avatar']['name']);
    $uploadPath = $uploadDir . $fileName;

    if(move_uploaded_file($_FILES['avatar']['tmp_name'], $uploadPath)) {
        $stmt = $pdo->prepare("UPDATE users SET avatar_url = ? WHERE id = ?");
        $stmt->execute([$uploadPath, $profile_id]);
        $user['avatar_url'] = $uploadPath;
    }
}

// Get user's videos
$stmt = $pdo->prepare("
    SELECT c.*, 
           COUNT(DISTINCT CASE WHEN i.type = 'view' THEN i.id END) as views,
           COUNT(DISTINCT CASE WHEN i.type = 'like' THEN i.id END) as likes,
           COUNT(DISTINCT CASE WHEN i.type = 'comment' THEN i.id END) as comments
    FROM content c
    LEFT JOIN interactions i ON c.id = i.content_id
    WHERE c.author_id = ?
    GROUP BY c.id
    ORDER BY c.created_at DESC
");
$stmt->execute([$profile_id]);
$videos = $stmt->fetchAll();

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($user['username']) ?> - ForMore</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Titillium+Web:wght@200;300;400;600;700;900&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Titillium Web', sans-serif;
        }
        .formore {
            font-weight: 900;
            background: linear-gradient(to right, #3b82f6, #1d4ed8);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
    </style>
</head>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white p-8">
    <!-- Nav -->
    <nav class="flex justify-between items-center mb-12">
        <a href="index.php" class="text-5xl formore">ForMore</a>
        <div class="flex items-center gap-6">
            <?php if(isset($_SESSION['user_id'])): ?>
                <?php if(!$isOwnProfile): ?>
                    <a href="profile.php" class="text-gray-300 hover:text-white">My Profile</a>
                <?php endif; ?>
                <a href="upload.php" class="bg-blue-600 hover:bg-blue-700 px-6 py-2 rounded-xl transition-all">Upload</a>
                <a href="logout.php" class="text-red-400 hover:text-red-300">Logout</a>
            <?php else: ?>
                <a href="login.php" class="text-blue-400 hover:text-blue-300">Login</a>
                <a href="signup.php" class="bg-blue-600 hover:bg-blue-700 px-6 py-2 rounded-xl transition-all">Sign Up</a>
            <?php endif; ?>
        </div>
    </nav>

    <!-- Profile Header -->
    <div class="bg-gray-800/50 backdrop-blur-sm rounded-2xl p-8 mb-8">
        <div class="flex items-center gap-8">
            <!-- Avatar -->
            <div class="relative group">
                <?php if($user['avatar_url']): ?>
                    <img src="<?= htmlspecialchars($user['avatar_url']) ?>" 
                         alt="Profile" 
                         class="w-32 h-32 rounded-full object-cover bg-gray-700">
                <?php else: ?>
                    <div class="w-32 h-32 rounded-full bg-gray-700 flex items-center justify-center text-4xl font-bold">
                        <?= strtoupper(substr($user['username'], 0, 1)) ?>
                    </div>
                <?php endif; ?>

                <?php if($isOwnProfile): ?>
                    <form id="avatarForm" class="hidden">
                        <input type="file" name="avatar" id="avatarInput" accept="image/*">
                    </form>
                    <button onclick="document.getElementById('avatarInput').click()" 
                            class="absolute inset-0 bg-black/50 rounded-full opacity-0 group-hover:opacity-100 transition-all flex items-center justify-center">
                        Change Photo
                    </button>
                <?php endif; ?>
            </div>

            <!-- Profile Info -->
            <div>
                <h1 class="text-3xl font-bold"><?= htmlspecialchars($user['username']) ?></h1>
                <p class="text-gray-400 mt-1">Member since <?= (new DateTime($user['created_at']))->format('F Y') ?></p>
                
                <!-- Stats -->
                <div class="flex gap-6 mt-4">
                    <div class="text-center">
                        <div class="text-2xl font-bold"><?= count($videos) ?></div>
                        <div class="text-gray-400">Videos</div>
                    </div>
                    <div class="text-center">
                        <div class="text-2xl font-bold">
                            <?= array_sum(array_column($videos, 'views')) ?>
                        </div>
                        <div class="text-gray-400">Views</div>
                    </div>
                    <div class="text-center">
                        <div class="text-2xl font-bold">
                            <?= array_sum(array_column($videos, 'likes')) ?>
                        </div>
                        <div class="text-gray-400">Likes</div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Videos Grid -->
    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
        <?php foreach($videos as $video): ?>
            <div class="bg-gray-800/50 backdrop-blur-sm rounded-2xl overflow-hidden hover:scale-105 transition-all cursor-pointer"
                 onclick="window.location.href='index.php?v=<?= $video['id'] ?>'">
                <div class="relative aspect-video">
                    <img src="<?= htmlspecialchars($video['picture_cover']) ?>" 
                         alt="" 
                         class="w-full h-full object-cover">
                    <div class="absolute bottom-2 right-2 bg-black/70 backdrop-blur-sm px-3 py-1 rounded-lg text-sm">
                        <?= htmlspecialchars($video['time']) ?>
                    </div>
                </div>
                <div class="p-4">
                    <h2 class="text-lg font-bold line-clamp-1"><?= htmlspecialchars($video['name']) ?></h2>
                    <p class="text-gray-400 text-sm mt-2 line-clamp-2"><?= htmlspecialchars($video['description']) ?></p>
                    <div class="flex justify-between mt-4 text-sm text-gray-400">
                        <span><?= number_format($video['views']) ?> views</span>
                        <span><?= number_format($video['likes']) ?> likes</span>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>

        <?php if(empty($videos)): ?>
            <div class="col-span-full text-center py-12 text-gray-400">
                No videos uploaded yet.
                <?php if($isOwnProfile): ?>
                    <br><a href="upload.php" class="text-blue-400 hover:text-blue-300">Upload your first video!</a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <?php if($isOwnProfile): ?>
    <script>
    document.getElementById('avatarInput').addEventListener('change', function() {
        const formData = new FormData(document.getElementById('avatarForm'));
        fetch(window.location.href, {
            method: 'POST',
            body: formData
        }).then(() => {
            location.reload();
        });
    });
    </script>
    <?php endif; ?>
</body>
</html>