<?php
require_once 'db.php';
session_start();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Management Dashboard</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'media',
            theme: {
                extend: {
                    fontFamily: {
                        'poppins': ['Poppins', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style>
        body {
            font-family: 'Poppins', sans-serif;
        }
        @media (prefers-color-scheme: dark) {
            body {
                background-color: #1a1a1a;
                color: #ffffff;
            }
        }
        .success-notification {
            animation: fadeOut 3s forwards;
            animation-delay: 2s;
        }
        @keyframes fadeOut {
            from {opacity: 1;}
            to {opacity: 0; display: none;}
        }
        .search-highlight {
            background-color: rgba(255, 255, 0, 0.3);
        }
    </style>
</head>
<body class="bg-white dark:bg-gray-900 text-gray-900 dark:text-white transition-colors duration-200">
    <!-- Notifications -->
    <?php if(isset($_GET['success'])): ?>
    <div class="success-notification fixed top-4 right-4 bg-green-500 text-white px-6 py-3 rounded-lg shadow-lg z-50">
        Operation completed successfully!
    </div>
    <?php endif; ?>

    <?php if(isset($_GET['error'])): ?>
    <div class="error-notification fixed top-4 right-4 bg-red-500 text-white px-6 py-3 rounded-lg shadow-lg z-50">
        An error occurred during the operation.
    </div>
    <?php endif; ?>

    <div class="container mx-auto px-4 py-8">
        <!-- Header with Logo -->
        <div class="flex flex-col md:flex-row items-center justify-between mb-8 space-y-4 md:space-y-0">
            <div class="flex items-center space-x-4">
                <img src="https://cdn.imators.com/logo.png" alt="Logo" class="h-12 w-auto">
                <h1 class="text-3xl font-bold">Imators Systems</h1>
                <p class="text-sm">Customer management dashboard and legal documents, reserved for administrative staff</p>
            </div>
            
            <!-- Global Search Bar -->
            <div class="w-full md:w-96">
                <div class="relative">
                    <input type="text" 
                           id="global-search" 
                           placeholder="Search users or documents..." 
                           class="w-full px-4 py-2 pl-10 pr-4 rounded-lg border border-gray-300 
                                  dark:border-gray-600 dark:bg-gray-700 focus:outline-none 
                                  focus:ring-2 focus:ring-black dark:focus:ring-white">
                    <div class="absolute left-3 top-2.5">
                        <svg class="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                                  d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                        </svg>
                    </div>
                </div>
            </div>
        </div>

        <!-- Tabs -->
        <div class="mb-8">
            <div class="border-b border-gray-200 dark:border-gray-700">
                <nav class="-mb-px flex space-x-8">
                    <button 
                        class="user-tab px-4 py-2 font-medium border-b-2 border-black dark:border-white"
                        data-tab="users">
                        User Management
                    </button>
                    <button 
                        class="doc-tab px-4 py-2 font-medium border-b-2 border-transparent"
                        data-tab="documents">
                        Document Management
                    </button>
                </nav>
            </div>
        </div>

        <!-- Users Content -->
        <div id="users-content" class="tab-content">
            <div class="mb-6">
                <div class="flex justify-between items-center">
                    <p class="text-sm text-gray-500 dark:text-gray-400" id="users-count">
                        Loading users...
                    </p>
                </div>
            </div>
            <div id="users-grid" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                <?php
                $db1 = getDB1Connection();
                $query = $db1->query("SELECT * FROM utilisateurs ORDER BY username");
                while($user = $query->fetch(PDO::FETCH_ASSOC)) {
                    ?>
                    <div class="user-card bg-gray-50 dark:bg-gray-800 p-6 rounded-lg cursor-pointer 
                              hover:shadow-lg transition-all duration-200" 
                         data-user-id="<?= htmlspecialchars($user['id']) ?>"
                         data-email="<?= htmlspecialchars($user['email']) ?>"
                         data-username="<?= htmlspecialchars($user['username']) ?>"
                         data-note="<?= htmlspecialchars($user['note']) ?>"
                         data-searchable="<?= htmlspecialchars($user['username'] . ' ' . $user['email'] . ' ' . $user['note']) ?>">
                        <div class="flex justify-between items-start">
                            <div>
                                <h3 class="font-medium text-lg username">
                                    <?= htmlspecialchars($user['username']) ?>
                                </h3>
                                <p class="text-sm text-gray-600 dark:text-gray-300 email">
                                    <?= htmlspecialchars($user['email']) ?>
                                </p>
                            </div>
                            <span class="text-xs px-2 py-1 bg-gray-200 dark:bg-gray-700 rounded-full">
                                Edit
                            </span>
                        </div>
                        <div class="mt-4">
                            <p class="text-sm text-gray-500 dark:text-gray-400 note">
                                Note: <?= $user['note'] ? htmlspecialchars($user['note']) : 'No note' ?>
                            </p>
                        </div>
                    </div>
                    <?php
                }
                ?>
            </div>
        </div>

        <!-- Documents Content -->
        <div id="documents-content" class="tab-content hidden">
            <div class="grid grid-cols-1 gap-6">
                <!-- Upload Form -->
                <div class="bg-gray-50 dark:bg-gray-800 p-6 rounded-lg">
                    <h2 class="text-xl font-medium mb-4">Upload New Document</h2>
                    <form action="upload_document.php" method="post" enctype="multipart/form-data" class="space-y-4">
                        <div>
                            <label class="block text-sm font-medium mb-2">Select PDF File</label>
                            <input type="file" name="pdf_file" accept=".pdf" 
                                class="block w-full text-sm text-gray-500 dark:text-gray-400
                                file:mr-4 file:py-2 file:px-4
                                file:rounded-full file:border-0
                                file:text-sm file:font-medium
                                file:bg-gray-200 file:dark:bg-gray-700
                                file:text-gray-700 file:dark:text-gray-300
                                hover:file:bg-gray-300 hover:file:dark:bg-gray-600">
                        </div>
                        <button type="submit" 
                            class="px-4 py-2 bg-black dark:bg-white text-white dark:text-black rounded-lg
                            hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors duration-200">
                            Upload Document
                        </button>
                    </form>
                </div>

                <!-- Documents List -->
                <div class="bg-gray-50 dark:bg-gray-800 p-6 rounded-lg">
                    <h2 class="text-xl font-medium mb-4">Existing Documents</h2>
                    <p class="text-sm text-gray-500 dark:text-gray-400 mb-4" id="documents-count">
                        Loading documents...
                    </p>
                    <div id="documents-list" class="space-y-2">
                        <?php
                        $db2 = getDB2Connection();
                        $query = $db2->query("SELECT * FROM documents ORDER BY created_at DESC");
                        while($doc = $query->fetch(PDO::FETCH_ASSOC)) {
                            ?>
                            <div class="document-item flex items-center justify-between p-4 bg-white dark:bg-gray-700 rounded-lg"
                                 data-searchable="<?= htmlspecialchars(basename($doc['pdf-link'])) ?>">
                                <a href="<?= htmlspecialchars($doc['pdf-link']) ?>" 
                                   target="_blank"
                                   class="text-blue-600 dark:text-blue-400 hover:underline document-name">
                                    <?= htmlspecialchars(basename($doc['pdf-link'])) ?>
                                </a>
                                <span class="text-xs text-gray-500 dark:text-gray-400">
                                    <?= date('Y-m-d H:i', strtotime($doc['created_at'])) ?>
                                </span>
                            </div>
                            <?php
                        }
                        ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- User Edit Modal -->
        <div id="user-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div class="bg-white dark:bg-gray-800 p-6 rounded-lg w-full max-w-md">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-xl font-bold">Edit User</h2>
                    <button id="close-modal" class="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <form id="edit-user-form" class="space-y-4">
                    <input type="hidden" id="user-id" name="user-id">
                    <div>
                        <label class="block text-sm font-medium">Email</label>
                        <input type="email" id="email" name="email" 
                            class="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 
                            dark:bg-gray-700 dark:text-white shadow-sm">
                    </div>
                    <div>
                        <label class="block text-sm font-medium">Username</label>
                        <input type="text" id="username" name="username" 
                            class="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 
                            dark:bg-gray-700 dark:text-white shadow-sm">
                    </div>
                    <div>
                        <label class="block text-sm font-medium">New Password</label>
                        <input type="password" id="password" name="password" 
                            class="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 
                            dark:bg-gray-700 dark:text-white shadow-sm"
                            placeholder="Leave empty to keep current password">
                    </div>
                    <div>
                        <label class="block text-sm font-medium">Note</label>
                        <textarea id="note" name="note" rows="3" 
                            class="mt-1 block w-full rounded-md border-gray-300 dark:border-gray-600 
                            dark:bg-gray-700 dark:text-white shadow-sm"></textarea>
                    </div>
                    <div class="flex justify-end space-x-3 pt-4">
                        <button type="button" class="close-modal px-4 py-2 border border-gray-300 dark:border-gray-600 
                            rounded-md text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700">
                            Cancel
                        </button>
                        <button type="submit" class="px-4 py-2 bg-black dark:bg-white text-white dark:text-black 
                            rounded-md hover:bg-gray-800 dark:hover:bg-gray-200">
                            Save Changes
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', () => {
            // Search functionality
            const searchInput = document.getElementById('global-search');
            const userCards = document.querySelectorAll('.user-card');
            const documentItems = document.querySelectorAll('.document-item');
            const usersCount = document.getElementById('users-count');
            const documentsCount = document.getElementById('documents-count');

            function updateCounts() {
                const visibleUsers = document.querySelectorAll('.user-card:not(.hidden)').length;
                const visibleDocs = document.querySelectorAll('.document-item:not(.hidden)').length;
                usersCount.textContent = `Showing ${visibleUsers} user${visibleUsers !== 1 ? 's' : ''}`;
                documentsCount.textContent = `Showing ${visibleDocs} document${visibleDocs !== 1 ? 's' : ''}`;
            }

            function highlightText(element, searchText) {
                const originalText = element.textContent;
                if (searchText && originalText.toLowerCase().includes(searchText.toLowerCase())) {
                    const regex = new RegExp(`(${searchText})`, 'gi');
                    element.innerHTML = originalText.replace(regex, '<mark class="search-highlight">$1</mark>');
                } else {
                    element.textContent = originalText;
                }
            }

            searchInput.addEventListener('input', (e) => {
                const searchText = e.target.value.toLowerCase();

                // Search in users
                userCards.forEach(card => {
                    const searchableText = card.dataset.searchable.toLowerCase();
                    const visible = searchableText.includes(searchText);
                    card.classList.toggle('hidden', !visible);

                    if (visible) {
                        // Highlight matching text
                        card.querySelectorAll('.username, .email, .note').forEach(element => {
                            highlightText(element, searchText);
                        });
                    }
                });

                // Search in documents
                documentItems.forEach(item => {
                    const searchableText = item.dataset.searchable.toLowerCase();
                    const visible = searchableText.includes(searchText);
                    item.classList.toggle('hidden', !visible);

                    if (visible) {
                        highlightText(item.querySelector('.document-name'), searchText);
                    }
                });

                updateCounts();
            });

            // Initial counts
            updateCounts();

            // Tab Management
            const tabs = document.querySelectorAll('[data-tab]');
            const contents = document.querySelectorAll('.tab-content');

            tabs.forEach(tab => {
                tab.addEventListener('click', () => {
                    tabs.forEach(t => t.classList.remove('border-black', 'dark:border-white'));
                    tabs.forEach(t => t.classList.add('border-transparent'));
                    
                    tab.classList.remove('border-transparent');
                    tab.classList.add('border-black', 'dark:border-white');
                    
                    contents.forEach(content => content.classList.add('hidden'));
                    document.getElementById(`${tab.dataset.tab}-content`).classList.remove('hidden');
                });
            });

            // Modal Management
            const modal = document.getElementById('user-modal');
            const closeButtons = document.querySelectorAll('.close-modal, #close-modal');
            
            userCards.forEach(card => {
                card.addEventListener('click', () => {
                    modal.classList.remove('hidden');
                    document.getElementById('user-id').value = card.dataset.userId;
                    document.getElementById('email').value = card.dataset.email;
                    document.getElementById('username').value = card.dataset.username;
                    document.getElementById('note').value = card.dataset.note;
                });
            });

            closeButtons.forEach(button => {
                button.addEventListener('click', () => {
                    modal.classList.add('hidden');
                });
            });

            // Form Submission
            document.getElementById('edit-user-form').addEventListener('submit', async (e) => {
                e.preventDefault();
                const formData = new FormData(e.target);
                
                try {
                    const response = await fetch('update_user.php', {
                        method: 'POST',
                        body: formData
                    });
                    
                    const result = await response.json();
                    if(result.success) {
                        modal.classList.add('hidden');
                        window.location.href = '?success=1';
                    } else {
                        window.location.href = '?error=1';
                    }
                } catch(error) {
                    console.error('Error:', error);
                    window.location.href = '?error=1';
                }
            });

            // Auto-hide notifications
            const notifications = document.querySelectorAll('.success-notification, .error-notification');
            notifications.forEach(notification => {
                setTimeout(() => {
                    notification.style.display = 'none';
                }, 5000);
            });
        });
    </script>
</body>
</html>