<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\DB;

Route::get('/', function () {
    return view('welcome');
});

Route::get('/auth', function() {
    $token = request('token');
    $expires = request('expires');
    $signature = request('signature');
    
    if (!$token || !$expires || !$signature) {
        return redirect('/')->with('error', 'Paramètres manquants');
    }
    
    if (time() > $expires) {
        return redirect('/')->with('error', 'Lien expiré');
    }
    
    $expectedSignature = hash_hmac('sha256', $token . $expires, '1MZEEO92K55S1LFKEG0NMVA6JHJTNJ54');
    
    if (!hash_equals($expectedSignature, $signature)) {
        return redirect('/')->with('error', 'Signature invalide');
    }
    
    $data = json_decode(base64_decode($token), true);
    
    if (!$data || !isset($data['user_id']) || !isset($data['ise_auth'])) {
        return redirect('/')->with('error', 'Token invalide');
    }
    
    session([
        'user_id' => $data['user_id'],
        'email' => $data['email'],
        'username' => $data['username'],
        'authenticated' => true,
        'auth_time' => time()
    ]);
    
    return redirect('/workspace');
});

Route::get('/workspace', function() {
    if (!session('authenticated')) {
        return redirect('/');
    }
    
    if (time() - session('auth_time', 0) > 3600) {
        session()->flush();
        return redirect('/')->with('error', 'Session expirée');
    }
    
    return view('workspace');
})->name('workspace');

Route::get('/logout', function() {
    session()->flush();
    return redirect('/');
});

Route::get('/editor', function() {
    if (!session('authenticated')) {
        return redirect('/');
    }
    
    if (time() - session('auth_time', 0) > 3600) {
        session()->flush();
        return redirect('/')->with('error', 'Session expirée');
    }
    
    return view('editor');
})->name('editor');

Route::get('/editor/{document_id?}', function($document_id = null) {
    if (!session('authenticated')) {
        return redirect('/');
    }
    
    if (time() - session('auth_time', 0) > 3600) {
        session()->flush();
        return redirect('/')->with('error', 'Session expirée');
    }
    
    return view('editor', compact('document_id'));
})->name('editor.document');

// Routes API pour l'éditeur
Route::middleware(['web'])->group(function () {
    // Documents
    Route::post('/api/documents/save', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $request = request();
        $title = $request->input('title', 'Document sans titre');
        $content = $request->input('content', '');
        $documentId = $request->input('document_id');

        try {
            if ($documentId) {
                // Mise à jour
                DB::table('documents')
                    ->where('id', $documentId)
                    ->where('user_id', session('user_id'))
                    ->update([
                        'title' => $title,
                        'content' => $content,
                        'updated_at' => now(),
                        'last_accessed_at' => now()
                    ]);
                return response()->json(['success' => true, 'document_id' => $documentId]);
            } else {
                // Création
                $documentId = DB::table('documents')->insertGetId([
                    'user_id' => session('user_id'),
                    'title' => $title,
                    'content' => $content,
                    'status' => 'draft',
                    'created_at' => now(),
                    'updated_at' => now(),
                    'last_accessed_at' => now()
                ]);
                return response()->json(['success' => true, 'document_id' => $documentId]);
            }
        } catch (Exception $e) {
            return response()->json(['error' => 'Erreur de sauvegarde'], 500);
        }
    });

    Route::get('/api/documents/{id}', function($id) {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $document = DB::table('documents')
            ->where('id', $id)
            ->where('user_id', session('user_id'))
            ->first();

        if (!$document) {
            return response()->json(['error' => 'Document non trouvé'], 404);
        }

        // Mettre à jour last_accessed_at
        DB::table('documents')
            ->where('id', $id)
            ->update(['last_accessed_at' => now()]);

        return response()->json($document);
    });

    Route::get('/api/documents', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $documents = DB::table('documents')
            ->where('user_id', session('user_id'))
            ->orderBy('updated_at', 'desc')
            ->select('id', 'title', 'status', 'created_at', 'updated_at', 'last_accessed_at')
            ->get();

        return response()->json($documents);
    });

    Route::delete('/api/documents/{id}', function($id) {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $deleted = DB::table('documents')
            ->where('id', $id)
            ->where('user_id', session('user_id'))
            ->delete();

        if ($deleted) {
            return response()->json(['success' => true]);
        }
        return response()->json(['error' => 'Document non trouvé'], 404);
    });

    // Partage de documents
    Route::post('/api/documents/{id}/share', function($id) {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $request = request();
        $permission = $request->input('permission', 'view');

        // Vérifier que le document appartient à l'utilisateur
        $document = DB::table('documents')
            ->where('id', $id)
            ->where('user_id', session('user_id'))
            ->first();

        if (!$document) {
            return response()->json(['error' => 'Document non trouvé'], 404);
        }

        // Générer token unique
        $shareToken = bin2hex(random_bytes(32));

        $shareId = DB::table('document_shares')->insertGetId([
            'document_id' => $id,
            'share_token' => $shareToken,
            'permission' => $permission,
            'is_active' => true,
            'expires_at' => null, // Pas d'expiration pour l'instant
            'created_at' => now(),
            'updated_at' => now()
        ]);

        $shareUrl = url("/shared/{$shareToken}");

        return response()->json([
            'success' => true,
            'share_url' => $shareUrl,
            'permission' => $permission
        ]);
    });

    // Claude API
    Route::post('/api/claude', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $request = request();
        $prompt = $request->input('prompt');
        $selectedText = $request->input('selected_text', '');
        $action = $request->input('action', 'custom');

        // Construire le prompt selon l'action
        $systemPrompt = "Tu es un assistant d'écriture expert. Réponds en français de manière concise et professionnelle.";
        
        switch($action) {
            case 'grammar':
                $fullPrompt = "Corrige la grammaire et l'orthographe de ce texte sans changer le sens : {$selectedText}";
                break;
            case 'expand':
                $fullPrompt = "Développe et enrichis ce texte en gardant le même style : {$selectedText}";
                break;
            case 'summarize':
                $fullPrompt = "Résume ce texte de manière concise : {$selectedText}";
                break;
            case 'rephrase':
                $fullPrompt = "Reformule ce texte de manière plus claire : {$selectedText}";
                break;
            default:
                $fullPrompt = $prompt . ($selectedText ? "\n\nTexte à traiter : {$selectedText}" : '');
        }

        try {
            // Simuler l'appel API Claude (remplace par ton API key)
            $response = [
                'success' => true,
                'content' => "Réponse simulée de Claude :\n\n" . $fullPrompt . "\n\n[Ici sera la vraie réponse de Claude]"
            ];
            
            return response()->json($response);
        } catch (Exception $e) {
            return response()->json(['error' => 'Erreur API Claude'], 500);
        }
    });

    // Perplexity API
    Route::post('/api/search', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $request = request();
        $query = $request->input('query');

        try {
            // Simuler l'appel API Perplexity (remplace par ton API key)
            $response = [
                'success' => true,
                'results' => [
                    [
                        'title' => 'Résultat de recherche 1',
                        'snippet' => 'Ceci est un extrait du premier résultat de recherche pour : ' . $query,
                        'url' => 'https://example.com/1',
                        'source' => 'Source fiable 1'
                    ],
                    [
                        'title' => 'Résultat de recherche 2', 
                        'snippet' => 'Voici un autre extrait informatif concernant : ' . $query,
                        'url' => 'https://example.com/2',
                        'source' => 'Source fiable 2'
                    ]
                ],
                'summary' => 'Résumé des informations trouvées concernant : ' . $query
            ];
            
            return response()->json($response);
        } catch (Exception $e) {
            return response()->json(['error' => 'Erreur API recherche'], 500);
        }
    });
});

// Route pour les documents partagés
Route::get('/shared/{token}', function($token) {
    $share = DB::table('document_shares')
        ->join('documents', 'document_shares.document_id', '=', 'documents.id')
        ->where('document_shares.share_token', $token)
        ->where('document_shares.is_active', true)
        ->select('documents.*', 'document_shares.permission')
        ->first();

    if (!$share) {
        abort(404, 'Document partagé non trouvé');
    }

    return view('shared-document', [
        'document' => $share,
        'permission' => $share->permission
    ]);
});

// Routes API pour l'éditeur - AJOUTER À LA FIN
Route::middleware(['web'])->group(function () {
    
    Route::post('/api/documents/save', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $request = request();
        $title = $request->input('title', 'Document sans titre');
        $content = $request->input('content', '');
        $documentId = $request->input('document_id');

        try {
            if ($documentId) {
                DB::table('documents')
                    ->where('id', $documentId)
                    ->where('user_id', session('user_id'))
                    ->update([
                        'title' => $title,
                        'content' => $content,
                        'updated_at' => now(),
                        'last_accessed_at' => now()
                    ]);
                return response()->json(['success' => true, 'document_id' => $documentId]);
            } else {
                $documentId = DB::table('documents')->insertGetId([
                    'user_id' => session('user_id'),
                    'title' => $title,
                    'content' => $content,
                    'status' => 'draft',
                    'created_at' => now(),
                    'updated_at' => now(),
                    'last_accessed_at' => now()
                ]);
                return response()->json(['success' => true, 'document_id' => $documentId]);
            }
        } catch (Exception $e) {
            return response()->json(['error' => 'Erreur: ' . $e->getMessage()], 500);
        }
    });

    Route::get('/api/documents/{id}', function($id) {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $document = DB::table('documents')
            ->where('id', $id)
            ->where('user_id', session('user_id'))
            ->first();

        if (!$document) {
            return response()->json(['error' => 'Document non trouvé'], 404);
        }

        return response()->json($document);
    });

    Route::get('/api/documents', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        $documents = DB::table('documents')
            ->where('user_id', session('user_id'))
            ->orderBy('updated_at', 'desc')
            ->get();

        return response()->json($documents);
    });

    Route::post('/api/claude', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        return response()->json([
            'success' => true,
            'content' => 'Réponse simulée de Claude pour le moment !'
        ]);
    });

    Route::post('/api/search', function() {
        if (!session('authenticated')) {
            return response()->json(['error' => 'Non authentifié'], 401);
        }

        return response()->json([
            'success' => true,
            'results' => [
                ['title' => 'Résultat 1', 'snippet' => 'Exemple de résultat', 'source' => 'Test']
            ],
            'summary' => 'Résumé de recherche simulé'
        ]);
    });
});