<?php
session_start();

require_once '../db.php';

$user_selected = false;
$user_data = null;
$identite = null;
$compte = null;
$casier = [];

if(isset($_GET['user_id'])) {
    $user_id = $_GET['user_id'];
    $user_selected = true;
    
    // Récupération des données utilisateur
    $sql = "SELECT * FROM users WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user_data = $result->fetch_assoc();
    
    // Récupération des données d'identité
    $sql = "SELECT * FROM identites WHERE user_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $identite = $result->fetch_assoc();
    
    // Récupération des données bancaires
    $sql = "SELECT * FROM comptes_bancaires WHERE user_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $compte = $result->fetch_assoc();
    
    // Récupération du casier judiciaire
    $sql = "SELECT * FROM casier_judiciaire WHERE user_id = ? ORDER BY date DESC";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while($row = $result->fetch_assoc()) {
        $casier[] = $row;
    }
}

// Traitement des actions
if(isset($_POST['action'])) {
    $action = $_POST['action'];
    $target_user_id = isset($_POST['user_id']) ? $_POST['user_id'] : null;
    
    switch($action) {
        case 'envoyer_message':
            $sujet = $_POST['sujet'];
            $contenu = $_POST['contenu'];
            $type = $_POST['type'];
            $montant = ($type == 'facture') ? $_POST['montant'] : null;
            
            $sql = "INSERT INTO courriers (destinataire_id, expediteur, sujet, contenu, type, montant) 
                    VALUES (?, 'Gouvernement de Peerkinton', ?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("isssd", $target_user_id, $sujet, $contenu, $type, $montant);
            
            if($stmt->execute()) {
                $message = "Message envoyé avec succès.";
                $message_type = "success";
            } else {
                $message = "Erreur lors de l'envoi du message.";
                $message_type = "error";
            }
            break;
            
        case 'modifier_notoriete':
            $nouvelle_notoriete = $_POST['nouvelle_notoriete'];
            $raison = $_POST['raison'];
            
            $sql = "UPDATE identites SET notoriete = ? WHERE user_id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("ii", $nouvelle_notoriete, $target_user_id);
            
            if($stmt->execute()) {
                // Journaliser la modification
                $sql = "INSERT INTO journal_gouvernement (admin_id, action, cible_id, details) 
                        VALUES (?, 'modification_notoriete', ?, ?)";
                $stmt = $conn->prepare($sql);
                $admin_id = $_SESSION['admin_id'];
                $details = "Modification de la notoriété à {$nouvelle_notoriete}. Raison: {$raison}";
                $stmt->bind_param("iis", $admin_id, $target_user_id, $details);
                $stmt->execute();
                
                $message = "Notoriété modifiée avec succès.";
                $message_type = "success";
            } else {
                $message = "Erreur lors de la modification de la notoriété.";
                $message_type = "error";
            }
            break;
            
        case 'ajouter_infraction':
            $type = $_POST['type_infraction'];
            $description = $_POST['description_infraction'];
            $sanction = $_POST['sanction_infraction'];
            $date = $_POST['date_infraction'];
            
            $sql = "INSERT INTO casier_judiciaire (user_id, type, description, sanction, date) 
                    VALUES (?, ?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("issss", $target_user_id, $type, $description, $sanction, $date);
            
            if($stmt->execute()) {
                // Journaliser l'action
                $sql = "INSERT INTO journal_gouvernement (admin_id, action, cible_id, details) 
                        VALUES (?, 'ajout_infraction', ?, ?)";
                $stmt = $conn->prepare($sql);
                $admin_id = $_SESSION['admin_id'];
                $details = "Ajout d'une infraction de type {$type}. Sanction: {$sanction}";
                $stmt->bind_param("iis", $admin_id, $target_user_id, $details);
                $stmt->execute();
                
                $message = "Infraction ajoutée au casier judiciaire.";
                $message_type = "success";
            } else {
                $message = "Erreur lors de l'ajout de l'infraction.";
                $message_type = "error";
            }
            break;
            
        case 'modification_solde':
            $montant = $_POST['montant_modification'];
            $raison = $_POST['raison_modification'];
            $operation = $_POST['operation_solde'];
            
            if($operation == 'ajouter') {
                $sql = "UPDATE comptes_bancaires SET solde = solde + ? WHERE user_id = ?";
            } else {
                $sql = "UPDATE comptes_bancaires SET solde = solde - ? WHERE user_id = ?";
            }
            
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("di", $montant, $target_user_id);
            
            if($stmt->execute()) {
                // Journaliser l'action
                $sql = "INSERT INTO journal_gouvernement (admin_id, action, cible_id, details) 
                        VALUES (?, 'modification_solde', ?, ?)";
                $stmt = $conn->prepare($sql);
                $admin_id = $_SESSION['admin_id'];
                $details = "{$operation} {$montant}€. Raison: {$raison}";
                $stmt->bind_param("iis", $admin_id, $target_user_id, $details);
                $stmt->execute();
                
                $message = "Solde modifié avec succès.";
                $message_type = "success";
            } else {
                $message = "Erreur lors de la modification du solde.";
                $message_type = "error";
            }
            break;
            
        case 'modifier_identite':
            $nom = $_POST['nom'];
            $prenom = $_POST['prenom'];
            $date_naissance = $_POST['date_naissance'];
            $lieu_naissance = $_POST['lieu_naissance'];
            $nationalite = $_POST['nationalite'];
            $sexe = $_POST['sexe'];
            $adresse = $_POST['adresse'];
            $telephone = $_POST['telephone'];
            $profession = $_POST['profession'];
            
            $sql = "UPDATE identites SET 
                    nom = ?, prenom = ?, date_naissance = ?, lieu_naissance = ?, 
                    nationalite = ?, sexe = ?, adresse = ?, telephone = ?, profession = ? 
                    WHERE user_id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("ssssssssi", $nom, $prenom, $date_naissance, $lieu_naissance, 
                             $nationalite, $sexe, $adresse, $telephone, $profession, $target_user_id);
            
            if($stmt->execute()) {
                // Journaliser l'action
                $sql = "INSERT INTO journal_gouvernement (admin_id, action, cible_id, details) 
                        VALUES (?, 'modification_identite', ?, ?)";
                $stmt = $conn->prepare($sql);
                $admin_id = $_SESSION['admin_id'];
                $details = "Modification des informations d'identité";
                $stmt->bind_param("iis", $admin_id, $target_user_id, $details);
                $stmt->execute();
                
                $message = "Identité modifiée avec succès.";
                $message_type = "success";
            } else {
                $message = "Erreur lors de la modification de l'identité.";
                $message_type = "error";
            }
            break;
            
        case 'bloquer_compte':
            $raison = $_POST['raison_blocage'];
            $duree = $_POST['duree_blocage']; // en jours
            
            $sql = "UPDATE users SET 
                    bloque = 1, 
                    raison_blocage = ?, 
                    date_fin_blocage = DATE_ADD(NOW(), INTERVAL ? DAY) 
                    WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("sii", $raison, $duree, $target_user_id);
            
            if($stmt->execute()) {
                // Journaliser l'action
                $sql = "INSERT INTO journal_gouvernement (admin_id, action, cible_id, details) 
                        VALUES (?, 'blocage_compte', ?, ?)";
                $stmt = $conn->prepare($sql);
                $admin_id = $_SESSION['admin_id'];
                $details = "Blocage du compte pour {$duree} jours. Raison: {$raison}";
                $stmt->bind_param("iis", $admin_id, $target_user_id, $details);
                $stmt->execute();
                
                $message = "Compte bloqué avec succès.";
                $message_type = "success";
            } else {
                $message = "Erreur lors du blocage du compte.";
                $message_type = "error";
            }
            break;
    }
    
    // Redirection pour éviter la resoumission du formulaire
    header("Location: gouvernement.php?user_id=" . $target_user_id . "&message=" . urlencode($message) . "&message_type=" . $message_type);
    exit();
}

// Récupération de tous les utilisateurs pour la sélection
$sql = "SELECT u.id, u.email, i.nom, i.prenom 
        FROM users u 
        LEFT JOIN identites i ON u.id = i.user_id 
        ORDER BY i.nom, i.prenom";
$stmt = $conn->prepare($sql);
$stmt->execute();
$result = $stmt->get_result();
$tous_utilisateurs = [];
while($row = $result->fetch_assoc()) {
    $tous_utilisateurs[] = $row;
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Administration Gouvernementale - Peerkinton</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" rel="stylesheet">
    <style>
        body {
            font-family: 'Poppins', sans-serif;
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }
    </style>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'sans': ['Poppins', 'sans-serif'],
                    },
                    colors: {
                        'gov': {
                            50: '#f0f9ff',
                            100: '#e0f2fe',
                            500: '#0369a1',
                            600: '#0284c7',
                            700: '#0369a1',
                            800: '#075985',
                            900: '#0c4a6e',
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gray-50 text-gray-900">
    <div class="min-h-screen flex flex-col">
        <!-- En-tête -->
        <header class="bg-gov-800 text-white shadow">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
                <div class="flex justify-between items-center">
                    <div class="flex items-center">
                        <i class="fas fa-landmark text-2xl mr-3"></i>
                        <h1 class="text-xl font-semibold">Gouvernement de Peerkinton</h1>
                    </div>
                    <div>
                        <a href="logout.php" class="text-white hover:text-gray-200">
                            <i class="fas fa-sign-out-alt mr-1"></i> Déconnexion
                        </a>
                    </div>
                </div>
            </div>
        </header>

        <!-- Contenu principal -->
        <main class="flex-grow py-6">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                
                <?php if(isset($_GET['message'])): ?>
                    <div class="mb-6 px-4 py-3 rounded <?php echo $_GET['message_type'] == 'success' ? 'bg-green-50 text-green-800' : 'bg-red-50 text-red-800'; ?>">
                        <?php echo $_GET['message']; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Sélection d'un citoyen -->
                <div class="bg-white shadow overflow-hidden sm:rounded-lg mb-6">
                    <div class="px-4 py-5 sm:px-6">
                        <h2 class="text-lg font-medium text-gray-900">Sélection d'un citoyen</h2>
                        <p class="mt-1 max-w-2xl text-sm text-gray-500">
                            Sélectionnez un citoyen pour effectuer des actions administratives. Chaque action est enregistré dans nos journaux d'observation a des fins de garentir la sécurité de nos systèmes envers vos actions.
                        </p>
                    </div>
                    <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                        <form action="" method="GET" class="flex items-center space-x-4">
                            <div class="flex-grow">
                                <select name="user_id" id="user_id" class="block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                    <option value="">Sélectionnez un citoyen</option>
                                    <?php foreach($tous_utilisateurs as $u): ?>
                                        <option value="<?php echo $u['id']; ?>" <?php echo (isset($_GET['user_id']) && $_GET['user_id'] == $u['id']) ? 'selected' : ''; ?>>
                                            <?php echo $u['nom'] ? $u['nom'] . ' ' . $u['prenom'] . ' (' . $u['email'] . ')' : $u['email']; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div>
                                <button type="submit" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-gov-600 hover:bg-gov-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gov-500">
                                    <i class="fas fa-search mr-2"></i> Sélectionner
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <?php if($user_selected && $user_data): ?>
                    <!-- Informations du citoyen -->
                    <div class="bg-white shadow overflow-hidden sm:rounded-lg mb-6">
                        <div class="px-4 py-5 sm:px-6 flex justify-between items-center">
                            <div>
                                <h2 class="text-lg font-medium text-gray-900">
                                    <?php echo $identite ? $identite['nom'] . ' ' . $identite['prenom'] : $user_data['email']; ?>
                                </h2>
                                <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                    ID: <?php echo $user_data['id']; ?> • 
                                    <?php echo $identite ? 'Carte d\'identité: ' . $identite['id_carte'] : 'Pas de carte d\'identité'; ?>
                                </p>
                            </div>
                            
                            <?php if($identite): ?>
                                <div class="text-right">
                                    <span class="px-3 py-1 text-sm font-medium rounded-full 
                                        <?php 
                                            if($identite['notoriete'] >= 80) echo 'bg-green-100 text-green-800';
                                            elseif($identite['notoriete'] >= 60) echo 'bg-blue-100 text-blue-800';
                                            elseif($identite['notoriete'] >= 40) echo 'bg-yellow-100 text-yellow-800';
                                            elseif($identite['notoriete'] >= 20) echo 'bg-orange-100 text-orange-800';
                                            else echo 'bg-red-100 text-red-800';
                                        ?>">
                                        Notoriété: <?php echo $identite['notoriete']; ?>/100
                                    </span>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Onglets -->
                    <div class="border-b border-gray-200 mb-6">
                        <nav class="-mb-px flex space-x-8">
                            <a href="#tab-actions" class="border-gov-500 text-gov-600 whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm tab-link" data-tab="tab-actions">
                                <i class="fas fa-cogs mr-2"></i> Actions
                            </a>
                            <a href="#tab-identite" class="border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm tab-link" data-tab="tab-identite">
                                <i class="fas fa-id-card mr-2"></i> Identité
                            </a>
                            <a href="#tab-casier" class="border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm tab-link" data-tab="tab-casier">
                                <i class="fas fa-gavel mr-2"></i> Casier judiciaire
                            </a>
                            <a href="#tab-messages" class="border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm tab-link" data-tab="tab-messages">
                                <i class="fas fa-envelope mr-2"></i> Envoyer un message
                            </a>
                        </nav>
                    </div>
                    
                    <!-- Contenu des onglets -->
                    <div class="tab-content active" id="tab-actions">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <!-- Modification de la notoriété -->
                            <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                                <div class="px-4 py-5 sm:px-6">
                                    <h3 class="text-lg leading-6 font-medium text-gray-900">
                                        Modifier la notoriété
                                    </h3>
                                </div>
                                <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                    <form action="" method="POST" class="space-y-4">
                                        <input type="hidden" name="action" value="modifier_notoriete">
                                        <input type="hidden" name="user_id" value="<?php echo $user_data['id']; ?>">
                                        
                                        <div>
                                            <label for="nouvelle_notoriete" class="block text-sm font-medium text-gray-700">Nouvelle valeur (0-100)</label>
                                            <input type="number" name="nouvelle_notoriete" id="nouvelle_notoriete" min="0" max="100" value="<?php echo $identite ? $identite['notoriete'] : 50; ?>" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                        </div>
                                        
                                        <div>
                                            <label for="raison" class="block text-sm font-medium text-gray-700">Raison</label>
                                            <textarea name="raison" id="raison" rows="3" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm"></textarea>
                                        </div>
                                        
                                        <div>
                                            <button type="submit" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-gov-600 hover:bg-gov-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gov-500">
                                                Appliquer la modification
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                            
                            <!-- Modification du solde -->
                            <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                                <div class="px-4 py-5 sm:px-6">
                                    <h3 class="text-lg leading-6 font-medium text-gray-900">
                                        Modifier le solde bancaire
                                    </h3>
                                    <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                        Solde actuel: <?php echo $compte ? number_format($compte['solde'], 2, ',', ' ') . ' €' : 'Aucun compte bancaire'; ?>
                                    </p>
                                </div>
                                <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                    <form action="" method="POST" class="space-y-4">
                                        <input type="hidden" name="action" value="modification_solde">
                                        <input type="hidden" name="user_id" value="<?php echo $user_data['id']; ?>">
                                        
                                        <div>
                                            <label for="operation_solde" class="block text-sm font-medium text-gray-700">Opération</label>
                                            <select name="operation_solde" id="operation_solde" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                                <option value="ajouter">Ajouter des fonds</option>
                                                <option value="retirer">Retirer des fonds</option>
                                            </select>
                                        </div>
                                        
                                        <div>
                                            <label for="montant_modification" class="block text-sm font-medium text-gray-700">Montant</label>
                                            <div class="mt-1 relative rounded-md shadow-sm">
                                                <input type="number" name="montant_modification" id="montant_modification" step="0.01" min="0.01" required class="block w-full pr-12 border-gray-300 rounded-md focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                                <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                                    <span class="text-gray-500 sm:text-sm">€</span>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div>
                                            <label for="raison_modification" class="block text-sm font-medium text-gray-700">Raison</label>
                                            <textarea name="raison_modification" id="raison_modification" rows="3" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm"></textarea>
                                        </div>
                                        
                                        <div>
                                            <button type="submit" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-gov-600 hover:bg-gov-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gov-500">
                                                Effectuer l'opération
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                            
                            <!-- Blocage de compte -->
                            <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                                <div class="px-4 py-5 sm:px-6">
                                    <h3 class="text-lg leading-6 font-medium text-gray-900">
                                        Bloquer l'accès
                                    </h3>
                                    <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                        État actuel: <?php echo $user_data['bloque'] ? 'Bloqué jusqu\'au ' . date('d/m/Y', strtotime($user_data['date_fin_blocage'])) : 'Actif'; ?>
                                    </p>
                                </div>
                                <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                    <form action="" method="POST" class="space-y-4">
                                        <input type="hidden" name="action" value="bloquer_compte">
                                        <input type="hidden" name="user_id" value="<?php echo $user_data['id']; ?>">
                                        
                                        <div>
                                            <label for="duree_blocage" class="block text-sm font-medium text-gray-700">Durée du blocage (jours)</label>
                                            <input type="number" name="duree_blocage" id="duree_blocage" min="1" max="365" value="7" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                        </div>
                                        
                                        <div>
                                            <label for="raison_blocage" class="block text-sm font-medium text-gray-700">Raison du blocage</label>
                                            <textarea name="raison_blocage" id="raison_blocage" rows="3" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm"></textarea>
                                        </div>
                                        
                                        <div>
                                            <button type="submit" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500" onclick="return confirm('Êtes-vous sûr de vouloir bloquer ce compte ?');">
                                                Bloquer l'accès
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="tab-content" id="tab-identite">
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    Carte d'identité
                                </h3>
                                <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                    Informations d'identité du citoyen.
                                </p>
                            </div>
                            
                            <?php if($identite): ?>
                                <!-- Aperçu de la carte d'identité -->
                                <div class="border-t border-gray-200">
                                    <div class="p-6 bg-gradient-to-r from-gov-600 to-gov-800 rounded-lg mx-4 my-5 text-white shadow-lg">
                                        <div class="flex justify-between items-start mb-6">
                                            <div>
                                                <h4 class="text-xl font-bold mb-1">Peerkinton</h4>
                                                <p class="text-sm opacity-80">Carte Nationale d'Identité</p>
                                            </div>
                                            <div class="text-right">
                                                <p class="text-sm opacity-80">ID: <?php echo $identite['id_carte']; ?></p>
                                            </div>
                                        </div>
                                        
                                        <div class="grid grid-cols-2 gap-4 mb-4">
                                            <div>
                                                <p class="text-xs opacity-80">Nom</p>
                                                <p class="font-medium"><?php echo $identite['nom']; ?></p>
                                            </div>
                                            <div>
                                                <p class="text-xs opacity-80">Prénom</p>
                                                <p class="font-medium"><?php echo $identite['prenom']; ?></p>
                                            </div>
                                            <div>
                                                <p class="text-xs opacity-80">Date de naissance</p>
                                                <p class="font-medium"><?php echo date('d/m/Y', strtotime($identite['date_naissance'])); ?></p>
                                            </div>
                                            <div>
                                                <p class="text-xs opacity-80">Lieu de naissance</p>
                                                <p class="font-medium"><?php echo $identite['lieu_naissance']; ?></p>
                                            </div>
                                            <div>
                                                <p class="text-xs opacity-80">Nationalité</p>
                                                <p class="font-medium"><?php echo $identite['nationalite']; ?></p>
                                            </div>
                                            <div>
                                                <p class="text-xs opacity-80">Sexe</p>
                                                <p class="font-medium"><?php echo $identite['sexe']; ?></p>
                                            </div>
                                        </div>
                                        
                                        <div class="mt-6 pt-4 border-t border-white border-opacity-20 flex justify-between items-center">
                                            <div>
                                                <p class="text-xs opacity-80">Date d'émission</p>
                                                <p class="font-medium"><?php echo date('d/m/Y', strtotime($identite['date_emission'])); ?></p>
                                            </div>
                                            <div class="text-right">
                                                <p class="text-xs opacity-80">Date d'expiration</p>
                                                <p class="font-medium"><?php echo date('d/m/Y', strtotime($identite['date_expiration'])); ?></p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Formulaire de modification d'identité -->
                                <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                    <h4 class="text-md font-medium text-gray-900 mb-4">Modifier les informations d'identité</h4>
                                    
                                    <form action="" method="POST" class="space-y-4">
                                        <input type="hidden" name="action" value="modifier_identite">
                                        <input type="hidden" name="user_id" value="<?php echo $user_data['id']; ?>">
                                        
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <label for="nom" class="block text-sm font-medium text-gray-700">Nom</label>
                                                <input type="text" name="nom" id="nom" value="<?php echo $identite['nom']; ?>" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            </div>
                                            
                                            <div>
                                                <label for="prenom" class="block text-sm font-medium text-gray-700">Prénom</label>
                                                <input type="text" name="prenom" id="prenom" value="<?php echo $identite['prenom']; ?>" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            </div>
                                            
                                            <div>
                                                <label for="date_naissance" class="block text-sm font-medium text-gray-700">Date de naissance</label>
                                                <input type="date" name="date_naissance" id="date_naissance" value="<?php echo $identite['date_naissance']; ?>" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            </div>
                                            
                                            <div>
                                                <label for="lieu_naissance" class="block text-sm font-medium text-gray-700">Lieu de naissance</label>
                                                <input type="text" name="lieu_naissance" id="lieu_naissance" value="<?php echo $identite['lieu_naissance']; ?>" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            </div>
                                            
                                            <div>
                                                <label for="nationalite" class="block text-sm font-medium text-gray-700">Nationalité</label>
                                                <input type="text" name="nationalite" id="nationalite" value="<?php echo $identite['nationalite']; ?>" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            </div>
                                            
                                            <div>
                                                <label for="sexe" class="block text-sm font-medium text-gray-700">Sexe</label>
                                                <select name="sexe" id="sexe" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                                    <option value="M" <?php echo $identite['sexe'] == 'M' ? 'selected' : ''; ?>>Masculin</option>
                                                    <option value="F" <?php echo $identite['sexe'] == 'F' ? 'selected' : ''; ?>>Féminin</option>
                                                </select>
                                            </div>
                                            
                                            <div>
                                                <label for="adresse" class="block text-sm font-medium text-gray-700">Adresse</label>
                                                <input type="text" name="adresse" id="adresse" value="<?php echo $identite['adresse']; ?>" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            </div>
                                            
                                            <div>
                                                <label for="telephone" class="block text-sm font-medium text-gray-700">Téléphone</label>
                                                <input type="text" name="telephone" id="telephone" value="<?php echo $identite['telephone']; ?>" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            </div>
                                            
                                            <div>
                                                <label for="profession" class="block text-sm font-medium text-gray-700">Profession</label>
                                                <input type="text" name="profession" id="profession" value="<?php echo $identite['profession']; ?>" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            </div>
                                        </div>
                                        
                                        <div>
                                            <button type="submit" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-gov-600 hover:bg-gov-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gov-500">
                                                Mettre à jour les informations
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            <?php else: ?>
                                <div class="border-t border-gray-200 px-4 py-5 sm:p-6 text-center">
                                    <div class="mb-4">
                                        <i class="fas fa-exclamation-circle text-yellow-500 text-4xl"></i>
                                    </div>
                                    <h4 class="text-lg font-medium text-gray-900 mb-2">Pas de carte d'identité</h4>
                                    <p class="text-gray-500 mb-4">Ce citoyen n'a pas encore de carte d'identité.</p>
                                    
                                    <a href="creer_identite.php?user_id=<?php echo $user_data['id']; ?>" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-gov-600 hover:bg-gov-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gov-500">
                                        Créer une carte d'identité
                                    </a>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="tab-content" id="tab-casier">
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    Casier judiciaire
                                </h3>
                                <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                    Historique des infractions du citoyen.
                                </p>
                            </div>
                            
                            <!-- Formulaire d'ajout d'infraction -->
                            <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                <h4 class="text-md font-medium text-gray-900 mb-4">Ajouter une infraction</h4>
                                
                                <form action="" method="POST" class="space-y-4">
                                    <input type="hidden" name="action" value="ajouter_infraction">
                                    <input type="hidden" name="user_id" value="<?php echo $user_data['id']; ?>">
                                    
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label for="type_infraction" class="block text-sm font-medium text-gray-700">Type d'infraction</label>
                                            <select name="type_infraction" id="type_infraction" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                                <option value="">Sélectionner...</option>
                                                <option value="Infraction routière">Infraction routière</option>
                                                <option value="Vol">Vol</option>
                                                <option value="Violence">Violence</option>
                                                <option value="Fraude">Fraude</option>
                                                <option value="Trafic">Trafic</option>
                                                <option value="Autre">Autre</option>
                                            </select>
                                        </div>
                                        
                                        <div>
                                            <label for="date_infraction" class="block text-sm font-medium text-gray-700">Date de l'infraction</label>
                                            <input type="date" name="date_infraction" id="date_infraction" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                        </div>
                                    </div>
                                    
                                    <div>
                                        <label for="description_infraction" class="block text-sm font-medium text-gray-700">Description</label>
                                        <textarea name="description_infraction" id="description_infraction" rows="3" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm"></textarea>
                                    </div>
                                    
                                    <div>
                                        <label for="sanction_infraction" class="block text-sm font-medium text-gray-700">Sanction</label>
                                        <input type="text" name="sanction_infraction" id="sanction_infraction" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm" placeholder="Ex: Amende de 500€, 3 mois de prison, etc.">
                                    </div>
                                    
                                    <div>
                                        <button type="submit" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500">
                                            Ajouter au casier judiciaire
                                        </button>
                                    </div>
                                </form>
                            </div>
                            
                            <!-- Liste des infractions -->
                            <div class="border-t border-gray-200">
                                <div class="px-4 py-5 sm:p-6">
                                    <h4 class="text-md font-medium text-gray-900 mb-4">Historique des infractions</h4>
                                    
                                    <?php if(count($casier) > 0): ?>
                                        <div class="overflow-x-auto">
                                            <table class="min-w-full divide-y divide-gray-200">
                                                <thead class="bg-gray-50">
                                                    <tr>
                                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Description</th>
                                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Sanction</th>
                                                    </tr>
                                                </thead>
                                                <tbody class="bg-white divide-y divide-gray-200">
                                                    <?php foreach($casier as $infraction): ?>
                                                        <tr>
                                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                                                <?php echo date('d/m/Y', strtotime($infraction['date'])); ?>
                                                            </td>
                                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                                <?php echo $infraction['type']; ?>
                                                            </td>
                                                            <td class="px-6 py-4 text-sm text-gray-500">
                                                                <?php echo $infraction['description']; ?>
                                                            </td>
                                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                                                <?php echo $infraction['sanction']; ?>
                                                            </td>
                                                        </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    <?php else: ?>
                                        <div class="text-center py-4">
                                            <div class="mb-4">
                                                <i class="fas fa-check-circle text-green-500 text-4xl"></i>
                                            </div>
                                            <h4 class="text-lg font-medium text-gray-900 mb-2">Casier judiciaire vierge</h4>
                                            <p class="text-gray-500">Ce citoyen n'a commis aucune infraction.</p>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="tab-content" id="tab-messages">
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    Envoyer un message officiel
                                </h3>
                                <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                    Communiquer avec le citoyen au nom du gouvernement.
                                </p>
                            </div>
                            <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                <form action="" method="POST" class="space-y-4">
                                    <input type="hidden" name="action" value="envoyer_message">
                                    <input type="hidden" name="user_id" value="<?php echo $user_data['id']; ?>">
                                    
                                    <div>
                                        <label for="type" class="block text-sm font-medium text-gray-700">Type de message</label>
                                        <select name="type" id="type" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm" onchange="toggleFactureFields()">
                                            <option value="communication">Communication générale</option>
                                            <option value="facture">Facture</option>
                                        </select>
                                    </div>
                                    
                                    <div>
                                        <label for="sujet" class="block text-sm font-medium text-gray-700">Sujet</label>
                                        <input type="text" name="sujet" id="sujet" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                    </div>
                                    
                                    <div id="facture-fields" style="display: none;">
                                        <label for="montant" class="block text-sm font-medium text-gray-700">Montant de la facture</label>
                                        <div class="mt-1 relative rounded-md shadow-sm">
                                            <input type="number" name="montant" id="montant" step="0.01" min="0.01" class="block w-full pr-12 border-gray-300 rounded-md focus:ring-gov-500 focus:border-gov-500 sm:text-sm">
                                            <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                                <span class="text-gray-500 sm:text-sm">€</span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div>
                                        <label for="contenu" class="block text-sm font-medium text-gray-700">Contenu du message</label>
                                        <textarea name="contenu" id="contenu" rows="6" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-gov-500 focus:border-gov-500 sm:text-sm"></textarea>
                                    </div>
                                    
                                    <div>
                                        <button type="submit" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-gov-600 hover:bg-gov-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gov-500">
                                            Envoyer le message
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="bg-white shadow overflow-hidden sm:rounded-lg px-4 py-5 sm:p-6 text-center">
                        <div class="mb-4">
                            <i class="fas fa-user-alt text-gray-400 text-5xl"></i>
                        </div>
                        <h3 class="text-lg font-medium text-gray-900 mb-2">Sélectionnez un citoyen</h3>
                        <p class="text-gray-500">Veuillez sélectionner un citoyen pour afficher ses informations et effectuer des actions administratives.</p>
                    </div>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const tabLinks = document.querySelectorAll('.tab-link');
            const tabContents = document.querySelectorAll('.tab-content');
            
            tabLinks.forEach(link => {
                link.addEventListener('click', function(e) {
                    e.preventDefault();
                    const tab = this.getAttribute('data-tab');
                    
                    tabLinks.forEach(t => {
                        t.classList.remove('border-gov-500', 'text-gov-600');
                        t.classList.add('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
                    });
                    
                    this.classList.add('border-gov-500', 'text-gov-600');
                    this.classList.remove('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
                    
                    tabContents.forEach(content => {
                        content.classList.remove('active');
                    });
                    
                    document.getElementById(tab).classList.add('active');
                });
            });
            
            toggleFactureFields();
        });
        
        function toggleFactureFields() {
            const typeSelect = document.getElementById('type');
            const factureFields = document.getElementById('facture-fields');
            const montantInput = document.getElementById('montant');
            
            if(typeSelect.value === 'facture') {
                factureFields.style.display = 'block';
                montantInput.required = true;
            } else {
                factureFields.style.display = 'none';
                montantInput.required = false;
            }
        }
    </script>
</body>
</html>