<?php
require_once '../db.php';

try {
    $usersStmt = $pdo->query("SELECT * FROM users ORDER BY created_at DESC");
    $users = $usersStmt->fetchAll();
    
    $projectsStmt = $pdo->query("
        SELECT p.*, u.name as user_name 
        FROM projects p 
        JOIN users u ON p.user_id = u.id 
        ORDER BY p.created_at DESC
    ");
    $projects = $projectsStmt->fetchAll();
    
    $requestsStmt = $pdo->query("
        SELECT r.*, u.name as user_name, p.project_name 
        FROM requests r 
        JOIN users u ON r.user_id = u.id 
        LEFT JOIN projects p ON r.project_id = p.id 
        ORDER BY r.created_at DESC
    ");
    $requests = $requestsStmt->fetchAll();
    
} catch (PDOException $e) {
    echo "Database error: " . $e->getMessage();
    exit();
}

function getStatusColor($status) {
    switch ($status) {
        case 'pending': return 'bg-yellow-100 text-yellow-800';
        case 'in_progress': return 'bg-blue-100 text-blue-800';
        case 'testing': return 'bg-purple-100 text-purple-800';
        case 'completed': return 'bg-green-100 text-green-800';
        case 'on_hold': return 'bg-red-100 text-red-800';
        case 'open': return 'bg-orange-100 text-orange-800';
        case 'resolved': return 'bg-green-100 text-green-800';
        case 'closed': return 'bg-gray-100 text-gray-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}

function getPriorityColor($priority) {
    switch ($priority) {
        case 'low': return 'bg-gray-100 text-gray-800';
        case 'medium': return 'bg-blue-100 text-blue-800';
        case 'high': return 'bg-orange-100 text-orange-800';
        case 'urgent': return 'bg-red-100 text-red-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}
?>
<!DOCTYPE html>
<html lang="en-GB">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - aktascorp</title>
    <link rel="icon" type="image/png" href="ac.png">
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Raleway:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Raleway', sans-serif;
            background-color: #f9f0e6;
        }
        .modal {
            display: none;
            position: fixed;
            z-index: 50;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
        }
        .modal.show {
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }
        .tab-button.active {
            background-color: #374151;
            color: white;
        }
    </style>
</head>
<body class="text-gray-800">

    <div class="min-h-screen">
        <nav class="bg-gray-900 text-white shadow-lg">
            <div class="max-w-7xl mx-auto px-4 py-4">
                <div class="flex items-center justify-between">
                    <div class="flex items-center space-x-4">
                        <h1 class="text-2xl font-light tracking-wider">aktascorp</h1>
                        <span class="text-sm text-gray-300">admin portal</span>
                    </div>
                    <div class="flex items-center space-x-4">
                        <span class="text-sm text-gray-300">Administrator</span>
                        <a href="/" class="text-sm text-red-400 hover:text-red-300 transition duration-200">Exit Admin</a>
                    </div>
                </div>
            </div>
        </nav>

        <div class="max-w-7xl mx-auto px-4 py-8">
            
            <div class="mb-12">
                <h2 class="text-4xl font-light mb-3">Admin Dashboard</h2>
                <p class="text-gray-600 text-lg">Manage clients, projects and requests</p>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-12">
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center">
                        <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center mr-4">
                            <svg class="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z"></path>
                            </svg>
                        </div>
                        <div>
                            <h3 class="text-2xl font-semibold"><?= count($users) ?></h3>
                            <p class="text-gray-600">Total Clients</p>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center">
                        <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center mr-4">
                            <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"></path>
                            </svg>
                        </div>
                        <div>
                            <h3 class="text-2xl font-semibold"><?= count($projects) ?></h3>
                            <p class="text-gray-600">Total Projects</p>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center">
                        <div class="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center mr-4">
                            <svg class="w-6 h-6 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z"></path>
                            </svg>
                        </div>
                        <div>
                            <h3 class="text-2xl font-semibold"><?= count($requests) ?></h3>
                            <p class="text-gray-600">Total Requests</p>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center">
                        <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center mr-4">
                            <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                        <div>
                            <h3 class="text-2xl font-semibold"><?= count(array_filter($requests, function($r) { return $r['status'] === 'open'; })) ?></h3>
                            <p class="text-gray-600">Open Requests</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tabs Navigation -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 mb-8">
                <div class="border-b border-gray-200">
                    <nav class="flex space-x-8 px-6">
                        <button onclick="showTab('clients')" class="tab-button active py-4 px-2 border-b-2 border-transparent font-medium text-sm hover:text-gray-700 hover:border-gray-300 transition duration-200">
                            Clients Management
                        </button>
                        <button onclick="showTab('projects')" class="tab-button py-4 px-2 border-b-2 border-transparent font-medium text-sm text-gray-500 hover:text-gray-700 hover:border-gray-300 transition duration-200">
                            Projects Management
                        </button>
                        <button onclick="showTab('requests')" class="tab-button py-4 px-2 border-b-2 border-transparent font-medium text-sm text-gray-500 hover:text-gray-700 hover:border-gray-300 transition duration-200">
                            Requests Management
                        </button>
                    </nav>
                </div>

                <!-- Clients Tab -->
                <div id="clients" class="tab-content active p-6">
                    <div class="flex items-center justify-between mb-6">
                        <h3 class="text-2xl font-light">Clients</h3>
                        <button onclick="showModal('addClientModal')" class="bg-gray-800 text-white px-4 py-2 rounded-lg text-sm hover:bg-gray-700 transition duration-200 flex items-center">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                            </svg>
                            Add Client
                        </button>
                    </div>

                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Client</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Email</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Company</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Access Code</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Projects</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($users as $user): ?>
                                    <?php $userProjects = array_filter($projects, function($p) use ($user) { return $p['user_id'] == $user['id']; }); ?>
                                    <tr class="hover:bg-gray-50">
                                        <td class="px-6 py-4 whitespace-nowrap">
                                            <div class="flex items-center">
                                                <div class="w-10 h-10 bg-gray-800 rounded-full flex items-center justify-center mr-3">
                                                    <span class="text-white font-medium"><?= strtoupper(substr($user['name'], 0, 1)) ?></span>
                                                </div>
                                                <div>
                                                    <div class="text-sm font-medium text-gray-900"><?= htmlspecialchars($user['name']) ?></div>
                                                    <div class="text-sm text-gray-500">ID: <?= $user['id'] ?></div>
                                                </div>
                                            </div>
                                        </td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?= htmlspecialchars($user['email']) ?></td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?= htmlspecialchars($user['company'] ?: 'N/A') ?></td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-mono text-gray-900"><?= htmlspecialchars($user['code']) ?></td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?= count($userProjects) ?> projects</td>
                                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                            <button onclick="manageClient(<?= $user['id'] ?>)" class="text-blue-600 hover:text-blue-900 mr-3">Manage</button>
                                            <button onclick="editClient(<?= $user['id'] ?>)" class="text-gray-600 hover:text-gray-900 mr-3">Edit</button>
                                            <button onclick="deleteClient(<?= $user['id'] ?>)" class="text-red-600 hover:text-red-900">Delete</button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Projects Tab -->
                <div id="projects" class="tab-content p-6">
                    <div class="flex items-center justify-between mb-6">
                        <h3 class="text-2xl font-light">Projects</h3>
                        <button onclick="showModal('addProjectModal')" class="bg-gray-800 text-white px-4 py-2 rounded-lg text-sm hover:bg-gray-700 transition duration-200 flex items-center">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                            </svg>
                            Add Project
                        </button>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        <?php foreach ($projects as $project): ?>
                            <div class="bg-gray-50 rounded-lg p-6 border border-gray-200 hover:shadow-md transition duration-200">
                                <div class="flex items-start justify-between mb-4">
                                    <h4 class="text-lg font-medium text-gray-900"><?= htmlspecialchars($project['project_name']) ?></h4>
                                    <span class="px-2 py-1 text-xs rounded-full <?= getStatusColor($project['status']) ?>">
                                        <?= ucfirst(str_replace('_', ' ', $project['status'])) ?>
                                    </span>
                                </div>
                                
                                <p class="text-sm text-gray-600 mb-3"><?= htmlspecialchars($project['description'] ?: 'No description') ?></p>
                                
                                <div class="text-sm text-gray-500 mb-4">
                                    <p><strong>Client:</strong> <?= htmlspecialchars($project['user_name']) ?></p>
                                    <?php if ($project['domain_name']): ?>
                                        <p><strong>Domain:</strong> <?= htmlspecialchars($project['domain_name']) ?></p>
                                    <?php endif; ?>
                                    <?php if ($project['visit_link']): ?>
                                        <p><strong>Link:</strong> <a href="<?= htmlspecialchars($project['visit_link']) ?>" target="_blank" class="text-blue-600 hover:text-blue-800">Visit</a></p>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="flex space-x-2">
                                    <button onclick="manageProject(<?= $project['id'] ?>)" class="flex-1 bg-gray-800 text-white py-2 px-3 rounded text-sm hover:bg-gray-700 transition duration-200">
                                        Manage
                                    </button>
                                    <button onclick="addProjectLog(<?= $project['id'] ?>)" class="flex-1 bg-blue-600 text-white py-2 px-3 rounded text-sm hover:bg-blue-700 transition duration-200">
                                        Add Log
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <!-- Requests Tab -->
                <div id="requests" class="tab-content p-6">
                    <div class="flex items-center justify-between mb-6">
                        <h3 class="text-2xl font-light">Requests</h3>
                        <button onclick="showModal('addRequestModal')" class="bg-gray-800 text-white px-4 py-2 rounded-lg text-sm hover:bg-gray-700 transition duration-200 flex items-center">
                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                            </svg>
                            Add Request
                        </button>
                    </div>

                    <div class="space-y-4">
                        <?php foreach ($requests as $request): ?>
                            <div class="bg-gray-50 rounded-lg p-6 border border-gray-200 hover:shadow-md transition duration-200">
                                <div class="flex items-start justify-between">
                                    <div class="flex-1">
                                        <div class="flex items-center space-x-3 mb-2">
                                            <h4 class="text-lg font-medium text-gray-900"><?= htmlspecialchars($request['subject']) ?></h4>
                                            <span class="px-2 py-1 text-xs rounded-full <?= getStatusColor($request['status']) ?>">
                                                <?= ucfirst(str_replace('_', ' ', $request['status'])) ?>
                                            </span>
                                            <span class="px-2 py-1 text-xs rounded-full <?= getPriorityColor($request['priority']) ?>">
                                                <?= ucfirst($request['priority']) ?>
                                            </span>
                                        </div>
                                        
                                        <p class="text-gray-600 mb-3"><?= htmlspecialchars(substr($request['message'], 0, 150)) ?>...</p>
                                        
                                        <div class="text-sm text-gray-500">
                                            <p><strong>Client:</strong> <?= htmlspecialchars($request['user_name']) ?></p>
                                            <?php if ($request['project_name']): ?>
                                                <p><strong>Project:</strong> <?= htmlspecialchars($request['project_name']) ?></p>
                                            <?php endif; ?>
                                            <p><strong>Created:</strong> <?= date('M j, Y H:i', strtotime($request['created_at'])) ?></p>
                                        </div>
                                    </div>
                                    
                                    <div class="ml-4 flex flex-col space-y-2">
                                        <button onclick="viewRequest(<?= $request['id'] ?>)" class="bg-gray-800 text-white py-2 px-4 rounded text-sm hover:bg-gray-700 transition duration-200">
                                            View
                                        </button>
                                        <button onclick="respondToRequest(<?= $request['id'] ?>)" class="bg-blue-600 text-white py-2 px-4 rounded text-sm hover:bg-blue-700 transition duration-200">
                                            Respond
                                        </button>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Client Modal -->
    <div id="addClientModal" class="modal">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full mx-4 p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-xl font-medium">Add New Client</h3>
                <button onclick="closeModal('addClientModal')" class="text-gray-500 hover:text-gray-700">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            
            <form method="POST" action="admin_actions.php">
                <input type="hidden" name="action" value="add_client">
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Name</label>
                        <input type="text" name="name" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Email</label>
                        <input type="email" name="email" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Company</label>
                        <input type="text" name="company" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Access Code</label>
                        <input type="text" name="code" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400" placeholder="8-digit numeric code">
                    </div>
                    
                    <div class="flex space-x-3 pt-4">
                        <button type="submit" class="flex-1 bg-gray-800 text-white py-2 rounded-lg hover:bg-gray-700 transition duration-200">
                            Add Client
                        </button>
                        <button type="button" onclick="closeModal('addClientModal')" class="flex-1 bg-gray-200 text-gray-800 py-2 rounded-lg hover:bg-gray-300 transition duration-200">
                            Cancel
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Add Project Modal -->
    <div id="addProjectModal" class="modal">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full mx-4 p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-xl font-medium">Add New Project</h3>
                <button onclick="closeModal('addProjectModal')" class="text-gray-500 hover:text-gray-700">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            
            <form method="POST" action="admin_actions.php">
                <input type="hidden" name="action" value="add_project">
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Client</label>
                        <select name="user_id" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                            <option value="">Select a client</option>
                            <?php foreach ($users as $user): ?>
                                <option value="<?= $user['id'] ?>"><?= htmlspecialchars($user['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Project Name</label>
                        <input type="text" name="project_name" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Description</label>
                        <textarea name="description" rows="3" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400"></textarea>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Domain Name</label>
                        <input type="text" name="domain_name" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Visit Link</label>
                        <input type="url" name="visit_link" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                        <select name="status" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                            <option value="pending">Pending</option>
                            <option value="in_progress">In Progress</option>
                            <option value="testing">Testing</option>
                            <option value="completed">Completed</option>
                            <option value="on_hold">On Hold</option>
                        </select>
                    </div>
                    
                    <div class="flex space-x-3 pt-4">
                        <button type="submit" class="flex-1 bg-gray-800 text-white py-2 rounded-lg hover:bg-gray-700 transition duration-200">
                            Add Project
                        </button>
                        <button type="button" onclick="closeModal('addProjectModal')" class="flex-1 bg-gray-200 text-gray-800 py-2 rounded-lg hover:bg-gray-300 transition duration-200">
                            Cancel
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Add Request Modal -->
    <div id="addRequestModal" class="modal">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full mx-4 p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-xl font-medium">Add New Request</h3>
                <button onclick="closeModal('addRequestModal')" class="text-gray-500 hover:text-gray-700">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            
            <form method="POST" action="admin_actions.php">
                <input type="hidden" name="action" value="add_request">
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Client</label>
                        <select name="user_id" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                            <option value="">Select a client</option>
                            <?php foreach ($users as $user): ?>
                                <option value="<?= $user['id'] ?>"><?= htmlspecialchars($user['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Related Project</label>
                        <select name="project_id" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                            <option value="">No specific project</option>
                            <?php foreach ($projects as $project): ?>
                                <option value="<?= $project['id'] ?>"><?= htmlspecialchars($project['project_name']) ?> (<?= htmlspecialchars($project['user_name']) ?>)</option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Subject</label>
                        <input type="text" name="subject" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Priority</label>
                        <select name="priority" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                            <option value="low">Low</option>
                            <option value="medium" selected>Medium</option>
                            <option value="high">High</option>
                            <option value="urgent">Urgent</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                        <select name="status" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                            <option value="open">Open</option>
                            <option value="in_progress">In Progress</option>
                            <option value="resolved">Resolved</option>
                            <option value="closed">Closed</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Message</label>
                        <textarea name="message" rows="4" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400"></textarea>
                    </div>
                    
                    <div class="flex space-x-3 pt-4">
                        <button type="submit" class="flex-1 bg-gray-800 text-white py-2 rounded-lg hover:bg-gray-700 transition duration-200">
                            Add Request
                        </button>
                        <button type="button" onclick="closeModal('addRequestModal')" class="flex-1 bg-gray-200 text-gray-800 py-2 rounded-lg hover:bg-gray-300 transition duration-200">
                            Cancel
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Add Project Log Modal -->
    <div id="addLogModal" class="modal">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full mx-4 p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-xl font-medium">Add Project Log</h3>
                <button onclick="closeModal('addLogModal')" class="text-gray-500 hover:text-gray-700">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            
            <form method="POST" action="admin_actions.php">
                <input type="hidden" name="action" value="add_log">
                <input type="hidden" name="project_id" id="logProjectId">
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Log Type</label>
                        <select name="log_type" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                            <option value="update">Update</option>
                            <option value="status_change">Status Change</option>
                            <option value="milestone">Milestone</option>
                            <option value="note">Note</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Title</label>
                        <input type="text" name="title" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Description</label>
                        <textarea name="description" rows="4" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400"></textarea>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Created By</label>
                        <input type="text" name="created_by" value="AktasCorp Team" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div class="flex space-x-3 pt-4">
                        <button type="submit" class="flex-1 bg-gray-800 text-white py-2 rounded-lg hover:bg-gray-700 transition duration-200">
                            Add Log
                        </button>
                        <button type="button" onclick="closeModal('addLogModal')" class="flex-1 bg-gray-200 text-gray-800 py-2 rounded-lg hover:bg-gray-300 transition duration-200">
                            Cancel
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Respond to Request Modal -->
    <div id="respondModal" class="modal">
        <div class="bg-white rounded-lg shadow-lg max-w-lg w-full mx-4 p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-xl font-medium">Respond to Request</h3>
                <button onclick="closeModal('respondModal')" class="text-gray-500 hover:text-gray-700">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            
            <form method="POST" action="admin_actions.php">
                <input type="hidden" name="action" value="respond_request">
                <input type="hidden" name="request_id" id="respondRequestId">
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Response</label>
                        <textarea name="message" rows="6" required class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400" placeholder="Enter your response to the client..."></textarea>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Update Request Status</label>
                        <select name="new_status" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                            <option value="">Don't change status</option>
                            <option value="in_progress">In Progress</option>
                            <option value="resolved">Resolved</option>
                            <option value="closed">Closed</option>
                        </select>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Responder Name</label>
                        <input type="text" name="responder_name" value="AktasCorp Team" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-gray-400">
                    </div>
                    
                    <div class="flex space-x-3 pt-4">
                        <button type="submit" class="flex-1 bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700 transition duration-200">
                            Send Response
                        </button>
                        <button type="button" onclick="closeModal('respondModal')" class="flex-1 bg-gray-200 text-gray-800 py-2 rounded-lg hover:bg-gray-300 transition duration-200">
                            Cancel
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Client Management Modal -->
    <div id="clientManagementModal" class="modal">
        <div class="bg-white rounded-lg shadow-lg max-w-4xl w-full mx-4 p-6 max-h-[90vh] overflow-y-auto">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-xl font-medium">Client Management</h3>
                <button onclick="closeModal('clientManagementModal')" class="text-gray-500 hover:text-gray-700">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            <div id="clientManagementContent">
                <!-- Content will be loaded here -->
            </div>
        </div>
    </div>

    <script>
        function showTab(tabName) {
            // Hide all tab contents
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            // Remove active class from all tab buttons
            document.querySelectorAll('.tab-button').forEach(button => {
                button.classList.remove('active');
            });
            
            // Show selected tab content
            document.getElementById(tabName).classList.add('active');
            
            // Add active class to clicked button
            event.target.classList.add('active');
        }

        function showModal(modalId) {
            document.getElementById(modalId).classList.add('show');
        }

        function closeModal(modalId) {
            document.getElementById(modalId).classList.remove('show');
        }

        function manageClient(clientId) {
            // Load client management interface
            document.getElementById('clientManagementContent').innerHTML = `
                <div class="text-center py-8">
                    <p class="text-gray-600">Loading client management for ID: ${clientId}...</p>
                    <div class="mt-4">
                        <button onclick="addProjectForClient(${clientId})" class="bg-blue-600 text-white px-4 py-2 rounded-lg mr-2">Add Project</button>
                        <button onclick="addRequestForClient(${clientId})" class="bg-green-600 text-white px-4 py-2 rounded-lg mr-2">Add Request</button>
                        <button onclick="viewClientHistory(${clientId})" class="bg-gray-600 text-white px-4 py-2 rounded-lg">View History</button>
                    </div>
                </div>
            `;
            showModal('clientManagementModal');
        }

        function editClient(clientId) {
            alert('Edit client functionality - ID: ' + clientId);
        }

        function deleteClient(clientId) {
            if (confirm('Are you sure you want to delete this client?')) {
                window.location.href = 'admin_actions.php?action=delete_client&id=' + clientId;
            }
        }

        function manageProject(projectId) {
            alert('Manage project functionality - ID: ' + projectId);
        }

        function addProjectLog(projectId) {
            document.getElementById('logProjectId').value = projectId;
            showModal('addLogModal');
        }

        function viewRequest(requestId) {
            alert('View request functionality - ID: ' + requestId);
        }

        function respondToRequest(requestId) {
            document.getElementById('respondRequestId').value = requestId;
            showModal('respondModal');
        }

        function addProjectForClient(clientId) {
            closeModal('clientManagementModal');
            // Pre-select the client in the add project modal
            document.querySelector('#addProjectModal select[name="user_id"]').value = clientId;
            showModal('addProjectModal');
        }

        function addRequestForClient(clientId) {
            closeModal('clientManagementModal');
            // Pre-select the client in the add request modal
            document.querySelector('#addRequestModal select[name="user_id"]').value = clientId;
            showModal('addRequestModal');
        }

        function viewClientHistory(clientId) {
            alert('View client history - ID: ' + clientId);
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            if (event.target.classList.contains('modal')) {
                event.target.classList.remove('show');
            }
        }
    </script>

</body>
</html>