import * as react from 'react';
import { ReactNode } from 'react';
import { ButtonProps } from '@heroui/button';
import { AlertVariantProps, SlotsToClasses, AlertSlots } from '@heroui/theme';
import { PropGetter, HTMLHeroUIProps } from '@heroui/system';
import { ReactRef } from '@heroui/react-utils';

interface Props extends HTMLHeroUIProps<"div", "title"> {
    /**
     * Ref to the DOM node.
     */
    ref?: ReactRef<HTMLDivElement | null>;
    /**
     * title of the alert message
     */
    title?: ReactNode;
    /**
     * description of the alert message
     */
    description?: ReactNode;
    /**
     * Icon to be displayed in the alert - overrides the default icon
     */
    icon?: ReactNode;
    /**
     * Content to be displayed in the end side of the alert
     */
    endContent?: ReactNode;
    /**
     * Content to be displayed in the start side of the alert
     */
    startContent?: ReactNode;
    /**
     * Whether the alert is visible.
     * @default false
     */
    isVisible?: boolean;
    /**
     * Whether the alert should be visible by default.
     * @default false
     */
    isDefaultVisible?: boolean;
    /**
     * The event handler for the alert visibility state.
     * @param isVisible boolean
     * @returns void
     */
    onVisibleChange?: (isVisible: boolean) => void;
    /**
     *  whether the alert can be closed by user
     */
    isClosable?: boolean;
    /**
     * Props for the close button
     */
    closeButtonProps?: Omit<ButtonProps, "children">;
    /**
     * function which is called when close button is clicked
     */
    onClose?: () => void;
    /**
     * Classname or List of classes to change the classNames of the element.
     * if `className` is passed, it will be added to the base slot.
     *
     * @example
     * ```ts
     * <Alert classNames={{
     *    base:"base-classes",
     *    mainWrapper: "mainWrapper-classes"
     *    description: "description-classes"
     *    title: "title-classes"
     *    closeButton: "closeButton-classes"
     *    closeIcon: "closeIcon-classes"
     * }} />
     * ```
     */
    classNames?: SlotsToClasses<AlertSlots>;
}
type UseAlertProps = Props & AlertVariantProps;
declare function useAlert(originalProps: UseAlertProps): {
    title: ReactNode;
    icon: ReactNode;
    children: ReactNode;
    description: ReactNode;
    isClosable: boolean | undefined;
    domRef: react.RefObject<HTMLDivElement>;
    endContent: ReactNode;
    startContent: ReactNode;
    getBaseProps: PropGetter;
    getMainWrapperProps: PropGetter;
    getDescriptionProps: PropGetter;
    getTitleProps: PropGetter;
    color: any;
    getCloseButtonProps: PropGetter;
    handleClose: () => void;
    isVisible: boolean;
    onClose: (() => void) | undefined;
    getAlertIconProps: PropGetter;
    getIconWrapperProps: PropGetter;
};

export { type UseAlertProps, useAlert };
