<?php
session_start();
include '../db.php';

header('Content-Type: application/json');

if (!isset($_SESSION['merchant_id'])) {
    echo json_encode(['error' => 'unauthorized']);
    exit;
}

$shop_id = $_SESSION['shop_id'];

$visits_today_stmt = $conn->prepare("SELECT COUNT(*) as count FROM card_usage WHERE shop_id = ? AND DATE(used_at) = CURDATE()");
$visits_today_stmt->bind_param("i", $shop_id);
$visits_today_stmt->execute();
$visits_today = $visits_today_stmt->get_result()->fetch_assoc()['count'];

$visits_yesterday_stmt = $conn->prepare("SELECT COUNT(*) as count FROM card_usage WHERE shop_id = ? AND DATE(used_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)");
$visits_yesterday_stmt->bind_param("i", $shop_id);
$visits_yesterday_stmt->execute();
$visits_yesterday = $visits_yesterday_stmt->get_result()->fetch_assoc()['count'];

$visits_week_stmt = $conn->prepare("SELECT COUNT(*) as count FROM card_usage WHERE shop_id = ? AND used_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
$visits_week_stmt->bind_param("i", $shop_id);
$visits_week_stmt->execute();
$visits_week = $visits_week_stmt->get_result()->fetch_assoc()['count'];

$visits_month_stmt = $conn->prepare("SELECT COUNT(*) as count FROM card_usage WHERE shop_id = ? AND used_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
$visits_month_stmt->bind_param("i", $shop_id);
$visits_month_stmt->execute();
$visits_month = $visits_month_stmt->get_result()->fetch_assoc()['count'];

$change_percent = $visits_yesterday > 0 ? (($visits_today - $visits_yesterday) / $visits_yesterday) * 100 : 0;

$top_client_stmt = $conn->prepare("
    SELECT r.first_name, r.last_name, COUNT(*) as visit_count 
    FROM card_usage cu 
    JOIN residents r ON cu.resident_id = r.id 
    WHERE cu.shop_id = ? AND cu.used_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) 
    GROUP BY cu.resident_id 
    ORDER BY visit_count DESC 
    LIMIT 1
");
$top_client_stmt->bind_param("i", $shop_id);
$top_client_stmt->execute();
$top_client_result = $top_client_stmt->get_result();
$top_client = $top_client_result->num_rows > 0 ? $top_client_result->fetch_assoc() : null;

$recent_visits_stmt = $conn->prepare("
    SELECT cu.*, r.first_name, r.last_name 
    FROM card_usage cu 
    JOIN residents r ON cu.resident_id = r.id 
    WHERE cu.shop_id = ? 
    ORDER BY cu.used_at DESC 
    LIMIT 10
");
$recent_visits_stmt->bind_param("i", $shop_id);
$recent_visits_stmt->execute();
$recent_visits = $recent_visits_stmt->get_result();

$recent_visits_array = [];
while($visit = $recent_visits->fetch_assoc()) {
    $recent_visits_array[] = [
        'name' => $visit['first_name'] . ' ' . substr($visit['last_name'], 0, 1) . '.',
        'time' => date('H:i', strtotime($visit['used_at']))
    ];
}

$hourly_stats_stmt = $conn->prepare("
    SELECT HOUR(used_at) as hour, COUNT(*) as count 
    FROM card_usage 
    WHERE shop_id = ? AND DATE(used_at) = CURDATE() 
    GROUP BY HOUR(used_at) 
    ORDER BY hour
");
$hourly_stats_stmt->bind_param("i", $shop_id);
$hourly_stats_stmt->execute();
$hourly_stats = $hourly_stats_stmt->get_result();

$hourly_data = array_fill(0, 24, 0);
while ($row = $hourly_stats->fetch_assoc()) {
    $hourly_data[$row['hour']] = $row['count'];
}
$hourly_labels = array_map(function($h) { return $h . 'h'; }, range(0, 23));

$daily_stats_stmt = $conn->prepare("
    SELECT DATE(used_at) as date, COUNT(*) as count 
    FROM card_usage 
    WHERE shop_id = ? AND used_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) 
    GROUP BY DATE(used_at) 
    ORDER BY date
");
$daily_stats_stmt->bind_param("i", $shop_id);
$daily_stats_stmt->execute();
$daily_stats = $daily_stats_stmt->get_result();

$daily_labels = [];
$daily_counts = [];
while ($row = $daily_stats->fetch_assoc()) {
    $daily_labels[] = date('d/m', strtotime($row['date']));
    $daily_counts[] = $row['count'];
}

echo json_encode([
    'visits_today' => $visits_today,
    'visits_week' => $visits_week,
    'visits_month' => $visits_month,
    'change_percent' => $change_percent,
    'top_client' => $top_client ? [
        'name' => $top_client['first_name'] . ' ' . $top_client['last_name'],
        'count' => $top_client['visit_count']
    ] : null,
    'recent_visits' => $recent_visits_array,
    'hourly_labels' => $hourly_labels,
    'hourly_data' => $hourly_data,
    'daily_labels' => $daily_labels,
    'daily_data' => $daily_counts
]);