<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Générateur de zones de livraison</title>
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 p-4">
    <div class="max-w-6xl mx-auto">
        <h1 class="text-2xl font-bold mb-6">🗺️ Générateur de zones de livraison</h1>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div class="lg:col-span-2 bg-white p-6 rounded-lg shadow">
                <h2 class="text-lg font-bold mb-4">Dessine tes zones</h2>
                <div id="map" class="h-96 rounded border"></div>
                <div class="flex space-x-2 mt-4">
                    <button onclick="startDrawing()" id="draw-btn" class="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600">
                        Dessiner une zone
                    </button>
                    <button onclick="finishZone()" id="finish-btn" class="px-4 py-2 bg-green-500 text-white rounded hover:bg-green-600" disabled>
                        Terminer la zone
                    </button>
                    <button onclick="clearAll()" class="px-4 py-2 bg-red-500 text-white rounded hover:bg-red-600">
                        Tout effacer
                    </button>
                </div>
                <p class="text-sm text-gray-600 mt-2">
                    <span id="status">Clique sur "Dessiner une zone" puis clique sur la carte pour délimiter ta zone</span>
                </p>
            </div>
            
            <div class="bg-white p-6 rounded-lg shadow">
                <h2 class="text-lg font-bold mb-4">Zones créées</h2>
                <div id="zones-list" class="space-y-3 mb-4 max-h-40 overflow-y-auto">
                    <p class="text-gray-500 text-sm">Aucune zone créée</p>
                </div>
                
                <div id="current-zone-form" class="space-y-3 hidden">
                    <h3 class="font-medium">Zone en cours :</h3>
                    <input type="text" id="zone-name" placeholder="Nom du quartier" class="w-full p-2 border rounded">
                    <select id="zone-color" class="w-full p-2 border rounded">
                        <option value="#000000">Noir</option>
                        <option value="#dc2626">Rouge</option>
                        <option value="#2563eb">Bleu</option>
                        <option value="#16a34a">Vert</option>
                        <option value="#ca8a04">Orange</option>
                        <option value="#9333ea">Violet</option>
                    </select>
                </div>
                
                <div class="mt-6">
                    <h3 class="font-bold mb-2">Code JavaScript :</h3>
                    <textarea id="generated-code" class="w-full h-32 p-2 border rounded text-xs font-mono" readonly placeholder="Le code apparaîtra ici..."></textarea>
                    <button onclick="copyCode()" class="w-full bg-green-500 text-white py-2 rounded hover:bg-green-600 mt-2">
                        Copier tout le code
                    </button>
                </div>
            </div>
        </div>
        
        <div class="bg-blue-50 border border-blue-200 p-4 rounded mt-6">
            <h3 class="font-bold text-blue-800 mb-2">📋 Instructions :</h3>
            <ol class="list-decimal list-inside text-sm text-blue-700 space-y-1">
                <li>Clique sur "Dessiner une zone"</li>
                <li>Clique sur la carte pour délimiter ta zone (au moins 3 points)</li>
                <li>Clique sur "Terminer la zone"</li>
                <li>Donne un nom et choisis une couleur</li>
                <li>Répète pour créer d'autres zones</li>
                <li>Copie le code final et colle-le dans ton fichier livraison.html</li>
            </ol>
        </div>
    </div>

    <script>
        let map;
        let isDrawing = false;
        let currentPoints = [];
        let currentMarkers = [];
        let zones = [];
        let currentPolygon = null;

        const colors = ['#000000', '#dc2626', '#2563eb', '#16a34a', '#ca8a04', '#9333ea'];
        let colorIndex = 0;

        function initMap() {
            map = L.map('map').setView([14.7250, -61.0750], 12);
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '© OpenStreetMap'
            }).addTo(map);

            map.on('click', function(e) {
                if (isDrawing) {
                    addPoint(e.latlng.lat, e.latlng.lng);
                }
            });
        }

        function startDrawing() {
            isDrawing = true;
            currentPoints = [];
            currentMarkers = [];
            if (currentPolygon) {
                map.removeLayer(currentPolygon);
            }
            
            document.getElementById('draw-btn').disabled = true;
            document.getElementById('finish-btn').disabled = false;
            document.getElementById('current-zone-form').classList.remove('hidden');
            document.getElementById('status').textContent = 'Clique sur la carte pour délimiter ta zone (au moins 3 points)';
            
            document.getElementById('zone-color').value = colors[colorIndex % colors.length];
        }

        function addPoint(lat, lng) {
            currentPoints.push([lat, lng]);
            
            const marker = L.marker([lat, lng], {
                icon: L.divIcon({
                    className: 'point-marker',
                    html: `<div style="background: red; color: white; width: 20px; height: 20px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 12px; font-weight: bold;">${currentPoints.length}</div>`,
                    iconSize: [20, 20],
                    iconAnchor: [10, 10]
                })
            }).addTo(map);
            
            currentMarkers.push(marker);
            
            if (currentPoints.length >= 3) {
                updatePolygon();
                document.getElementById('status').textContent = `${currentPoints.length} points placés. Tu peux terminer la zone ou continuer.`;
            } else {
                document.getElementById('status').textContent = `${currentPoints.length} points placés. Il faut au moins 3 points.`;
            }
        }

        function updatePolygon() {
            if (currentPolygon) {
                map.removeLayer(currentPolygon);
            }
            
            if (currentPoints.length >= 3) {
                const color = document.getElementById('zone-color').value;
                currentPolygon = L.polygon(currentPoints, {
                    color: color,
                    weight: 2,
                    fillOpacity: 0.3
                }).addTo(map);
            }
        }

        function finishZone() {
            if (currentPoints.length < 3) {
                alert('Il faut au moins 3 points pour créer une zone');
                return;
            }
            
            const zoneName = document.getElementById('zone-name').value.trim();
            if (!zoneName) {
                alert('Donne un nom à ta zone');
                return;
            }
            
            const color = document.getElementById('zone-color').value;
            
            // Calculer les bounds
            const lats = currentPoints.map(p => p[0]);
            const lngs = currentPoints.map(p => p[1]);
            const minLat = Math.min(...lats);
            const maxLat = Math.max(...lats);
            const minLng = Math.min(...lngs);
            const maxLng = Math.max(...lngs);
            const centerLat = (minLat + maxLat) / 2;
            const centerLng = (minLng + maxLng) / 2;
            
            // Sauvegarder la zone
            zones.push({
                name: zoneName,
                points: [...currentPoints],
                bounds: [[minLat, minLng], [maxLat, maxLng]],
                center: [centerLat, centerLng],
                color: color
            });
            
            // Nettoyer les marqueurs temporaires
            currentMarkers.forEach(marker => map.removeLayer(marker));
            
            // Créer le polygone final
            if (currentPolygon) {
                map.removeLayer(currentPolygon);
            }
            
            const finalPolygon = L.polygon(currentPoints, {
                color: color,
                weight: 2,
                fillOpacity: 0.2
            }).addTo(map);
            
            finalPolygon.bindPopup(`<strong>${zoneName}</strong>`);
            
            // Reset
            isDrawing = false;
            currentPoints = [];
            currentMarkers = [];
            currentPolygon = null;
            colorIndex++;
            
            document.getElementById('draw-btn').disabled = false;
            document.getElementById('finish-btn').disabled = true;
            document.getElementById('current-zone-form').classList.add('hidden');
            document.getElementById('zone-name').value = '';
            document.getElementById('status').textContent = 'Zone créée ! Tu peux en créer une autre.';
            
            updateZonesList();
            generateCode();
        }

        function updateZonesList() {
            const zonesList = document.getElementById('zones-list');
            
            if (zones.length === 0) {
                zonesList.innerHTML = '<p class="text-gray-500 text-sm">Aucune zone créée</p>';
                return;
            }
            
            zonesList.innerHTML = zones.map((zone, index) => `
                <div class="flex items-center justify-between p-2 border rounded">
                    <div class="flex items-center space-x-2">
                        <div class="w-4 h-4 rounded" style="background: ${zone.color}"></div>
                        <span class="text-sm font-medium">${zone.name}</span>
                    </div>
                    <button onclick="removeZone(${index})" class="text-red-500 hover:text-red-700 text-sm">
                        Supprimer
                    </button>
                </div>
            `).join('');
        }

        function removeZone(index) {
            zones.splice(index, 1);
            updateZonesList();
            generateCode();
            
            // Redessiner la carte
            map.eachLayer(layer => {
                if (layer instanceof L.Polygon) {
                    map.removeLayer(layer);
                }
            });
            
            zones.forEach(zone => {
                const polygon = L.polygon(zone.points, {
                    color: zone.color,
                    weight: 2,
                    fillOpacity: 0.2
                }).addTo(map);
                polygon.bindPopup(`<strong>${zone.name}</strong>`);
            });
        }

        function generateCode() {
            if (zones.length === 0) {
                document.getElementById('generated-code').value = '';
                return;
            }
            
            const codeLines = zones.map(zone => 
                `    '${zone.name}': {
        bounds: [[${zone.bounds[0][0].toFixed(6)}, ${zone.bounds[0][1].toFixed(6)}], [${zone.bounds[1][0].toFixed(6)}, ${zone.bounds[1][1].toFixed(6)}]],
        center: [${zone.center[0].toFixed(6)}, ${zone.center[1].toFixed(6)}],
        color: '${zone.color}'
    }`
            );
            
            const finalCode = `const deliveryZones = {
${codeLines.join(',\n')}
};`;
            
            document.getElementById('generated-code').value = finalCode;
        }

        function copyCode() {
            const textarea = document.getElementById('generated-code');
            if (textarea.value.trim() === '') {
                alert('Crée d\'abord des zones');
                return;
            }
            
            textarea.select();
            document.execCommand('copy');
            alert('Code copié ! Colle-le dans ton fichier livraison.html');
        }

        function clearAll() {
            if (zones.length === 0 && !isDrawing) return;
            
            if (confirm('Supprimer toutes les zones ?')) {
                zones = [];
                isDrawing = false;
                currentPoints = [];
                
                // Nettoyer la carte
                map.eachLayer(layer => {
                    if (layer instanceof L.Polygon || layer instanceof L.Marker) {
                        map.removeLayer(layer);
                    }
                });
                
                currentMarkers.forEach(marker => map.removeLayer(marker));
                currentMarkers = [];
                
                if (currentPolygon) {
                    map.removeLayer(currentPolygon);
                    currentPolygon = null;
                }
                
                document.getElementById('draw-btn').disabled = false;
                document.getElementById('finish-btn').disabled = true;
                document.getElementById('current-zone-form').classList.add('hidden');
                document.getElementById('zone-name').value = '';
                document.getElementById('status').textContent = 'Clique sur "Dessiner une zone" puis clique sur la carte pour délimiter ta zone';
                
                updateZonesList();
                generateCode();
                colorIndex = 0;
            }
        }

        document.getElementById('zone-color').addEventListener('change', function() {
            if (currentPolygon) {
                updatePolygon();
            }
        });

        initMap();
    </script>
</body>
</html>