<?php
session_start();

if(!isset($_SESSION['user_id']) || $_SESSION['user_id'] != 1) {
    header("Location: login.php");
    exit();
}

require_once '../db.php';
require_once 'functions_loterie.php';

$user_id = $_SESSION['user_id'];
$sql = "SELECT * FROM users WHERE id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();

$message = '';
$success = false;

if(isset($_POST['action'])) {
    switch($_POST['action']) {
        case 'creer_loterie':
            $nom = $_POST['nom_loterie'];
            $date_tirage = $_POST['date_tirage'];
            $num1 = $_POST['numero_1'];
            $num2 = $_POST['numero_2'];
            $num3 = $_POST['numero_3'];
            $jackpot = $_POST['jackpot_initial'];
            
            if(empty($nom) || empty($date_tirage) || empty($num1) || empty($num2) || empty($num3)) {
                $message = "Tous les champs sont obligatoires.";
            } elseif($num1 == $num2 || $num1 == $num3 || $num2 == $num3) {
                $message = "Les trois numéros doivent être différents.";
            } elseif($num1 < 1 || $num1 > 10 || $num2 < 1 || $num2 > 10 || $num3 < 1 || $num3 > 10) {
                $message = "Les numéros doivent être entre 1 et 10.";
            } else {
                $numeros_string = $num1 . '-' . $num2 . '-' . $num3;
                $cle_cryptage = 'peerkinton_secret_' . date('Y');
                $numeros_cryptes = crypterNumeros($numeros_string, $cle_cryptage);
                
                $sql = "INSERT INTO loteries (nom, date_tirage, numeros_gagnants, cle_cryptage, jackpot, statut) VALUES (?, ?, ?, ?, ?, 'ouvert')";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssssd", $nom, $date_tirage, $numeros_cryptes, $cle_cryptage, $jackpot);
                
                if($stmt->execute()) {
                    $success = true;
                    $message = "Loterie créée avec succès ! Numéros gagnants : " . $numeros_string;
                } else {
                    $message = "Erreur lors de la création de la loterie.";
                }
            }
            break;
            
        case 'forcer_tirage':
            $loterie_id = $_POST['loterie_id'];
            
            $sql = "UPDATE loteries SET date_tirage = CURDATE() WHERE id = ? AND statut = 'ouvert'";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("i", $loterie_id);
            
            if($stmt->execute() && $stmt->affected_rows > 0) {
                ob_start();
                $_GET['force_tirage'] = $loterie_id;
                include 'cron_tirage_loterie.php';
                $tirage_output = ob_get_clean();
                
                $success = true;
                $message = "Tirage forcé avec succès !";
            } else {
                $message = "Erreur : loterie introuvable ou déjà tirée.";
            }
            break;
            
        case 'generer_auto':
            $numeros_string = genererNumerosAleatoires();
            $nom = "Loterie Auto #" . date('W');
            $date_tirage = date('Y-m-d', strtotime('+7 days'));
            $jackpot = 1000;
            $cle_cryptage = 'peerkinton_secret_' . date('Y');
            $numeros_cryptes = crypterNumeros($numeros_string, $cle_cryptage);
            
            $sql = "INSERT INTO loteries (nom, date_tirage, numeros_gagnants, cle_cryptage, jackpot, statut) VALUES (?, ?, ?, ?, ?, 'ouvert')";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("ssssd", $nom, $date_tirage, $numeros_cryptes, $cle_cryptage, $jackpot);
            
            if($stmt->execute()) {
                $success = true;
                $message = "Loterie automatique créée ! Date: $date_tirage, Numéros: $numeros_string";
            } else {
                $message = "Erreur lors de la création automatique.";
            }
            break;
    }
}

$onglet = isset($_GET['tab']) ? $_GET['tab'] : 'overview';
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Administration Loterie - Peerkinton</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" rel="stylesheet">
    <style>
        body {
            font-family: 'Poppins', sans-serif;
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }
    </style>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'sans': ['Poppins', 'sans-serif'],
                    },
                    colors: {
                        primary: {
                            50: '#f0f9ff',
                            100: '#e0f2fe',
                            200: '#bae6fd',
                            300: '#7dd3fc',
                            400: '#38bdf8',
                            500: '#0ea5e9',
                            600: '#0284c7',
                            700: '#0369a1',
                            800: '#075985',
                            900: '#0c4a6e',
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-gray-50 text-gray-900">
    <div class="min-h-screen flex flex-col">
        <nav class="bg-white shadow">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div class="flex justify-between h-16">
                    <div class="flex">
                        <div class="flex-shrink-0 flex items-center">
                            <img src="../logo.png" alt="Peerkinton Logo" class="h-8">
                            <a href="index.php" class="ml-4 flex items-center px-3 py-2 text-gray-700 hover:text-primary-600 transition-colors">
                                <i class="fas fa-arrow-left mr-2"></i>
                                Retour Dashboard
                            </a>
                        </div>
                    </div>
                    <div class="flex items-center">
                        <div class="ml-3 relative">
                            <div class="flex items-center space-x-4">
                                <div class="bg-red-100 text-red-800 px-3 py-1 rounded-full text-xs font-semibold">
                                    <i class="fas fa-crown mr-1"></i>ADMIN
                                </div>
                                <span class="text-sm font-medium"><?php echo htmlspecialchars($user['email']); ?></span>
                                <a href="logout.php" class="text-gray-500 hover:text-gray-700">
                                    <i class="fas fa-sign-out-alt"></i>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </nav>

        <div class="flex-grow py-6">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div class="mb-6">
                    <h1 class="text-3xl font-bold text-gray-900 flex items-center">
                        <i class="fas fa-dice mr-3 text-purple-600"></i>
                        Administration Loterie
                    </h1>
                    <p class="mt-1 text-sm text-gray-600">Gestion complète du système de loterie Peerkinton</p>
                </div>

                <?php if($message): ?>
                    <div class="mb-6 px-4 py-3 rounded-lg <?php echo $success ? 'bg-green-50 border border-green-200 text-green-800' : 'bg-red-50 border border-red-200 text-red-800'; ?>">
                        <div class="flex items-center">
                            <i class="<?php echo $success ? 'fas fa-check-circle' : 'fas fa-exclamation-triangle'; ?> mr-2"></i>
                            <?php echo htmlspecialchars($message); ?>
                        </div>
                    </div>
                <?php endif; ?>

                <div class="border-b border-gray-200 mb-6">
                    <nav class="-mb-px flex space-x-8">
                        <button onclick="showTab('overview')" class="tab-link <?php echo $onglet == 'overview' ? 'border-primary-500 text-primary-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?> whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm">
                            <i class="fas fa-chart-line mr-2"></i> Vue d'ensemble
                        </button>
                        <button onclick="showTab('create')" class="tab-link <?php echo $onglet == 'create' ? 'border-primary-500 text-primary-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?> whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm">
                            <i class="fas fa-plus-circle mr-2"></i> Créer Loterie
                        </button>
                        <button onclick="showTab('manage')" class="tab-link <?php echo $onglet == 'manage' ? 'border-primary-500 text-primary-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?> whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm">
                            <i class="fas fa-cogs mr-2"></i> Gérer Loteries
                        </button>
                        <button onclick="showTab('stats')" class="tab-link <?php echo $onglet == 'stats' ? 'border-primary-500 text-primary-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'; ?> whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm">
                            <i class="fas fa-chart-pie mr-2"></i> Statistiques
                        </button>
                    </nav>
                </div>

                <!-- VUE D'ENSEMBLE -->
                <div class="tab-content <?php echo $onglet == 'overview' ? 'active' : ''; ?>" id="tab-overview">
                    <?php
                    // Statistiques globales
                    $sql = "SELECT COUNT(*) as total_loteries FROM loteries";
                    $total_loteries = $conn->query($sql)->fetch_assoc()['total_loteries'];
                    
                    $sql = "SELECT COUNT(*) as loteries_actives FROM loteries WHERE statut = 'ouvert'";
                    $loteries_actives = $conn->query($sql)->fetch_assoc()['loteries_actives'];
                    
                    $sql = "SELECT SUM(prix_paye) as revenus_total FROM loterie_participations";
                    $revenus_total = $conn->query($sql)->fetch_assoc()['revenus_total'] ?: 0;
                    
                    $sql = "SELECT SUM(gains) as gains_distribues FROM loterie_participations WHERE gains > 0";
                    $gains_distribues = $conn->query($sql)->fetch_assoc()['gains_distribues'] ?: 0;
                    ?>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                        <div class="bg-white overflow-hidden shadow rounded-lg">
                            <div class="p-5">
                                <div class="flex items-center">
                                    <div class="flex-shrink-0">
                                        <i class="fas fa-dice text-2xl text-purple-600"></i>
                                    </div>
                                    <div class="ml-5 w-0 flex-1">
                                        <dl>
                                            <dt class="text-sm font-medium text-gray-500 truncate">Total Loteries</dt>
                                            <dd class="text-lg font-medium text-gray-900"><?php echo $total_loteries; ?></dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="bg-white overflow-hidden shadow rounded-lg">
                            <div class="p-5">
                                <div class="flex items-center">
                                    <div class="flex-shrink-0">
                                        <i class="fas fa-play-circle text-2xl text-green-600"></i>
                                    </div>
                                    <div class="ml-5 w-0 flex-1">
                                        <dl>
                                            <dt class="text-sm font-medium text-gray-500 truncate">Loteries Actives</dt>
                                            <dd class="text-lg font-medium text-gray-900"><?php echo $loteries_actives; ?></dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="bg-white overflow-hidden shadow rounded-lg">
                            <div class="p-5">
                                <div class="flex items-center">
                                    <div class="flex-shrink-0">
                                        <i class="fas fa-coins text-2xl text-yellow-600"></i>
                                    </div>
                                    <div class="ml-5 w-0 flex-1">
                                        <dl>
                                            <dt class="text-sm font-medium text-gray-500 truncate">Revenus Total</dt>
                                            <dd class="text-lg font-medium text-gray-900"><?php echo number_format($revenus_total, 0, ',', ' '); ?> F&</dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="bg-white overflow-hidden shadow rounded-lg">
                            <div class="p-5">
                                <div class="flex items-center">
                                    <div class="flex-shrink-0">
                                        <i class="fas fa-trophy text-2xl text-orange-600"></i>
                                    </div>
                                    <div class="ml-5 w-0 flex-1">
                                        <dl>
                                            <dt class="text-sm font-medium text-gray-500 truncate">Gains Distribués</dt>
                                            <dd class="text-lg font-medium text-gray-900"><?php echo number_format($gains_distribues, 0, ',', ' '); ?> F&</dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Loteries récentes -->
                    <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                        <div class="px-4 py-5 sm:px-6">
                            <h3 class="text-lg leading-6 font-medium text-gray-900">Loteries Récentes</h3>
                        </div>
                        <div class="border-t border-gray-200">
                            <div class="overflow-x-auto">
                                <table class="min-w-full divide-y divide-gray-200">
                                    <thead class="bg-gray-50">
                                        <tr>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nom</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date Tirage</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Jackpot</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Statut</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Participants</th>
                                        </tr>
                                    </thead>
                                    <tbody class="bg-white divide-y divide-gray-200">
                                        <?php
                                        $sql = "SELECT l.*, COUNT(lp.id) as nb_participants 
                                                FROM loteries l 
                                                LEFT JOIN loterie_participations lp ON l.id = lp.loterie_id 
                                                GROUP BY l.id 
                                                ORDER BY l.date_creation DESC 
                                                LIMIT 10";
                                        $result = $conn->query($sql);
                                        
                                        while($loterie = $result->fetch_assoc()) {
                                            $statut_class = '';
                                            $statut_text = '';
                                            switch($loterie['statut']) {
                                                case 'ouvert':
                                                    $statut_class = 'bg-green-100 text-green-800';
                                                    $statut_text = 'Ouvert';
                                                    break;
                                                case 'ferme':
                                                    $statut_class = 'bg-yellow-100 text-yellow-800';
                                                    $statut_text = 'Fermé';
                                                    break;
                                                case 'tire':
                                                    $statut_class = 'bg-blue-100 text-blue-800';
                                                    $statut_text = 'Tiré';
                                                    break;
                                            }
                                            
                                            echo '<tr>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">' . $loterie['id'] . '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">' . htmlspecialchars($loterie['nom']) . '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">' . date('d/m/Y', strtotime($loterie['date_tirage'])) . '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-semibold">' . number_format($loterie['jackpot'], 0, ',', ' ') . ' F&</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap">';
                                            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ' . $statut_class . '">' . $statut_text . '</span>';
                                            echo '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">' . $loterie['nb_participants'] . '</td>';
                                            echo '</tr>';
                                        }
                                        ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- CRÉER LOTERIE -->
                <div class="tab-content <?php echo $onglet == 'create' ? 'active' : ''; ?>" id="tab-create">
                    <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
                        <!-- Création manuelle -->
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    <i class="fas fa-edit mr-2 text-primary-600"></i>Création Manuelle
                                </h3>
                                <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                    Créez une loterie avec des paramètres personnalisés.
                                </p>
                            </div>
                            
                            <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                <form action="" method="POST" class="space-y-6">
                                    <input type="hidden" name="action" value="creer_loterie">
                                    
                                    <div>
                                        <label for="nom_loterie" class="block text-sm font-medium text-gray-700">Nom de la loterie</label>
                                        <input type="text" name="nom_loterie" id="nom_loterie" required class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-primary-500 focus:border-primary-500 sm:text-sm" placeholder="Ex: Loterie Spéciale Noël">
                                    </div>
                                    
                                    <div>
                                        <label for="date_tirage" class="block text-sm font-medium text-gray-700">Date du tirage</label>
                                        <input type="date" name="date_tirage" id="date_tirage" required min="<?php echo date('Y-m-d', strtotime('+1 day')); ?>" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-primary-500 focus:border-primary-500 sm:text-sm">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-3">Numéros gagnants (seront cryptés)</label>
                                        <div class="grid grid-cols-3 gap-4">
                                            <div>
                                                <label class="block text-xs text-gray-500 mb-1">1er numéro</label>
                                                <select name="numero_1" required class="w-full border-gray-300 rounded-md shadow-sm focus:ring-primary-500 focus:border-primary-500 text-center font-bold">
                                                    <?php for($i = 1; $i <= 10; $i++): ?>
                                                        <option value="<?php echo $i; ?>"><?php echo $i; ?></option>
                                                    <?php endfor; ?>
                                                </select>
                                            </div>
                                            <div>
                                                <label class="block text-xs text-gray-500 mb-1">2ème numéro</label>
                                                <select name="numero_2" required class="w-full border-gray-300 rounded-md shadow-sm focus:ring-primary-500 focus:border-primary-500 text-center font-bold">
                                                    <?php for($i = 1; $i <= 10; $i++): ?>
                                                        <option value="<?php echo $i; ?>"><?php echo $i; ?></option>
                                                    <?php endfor; ?>
                                                </select>
                                            </div>
                                            <div>
                                                <label class="block text-xs text-gray-500 mb-1">3ème numéro</label>
                                                <select name="numero_3" required class="w-full border-gray-300 rounded-md shadow-sm focus:ring-primary-500 focus:border-primary-500 text-center font-bold">
                                                    <?php for($i = 1; $i <= 10; $i++): ?>
                                                        <option value="<?php echo $i; ?>"><?php echo $i; ?></option>
                                                    <?php endfor; ?>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div>
                                        <label for="jackpot_initial" class="block text-sm font-medium text-gray-700">Jackpot initial (F&)</label>
                                        <input type="number" name="jackpot_initial" id="jackpot_initial" required min="100" value="1000" class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-primary-500 focus:border-primary-500 sm:text-sm">
                                    </div>
                                    
                                    <div>
                                        <button type="submit" class="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-primary-600 hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500">
                                            <i class="fas fa-plus mr-2"></i>
                                            Créer la Loterie
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                        
                        <!-- Création automatique -->
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    <i class="fas fa-magic mr-2 text-purple-600"></i>Création Automatique
                                </h3>
                                <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                    Génération automatique avec numéros aléatoires.
                                </p>
                            </div>
                            
                            <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                <div class="text-center">
                                    <div class="bg-purple-100 p-6 rounded-full w-24 h-24 mx-auto mb-6 flex items-center justify-center">
                                        <i class="fas fa-dice text-3xl text-purple-600"></i>
                                    </div>
                                    
                                    <h4 class="text-lg font-semibold text-gray-900 mb-4">Génération Automatique</h4>
                                    
                                    <div class="bg-gray-50 rounded-lg p-4 mb-6">
                                        <div class="grid grid-cols-2 gap-4 text-sm">
                                            <div>
                                                <span class="text-gray-600">Nom :</span>
                                                <span class="font-medium">Loterie Auto #<?php echo date('W'); ?></span>
                                            </div>
                                            <div>
                                                <span class="text-gray-600">Date :</span>
                                                <span class="font-medium"><?php echo date('d/m/Y', strtotime('+7 days')); ?></span>
                                            </div>
                                            <div>
                                                <span class="text-gray-600">Jackpot :</span>
                                                <span class="font-medium">1,000 F&</span>
                                            </div>
                                            <div>
                                                <span class="text-gray-600">Numéros :</span>
                                                <span class="font-medium">Aléatoires</span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <form action="" method="POST">
                                        <input type="hidden" name="action" value="generer_auto">
                                        <button type="submit" class="w-full flex justify-center py-3 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-purple-600 hover:bg-purple-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-purple-500">
                                            <i class="fas fa-magic mr-2"></i>
                                            Générer Automatiquement
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- GÉRER LOTERIES -->
                <div class="tab-content <?php echo $onglet == 'manage' ? 'active' : ''; ?>" id="tab-manage">
                    <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                        <div class="px-4 py-5 sm:px-6">
                            <h3 class="text-lg leading-6 font-medium text-gray-900">
                                <i class="fas fa-cogs mr-2 text-primary-600"></i>Gestion des Loteries
                            </h3>
                            <p class="mt-1 max-w-2xl text-sm text-gray-500">
                                Administrez toutes les loteries existantes.
                            </p>
                        </div>
                        
                        <div class="border-t border-gray-200">
                            <div class="overflow-x-auto">
                                <table class="min-w-full divide-y divide-gray-200">
                                    <thead class="bg-gray-50">
                                        <tr>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nom</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date Tirage</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Jackpot</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Participants</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Statut</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody class="bg-white divide-y divide-gray-200">
                                        <?php
                                        $sql = "SELECT l.*, COUNT(lp.id) as nb_participants, 
                                                       SUM(CASE WHEN lp.gagnant = 'oui' THEN 1 ELSE 0 END) as nb_gagnants
                                                FROM loteries l 
                                                LEFT JOIN loterie_participations lp ON l.id = lp.loterie_id 
                                                GROUP BY l.id 
                                                ORDER BY l.date_tirage DESC";
                                        $result = $conn->query($sql);
                                        
                                        while($loterie = $result->fetch_assoc()) {
                                            $statut_class = '';
                                            $statut_text = '';
                                            $can_force_tirage = false;
                                            
                                            switch($loterie['statut']) {
                                                case 'ouvert':
                                                    $statut_class = 'bg-green-100 text-green-800';
                                                    $statut_text = 'Ouvert';
                                                    $can_force_tirage = true;
                                                    break;
                                                case 'ferme':
                                                    $statut_class = 'bg-yellow-100 text-yellow-800';
                                                    $statut_text = 'Fermé';
                                                    break;
                                                case 'tire':
                                                    $statut_class = 'bg-blue-100 text-blue-800';
                                                    $statut_text = 'Tiré';
                                                    break;
                                            }
                                            
                                            echo '<tr>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">' . $loterie['id'] . '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">' . htmlspecialchars($loterie['nom']) . '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">' . date('d/m/Y', strtotime($loterie['date_tirage'])) . '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-semibold">' . number_format($loterie['jackpot'], 0, ',', ' ') . ' F&</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">' . $loterie['nb_participants'] . '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap">';
                                            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ' . $statut_class . '">' . $statut_text . '</span>';
                                            echo '</td>';
                                            echo '<td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">';
                                            
                                            if($can_force_tirage) {
                                                echo '<form action="" method="POST" class="inline">';
                                                echo '<input type="hidden" name="action" value="forcer_tirage">';
                                                echo '<input type="hidden" name="loterie_id" value="' . $loterie['id'] . '">';
                                                echo '<button type="submit" onclick="return confirm(\'Êtes-vous sûr de vouloir forcer le tirage ?\')" class="text-red-600 hover:text-red-900">';
                                                echo '<i class="fas fa-play-circle mr-1"></i>Forcer Tirage';
                                                echo '</button>';
                                                echo '</form>';
                                            }
                                            
                                            if($loterie['statut'] == 'tire') {
                                                echo '<button onclick="showResults(' . $loterie['id'] . ')" class="text-blue-600 hover:text-blue-900 ml-2">';
                                                echo '<i class="fas fa-eye mr-1"></i>Voir Résultats';
                                                echo '</button>';
                                            }
                                            
                                            echo '</td>';
                                            echo '</tr>';
                                        }
                                        ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- STATISTIQUES -->
                <div class="tab-content <?php echo $onglet == 'stats' ? 'active' : ''; ?>" id="tab-stats">
                    <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
                        <!-- Revenus par mois -->
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    <i class="fas fa-chart-line mr-2 text-green-600"></i>Revenus Mensuels
                                </h3>
                            </div>
                            <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                <?php
                                $sql = "SELECT YEAR(lp.date_participation) as annee, MONTH(lp.date_participation) as mois, SUM(lp.prix_paye) as revenus
                                        FROM loterie_participations lp 
                                        WHERE lp.date_participation >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
                                        GROUP BY YEAR(lp.date_participation), MONTH(lp.date_participation)
                                        ORDER BY annee DESC, mois DESC
                                        LIMIT 6";
                                $result = $conn->query($sql);
                                ?>
                                
                                <div class="space-y-3">
                                    <?php while($row = $result->fetch_assoc()): ?>
                                        <?php 
                                        $mois_nom = date('F Y', mktime(0, 0, 0, $row['mois'], 1, $row['annee']));
                                        $pourcentage = ($row['revenus'] / 5000) * 100; // Base 5000 pour la barre
                                        ?>
                                        <div>
                                            <div class="flex justify-between text-sm">
                                                <span class="font-medium text-gray-900"><?php echo $mois_nom; ?></span>
                                                <span class="text-gray-600"><?php echo number_format($row['revenus'], 0, ',', ' '); ?> F&</span>
                                            </div>
                                            <div class="mt-1 bg-gray-200 rounded-full h-2">
                                                <div class="bg-green-600 h-2 rounded-full" style="width: <?php echo min($pourcentage, 100); ?>%"></div>
                                            </div>
                                        </div>
                                    <?php endwhile; ?>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Top joueurs -->
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    <i class="fas fa-users mr-2 text-blue-600"></i>Top Joueurs
                                </h3>
                            </div>
                            <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                <?php
                                $sql = "SELECT u.email, COUNT(lp.id) as nb_participations, SUM(lp.prix_paye) as total_depense, SUM(lp.gains) as total_gains
                                        FROM loterie_participations lp
                                        JOIN users u ON lp.user_id = u.id
                                        GROUP BY lp.user_id
                                        ORDER BY nb_participations DESC
                                        LIMIT 10";
                                $result = $conn->query($sql);
                                ?>
                                
                                <div class="space-y-4">
                                    <?php $position = 1; ?>
                                    <?php while($row = $result->fetch_assoc()): ?>
                                        <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                            <div class="flex items-center">
                                                <div class="flex-shrink-0 w-8 h-8 bg-primary-600 text-white rounded-full flex items-center justify-center text-sm font-bold">
                                                    <?php echo $position; ?>
                                                </div>
                                                <div class="ml-3">
                                                    <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($row['email']); ?></p>
                                                    <p class="text-xs text-gray-500"><?php echo $row['nb_participations']; ?> participations</p>
                                                </div>
                                            </div>
                                            <div class="text-right">
                                                <p class="text-sm font-semibold text-gray-900">
                                                    <?php echo number_format($row['total_gains'] - $row['total_depense'], 0, ',', ' '); ?> F&
                                                </p>
                                                <p class="text-xs text-gray-500">
                                                    <?php echo $row['total_gains'] > $row['total_depense'] ? 'Bénéfice' : 'Perte'; ?>
                                                </p>
                                            </div>
                                        </div>
                                        <?php $position++; ?>
                                    <?php endwhile; ?>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Taux de gain -->
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    <i class="fas fa-percentage mr-2 text-purple-600"></i>Taux de Gains
                                </h3>
                            </div>
                            <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                <?php
                                $sql = "SELECT 
                                    COUNT(*) as total_participations,
                                    SUM(CASE WHEN gagnant = 'oui' THEN 1 ELSE 0 END) as total_gagnants,
                                    SUM(CASE WHEN gains >= 200 THEN 1 ELSE 0 END) as jackpots_partiels,
                                    SUM(CASE WHEN gains >= 1000 THEN 1 ELSE 0 END) as jackpots_complets
                                    FROM loterie_participations";
                                $stats = $conn->query($sql)->fetch_assoc();
                                
                                $taux_gain = $stats['total_participations'] > 0 ? 
                                    ($stats['total_gagnants'] / $stats['total_participations']) * 100 : 0;
                                ?>
                                
                                <div class="space-y-4">
                                    <div class="text-center">
                                        <div class="text-3xl font-bold text-purple-600"><?php echo number_format($taux_gain, 1); ?>%</div>
                                        <div class="text-sm text-gray-500">Taux de gain global</div>
                                    </div>
                                    
                                    <div class="grid grid-cols-2 gap-4 text-center">
                                        <div class="bg-yellow-50 p-3 rounded-lg">
                                            <div class="text-lg font-semibold text-yellow-600"><?php echo $stats['jackpots_partiels']; ?></div>
                                            <div class="text-xs text-gray-600">Gains partiels</div>
                                        </div>
                                        <div class="bg-green-50 p-3 rounded-lg">
                                            <div class="text-lg font-semibold text-green-600"><?php echo $stats['jackpots_complets']; ?></div>
                                            <div class="text-xs text-gray-600">Jackpots</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Prochains tirages -->
                        <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                            <div class="px-4 py-5 sm:px-6">
                                <h3 class="text-lg leading-6 font-medium text-gray-900">
                                    <i class="fas fa-calendar-alt mr-2 text-orange-600"></i>Prochains Tirages
                                </h3>
                            </div>
                            <div class="border-t border-gray-200 px-4 py-5 sm:p-6">
                                <?php
                                $sql = "SELECT nom, date_tirage, jackpot FROM loteries 
                                        WHERE statut = 'ouvert' AND date_tirage >= CURDATE() 
                                        ORDER BY date_tirage ASC LIMIT 5";
                                $result = $conn->query($sql);
                                ?>
                                
                                <div class="space-y-3">
                                    <?php if($result->num_rows > 0): ?>
                                        <?php while($row = $result->fetch_assoc()): ?>
                                            <div class="flex justify-between items-center p-3 bg-orange-50 rounded-lg">
                                                <div>
                                                    <p class="text-sm font-medium text-gray-900"><?php echo htmlspecialchars($row['nom']); ?></p>
                                                    <p class="text-xs text-gray-500"><?php echo date('d/m/Y', strtotime($row['date_tirage'])); ?></p>
                                                </div>
                                                <div class="text-right">
                                                    <p class="text-sm font-semibold text-orange-600">
                                                        <?php echo number_format($row['jackpot'], 0, ',', ' '); ?> F&
                                                    </p>
                                                </div>
                                            </div>
                                        <?php endwhile; ?>
                                    <?php else: ?>
                                        <div class="text-center text-gray-500 py-8">
                                            <i class="fas fa-calendar-times text-3xl mb-3"></i>
                                            <p>Aucun tirage programmé</p>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal pour les résultats -->
    <div id="resultModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden">
        <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-3/4 lg:w-1/2 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="text-lg font-medium text-gray-900" id="modalTitle">Résultats du Tirage</h3>
                    <button onclick="closeModal()" class="text-gray-400 hover:text-gray-600">
                        <i class="fas fa-times text-xl"></i>
                    </button>
                </div>
                <div id="modalContent" class="space-y-4">
                    <!-- Contenu chargé dynamiquement -->
                </div>
            </div>
        </div>
    </div>

    <script>
        function showTab(tabName) {
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            document.querySelectorAll('.tab-link').forEach(link => {
                link.classList.remove('border-primary-500', 'text-primary-600');
                link.classList.add('border-transparent', 'text-gray-500');
            });
            
            document.getElementById('tab-' + tabName).classList.add('active');
            event.target.classList.remove('border-transparent', 'text-gray-500');
            event.target.classList.add('border-primary-500', 'text-primary-600');
        }

        function showResults(loterieId) {
            // Simuler l'affichage des résultats (à implémenter avec AJAX)
            document.getElementById('modalTitle').textContent = 'Résultats du Tirage #' + loterieId;
            document.getElementById('modalContent').innerHTML = `
                <div class="text-center">
                    <div class="bg-blue-100 p-4 rounded-lg mb-4">
                        <p class="font-semibold">Chargement des résultats...</p>
                    </div>
                </div>
            `;
            document.getElementById('resultModal').classList.remove('hidden');
        }

        function closeModal() {
            document.getElementById('resultModal').classList.add('hidden');
        }

        // Fermer modal en cliquant à l'extérieur
        window.onclick = function(event) {
            const modal = document.getElementById('resultModal');
            if (event.target == modal) {
                closeModal();
            }
        }
    </script>
</body>
</html>