<?php
require_once 'config/db.php';

// Vérifier si l'ID est fourni
if (!isset($_GET['id'])) {
    header('Location: index.php');
    exit();
}

$profile_id = intval($_GET['id']);

// Récupérer les informations détaillées du profil
$query = "SELECT users.*, 
          COUNT(DISTINCT views.id) as view_count,
          COUNT(DISTINCT likes.id) as like_count,
          COUNT(DISTINCT hits.id) as hit_count,
          videos.video_url,
          videos.thumbnail_url,
          videos.title as video_title,
          videos.description as video_description
          FROM users 
          LEFT JOIN views ON users.id = views.user_id
          LEFT JOIN likes ON users.id = likes.user_id
          LEFT JOIN hits ON users.id = hits.user_id
          LEFT JOIN videos ON users.id = videos.user_id
          WHERE users.id = ?
          GROUP BY users.id";

$stmt = $conn->prepare($query);
$stmt->bind_param("i", $profile_id);
$stmt->execute();
$result = $stmt->get_result();
$profile = $result->fetch_assoc();

if (!$profile) {
    header('Location: index.php');
    exit();
}

// Enregistrer la vue
if (isset($_SESSION['user_id']) && $_SESSION['user_id'] != $profile_id) {
    $view_query = "INSERT INTO views (user_id, viewer_id) VALUES (?, ?)";
    $view_stmt = $conn->prepare($view_query);
    $view_stmt->bind_param("ii", $profile_id, $_SESSION['user_id']);
    $view_stmt->execute();
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($profile['username']); ?> - Hothings</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body class="bg-gray-100">
    <nav class="bg-white shadow-lg">
        <div class="max-w-6xl mx-auto px-4">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center space-x-8">
                    <a href="/" class="flex items-center space-x-3">
                        <span class="text-2xl font-bold text-purple-600">Hothings</span>
                    </a>
                </div>
                <div class="flex items-center space-x-4">
                    <a href="#" class="text-gray-600 hover:text-purple-600">
                        <i class="fas fa-search text-xl"></i>
                    </a>
                    <a href="#" class="bg-purple-600 text-white px-4 py-2 rounded-lg hover:bg-purple-700">
                        Connexion
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <main class="max-w-6xl mx-auto px-4 py-8">
        <div class="bg-white rounded-lg shadow-xl overflow-hidden">
            <!-- En-tête du profil -->
            <div class="relative h-64 bg-purple-100">
                <img src="<?php echo htmlspecialchars($profile['profile_picture']); ?>" 
                     alt="Photo de profil de <?php echo htmlspecialchars($profile['username']); ?>"
                     class="absolute bottom-0 left-8 transform translate-y-1/2 w-40 h-40 rounded-full border-4 border-white object-cover shadow-lg">
            </div>

            <!-- Informations du profil -->
            <div class="mt-24 px-8 pb-8">
                <div class="flex justify-between items-start">
                    <div>
                        <div class="flex items-center space-x-4">
                            <h1 class="text-3xl font-bold text-gray-900">
                                <?php echo htmlspecialchars($profile['username']); ?>
                            </h1>
                            <?php if ($profile['is_verified']): ?>
                            <span class="bg-blue-500 text-white px-3 py-1 rounded-full text-sm">
                                <i class="fas fa-check-circle"></i> Vérifié
                            </span>
                            <?php endif; ?>
                        </div>
                        <p class="text-xl text-gray-600 mt-2"><?php echo $profile['age']; ?> ans</p>
                    </div>
                    
                    <!-- Actions -->
                    <div class="flex space-x-4">
                        <button onclick="likeProfile(<?php echo $profile_id; ?>)" 
                                class="flex items-center space-x-2 px-4 py-2 bg-pink-100 text-pink-600 rounded-lg hover:bg-pink-200">
                            <i class="fas fa-heart"></i>
                            <span id="likeCount"><?php echo number_format($profile['like_count']); ?></span>
                        </button>
                        <button onclick="sendHit(<?php echo $profile_id; ?>)" 
                                class="flex items-center space-x-2 px-4 py-2 bg-orange-100 text-orange-600 rounded-lg hover:bg-orange-200">
                            <i class="fas fa-fire"></i>
                            <span id="hitCount"><?php echo number_format($profile['hit_count']); ?></span>
                        </button>
                        <?php if ($profile['instagram_handle']): ?>
                        <a href="https://instagram.com/<?php echo htmlspecialchars($profile['instagram_handle']); ?>" 
                           target="_blank" 
                           class="flex items-center space-x-2 px-4 py-2 bg-gradient-to-r from-purple-500 to-pink-500 text-white rounded-lg hover:from-purple-600 hover:to-pink-600">
                            <i class="fab fa-instagram"></i>
                            <span>Instagram</span>
                        </a>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Description -->
                <div class="mt-8">
                    <h2 class="text-2xl font-semibold text-gray-900 mb-4">À propos</h2>
                    <p class="text-gray-700 text-lg leading-relaxed">
                        <?php echo nl2br(htmlspecialchars($profile['description'])); ?>
                    </p>
                </div>

                <!-- Vidéo -->
                <?php if ($profile['video_url']): ?>
                <div class="mt-12">
                    <h2 class="text-2xl font-semibold text-gray-900 mb-4">
                        <?php echo htmlspecialchars($profile['video_title']); ?>
                    </h2>
                    <div class="relative pt-[56.25%]">
                        <video 
                            class="absolute top-0 left-0 w-full h-full rounded-lg"
                            poster="<?php echo htmlspecialchars($profile['thumbnail_url']); ?>"
                            controls>
                            <source src="<?php echo htmlspecialchars($profile['video_url']); ?>" type="video/mp4">
                            Votre navigateur ne supporte pas la lecture de vidéos.
                        </video>
                    </div>
                    <?php if ($profile['video_description']): ?>
                    <p class="mt-4 text-gray-700">
                        <?php echo nl2br(htmlspecialchars($profile['video_description'])); ?>
                    </p>
                    <?php endif; ?>
                </div>
                <?php endif; ?>

                <!-- Statistiques -->
                <div class="mt-12 grid grid-cols-3 gap-6">
                    <div class="bg-gray-50 rounded-lg p-6 text-center">
                        <div class="text-3xl font-bold text-gray-900">
                            <?php echo number_format($profile['view_count']); ?>
                        </div>
                        <div class="text-gray-600 mt-1">Vues</div>
                    </div>
                    <div class="bg-gray-50 rounded-lg p-6 text-center">
                        <div class="text-3xl font-bold text-gray-900">
                            <?php echo number_format($profile['like_count']); ?>
                        </div>
                        <div class="text-gray-600 mt-1">Likes</div>
                    </div>
                    <div class="bg-gray-50 rounded-lg p-6 text-center">
                        <div class="text-3xl font-bold text-gray-900">
                            <?php echo number_format($profile['hit_count']); ?>
                        </div>
                        <div class="text-gray-600 mt-1">Hits</div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script>
    async function likeProfile(profileId) {
        try {
            const response = await fetch('actions/like.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ profileId })
            });
            const data = await response.json();
            if (data.success) {
                document.getElementById('likeCount').textContent = data.newCount;
            }
        } catch (error) {
            console.error('Erreur:', error);
        }
    }

    async function sendHit(profileId) {
        try {
            const response = await fetch('actions/hit.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ profileId })
            });
            const data = await response.json();
            if (data.success) {
                document.getElementById('hitCount').textContent = data.newCount;
            }
        } catch (error) {
            console.error('Erreur:', error);
        }
    }
    </script>
</body>
</html>