<?php
session_start();

// Initialiser les données de jeu si elles n'existent pas
if (!isset($_SESSION['game_data'])) {
    $_SESSION['game_data'] = [
        'score' => 0,
        'level' => 1,
        'filesProcessed' => 0,
        'spamsCaught' => 0,
        'virusesDetected' => 0,
        'systemCrashes' => 0,
        'errors' => 0,
        'startTime' => time(),
        'lastSave' => time(),
        'achievements' => [],
        'notes' => '',
        'settings' => [
            'sound' => true,
            'notifications' => true,
            'autoSave' => true,
            'difficulty' => 'normal'
        ],
        'statistics' => [
            'totalPlayTime' => 0,
            'emailsProcessed' => 0,
            'messagesRead' => 0,
            'scansCompleted' => 0,
            'threatsNeutralized' => 0
        ]
    ];
}

// Gérer les différentes actions
if (isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    switch($_POST['action']) {
        case 'save':
            if (isset($_POST['gameState'])) {
                $newState = json_decode($_POST['gameState'], true);
                $_SESSION['game_data'] = array_merge($_SESSION['game_data'], $newState);
                $_SESSION['game_data']['lastSave'] = time();
                
                // Vérifier les achievements
                checkAchievements();
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Partie sauvegardée',
                    'timestamp' => date('H:i:s')
                ]);
            }
            exit;
            
        case 'load':
            echo json_encode([
                'success' => true,
                'data' => $_SESSION['game_data']
            ]);
            exit;
            
        case 'reset':
            session_destroy();
            echo json_encode([
                'success' => true,
                'message' => 'Progression réinitialisée'
            ]);
            exit;
            
        case 'export':
            $saveData = base64_encode(json_encode($_SESSION['game_data']));
            echo json_encode([
                'success' => true,
                'saveCode' => $saveData
            ]);
            exit;
            
        case 'import':
            if (isset($_POST['saveCode'])) {
                try {
                    $data = json_decode(base64_decode($_POST['saveCode']), true);
                    if ($data) {
                        $_SESSION['game_data'] = $data;
                        echo json_encode([
                            'success' => true,
                            'message' => 'Sauvegarde importée avec succès'
                        ]);
                    } else {
                        echo json_encode([
                            'success' => false,
                            'message' => 'Code de sauvegarde invalide'
                        ]);
                    }
                } catch (Exception $e) {
                    echo json_encode([
                        'success' => false,
                        'message' => 'Erreur lors de l\'import'
                    ]);
                }
            }
            exit;
            
        case 'updateStat':
            $stat = $_POST['stat'];
            $value = $_POST['value'];
            
            if (isset($_SESSION['game_data']['statistics'][$stat])) {
                $_SESSION['game_data']['statistics'][$stat] += $value;
            } else {
                $_SESSION['game_data'][$stat] = $value;
            }
            
            echo json_encode(['success' => true]);
            exit;
    }
}

function checkAchievements() {
    $achievements = &$_SESSION['game_data']['achievements'];
    $data = $_SESSION['game_data'];
    
    // Achievement: Premier spam détecté
    if ($data['spamsCaught'] >= 1 && !in_array('first_spam', $achievements)) {
        $achievements[] = 'first_spam';
    }
    
    // Achievement: 100 fichiers traités
    if ($data['filesProcessed'] >= 100 && !in_array('file_master', $achievements)) {
        $achievements[] = 'file_master';
    }
    
    // Achievement: Survivre 10 crashs
    if ($data['systemCrashes'] >= 10 && !in_array('survivor', $achievements)) {
        $achievements[] = 'survivor';
    }
    
    // Achievement: Score de 1000
    if ($data['score'] >= 1000 && !in_array('high_scorer', $achievements)) {
        $achievements[] = 'high_scorer';
    }
}

// Si c'est un accès direct, afficher l'interface
if (!isset($_POST['action'])):
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>AWPO - Gestionnaire de Sauvegarde</title>
    <style>
        body {
            font-family: Tahoma, sans-serif;
            font-size: 11px;
            background: #ECE9D8;
            margin: 0;
            padding: 20px;
        }
        
        .save-container {
            max-width: 600px;
            margin: 0 auto;
            background: white;
            border: 2px solid #0054E3;
            box-shadow: 2px 2px 5px rgba(0,0,0,0.3);
        }
        
        .save-header {
            background: linear-gradient(to right, #0054E3, #4B9BFF);
            color: white;
            padding: 5px 10px;
            font-weight: bold;
            font-size: 12px;
        }
        
        .save-content {
            padding: 15px;
        }
        
        .stat-box {
            background: #F0F0F0;
            border: 1px solid #919B9C;
            padding: 10px;
            margin: 10px 0;
        }
        
        .stat-row {
            display: flex;
            justify-content: space-between;
            padding: 3px 0;
        }
        
        .stat-label {
            color: #333;
        }
        
        .stat-value {
            font-weight: bold;
            color: #0054E3;
        }
        
        .button {
            background: #ECE9D8;
            border: 1px solid #003C74;
            padding: 5px 15px;
            cursor: pointer;
            margin: 5px;
        }
        
        .button:hover {
            background: #DDD6C1;
        }
        
        .button-danger {
            background: #FFE0E0;
            border-color: #800000;
        }
        
        .button-success {
            background: #E0FFE0;
            border-color: #008000;
        }
        
        .achievement {
            display: inline-block;
            background: #FFD700;
            border: 1px solid #B8860B;
            padding: 2px 8px;
            margin: 2px;
            border-radius: 3px;
        }
        
        .save-code-box {
            width: 100%;
            height: 60px;
            font-family: monospace;
            font-size: 10px;
            border: 1px solid #919B9C;
            padding: 5px;
        }
        
        .tab-buttons {
            display: flex;
            border-bottom: 1px solid #919B9C;
            background: #F0F0F0;
        }
        
        .tab-button {
            padding: 8px 15px;
            background: #E0E0E0;
            border: none;
            cursor: pointer;
            border-right: 1px solid #919B9C;
        }
        
        .tab-button.active {
            background: white;
            font-weight: bold;
        }
        
        .tab-content {
            display: none;
            padding: 15px;
        }
        
        .tab-content.active {
            display: block;
        }
    </style>
</head>
<body>
    <div class="save-container">
        <div class="save-header">
            💾 AWPO - Gestionnaire de Sauvegarde
        </div>
        
        <div class="tab-buttons">
            <button class="tab-button active" onclick="showTab('stats')">📊 Statistiques</button>
            <button class="tab-button" onclick="showTab('save')">💾 Sauvegarde</button>
            <button class="tab-button" onclick="showTab('achievements')">🏆 Succès</button>
            <button class="tab-button" onclick="showTab('settings')">⚙️ Paramètres</button>
        </div>
        
        <div class="save-content">
            
            <!-- Tab Statistiques -->
            <div id="tab-stats" class="tab-content active">
                <h3>Statistiques de jeu</h3>
                
                <div class="stat-box">
                    <h4>Progression actuelle</h4>
                    <div class="stat-row">
                        <span class="stat-label">Score:</span>
                        <span class="stat-value"><?= $_SESSION['game_data']['score'] ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Niveau:</span>
                        <span class="stat-value"><?= $_SESSION['game_data']['level'] ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Fichiers traités:</span>
                        <span class="stat-value"><?= $_SESSION['game_data']['filesProcessed'] ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Spams détectés:</span>
                        <span class="stat-value"><?= $_SESSION['game_data']['spamsCaught'] ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Virus neutralisés:</span>
                        <span class="stat-value"><?= $_SESSION['game_data']['virusesDetected'] ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Crashs système:</span>
                        <span class="stat-value" style="color: #FF0000;"><?= $_SESSION['game_data']['systemCrashes'] ?></span>
                    </div>
                </div>
                
                <div class="stat-box">
                    <h4>Statistiques globales</h4>
                    <div class="stat-row">
                        <span class="stat-label">Temps de jeu total:</span>
                        <span class="stat-value"><?= gmdate("H:i:s", time() - $_SESSION['game_data']['startTime']) ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Dernière sauvegarde:</span>
                        <span class="stat-value"><?= date('d/m/Y H:i:s', $_SESSION['game_data']['lastSave']) ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Tab Sauvegarde -->
            <div id="tab-save" class="tab-content">
                <h3>Gestion de la sauvegarde</h3>
                
                <div class="stat-box">
                    <h4>Sauvegarde manuelle</h4>
                    <p>Sauvegardez votre progression maintenant:</p>
                    <button class="button button-success" onclick="saveGame()">
                        💾 Sauvegarder maintenant
                    </button>
                    <div id="saveMessage" style="margin-top: 10px; color: green;"></div>
                </div>
                
                <div class="stat-box">
                    <h4>Exporter/Importer</h4>
                    <p>Exportez votre sauvegarde pour la conserver ou la partager:</p>
                    <button class="button" onclick="exportSave()">📤 Exporter la sauvegarde</button>
                    <textarea id="saveCode" class="save-code-box" placeholder="Code de sauvegarde apparaîtra ici..."></textarea>
                    
                    <p style="margin-top: 15px;">Importez une sauvegarde:</p>
                    <textarea id="importCode" class="save-code-box" placeholder="Collez votre code de sauvegarde ici..."></textarea>
                    <button class="button" onclick="importSave()">📥 Importer la sauvegarde</button>
                </div>
                
                <div class="stat-box" style="background: #FFE0E0;">
                    <h4>⚠️ Zone dangereuse</h4>
                    <p>Réinitialiser complètement votre progression:</p>
                    <button class="button button-danger" onclick="resetGame()">
                        🗑️ Réinitialiser la partie
                    </button>
                </div>
            </div>
            
            <!-- Tab Achievements -->
            <div id="tab-achievements" class="tab-content">
                <h3>Succès débloqués</h3>
                
                <div class="stat-box">
                    <?php 
                    $allAchievements = [
                        'first_spam' => '🎯 Premier Spam - Détecter votre premier spam',
                        'file_master' => '📁 Maître des Fichiers - Traiter 100 fichiers',
                        'survivor' => '💀 Survivant - Survivre à 10 crashs système',
                        'high_scorer' => '⭐ High Score - Atteindre 1000 points',
                        'virus_hunter' => '🦠 Chasseur de Virus - Détecter 50 virus',
                        'paranormal' => '👻 Activité Paranormale - Rencontrer l\'inconnu'
                    ];
                    
                    foreach ($allAchievements as $key => $desc): 
                        $unlocked = in_array($key, $_SESSION['game_data']['achievements']);
                    ?>
                        <div style="padding: 5px; opacity: <?= $unlocked ? '1' : '0.3' ?>;">
                            <?php if ($unlocked): ?>
                                <span class="achievement">✓</span>
                            <?php else: ?>
                                <span class="achievement" style="background: #CCC;">🔒</span>
                            <?php endif; ?>
                            <?= $desc ?>
                        </div>
                    <?php endforeach; ?>
                </div>
                
                <div class="stat-box">
                    <p>Succès débloqués: <strong><?= count($_SESSION['game_data']['achievements']) ?>/6</strong></p>
                    <div style="border: 1px solid #919B9C; height: 20px; background: white;">
                        <div style="height: 100%; background: #4169C7; width: <?= (count($_SESSION['game_data']['achievements']) / 6 * 100) ?>%;"></div>
                    </div>
                </div>
            </div>
            
            <!-- Tab Settings -->
            <div id="tab-settings" class="tab-content">
                <h3>Paramètres de sauvegarde</h3>
                
                <div class="stat-box">
                    <h4>Options</h4>
                    <label style="display: block; margin: 5px 0;">
                        <input type="checkbox" <?= $_SESSION['game_data']['settings']['autoSave'] ? 'checked' : '' ?> onchange="toggleSetting('autoSave', this.checked)">
                        Sauvegarde automatique (toutes les 5 secondes)
                    </label>
                    <label style="display: block; margin: 5px 0;">
                        <input type="checkbox" <?= $_SESSION['game_data']['settings']['sound'] ? 'checked' : '' ?> onchange="toggleSetting('sound', this.checked)">
                        Effets sonores
                    </label>
                    <label style="display: block; margin: 5px 0;">
                        <input type="checkbox" <?= $_SESSION['game_data']['settings']['notifications'] ? 'checked' : '' ?> onchange="toggleSetting('notifications', this.checked)">
                        Notifications système
                    </label>
                </div>
                
                <div class="stat-box">
                    <h4>Difficulté</h4>
                    <select onchange="changeDifficulty(this.value)" style="padding: 5px;">
                        <option value="easy" <?= $_SESSION['game_data']['settings']['difficulty'] == 'easy' ? 'selected' : '' ?>>Facile</option>
                        <option value="normal" <?= $_SESSION['game_data']['settings']['difficulty'] == 'normal' ? 'selected' : '' ?>>Normal</option>
                        <option value="hard" <?= $_SESSION['game_data']['settings']['difficulty'] == 'hard' ? 'selected' : '' ?>>Difficile</option>
                        <option value="paranormal" <?= $_SESSION['game_data']['settings']['difficulty'] == 'paranormal' ? 'selected' : '' ?>>PARANORMAL</option>
                    </select>
                </div>
            </div>
            
        </div>
    </div>
    
    <script>
        function showTab(tabName) {
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            document.querySelectorAll('.tab-button').forEach(btn => {
                btn.classList.remove('active');
            });
            
            document.getElementById('tab-' + tabName).classList.add('active');
            event.target.classList.add('active');
        }
        
        function saveGame() {
            fetch('save_game.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'action=save&gameState=' + encodeURIComponent(JSON.stringify({}))
            })
            .then(response => response.json())
            .then(data => {
                document.getElementById('saveMessage').textContent = '✓ Sauvegarde effectuée à ' + data.timestamp;
                setTimeout(() => {
                    document.getElementById('saveMessage').textContent = '';
                }, 3000);
            });
        }
        
        function exportSave() {
            fetch('save_game.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'action=export'
            })
            .then(response => response.json())
            .then(data => {
                document.getElementById('saveCode').value = data.saveCode;
            });
        }
        
        function importSave() {
            const code = document.getElementById('importCode').value;
            if (!code) {
                alert('Veuillez coller un code de sauvegarde');
                return;
            }
            
            fetch('save_game.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'action=import&saveCode=' + encodeURIComponent(code)
            })
            .then(response => response.json())
            .then(data => {
                alert(data.message);
                if (data.success) {
                    location.reload();
                }
            });
        }
        
        function resetGame() {
            if (confirm('ATTENTION: Ceci effacera TOUTE votre progression!\n\nÊtes-vous sûr?')) {
                if (confirm('Dernière chance! Voulez-vous vraiment tout effacer?')) {
                    fetch('save_game.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                        body: 'action=reset'
                    })
                    .then(response => response.json())
                    .then(data => {
                        alert('Progression réinitialisée');
                        location.reload();
                    });
                }
            }
        }
        
        function toggleSetting(setting, value) {
            // Sauvegarder le paramètre
            console.log('Setting ' + setting + ' to ' + value);
        }
        
        function changeDifficulty(value) {
            if (value === 'paranormal') {
                alert('Mode PARANORMAL activé!\n\nAttention: Des choses étranges vont se produire...');
            }
        }
    </script>
</body>
</html>

<?php endif; ?>