<?php 
require_once 'includes/auth.php';

$stmt = $pdo->prepare("SELECT * FROM profiles WHERE user_id = ? ORDER BY is_main DESC, created_at ASC");
$stmt->execute([$current_user['id']]);
$profiles = $stmt->fetchAll(PDO::FETCH_ASSOC);

$available_avatars = [];
$avatar_dir = 'profil-picture-icon/';
if (is_dir($avatar_dir)) {
    $categories = array_diff(scandir($avatar_dir), ['.', '..']);
    foreach ($categories as $category) {
        $category_path = $avatar_dir . $category . '/';
        if (is_dir($category_path)) {
            $files = scandir($category_path);
            foreach ($files as $file) {
                if (in_array(strtolower(pathinfo($file, PATHINFO_EXTENSION)), ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                    if (!isset($available_avatars[$category])) {
                        $available_avatars[$category] = [];
                    }
                    $available_avatars[$category][] = $category_path . $file;
                }
            }
        }
    }
}

$stmt = $pdo->prepare("SELECT COUNT(*) as uploads FROM content WHERE uploaded_by = ?");
$stmt->execute([$current_user['id']]);
$user_uploads = $stmt->fetchColumn() ?? 0;

$stmt = $pdo->prepare("SELECT COUNT(*) as pending FROM content WHERE uploaded_by = ? AND status = 'pending'");
$stmt->execute([$current_user['id']]);
$pending_uploads = $stmt->fetchColumn() ?? 0;

$error = '';
$success = '';

function verifyPasswordWithImators($user_id, $password) {
    if (isset($_SESSION['user_password_hash'])) {
        return password_verify($password, $_SESSION['user_password_hash']);
    }
    return true;
}

function isValidAvatar($avatar, $available_avatars) {
    foreach ($available_avatars as $category => $avatars) {
        if (in_array($avatar, $avatars)) {
            return true;
        }
    }
    return false;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $password = $_POST['password'] ?? '';
    
    if (empty($password)) {
        $error = 'Password is required for any modification';
    } else {
        $password_valid = verifyPasswordWithImators($current_user['id'], $password);
        
        if (!$password_valid) {
            $error = 'Incorrect password';
        } else {
            if (isset($_POST['logout'])) {
                setcookie('formore_auth_token', '', time() - 3600, '/', '', true, true);
                session_destroy();
                header('Location: https://idsma.imators.com/login.php?formore-disconnected=single');
                exit;
            }
            
            if (isset($_POST['logout_all'])) {
                setcookie('formore_auth_token', '', time() - 3600, '/', '', true, true);
                session_destroy();
                header('Location: https://idsma.imators.com/login.php?formore-disconnected=all');
                exit;
            }
            
            if (isset($_POST['add_profile'])) {
                $name = trim($_POST['profile_name']);
                $avatar = $_POST['profile_avatar'];
                $age_limit = (int)$_POST['age_limit'];
                
                if (!empty($name) && !empty($avatar) && $age_limit > 0) {
                    if (isValidAvatar($avatar, $available_avatars)) {
                        $stmt = $pdo->prepare("INSERT INTO profiles (user_id, name, avatar, age_limit, is_main) VALUES (?, ?, ?, ?, 0)");
                        if ($stmt->execute([$current_user['id'], $name, $avatar, $age_limit])) {
                            $success = 'Profile added successfully';
                            header('Location: settings.php?success=profile_added');
                            exit;
                        } else {
                            $error = 'Failed to add profile. Please try again.';
                        }
                    } else {
                        $error = 'Invalid avatar selected. Please choose from the available options.';
                    }
                } else {
                    $error = 'Please fill all fields correctly. Make sure to select an avatar.';
                }
            }
            
            if (isset($_POST['edit_profile'])) {
                $profile_id = (int)$_POST['profile_id'];
                $name = trim($_POST['profile_name']);
                $avatar = $_POST['profile_avatar'];
                $age_limit = (int)$_POST['age_limit'];
                
                if (!empty($name) && !empty($avatar) && $age_limit > 0) {
                    if (isValidAvatar($avatar, $available_avatars)) {
                        $stmt = $pdo->prepare("UPDATE profiles SET name = ?, avatar = ?, age_limit = ? WHERE id = ? AND user_id = ?");
                        if ($stmt->execute([$name, $avatar, $age_limit, $profile_id, $current_user['id']])) {
                            $success = 'Profile updated successfully';
                            header('Location: settings.php?success=profile_updated');
                            exit;
                        } else {
                            $error = 'Failed to update profile. Please try again.';
                        }
                    } else {
                        $error = 'Invalid avatar selected. Please choose from the available options.';
                    }
                } else {
                    $error = 'Please fill all fields correctly. Make sure to select an avatar.';
                }
            }
            
            if (isset($_POST['delete_profile'])) {
                $profile_id = (int)$_POST['profile_id'];
                
                $stmt = $pdo->prepare("SELECT is_main FROM profiles WHERE id = ? AND user_id = ?");
                $stmt->execute([$profile_id, $current_user['id']]);
                $profile = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($profile && !$profile['is_main']) {
                    $stmt = $pdo->prepare("DELETE FROM profiles WHERE id = ? AND user_id = ?");
                    if ($stmt->execute([$profile_id, $current_user['id']])) {
                        if (isset($_SESSION['selected_profile']) && $_SESSION['selected_profile'] == $profile_id) {
                            unset($_SESSION['selected_profile']);
                        }
                        $success = 'Profile deleted successfully';
                        header('Location: settings.php?success=profile_deleted');
                        exit;
                    } else {
                        $error = 'Failed to delete profile. Please try again.';
                    }
                } else {
                    $error = 'Cannot delete the main profile or profile not found.';
                }
            }
            
            if (isset($_POST['update_email'])) {
                $new_email = filter_var($_POST['email'], FILTER_VALIDATE_EMAIL);
                
                if (!$new_email) {
                    $error = 'Please provide a valid email address.';
                } else {
                    $stmt = $pdo->prepare("SELECT email FROM users WHERE email = ? AND id != ?");
                    $stmt->execute([$new_email, $current_user['id']]);
                    if ($stmt->rowCount() > 0) {
                        $error = 'This email address is already in use.';
                    } else {
                        $stmt = $pdo->prepare("UPDATE users SET email = ? WHERE id = ?");
                        if ($stmt->execute([$new_email, $current_user['id']])) {
                            $_SESSION['user']['email'] = $new_email;
                            $current_user['email'] = $new_email;
                            $success = 'Email updated successfully';
                            header('Location: settings.php?success=email_updated');
                            exit;
                        } else {
                            $error = 'Failed to update email. Please try again.';
                        }
                    }
                }
            }
            
            if (isset($_POST['update_password'])) {
                $new_password = $_POST['new_password'] ?? '';
                $confirm_password = $_POST['confirm_password'] ?? '';
                
                if (strlen($new_password) < 8) {
                    $error = 'Password must be at least 8 characters long.';
                } else if ($new_password !== $confirm_password) {
                    $error = 'Passwords do not match.';
                } else {
                    $hashed = password_hash($new_password, PASSWORD_BCRYPT);
                    $_SESSION['user_password_hash'] = $hashed;
                    $success = 'Password updated successfully';
                    header('Location: settings.php?success=password_updated');
                    exit;
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - formore</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        :root {
            color-scheme: light dark;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background-color: light-dark(#f8f9fa, #1a1a1a);
            color: light-dark(#333, #e0e0e0);
            transition: background-color 0.3s, color 0.3s;
            line-height: 1.6;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 1.5rem;
        }

        header {
            margin-bottom: 2rem;
            border-bottom: 1px solid light-dark(#e0e0e0, #333);
            padding-bottom: 1.5rem;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .header-content {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .back-btn {
            padding: 0.6rem 1rem;
            border: none;
            background: none;
            cursor: pointer;
            font-size: 1rem;
            border-radius: 4px;
            transition: all 0.2s;
            color: light-dark(#666, #aaa);
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .back-btn:hover {
            background-color: light-dark(#e8e8e8, #2a2a2a);
            color: light-dark(#333, #e0e0e0);
        }

        .logo-section {
            display: flex;
            align-items: center;
            gap: 1.5rem;
        }

        .logo-box {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .logo-icon {
            width: 40px;
            height: 40px;
            background-color: light-dark(#333, #e0e0e0);
            color: light-dark(#fff, #000);
            border-radius: 4px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
        }

        .logo-text h3 {
            margin: 0;
            font-size: 0.9rem;
            font-weight: 600;
            color: light-dark(#333, #e0e0e0);
        }

        .logo-text p {
            margin: 0;
            font-size: 0.75rem;
            color: light-dark(#666, #aaa);
        }

        .alert {
            padding: 1rem;
            margin-bottom: 1.5rem;
            border-radius: 4px;
            animation: slideIn 0.3s ease;
        }

        .alert-error {
            background-color: light-dark(#fee, #4a2a2a);
            color: light-dark(#c33, #ff6b6b);
        }

        .alert-success {
            background-color: light-dark(#efe, #2a4a2a);
            color: light-dark(#3c3, #6bff6b);
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .content-wrapper {
            display: grid;
            grid-template-columns: 200px 1fr;
            gap: 2rem;
        }

        nav {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .nav-link {
            padding: 0.75rem 1rem;
            cursor: pointer;
            border: none;
            background: none;
            font-size: 1rem;
            text-align: left;
            border-radius: 4px;
            transition: all 0.2s;
            color: light-dark(#666, #aaa);
        }

        .nav-link:hover {
            background-color: light-dark(#e8e8e8, #2a2a2a);
            color: light-dark(#333, #e0e0e0);
        }

        .nav-link.active {
            background-color: light-dark(#ddd, #333);
            color: light-dark(#333, #e0e0e0);
            font-weight: 600;
        }

        .sections {
            display: flex;
            flex-direction: column;
        }

        .content-section {
            display: none;
        }

        .content-section.active {
            display: block;
            animation: fadeIn 0.3s ease;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
            }
            to {
                opacity: 1;
            }
        }

        .section-title {
            font-size: 1.4rem;
            margin-bottom: 1.5rem;
        }

        .profile-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .profile-card {
            padding: 1.5rem;
            border: 1px solid light-dark(#e0e0e0, #333);
            border-radius: 4px;
            transition: all 0.2s;
        }

        .profile-card:hover {
            box-shadow: 0 2px 8px light-dark(rgba(0,0,0,0.1), rgba(255,255,255,0.1));
        }

        .profile-header {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 1rem;
        }

        .profile-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            object-fit: cover;
        }

        .profile-info h3 {
            margin-bottom: 0.3rem;
        }

        .profile-badge {
            font-size: 0.8rem;
            padding: 0.25rem 0.5rem;
            border-radius: 2px;
            background-color: light-dark(#e0e0e0, #333);
            display: inline-block;
        }

        .profile-actions {
            display: flex;
            gap: 0.5rem;
            margin-top: 1rem;
        }

        .btn {
            padding: 0.6rem 1rem;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 0.9rem;
            transition: all 0.2s;
            font-weight: 500;
        }

        .btn-primary {
            background-color: light-dark(#333, #e0e0e0);
            color: light-dark(#fff, #000);
        }

        .btn-primary:hover {
            opacity: 0.8;
        }

        .btn-secondary {
            background-color: light-dark(#ddd, #333);
            color: light-dark(#333, #e0e0e0);
        }

        .btn-secondary:hover {
            background-color: light-dark(#ccc, #444);
        }

        .btn-danger {
            background-color: light-dark(#dc3545, #8b3a3a);
            color: #fff;
        }

        .btn-danger:hover {
            opacity: 0.8;
        }

        .btn-small {
            padding: 0.4rem 0.8rem;
            font-size: 0.8rem;
            flex: 1;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
        }

        .form-input,
        select {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid light-dark(#ddd, #444);
            border-radius: 4px;
            background-color: light-dark(#fff, #2a2a2a);
            color: light-dark(#333, #e0e0e0);
            font-size: 1rem;
            font-family: inherit;
        }

        .form-input:focus,
        select:focus {
            outline: none;
            border-color: light-dark(#666, #777);
            box-shadow: 0 0 0 2px light-dark(rgba(0,0,0,0.05), rgba(255,255,255,0.05));
        }

        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }

        .stat-box {
            padding: 1rem;
            border: 1px solid light-dark(#e0e0e0, #333);
            border-radius: 4px;
            text-align: center;
        }

        .stat-number {
            font-size: 1.8rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
        }

        .stat-label {
            font-size: 0.9rem;
            color: light-dark(#666, #aaa);
        }

        .modal-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: light-dark(rgba(0,0,0,0.5), rgba(0,0,0,0.8));
            z-index: 1000;
            align-items: center;
            justify-content: center;
            padding: 1rem;
        }

        .modal-overlay.active {
            display: flex;
        }

        .modal-content {
            background-color: light-dark(#fff, #1a1a1a);
            border-radius: 8px;
            box-shadow: 0 20px 60px light-dark(rgba(0,0,0,0.3), rgba(0,0,0,0.9));
            display: flex;
            flex-direction: column;
            max-height: 90vh;
            width: 100%;
        }

        #profileModal .modal-content {
            max-width: 600px;
        }

        #deleteModal .modal-content {
            max-width: 450px;
        }

        .modal-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 2rem;
            border-bottom: 1px solid light-dark(#e0e0e0, #333);
            flex-shrink: 0;
        }

        .modal-title {
            font-size: 1.6rem;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.8rem;
            color: light-dark(#333, #e0e0e0);
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.8rem;
            cursor: pointer;
            color: light-dark(#666, #aaa);
            transition: all 0.2s;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
        }

        .modal-close:hover {
            background-color: light-dark(#e8e8e8, #2a2a2a);
            color: light-dark(#333, #e0e0e0);
        }

        .modal-body {
            overflow-y: auto;
            flex: 1;
            padding: 2rem;
        }

        .modal-footer {
            display: flex;
            gap: 1rem;
            justify-content: flex-end;
            padding: 2rem;
            border-top: 1px solid light-dark(#e0e0e0, #333);
            flex-shrink: 0;
            background-color: light-dark(#f8f9fa, #0a0a0a);
        }

        #profileForm {
            display: contents;
        }

        #deleteForm {
            display: contents;
        }

        .avatar-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(80px, 1fr));
            gap: 1rem;
            margin-bottom: 1.5rem;
        }

        .avatar-option {
            width: 100%;
            aspect-ratio: 1;
            border: 3px solid light-dark(#ddd, #444);
            border-radius: 8px;
            cursor: pointer;
            object-fit: cover;
            transition: all 0.3s;
        }

        .avatar-option:hover {
            border-color: light-dark(#999, #666);
            transform: scale(1.05);
        }

        .avatar-option.selected {
            border-color: light-dark(#333, #e0e0e0);
            box-shadow: 0 0 0 4px light-dark(rgba(0,0,0,0.1), rgba(255,255,255,0.1));
        }

        .avatar-tabs {
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }

        .avatar-tab-buttons {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }

        .avatar-tab-btn {
            padding: 0.5rem 1rem;
            border: 2px solid light-dark(#ddd, #444);
            background: light-dark(#f5f5f5, #2a2a2a);
            color: light-dark(#333, #e0e0e0);
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.95rem;
            transition: all 0.3s;
        }

        .avatar-tab-btn:hover {
            border-color: light-dark(#999, #666);
            background: light-dark(#e8e8e8, #333);
        }

        .avatar-tab-btn.active {
            border-color: light-dark(#333, #e0e0e0);
            background: light-dark(#333, #e0e0e0);
            color: light-dark(#f5f5f5, #1a1a1a);
        }

        .avatar-tab-content {
            display: none;
        }

        .avatar-tab-content.active {
            display: block;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group:last-of-type {
            margin-bottom: 0;
        }

        .form-label {
            display: block;
            margin-bottom: 0.6rem;
            font-weight: 600;
            font-size: 0.95rem;
            color: light-dark(#333, #e0e0e0);
        }

        .form-input,
        select {
            width: 100%;
            padding: 0.85rem;
            border: 1px solid light-dark(#ddd, #444);
            border-radius: 6px;
            background-color: light-dark(#fff, #2a2a2a);
            color: light-dark(#333, #e0e0e0);
            font-size: 1rem;
            font-family: inherit;
            transition: all 0.2s;
        }

        .form-input:focus,
        select:focus {
            outline: none;
            border-color: light-dark(#333, #777);
            box-shadow: 0 0 0 3px light-dark(rgba(0,0,0,0.05), rgba(255,255,255,0.05));
        }

        @media (max-width: 768px) {
            .modal-header {
                padding: 1.5rem;
            }

            .modal-body {
                padding: 1.5rem;
            }

            .modal-footer {
                padding: 1.5rem;
                gap: 0.75rem;
            }

            .modal-title {
                font-size: 1.3rem;
            }

            .avatar-grid {
                grid-template-columns: repeat(auto-fill, minmax(70px, 1fr));
                gap: 0.75rem;
            }

            .form-group {
                margin-bottom: 1.2rem;
            }

            .btn {
                flex: 1;
            }
        }

        @media (max-width: 480px) {
            .modal-overlay {
                padding: 0.5rem;
            }

            #profileModal .modal-content {
                max-width: 100%;
                max-height: 100vh;
            }

            #deleteModal .modal-content {
                max-width: 100%;
            }

            .modal-header {
                padding: 1rem;
            }

            .modal-body {
                padding: 1rem;
            }

            .modal-footer {
                padding: 1rem;
                flex-direction: column;
            }

            .modal-title {
                font-size: 1.1rem;
            }

            .avatar-grid {
                grid-template-columns: repeat(3, 1fr);
                gap: 0.5rem;
            }

            .avatar-option {
                border-width: 2px;
            }

            .form-label {
                font-size: 0.9rem;
            }

            .form-input,
            select {
                padding: 0.75rem;
                font-size: 16px;
            }

            .btn {
                padding: 0.7rem 1rem;
                font-size: 0.9rem;
                width: 100%;
            }
        }

        .avatar-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 0.5rem;
            margin-bottom: 1rem;
        }

        .avatar-option {
            width: 100%;
            aspect-ratio: 1;
            border: 2px solid light-dark(#ddd, #444);
            border-radius: 4px;
            cursor: pointer;
            object-fit: cover;
            transition: all 0.2s;
        }

        .avatar-option:hover {
            border-color: light-dark(#666, #777);
        }

        .avatar-option.selected {
            border-color: light-dark(#333, #e0e0e0);
            box-shadow: 0 0 0 3px light-dark(rgba(0,0,0,0.1), rgba(255,255,255,0.1));
        }

        .action-buttons {
            display: flex;
            gap: 0.5rem;
            justify-content: flex-end;
            margin-top: 2rem;
        }

        .add-profile-btn {
            margin-bottom: 1.5rem;
        }

        @media (max-width: 768px) {
            .container {
                padding: 1rem;
            }

            .content-wrapper {
                grid-template-columns: 1fr;
                gap: 1rem;
            }

            nav {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
                gap: 0.5rem;
                margin-bottom: 1rem;
            }

            .nav-link {
                padding: 0.6rem 0.8rem;
                font-size: 0.9rem;
                text-align: center;
            }

            .profile-grid {
                grid-template-columns: 1fr;
            }

            .avatar-grid {
                grid-template-columns: repeat(3, 1fr);
            }

            #profileModal .modal-content {
                width: 95vw;
                height: 95vh;
                padding: 1.5rem;
            }

            h1 {
                font-size: 1.4rem;
            }

            .section-title {
                font-size: 1.2rem;
            }

            .action-buttons {
                flex-direction: column;
                margin-top: auto;
                gap: 0.75rem;
            }

            .btn {
                padding: 0.5rem 0.8rem;
                font-size: 0.85rem;
            }

            .stats {
                grid-template-columns: repeat(2, 1fr);
            }
        }

        @media (max-width: 480px) {
            .container {
                padding: 0.75rem;
            }

            nav {
                grid-template-columns: 1fr;
            }

            .profile-header {
                flex-direction: column;
                text-align: center;
            }

            .profile-avatar {
                width: 50px;
                height: 50px;
            }

            .profile-actions {
                flex-direction: column;
            }

            .avatar-grid {
                grid-template-columns: repeat(2, 1fr);
            }

            .stats {
                grid-template-columns: 1fr;
            }

            .action-buttons {
                flex-direction: column;
                gap: 0.75rem;
            }

            .btn {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header>
            <div class="header-content">
                <button class="back-btn">
                    <i class="fas fa-arrow-left"></i>
                    <a href="/home">Back</a>
                </button>
                <div>
                    <h1 class="text-3xl font-bold">Settings</h1>
                    <p class="text-ms">Manage your formore account and profiles</p>
                </div>
            </div>
        </header>

        <?php if ($error): ?>
            <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
        <?php endif; ?>

        <div class="content-wrapper">
            <nav>
                <button class="nav-link active" onclick="showSection('profiles', this)"><i class="fas fa-user"></i> Profiles</button>
                <button class="nav-link" onclick="showSection('account', this)"><i class="fas fa-cog"></i> Account</button>
                <button class="nav-link" onclick="showSection('security', this)"><i class="fas fa-shield"></i> Security</button>
                <button class="nav-link" onclick="showSection('devices', this)"><i class="fas fa-laptop"></i> Devices</button>
            </nav>

            <div class="sections">
                <div id="profiles" class="content-section active">
                    <h2 class="section-title">Your Profiles</h2>

                    <div class="stats">
                        <div class="stat-box">
                            <div class="stat-number"><?php echo count($profiles); ?></div>
                            <div class="stat-label">Total Profiles</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-number"><?php echo $user_uploads; ?></div>
                            <div class="stat-label">Your uploads content</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-number"><?php echo $pending_uploads; ?></div>
                            <div class="stat-label">Pending content</div>
                        </div>
                    </div>

                    <div class="add-profile-btn">
                        <button class="btn btn-primary" onclick="addProfile()"><i class="fas fa-plus"></i> Add New Profile</button>
                    </div>

                    <div class="profile-grid">
                        <?php if (!empty($profiles)): ?>
                            <?php foreach ($profiles as $profile): ?>
                                <div class="profile-card">
                                    <div class="profile-header">
                                        <img src="<?php echo htmlspecialchars($profile['avatar']); ?>" alt="<?php echo htmlspecialchars($profile['name']); ?>" class="profile-avatar">
                                        <div class="profile-info">
                                            <h3><?php echo htmlspecialchars($profile['name']); ?></h3>
                                            <?php if ($profile['is_main']): ?>
                                                <span class="profile-badge">Main Profile</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div>
                                        <p><strong>Age Limit:</strong> <?php echo $profile['age_limit']; ?>+</p>
                                        <p><strong>Created:</strong> <?php echo date('M d, Y', strtotime($profile['created_at'])); ?></p>
                                    </div>
                                    <div class="profile-actions">
                                        <?php if (!$profile['is_main']): ?>
                                            <button class="btn btn-secondary btn-small" onclick="editProfile(<?php echo htmlspecialchars(json_encode($profile)); ?>)"><i class="fas fa-edit"></i> Edit</button>
                                            <button class="btn btn-danger btn-small" onclick="deleteProfile(<?php echo $profile['id']; ?>, '<?php echo htmlspecialchars($profile['name']); ?>')"><i class="fas fa-trash"></i> Delete</button>
                                        <?php else: ?>
                                            <button class="btn btn-secondary btn-small" onclick="editProfile(<?php echo htmlspecialchars(json_encode($profile)); ?>)"><i class="fas fa-edit"></i> Edit</button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <p>No profiles found. Create one to get started.</p>
                        <?php endif; ?>
                    </div>
                </div>

                <div id="account" class="content-section">
                    <h2 class="section-title">Account Settings</h2>

                    <div style="padding: 1rem; background-color: light-dark(#f0f0f0, #2a2a2a); border-radius: 4px; margin-bottom: 1.5rem; border-left: 3px solid light-dark(#333, #e0e0e0);">
                        <p style="margin: 0; font-size: 0.95rem; color: light-dark(#333, #aaa);"><i class="fas fa-info-circle"></i> Your Formore account is securely managed and authenticated through <strong>Imators IDSMA</strong>. All account changes are synchronized across our platform.</p>
                    </div>

                    <div class="stats">
                        <div class="stat-box">
                            <div class="stat-number"><?php echo htmlspecialchars($current_user['username']); ?></div>
                            <div class="stat-label">Username</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-number"><?php echo htmlspecialchars($current_user['created_at'] ? date('Y-m-d', strtotime($current_user['created_at'])) : 'N/A'); ?></div>
                            <div class="stat-label">Member Since</div>
                        </div>
                    </div>

                    <form method="POST">
                        <div class="form-group">
                            <label class="form-label">Email Address</label>
                            <input type="email" name="email" class="form-input" value="<?php echo htmlspecialchars($current_user['email']); ?>" required>
                        </div>

                        <div class="form-group">
                            <label class="form-label">Password (to confirm changes)</label>
                            <input type="password" name="password" class="form-input" placeholder="Your password" required>
                        </div>

                        <button type="submit" name="update_email" class="btn btn-primary"><i class="fas fa-save"></i> Update Email</button>
                    </form>
                </div>

                <div id="security" class="content-section">
                    <h2 class="section-title">Security Settings</h2>

                    <div style="padding: 1rem; background-color: light-dark(#f0f0f0, #2a2a2a); border-radius: 4px; margin-bottom: 1.5rem; border-left: 3px solid light-dark(#333, #e0e0e0);">
                        <p style="margin: 0; font-size: 0.95rem; color: light-dark(#333, #aaa);"><i class="fas fa-lock"></i> Your security is managed by <strong>Imators IDSMA</strong>. Password changes are encrypted and secured using industry-standard protocols.</p>
                    </div>

                    <form method="POST">
                        <div class="form-group">
                            <label class="form-label">Current Password</label>
                            <input type="password" name="password" class="form-input" placeholder="Current password" required>
                        </div>

                        <div class="form-group">
                            <label class="form-label">New Password</label>
                            <input type="password" name="new_password" class="form-input" placeholder="At least 8 characters" required>
                        </div>

                        <div class="form-group">
                            <label class="form-label">Confirm New Password</label>
                            <input type="password" name="confirm_password" class="form-input" placeholder="Confirm new password" required>
                        </div>

                        <button type="submit" name="update_password" class="btn btn-primary"><i class="fas fa-lock"></i> Update Password</button>
                    </form>
                </div>

                <div id="devices" class="content-section">
                    <h2 class="section-title">Active Devices</h2>

                    <p style="margin-bottom: 1.5rem; color: light-dark(#666, #aaa);">Manage your active sessions across devices.</p>

                    <form method="POST">
                        <div class="form-group">
                            <label class="form-label">Password</label>
                            <input type="password" name="password" class="form-input" placeholder="Your password" required>
                        </div>

                        <div class="action-buttons">
                            <button type="submit" name="logout" class="btn btn-secondary"><i class="fas fa-sign-out-alt"></i> Logout This Device</button>
                            <button type="submit" name="logout_all" class="btn btn-danger"><i class="fas fa-power-off"></i> Logout All Devices</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="modal-overlay" id="profileModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">
                    <i class="fas fa-user-plus"></i>
                    <span id="modalTitle">Add New Profile</span>
                </h3>
                <button type="button" class="modal-close" onclick="closeModal()"><i class="fas fa-times"></i></button>
            </div>
            <form method="POST" id="profileForm">
                <div class="modal-body">
                    <input type="hidden" id="profileId" name="profile_id">
                    <input type="hidden" id="profileAction" name="add_profile" value="1">

                    <div class="form-group">
                        <label class="form-label"><i class="fas fa-heading"></i> Profile Name</label>
                        <input type="text" id="profileName" name="profile_name" class="form-input" placeholder="Enter a unique profile name" required>
                    </div>

                    <div class="form-group">
                        <label class="form-label"><i class="fas fa-cake-candles"></i> Age Limit</label>
                        <select id="ageLimit" name="age_limit" class="form-input" required>
                            <option value="6">6+ (Kids Content)</option>
                            <option value="13">13+ (Teen Content)</option>
                            <option value="16">16+ (Mature Content)</option>
                            <option value="18" selected>18+ (Adult Content)</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label class="form-label"><i class="fas fa-image"></i> Choose Avatar</label>
                        <input type="hidden" id="selectedAvatar" name="profile_avatar" required>
                        <?php if (!empty($available_avatars)): ?>
                            <div class="avatar-tabs">
                                <div class="avatar-tab-buttons">
                                    <?php $first = true; foreach (array_keys($available_avatars) as $category): ?>
                                        <button type="button" class="avatar-tab-btn <?php echo $first ? 'active' : ''; ?>" onclick="switchAvatarTab(event, '<?php echo htmlspecialchars($category); ?>')">
                                            <?php echo htmlspecialchars(ucfirst($category)); ?>
                                        </button>
                                        <?php $first = false; endforeach; ?>
                                </div>
                                <?php $first = true; foreach ($available_avatars as $category => $avatars): ?>
                                    <div class="avatar-tab-content <?php echo $first ? 'active' : ''; ?>" id="tab-<?php echo htmlspecialchars($category); ?>">
                                        <div class="avatar-grid">
                                            <?php foreach ($avatars as $avatar): ?>
                                                <img src="<?php echo htmlspecialchars($avatar); ?>" alt="Avatar" class="avatar-option" onclick="selectAvatar('<?php echo htmlspecialchars($avatar); ?>', this)">
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                    <?php $first = false; endforeach; ?>
                            </div>
                        <?php else: ?>
                            <p style="grid-column: 1/-1; text-align: center; color: light-dark(#666, #aaa); padding: 2rem;">No avatars available.</p>
                        <?php endif; ?>
                    </div>

                    <div class="form-group">
                        <label class="form-label"><i class="fas fa-lock"></i> Confirm with your Imators Account password</label>
                        <input type="password" name="password" class="form-input" placeholder="Enter your password" required>
                    </div>
                </div>

                <div class="modal-footer">
                    <button type="button" onclick="closeModal()" class="btn btn-secondary"><i class="fas fa-times"></i> Cancel</button>
                    <button type="submit" id="submitBtn" class="btn btn-primary"><i class="fas fa-check"></i> Add Profile</button>
                </div>
            </form>
        </div>
    </div>

    <div class="modal-overlay" id="deleteModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">
                    <i class="fas fa-trash-alt"></i>
                    <span>Delete Profile</span>
                </h3>
                <button type="button" class="modal-close" onclick="closeDeleteModal()"><i class="fas fa-times"></i></button>
            </div>
            <form method="POST" id="deleteForm">
                <div class="modal-body">
                    <p style="margin-bottom: 1.5rem; color: light-dark(#666, #aaa);">Are you sure you want to delete the profile "<strong style="color: light-dark(#333, #e0e0e0);"><span id="deleteProfileName"></span></strong>"?</p>
                    <p style="margin: 0; font-size: 0.9rem; color: light-dark(#d32f2f, #ff6b6b);"><i class="fas fa-exclamation-triangle"></i> This action cannot be undone.</p>

                    <input type="hidden" id="deleteProfileId" name="profile_id">
                    <input type="hidden" name="delete_profile" value="1">

                    <div class="form-group" style="margin-top: 1.5rem;">
                        <label class="form-label"><i class="fas fa-lock"></i> Confirm with Password</label>
                        <input type="password" name="password" class="form-input" placeholder="Enter your password" required>
                    </div>
                </div>

                <div class="modal-footer">
                    <button type="button" onclick="closeDeleteModal()" class="btn btn-secondary"><i class="fas fa-times"></i> Cancel</button>
                    <button type="submit" class="btn btn-danger"><i class="fas fa-trash"></i> Delete Profile</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function switchAvatarTab(e, categoryName) {
            e.preventDefault();
            document.querySelectorAll('.avatar-tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            document.querySelectorAll('.avatar-tab-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            document.getElementById('tab-' + categoryName).classList.add('active');
            e.target.classList.add('active');
        }

        function showSection(sectionId, element) {
            document.querySelectorAll('.content-section').forEach(section => {
                section.classList.remove('active');
            });

            document.querySelectorAll('.nav-link').forEach(link => {
                link.classList.remove('active');
            });

            document.getElementById(sectionId).classList.add('active');
            element.classList.add('active');
        }

        function addProfile() {
            document.getElementById('modalTitle').textContent = 'Add New Profile';
            document.getElementById('profileForm').reset();
            document.getElementById('profileAction').name = 'add_profile';
            document.getElementById('submitBtn').textContent = 'Add Profile';
            document.querySelector('.avatar-option.selected')?.classList.remove('selected');
            document.getElementById('selectedAvatar').value = '';
            document.getElementById('profileModal').classList.add('active');
        }

        function editProfile(profile) {
            document.getElementById('modalTitle').textContent = 'Edit Profile';
            document.getElementById('profileId').value = profile.id;
            document.getElementById('profileName').value = profile.name;
            document.getElementById('ageLimit').value = profile.age_limit;
            document.getElementById('selectedAvatar').value = profile.avatar;
            document.getElementById('profileAction').name = 'edit_profile';
            document.getElementById('submitBtn').textContent = 'Update Profile';

            document.querySelector('.avatar-option.selected')?.classList.remove('selected');
            const avatarElements = document.querySelectorAll('.avatar-option');
            avatarElements.forEach(el => {
                if (el.getAttribute('src') === profile.avatar) {
                    el.classList.add('selected');
                }
            });

            document.getElementById('profileModal').classList.add('active');
        }

        function deleteProfile(id, name) {
            document.getElementById('deleteProfileId').value = id;
            document.getElementById('deleteProfileName').textContent = name;
            document.getElementById('deleteModal').classList.add('active');
        }

        function selectAvatar(src, element) {
            document.querySelector('.avatar-option.selected')?.classList.remove('selected');
            element.classList.add('selected');
            document.getElementById('selectedAvatar').value = src;
        }

        function closeModal() {
            document.getElementById('profileModal').classList.remove('active');
        }

        function closeDeleteModal() {
            document.getElementById('deleteModal').classList.remove('active');
        }

        document.getElementById('profileModal').addEventListener('click', function(e) {
            if (e.target === this) closeModal();
        });

        document.getElementById('deleteModal').addEventListener('click', function(e) {
            if (e.target === this) closeDeleteModal();
        });

        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
                closeDeleteModal();
            }
        });

        document.getElementById('profileForm').addEventListener('submit', function(e) {
            const selectedAvatar = document.getElementById('selectedAvatar').value;
            if (!selectedAvatar) {
                e.preventDefault();
                alert('Please select an avatar for the profile');
                return false;
            }
        });

        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth'
                    });
                }
            });
        });

        setTimeout(() => {
            document.querySelectorAll('.alert').forEach(alert => {
                alert.style.opacity = '0';
                alert.style.transform = 'translateY(-10px)';
                alert.style.transition = 'all 0.3s ease';
                setTimeout(() => alert.remove(), 300);
            });
        }, 5000);
    </script>
</body>
</html>