<?php
require_once '../client/guest/auth_check.php';
require_once '../client/guest/db.php';
session_start();

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$uploadDir = dirname(__FILE__) . '/../client/guest/documents/';
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0777, true);
}

// Gestion de l'upload de fichiers
if (isset($_POST['upload']) && !empty($_FILES['files']['name'][0])) {
    $allowedTypes = ['pdf'];
    $maxFileSize = 20 * 1024 * 1024; // 20MB
    
    $uploadedFiles = [];
    $errors = [];
    
    foreach ($_FILES['files']['tmp_name'] as $key => $tmp_name) {
        $fileName = $_FILES['files']['name'][$key];
        $fileSize = $_FILES['files']['size'][$key];
        $fileType = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        
        if (!in_array($fileType, $allowedTypes)) {
            $errors[] = "$fileName: Only PDF files are allowed.";
            continue;
        }
        
        if ($fileSize > $maxFileSize) {
            $errors[] = "$fileName: File size must be less than 20MB.";
            continue;
        }
        
        $newFileName = uniqid() . '_' . str_replace(' ', '_', $fileName);
        $destination = $uploadDir . $newFileName;
        
        if (move_uploaded_file($tmp_name, $destination)) {
            chmod($destination, 0644);
            
            try {
                $db2 = getDB2Connection();
                $stmt = $db2->prepare("INSERT INTO documents (`pdf-link`, created_at) VALUES (?, NOW())");
                $stmt->execute([$newFileName]);
                $uploadedFiles[] = $newFileName;
            } catch (PDOException $e) {
                $errors[] = "$fileName: Upload failed. Please try again.";
                if (file_exists($destination)) {
                    unlink($destination);
                }
            }
        } else {
            $errors[] = "$fileName: Upload failed. Please try again.";
        }
    }
}

// Gestion du partage de fichiers
if (isset($_POST['share_files']) && !empty($_POST['selected_files'])) {
    $selectedFiles = $_POST['selected_files'];
    $shareId = uniqid('share_', true);
    $expiryDate = date('Y-m-d H:i:s', strtotime('+7 days'));

    try {
        $db2 = getDB2Connection();
        $stmt = $db2->prepare("INSERT INTO shares (share_id, files, expiry_date, created_by) VALUES (?, ?, ?, ?)");
        $filesJson = json_encode($selectedFiles);
        $stmt->execute([$shareId, $filesJson, $expiryDate, $_SESSION['user_id']]);
        
        $shareUrl = "https://" . $_SERVER['HTTP_HOST'] . "/share/download.php?id=" . $shareId;
        header("Location: share.php?success=1&url=" . urlencode($shareUrl));
        exit;
    } catch (PDOException $e) {
        $errors[] = "Failed to create share link. Please try again.";
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Share Files - Imators Systems</title>
    <link href="https://cdn.imators.com/logo.png" rel="icon" type="image/png" />
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {
                    fontFamily: {
                        'poppins': ['Poppins', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style>
        .drag-over {
            border-color: #3B82F6;
            background-color: rgba(59, 130, 246, 0.1);
        }
    </style>
</head>
<body class="bg-white dark:bg-black text-black dark:text-white min-h-screen transition-colors duration-200 p-4 md:p-6 lg:p-8">
    <div class="max-w-[1400px] mx-auto">
        <?php if(isset($_GET['success']) && isset($_GET['url'])): ?>
        <div class="success-notification fixed top-4 right-4 bg-green-500 text-white px-6 py-3 rounded-lg shadow-lg z-50">
            <div class="flex items-center space-x-2">
                <i class="fas fa-check-circle"></i>
                <span>Share link created successfully!</span>
            </div>
            <div class="mt-2">
                <input type="text" value="<?= htmlspecialchars($_GET['url']) ?>" 
                       class="w-full px-3 py-1 rounded bg-white/20 text-white" readonly>
                <button onclick="copyLink(this)" class="text-sm hover:underline mt-1">
                    <i class="fas fa-copy"></i> Copy link
                </button>
            </div>
        </div>
        <?php endif; ?>

        <?php if(isset($_GET['error'])): ?>
        <div class="error-notification fixed top-4 right-4 bg-red-500 text-white px-6 py-3 rounded-lg shadow-lg z-50">
            <div class="flex items-center space-x-2">
                <i class="fas fa-exclamation-circle"></i>
                <span><?= htmlspecialchars($_GET['message']) ?></span>
            </div>
        </div>
        <?php endif; ?>

        <?php if(!empty($errors)): ?>
        <div class="fixed top-4 right-4 bg-red-500 text-white px-6 py-3 rounded-lg shadow-lg z-50">
            <?php foreach($errors as $error): ?>
            <div class="flex items-center space-x-2 mb-2">
                <i class="fas fa-exclamation-circle"></i>
                <span><?= htmlspecialchars($error) ?></span>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>

        <?php if(!empty($uploadedFiles)): ?>
        <div class="fixed top-4 right-4 bg-green-500 text-white px-6 py-3 rounded-lg shadow-lg z-50">
            <div class="flex items-center space-x-2">
                <i class="fas fa-check-circle"></i>
                <span><?= count($uploadedFiles) ?> file(s) uploaded successfully!</span>
            </div>
        </div>
        <?php endif; ?>

        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
            <div class="flex items-center justify-between mb-8">
                <div class="flex items-center space-x-4">
                    <img src="https://cdn.imators.com/logo.png" alt="Logo" class="h-10 w-auto">
                    <div>
                        <h1 class="text-2xl font-bold">Share Files</h1>
                        <p class="text-sm text-gray-600 dark:text-gray-400">Upload and share files securely</p>
                    </div>
                </div>
                <button id="theme-toggle" class="p-2 rounded-lg bg-gray-100 dark:bg-gray-800">
                    <i class="fas fa-sun dark:hidden"></i>
                    <i class="fas fa-moon hidden dark:block"></i>
                </button>
            </div>

            <!-- Upload Section -->
            <div class="mb-8">
                <div class="bg-gray-100 dark:bg-gray-800 rounded-lg p-6">
                    <h2 class="text-xl font-medium flex items-center space-x-2 mb-4">
                        <i class="fas fa-cloud-upload-alt text-gray-600 dark:text-gray-400"></i>
                        <span>Upload Files</span>
                    </h2>

                    <form action="" method="POST" enctype="multipart/form-data" class="space-y-4">
                        <div id="drop-zone" 
                             class="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-8 text-center">
                            <input type="file" id="file-input" name="files[]" multiple accept=".pdf" class="hidden">
                            <div class="space-y-2">
                                <i class="fas fa-cloud-upload-alt text-4xl text-gray-400"></i>
                                <p class="text-lg">Drag and drop your files here</p>
                                <p class="text-sm text-gray-500">or</p>
                                <button type="button" onclick="document.getElementById('file-input').click()" 
                                        class="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors">
                                    Browse Files
                                </button>
                            </div>
                            <p class="text-sm text-gray-500 mt-2">Maximum file size: 20MB - PDF files only</p>
                        </div>

                        <div id="file-list" class="space-y-2"></div>

                        <div class="flex justify-end">
                            <button type="submit" name="upload" 
                                    class="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 
                                           transition-colors flex items-center space-x-2">
                                <i class="fas fa-upload"></i>
                                <span>Upload Files</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Share Section -->
            <form action="" method="POST" class="space-y-6">
                <div class="bg-gray-100 dark:bg-gray-800 rounded-lg p-6">
                    <div class="mb-4">
                        <h2 class="text-xl font-medium flex items-center space-x-2">
                            <i class="fas fa-share-alt text-gray-600 dark:text-gray-400"></i>
                            <span>Share Files</span>
                        </h2>
                        <p class="text-sm text-gray-600 dark:text-gray-400 mt-1">
                            Select the files you want to share
                        </p>
                    </div>

                    <div class="space-y-2">
                        <?php
                        $db2 = getDB2Connection();
                        $query = $db2->query("SELECT * FROM documents ORDER BY created_at DESC");
                        while($doc = $query->fetch(PDO::FETCH_ASSOC)) {
                            $filePath = $doc['pdf-link'];
                            $fullPath = '../client/guest/documents/' . $filePath;
                            if (file_exists($fullPath)) {
                        ?>
                            <label class="flex items-center space-x-3 p-4 bg-white dark:bg-gray-900 rounded-lg cursor-pointer 
                                       hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                                <input type="checkbox" name="selected_files[]" 
                                       value="<?= htmlspecialchars($fullPath) ?>" 
                                       class="rounded border-gray-300 dark:border-gray-600 text-blue-500">
                                <div class="flex items-center space-x-3 flex-grow">
                                    <i class="fas fa-file-pdf text-red-500"></i>
                                    <span class="flex-grow"><?= htmlspecialchars(basename($filePath)) ?></span>
                                    <span class="text-xs text-gray-500">
                                        <?= date('Y-m-d H:i', strtotime($doc['created_at'])) ?>
                                    </span>
                                </div>
                            </label>
                        <?php
                            }
                        }
                        ?>
                    </div>
                </div>

                <div class="flex justify-end space-x-3">
                    <a href="../client/dashboard.php" class="px-4 py-2 border border-gray-200 dark:border-gray-600 rounded-lg 
                              text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 
                              transition-colors flex items-center space-x-2">
                        <i class="fas fa-arrow-left"></i>
                        <span>Back</span>
                    </a>
                    <button type="submit" name="share_files" class="px-4 py-2 bg-black dark:bg-white text-white dark:text-black 
                                   rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors 
                                   flex items-center space-x-2">
                        <i class="fas fa-share"></i>
                        <span>Generate Share Link</span>
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
    // Copier le lien
    function copyLink(button) {
        const input = button.previousElementSibling;
        input.select();
        document.execCommand('copy');
        
        const originalText = button.innerHTML;
        button.innerHTML = '<i class="fas fa-check"></i> Copied!';
        setTimeout(() => {
            button.innerHTML = originalText;
        }, 2000);
    }

    // Mode sombre
    document.getElementById('theme-toggle').addEventListener('click', () => {
        document.documentElement.classList.toggle('dark');
        localStorage.theme = document.documentElement.classList.contains('dark') ? 'dark' : 'light';
    });

    if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
        document.documentElement.classList.add('dark');
    } else {
        document.documentElement.classList.remove('dark');
    }

    // Gestion du drag & drop
    const dropZone = document.getElementById('drop-zone');
    const fileInput = document.getElementById('file-input');
    const fileList = document.getElementById('file-list');

    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        dropZone.addEventListener(eventName, preventDefaults, false);
    });

    function preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }

    ['dragenter', 'dragover'].forEach(eventName => {
        dropZone.addEventListener(eventName, highlight, false);
    });

    ['dragleave', 'drop'].forEach(eventName => {
        dropZone.addEventListener(eventName, unhighlight, false);
    });

    function highlight(e) {
        dropZone.classList.add('drag-over');
    }

    function unhighlight(e) {
        dropZone.classList.remove('drag-over');
    }

    dropZone.addEventListener('drop', handleDrop, false);
    fileInput.addEventListener('change', handleFiles, false);

    function handleDrop(e) {
        const dt = e.dataTransfer;
        const files = dt.files;
        handleFiles({ target: { files: files } });
    }

    function handleFiles(e) {
        const files = [...e.target.files];
        updateFileList(files);
    }

    function updateFileList(files) {
        fileInput.files = new FileListItem(files);
        fileList.innerHTML = '';

        files.forEach((file, index) => {
            const isValid = validateFile(file);
            const div = document.createElement('div');
            div.className = `flex items-center justify-between p-4 bg-white dark:bg-gray-900 rounded-lg ${isValid ? '' : 'border-red-500 border'}`;
            
            div.innerHTML = `
                <div class="flex items-center space-x-3">
                    <i class="fas fa-file-pdf text-red-500"></i>
                    <div>
                        <div class="font-medium">${file.name}</div>
                        <div class="text-sm text-gray-500">${formatFileSize(file.size)}</div>
                    </div>
                </div>
                <div class="flex items-center space-x-2">
                    ${!isValid ? `<span class="text-sm text-red-500">Invalid file</span>` : ''}
                    <button type="button" onclick="removeFile(${index})" class="text-red-500 hover:text-red-600">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `;
            
            fileList.appendChild(div);
        });
    }

    function validateFile(file) {
        const maxSize = 20 * 1024 * 1024; // 20MB
        const validTypes = ['application/pdf'];
        return file.size <= maxSize && validTypes.includes(file.type);
    }

    function formatFileSize(bytes) {
        const units = ['B', 'KB', 'MB', 'GB'];
        let size = bytes;
        let unitIndex = 0;
        
        while (size >= 1024 && unitIndex < units.length - 1) {
            size /= 1024;
            unitIndex++;
        }
        
        return `${size.toFixed(1)} ${units[unitIndex]}`;
    }

    function removeFile(index) {
        const files = [...fileInput.files];
        files.splice(index, 1);
        updateFileList(files);
    }

    // Classe utilitaire pour créer un FileList à partir d'un tableau de fichiers
    function FileListItem(files) {
        const b = new ClipboardEvent("").clipboardData || new DataTransfer();
        for (let file of files) b.items.add(file);
        return b.files;
    }

    // Auto-hide notifications
    document.querySelectorAll('.success-notification, .error-notification').forEach(notification => {
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transition = 'opacity 0.3s ease-out';
            setTimeout(() => notification.remove(), 300);
        }, 5000);
    });
</script>
</body>
</html>