<?php
require_once 'config.php';

if(!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

if(isset($_POST['upload'])) {
    $name = $_POST['name'];
    $description = $_POST['description'];
    $tags = $_POST['tags'];
    $time = $_POST['time'];
    
    $uploadDir = 'uploads/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }

    $video = $_FILES['video_file'];
    $cover = $_FILES['cover_file'];
    $allowedVideoTypes = ['video/mp4', 'video/webm', 'video/ogg'];
    $allowedImageTypes = ['image/jpeg', 'image/png', 'image/webp'];
    $maxFileSize = 500 * 1024 * 1024;

    if($video['size'] > $maxFileSize || $cover['size'] > $maxFileSize) {
        $error = "File size too large. Maximum size is 500MB";
    } else {
        $videoFileType = mime_content_type($video['tmp_name']);
        $coverFileType = mime_content_type($cover['tmp_name']);

        if(!in_array($videoFileType, $allowedVideoTypes)) {
            $error = "Invalid video format. Allowed formats: MP4, WebM, OGG";
        } elseif(!in_array($coverFileType, $allowedImageTypes)) {
            $error = "Invalid image format. Allowed formats: JPEG, PNG, WebP";
        } else {
            $videoFileName = uniqid() . '_' . basename($video['name']);
            $coverFileName = uniqid() . '_' . basename($cover['name']);
            
            $videoPath = $uploadDir . $videoFileName;
            $coverPath = $uploadDir . $coverFileName;

            if(move_uploaded_file($video['tmp_name'], $videoPath) && 
               move_uploaded_file($cover['tmp_name'], $coverPath)) {
                
                $stmt = $pdo->prepare("INSERT INTO content (name, description, tags, time, url_video, picture_cover, author_id) VALUES (?, ?, ?, ?, ?, ?, ?)");
                try {
                    $stmt->execute([
                        $name, 
                        $description, 
                        $tags, 
                        $time, 
                        $videoPath, 
                        $coverPath, 
                        $_SESSION['user_id']
                    ]);
                    header('Location: index.php');
                    exit;
                } catch(PDOException $e) {
                    $error = "Error uploading video to database";
                    unlink($videoPath);
                    unlink($coverPath);
                }
            } else {
                $error = "Error uploading files";
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload Video - ForMore</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Titillium+Web:wght@200;300;400;600;700;900&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Titillium Web', sans-serif;
        }
        .upload-zone {
            transition: all 0.3s ease;
            border: 2px dashed #4B5563;
        }
        .upload-zone:hover, .upload-zone.dragging {
            border-color: #60A5FA;
            background-color: rgba(37, 99, 235, 0.1);
        }
    </style>
</head>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white p-8">
    <nav class="flex justify-between items-center mb-12">
        <a href="index.php" class="text-4xl font-black formore">ForMore</a>
        <a href="index.php" class="text-blue-400 hover:text-blue-300 transition-colors">Back to Home</a>
    </nav>

    <div class="max-w-4xl mx-auto">
        <div class="bg-gray-800 rounded-xl p-8 shadow-2xl">
            <h2 class="text-3xl font-black mb-8">Upload Your Video</h2>

            <?php if(isset($error)): ?>
                <div class="bg-red-500 text-white p-4 rounded-lg mb-6"><?= $error ?></div>
            <?php endif; ?>

            <form method="POST" enctype="multipart/form-data" class="space-y-6">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-gray-300 font-semibold mb-2">Video Title</label>
                        <input type="text" name="name" required 
                            class="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 focus:outline-none focus:border-blue-500 transition-colors">
                    </div>

                    <div>
                        <label class="block text-gray-300 font-semibold mb-2">Duration</label>
                        <input type="text" name="time" required pattern="[0-9]+:[0-9]+" placeholder="5:30"
                            class="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 focus:outline-none focus:border-blue-500 transition-colors">
                    </div>
                </div>

                <div>
                    <label class="block text-gray-300 font-semibold mb-2">Video File</label>
                    <div class="upload-zone rounded-lg p-8 text-center" id="videoDrop">
                        <input type="file" name="video_file" accept="video/*" required
                            class="hidden" id="videoInput">
                        <label for="videoInput" class="cursor-pointer">
                            <div class="flex flex-col items-center">
                                <svg class="w-12 h-12 text-gray-400 mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"/>
                                </svg>
                                <span class="text-gray-300">Drag & drop your video or click to browse</span>
                                <span class="text-gray-500 text-sm mt-2">MP4, WebM, OGG (max. 500MB)</span>
                            </div>
                        </label>
                        <div id="videoName" class="mt-4 text-blue-400"></div>
                    </div>
                </div>

                <div>
                    <label class="block text-gray-300 font-semibold mb-2">Cover Image</label>
                    <div class="upload-zone rounded-lg p-8 text-center" id="coverDrop">
                        <input type="file" name="cover_file" accept="image/*" required
                            class="hidden" id="coverInput">
                        <label for="coverInput" class="cursor-pointer">
                            <div class="flex flex-col items-center">
                                <svg class="w-12 h-12 text-gray-400 mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                </svg>
                                <span class="text-gray-300">Drag & drop your cover image or click to browse</span>
                                <span class="text-gray-500 text-sm mt-2">JPEG, PNG, WebP (max. 500MB)</span>
                            </div>
                        </label>
                        <div id="coverPreview" class="mt-4"></div>
                    </div>
                </div>

                <div>
                    <label class="block text-gray-300 font-semibold mb-2">Tags</label>
                    <input type="text" name="tags" required 
                        class="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 focus:outline-none focus:border-blue-500 transition-colors"
                        placeholder="gaming, minecraft, tutorial">
                    <p class="text-gray-400 text-sm mt-1">Separate tags with commas</p>
                </div>

                <div>
                    <label class="block text-gray-300 font-semibold mb-2">Description</label>
                    <textarea name="description" required rows="4" 
                        class="w-full bg-gray-700 border border-gray-600 rounded-lg px-4 py-3 focus:outline-none focus:border-blue-500 transition-colors"></textarea>
                </div>

                <button type="submit" name="upload" 
                    class="w-full bg-blue-600 hover:bg-blue-700 text-white font-bold py-3 px-4 rounded-lg transition-colors">
                    Upload Video
                </button>
            </form>
        </div>
    </div>

    <script>
        function handleDrop(e, input, preview) {
            e.preventDefault();
            e.stopPropagation();
            e.currentTarget.classList.remove('dragging');
            
            const file = e.dataTransfer.files[0];
            if (file) {
                input.files = e.dataTransfer.files;
                updatePreview(file, preview);
            }
        }

        function handleDragOver(e) {
            e.preventDefault();
            e.stopPropagation();
            e.currentTarget.classList.add('dragging');
        }

        function handleDragLeave(e) {
            e.currentTarget.classList.remove('dragging');
        }

        function updatePreview(file, previewElement) {
            if (file.type.startsWith('video/')) {
                previewElement.textContent = `Selected: ${file.name}`;
            } else if (file.type.startsWith('image/')) {
                const img = document.createElement('img');
                img.classList.add('mx-auto', 'max-h-32', 'rounded');
                img.file = file;

                const reader = new FileReader();
                reader.onload = (e) => img.src = e.target.result;
                reader.readAsDataURL(file);

                previewElement.innerHTML = '';
                previewElement.appendChild(img);
            }
        }

        ['video', 'cover'].forEach(type => {
            const dropZone = document.getElementById(`${type}Drop`);
            const input = document.getElementById(`${type}Input`);
            const preview = document.getElementById(type === 'video' ? 'videoName' : 'coverPreview');

            dropZone.addEventListener('dragover', handleDragOver);
            dropZone.addEventListener('dragleave', handleDragLeave);
            dropZone.addEventListener('drop', e => handleDrop(e, input, preview));

            input.addEventListener('change', e => {
                const file = e.target.files[0];
                if (file) updatePreview(file, preview);
            });
        });
    </script>
</body>
</html>