<?php
ini_set('session.gc_maxlifetime', 31536000);
ini_set('session.cookie_lifetime', 31536000);
session_set_cookie_params([
    'lifetime' => 31536000,
    'path' => '/',
    'domain' => $_SERVER['HTTP_HOST'],
    'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
    'httponly' => true,
    'samesite' => 'Lax'
]);
session_start();
include 'db.php';
include 'header-mobile.php';

if (!isset($_SESSION['resident_id'])) {
    header('Location: login.php');
    exit;
}

$resident_id = $_SESSION['resident_id'];
$stmt = $conn->prepare("SELECT points, first_name FROM residents WHERE id = ?");
$stmt->bind_param("i", $resident_id);
$stmt->execute();
$resident = $stmt->get_result()->fetch_assoc();

$points = $resident['points'];
$first_name = $resident['first_name'];

$tiers = [
    ['name' => 'Basic', 'min' => 0, 'max' => 100, 'reward' => null, 'reward_at' => null],
    ['name' => 'Evolve', 'min' => 100, 'max' => 500, 'reward' => 'Bon de 20€ chez un commerce au choix', 'reward_at' => 500],
    ['name' => 'Hunter', 'min' => 500, 'max' => 1000, 'reward' => null, 'reward_at' => null],
    ['name' => 'Quarter', 'min' => 1000, 'max' => 1500, 'reward' => 'Panier cadeau d\'une valeur de 50€', 'reward_at' => 1500]
];

$currentTier = null;
$nextTier = null;
$progress = 0;

foreach ($tiers as $index => $tier) {
    if ($points >= $tier['min'] && $points < $tier['max']) {
        $currentTier = $tier;
        if (isset($tiers[$index + 1])) {
            $nextTier = $tiers[$index + 1];
        }
        break;
    }
}

if ($points >= 1500) {
    $currentTier = end($tiers);
    $progress = 100;
} elseif ($currentTier && $nextTier) {
    $progress = (($points - $currentTier['min']) / ($nextTier['min'] - $currentTier['min'])) * 100;
    $progress = min(100, max(0, $progress));
}

$pointsToNext = $nextTier ? ($nextTier['min'] - $points) : 0;

$usage_stmt = $conn->prepare("SELECT shop_id, used_at FROM card_usage WHERE resident_id = ? ORDER BY used_at DESC LIMIT 10");
$usage_stmt->bind_param("i", $resident_id);
$usage_stmt->execute();
$usage_result = $usage_stmt->get_result();

$usages = [];
while ($usage = $usage_result->fetch_assoc()) {
    $shop_stmt = $conn->prepare("SELECT title FROM shop WHERE id = ?");
    $shop_stmt->bind_param("i", $usage['shop_id']);
    $shop_stmt->execute();
    $shop = $shop_stmt->get_result()->fetch_assoc();
    
    $usages[] = [
        'shop' => $shop['title'] ?? 'Commerce inconnu',
        'date' => date('d/m/Y à H:i', strtotime($usage['used_at']))
    ];
}

$rewards_stmt = $conn->prepare("SELECT * FROM rewards WHERE resident_id = ? ORDER BY points_required ASC");
$rewards_stmt->bind_param("i", $resident_id);
$rewards_stmt->execute();
$rewards_result = $rewards_stmt->get_result();
$rewards = [];
while ($reward = $rewards_result->fetch_assoc()) {
    $rewards[] = $reward;
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mes Points - Vert Chasseur</title>
    <link rel="icon" type="image/png" href="logo_new.png">
    
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: { extend: {} }
        }
    </script>
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            letter-spacing: -0.01em;
        }

        .circle-progress {
            transform: rotate(-90deg);
        }

        .circle-bg {
            stroke: rgba(0, 0, 0, 0.05);
        }

        .dark .circle-bg {
            stroke: rgba(255, 255, 255, 0.05);
        }

        .circle-fill {
            stroke: #059669;
            stroke-linecap: round;
        }

        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .animate-fade-in-up {
            animation: fadeInUp 0.8s ease-out forwards;
            opacity: 0;
        }

        .tier-item {
            transition: all 0.3s ease;
        }

        .tier-item:hover {
            transform: translateX(4px);
        }
    </style>
</head>

<body class="bg-white dark:bg-black text-black dark:text-white transition-colors duration-300">
    <script>
        if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            document.documentElement.classList.add('dark');
        }
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', e => {
            if (e.matches) {
                document.documentElement.classList.add('dark');
            } else {
                document.documentElement.classList.remove('dark');
            }
        });
    </script>

    <main class="px-4 py-8 md:py-16 max-w-4xl mx-auto">
        <div class="flex flex-col items-center justify-center mb-12 text-center">
            <h1 class="mt-4 md:mt-12 text-3xl md:text-5xl font-light text-stone-800 dark:text-white mb-2 tracking-tight">
                Mes Points
            </h1>
            <p class="text-lg text-stone-600 dark:text-stone-400 font-light">
                Salutations <?= htmlspecialchars($first_name) ?>, voici ta progression
            </p>
        </div>

        <div class="bg-white dark:bg-stone-900 rounded-3xl p-8 md:p-12 border border-stone-200 dark:border-stone-800 mb-8 animate-fade-in-up" style="animation-delay: 0.2s">
            <div class="flex flex-col items-center">
                <div class="relative mb-8">
                    <svg class="circle-progress" width="280" height="280" viewBox="0 0 280 280">
                        <circle class="circle-bg" cx="140" cy="140" r="120" stroke-width="20" fill="none"/>
                        <circle class="circle-fill" cx="140" cy="140" r="120" stroke-width="20" fill="none"
                                stroke-dasharray="753.98" 
                                stroke-dashoffset="753.98"
                                id="progressCircle"/>
                    </svg>
                    
                    <div class="absolute inset-0 flex flex-col items-center justify-center">
                        <div class="text-5xl md:text-6xl font-light text-black dark:text-white mb-1" id="pointsCounter">0</div>
                        <div class="text-sm text-stone-500 dark:text-stone-400">points</div>
                    </div>
                </div>

                <div class="text-center mb-8">
                    <div class="text-xl font-medium text-black dark:text-white mb-2">
                        Palier <?= htmlspecialchars($currentTier['name']) ?>
                    </div>
                    <?php if ($nextTier): ?>
                    <div class="text-sm text-stone-600 dark:text-stone-400">
                        Plus que <span class="font-semibold text-green-600 dark:text-green-400"><?= $pointsToNext ?> points</span> pour atteindre <?= htmlspecialchars($nextTier['name']) ?>
                    </div>
                    <?php else: ?>
                    <div class="text-sm text-green-600 dark:text-green-400 font-medium">
                        Félicitations ! Tu as atteint le palier maximum
                    </div>
                    <?php endif; ?>
                </div>

                <div class="w-full max-w-md">
                    <div class="text-xs font-medium text-stone-500 dark:text-stone-400 mb-2 text-center">
                        Progression : <span id="progressPercent">0</span>%
                    </div>
                    <div class="h-2 bg-stone-100 dark:bg-stone-800 rounded-full overflow-hidden">
                        <div class="h-full bg-gradient-to-r from-green-500 to-green-600 rounded-full transition-all"
                             style="width: 0%"
                             id="progressBar"></div>
                    </div>
                </div>
            </div>
        </div>

        <div class="bg-white dark:bg-stone-900 rounded-3xl p-8 border border-stone-200 dark:border-stone-800 mb-8 animate-fade-in-up" style="animation-delay: 0.4s">
            <h2 class="text-2xl font-light text-black dark:text-white mb-6">Paliers et récompenses</h2>
            
            <div class="space-y-4">
                <?php foreach ($tiers as $tier): ?>
                <div class="tier-item p-5 rounded-2xl border <?= $currentTier['name'] === $tier['name'] ? 'border-green-500 bg-green-50 dark:bg-green-900/10' : 'border-stone-200 dark:border-stone-800' ?>">
                    <div class="flex items-start justify-between mb-2">
                        <div class="flex-1">
                            <div class="flex items-center gap-3 mb-1">
                                <h3 class="text-lg font-semibold text-black dark:text-white"><?= htmlspecialchars($tier['name']) ?></h3>
                                <?php if ($currentTier['name'] === $tier['name']): ?>
                                <span class="text-xs px-3 py-1 rounded-full bg-green-600 text-white">Actuel</span>
                                <?php elseif ($points >= $tier['max']): ?>
                                <span class="text-xs px-3 py-1 rounded-full bg-stone-200 dark:bg-stone-700 text-stone-700 dark:text-stone-300">Complété</span>
                                <?php endif; ?>
                            </div>
                            <p class="text-sm text-stone-600 dark:text-stone-400"><?= $tier['min'] ?> - <?= $tier['max'] ?> points</p>
                        </div>
                    </div>
                    
                    <?php if ($tier['reward']): ?>
                    <div class="mt-3 pt-3 border-t border-stone-200 dark:border-stone-700">
                        <div class="flex items-start gap-2 mb-2">
                            <svg class="w-4 h-4 text-green-600 dark:text-green-400 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v13m0-13V6a2 2 0 112 2h-2zm0 0V5.5A2.5 2.5 0 109.5 8H12zm-7 4h14M5 12a2 2 0 110-4h14a2 2 0 110 4M5 12v7a2 2 0 002 2h10a2 2 0 002-2v-7"/>
                            </svg>
                            <span class="text-sm text-green-700 dark:text-green-400 font-medium"><?= htmlspecialchars($tier['reward']) ?></span>
                        </div>
                        <div class="ml-6 text-xs text-stone-500 dark:text-stone-400">
                            <?php
                            $hasReward = false;
                            foreach ($rewards as $reward) {
                                if ($reward['tier'] === $tier['name']) {
                                    $hasReward = true;
                                    if ($reward['status'] === 'pending') {
                                        echo '<span class="text-amber-600 dark:text-amber-400 font-medium">⏳ Récompense disponible ! Contacte-nous pour la récupérer</span>';
                                    } else {
                                        echo '<span class="text-green-600 dark:text-green-400">✓ Récompense réclamée le ' . date('d/m/Y', strtotime($reward['claimed_at'])) . '</span>';
                                    }
                                    break;
                                }
                            }
                            if (!$hasReward) {
                                if ($points >= $tier['reward_at']) {
                                    echo '<span class="text-amber-600 dark:text-amber-400 font-medium">✓ Récompense débloquée ! Contacte-nous pour la récupérer</span>';
                                } else {
                                    echo '<span class="text-stone-400 dark:text-stone-500">Récompense disponible à partir de ' . $tier['reward_at'] . ' points</span>';
                                }
                            }
                            ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            </div>
            
            <div class="mt-6 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-xl border border-blue-200 dark:border-blue-800">
                <div class="flex items-start gap-3">
                    <svg class="w-5 h-5 text-blue-600 dark:text-blue-400 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    <div class="text-sm text-blue-800 dark:text-blue-200">
                        <p class="font-medium mb-1">Comment récupérer mes récompenses ?</p>
                        <p class="text-xs">Les récompenses se débloquent à <strong>500 points</strong> (bon de 20€) et <strong>1500 points</strong> (panier cadeau 50€). Tu peux nous contacter via les paramètres de l'application pour organiser la récupération de ta récompense.</p>
                    </div>
                </div>
            </div>
        </div>

        <?php if (!empty($usages)): ?>
        <div class="bg-white dark:bg-stone-900 rounded-3xl p-8 border border-stone-200 dark:border-stone-800 animate-fade-in-up" style="animation-delay: 0.6s">
            <h2 class="text-2xl font-light text-black dark:text-white mb-6">Historique récent</h2>
            
            <div class="space-y-3">
                <?php foreach ($usages as $usage): ?>
                <div class="flex items-center justify-between p-4 rounded-xl bg-stone-50 dark:bg-stone-800/50">
                    <div class="flex items-center gap-3">
                        <div class="w-10 h-10 rounded-full bg-green-100 dark:bg-green-900/30 flex items-center justify-center flex-shrink-0">
                            <svg class="w-5 h-5 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/>
                            </svg>
                        </div>
                        <div>
                            <div class="text-sm font-medium text-black dark:text-white"><?= htmlspecialchars($usage['shop']) ?></div>
                            <div class="text-xs text-stone-500 dark:text-stone-400"><?= htmlspecialchars($usage['date']) ?></div>
                        </div>
                    </div>
                    <div class="text-sm font-semibold text-green-600 dark:text-green-400 flex-shrink-0">+5 pts</div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>

        <div class="text-center mt-8 text-sm text-stone-500 dark:text-stone-400 animate-fade-in-up" style="animation-delay: 0.8s">
            <p>Gagne 5 points à chaque utilisation de ta carte membre</p>
        </div>
    </main>

    <script>
        const targetPoints = <?= $points ?>;
        const targetProgress = <?= $progress ?>;
        const circumference = 753.98;

        function animateValue(callback, duration) {
            const startTime = performance.now();
            
            function animate(currentTime) {
                const elapsed = currentTime - startTime;
                const progress = Math.min(elapsed / duration, 1);
                
                callback(progress);
                
                if (progress < 1) {
                    requestAnimationFrame(animate);
                }
            }
            
            requestAnimationFrame(animate);
        }

        window.addEventListener('load', () => {
            const counter = document.getElementById('pointsCounter');
            const circle = document.getElementById('progressCircle');
            const bar = document.getElementById('progressBar');
            const percent = document.getElementById('progressPercent');
            
            animateValue((progress) => {
                const currentPoints = Math.floor(targetPoints * progress);
                counter.textContent = currentPoints;
                
                const currentPercent = targetProgress * progress;
                const offset = circumference - (circumference * currentPercent / 100);
                circle.style.strokeDashoffset = offset;
                bar.style.width = currentPercent + '%';
                percent.textContent = Math.round(currentPercent);
            }, 1500);
        });
    </script>
</body>
</html>