<?php
require_once 'db.php';
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Méthode non autorisée']);
    exit;
}

if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
    http_response_code(403);
    echo json_encode(['error' => 'Token CSRF invalide']);
    exit;
}

$action = $_POST['action'] ?? '';

switch ($action) {
    case 'search_addresses':
        $query = sanitizeInput($_POST['query']);
        $addresses = searchAddresses($query);
        echo json_encode($addresses);
        break;
        
    case 'get_order_updates':
        $commandeId = (int)$_POST['commande_id'];
        $lastUpdate = $_POST['last_update'] ?? '1970-01-01 00:00:00';
        $updates = getOrderUpdates($commandeId, $lastUpdate);
        echo json_encode($updates);
        break;
        
    case 'get_available_drivers':
        $creneauDebut = $_POST['creneau_debut'];
        $creneauFin = $_POST['creneau_fin'];
        $latitude = floatval($_POST['latitude']);
        $longitude = floatval($_POST['longitude']);
        $drivers = getAvailableDrivers($creneauDebut, $creneauFin, $latitude, $longitude);
        echo json_encode($drivers);
        break;
        
    case 'calculate_delivery_cost':
        $driverLat = floatval($_POST['driver_lat']);
        $driverLon = floatval($_POST['driver_lon']);
        $clientLat = floatval($_POST['client_lat']);
        $clientLon = floatval($_POST['client_lon']);
        $nbArticles = (int)$_POST['nb_articles'];
        $cost = calculateDeliveryCost($driverLat, $driverLon, $clientLat, $clientLon, $nbArticles);
        echo json_encode($cost);
        break;
        
    case 'validate_address':
        $address = sanitizeInput($_POST['address']);
        $validation = validateDeliveryAddress($address);
        echo json_encode($validation);
        break;
        
    case 'get_driver_location':
        session_start();
        if (!isset($_SESSION['livreur_id'])) {
            echo json_encode(['error' => 'Non connecté']);
            exit;
        }
        $location = getDriverLocation($_SESSION['livreur_id']);
        echo json_encode($location);
        break;
        
    case 'update_driver_location':
        session_start();
        if (!isset($_SESSION['livreur_id'])) {
            echo json_encode(['error' => 'Non connecté']);
            exit;
        }
        $latitude = floatval($_POST['latitude']);
        $longitude = floatval($_POST['longitude']);
        $result = updateDriverLocation($_SESSION['livreur_id'], $latitude, $longitude);
        echo json_encode($result);
        break;
        
    case 'get_live_orders':
        session_start();
        if (!isset($_SESSION['livreur_id'])) {
            echo json_encode(['error' => 'Non connecté']);
            exit;
        }
        $orders = getLiveOrders($_SESSION['livreur_id']);
        echo json_encode($orders);
        break;
        
    case 'get_unread_messages':
        $commandeId = (int)$_POST['commande_id'];
        $userType = sanitizeInput($_POST['user_type']);
        $messages = getUnreadMessages($commandeId, $userType);
        echo json_encode($messages);
        break;
        
    case 'mark_messages_read':
        $commandeId = (int)$_POST['commande_id'];
        $userType = sanitizeInput($_POST['user_type']);
        $result = markMessagesRead($commandeId, $userType);
        echo json_encode($result);
        break;
        
    case 'get_order_tracking':
        $commandeId = (int)$_POST['commande_id'];
        $tracking = getOrderTracking($commandeId);
        echo json_encode($tracking);
        break;
        
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Action non reconnue']);
}

function searchAddresses($query) {
    if (strlen($query) < 3) {
        return ['results' => []];
    }
    
    $url = "https://nominatim.openstreetmap.org/search?format=json&q=" . urlencode($query . ", Brussels, Belgium") . "&limit=5&addressdetails=1";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_USERAGENT, 'VertChasseur/1.0');
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && $response) {
        $data = json_decode($response, true);
        
        $filteredResults = array_filter($data, function($result) {
            $address = strtolower($result['display_name']);
            return strpos($address, 'uccle') !== false || strpos($address, 'fort jaco') !== false;
        });
        
        $results = array_map(function($result) {
            return [
                'display_name' => $result['display_name'],
                'lat' => floatval($result['lat']),
                'lon' => floatval($result['lon']),
                'address' => $result['address'] ?? []
            ];
        }, array_values($filteredResults));
        
        return ['results' => $results];
    }
    
    return ['results' => []];
}

function getOrderUpdates($commandeId, $lastUpdate) {
    $commande = db()->fetchOne("
        SELECT c.*, l.prenom as livreur_prenom, l.nom as livreur_nom, l.photo_profil,
               r.first_name, r.last_name
        FROM vr_commandes c
        LEFT JOIN vr_livreurs l ON c.livreur_id = l.id
        LEFT JOIN residents r ON c.resident_id = r.id
        WHERE c.id = ? AND (c.date_commande > ? OR c.date_confirmation > ? OR c.date_livraison > ?)
    ", [$commandeId, $lastUpdate, $lastUpdate, $lastUpdate]);
    
    $messages = db()->fetchAll("
        SELECT * FROM vr_messages 
        WHERE commande_id = ? AND created_at > ?
        ORDER BY created_at ASC
    ", [$commandeId, $lastUpdate]);
    
    return [
        'commande' => $commande,
        'messages' => $messages,
        'timestamp' => date('Y-m-d H:i:s')
    ];
}

function getAvailableDrivers($creneauDebut, $creneauFin, $latitude, $longitude) {
    $currentDay = date('w');
    
    $drivers = db()->fetchAll("
        SELECT l.*, AVG(c.note_client) as note_moyenne,
               (6371 * acos(cos(radians(?)) * cos(radians(l.latitude)) * 
               cos(radians(l.longitude) - radians(?)) + sin(radians(?)) * 
               sin(radians(l.latitude)))) AS distance
        FROM vr_livreurs l 
        LEFT JOIN vr_disponibilites d ON l.id = d.livreur_id 
        LEFT JOIN vr_commandes c ON l.id = c.livreur_id AND c.note_client IS NOT NULL
        WHERE l.actif = 1 
        AND d.jour_semaine = ? 
        AND d.heure_debut <= ? 
        AND d.heure_fin >= ?
        AND l.id NOT IN (
            SELECT livreur_id FROM vr_commandes 
            WHERE statut IN ('confirmee', 'en_route_commerce', 'commande_recue', 'en_route_client')
            AND livreur_id IS NOT NULL
        )
        GROUP BY l.id
        HAVING distance <= 15
        ORDER BY distance ASC, note_moyenne DESC
        LIMIT 5
    ", [$latitude, $longitude, $latitude, $currentDay, $creneauDebut, $creneauFin]);
    
    return ['drivers' => $drivers];
}

function calculateDeliveryCost($driverLat, $driverLon, $clientLat, $clientLon, $nbArticles) {
    $distance = calculateDistance($driverLat, $driverLon, $clientLat, $clientLon);
    $fraisDistance = $distance * 4.99;
    $fraisArticles = $nbArticles * 0.010;
    $total = $fraisDistance + $fraisArticles;
    
    return [
        'distance' => round($distance, 2),
        'frais_distance' => round($fraisDistance, 2),
        'frais_articles' => round($fraisArticles, 2),
        'total' => round($total, 2)
    ];
}

function validateDeliveryAddress($address) {
    $isValid = isValidAddress($address);
    
    if ($isValid) {
        $geocoding = searchAddresses($address);
        if (!empty($geocoding['results'])) {
            return [
                'valid' => true,
                'coordinates' => [
                    'lat' => $geocoding['results'][0]['lat'],
                    'lon' => $geocoding['results'][0]['lon']
                ],
                'formatted_address' => $geocoding['results'][0]['display_name']
            ];
        }
    }
    
    return [
        'valid' => false,
        'error' => 'Adresse non valide ou hors zone de livraison (Uccle et Fort Jaco uniquement)'
    ];
}

function getDriverLocation($livreurId) {
    $location = db()->fetchOne("
        SELECT latitude, longitude, 
               TIMESTAMPDIFF(MINUTE, last_location_update, NOW()) as minutes_ago
        FROM vr_livreurs 
        WHERE id = ?
    ", [$livreurId]);
    
    return $location ?: ['error' => 'Position non disponible'];
}

function updateDriverLocation($livreurId, $latitude, $longitude) {
    try {
        db()->update('vr_livreurs', [
            'latitude' => $latitude,
            'longitude' => $longitude,
            'last_location_update' => date('Y-m-d H:i:s')
        ], 'id = ?', [$livreurId]);
        
        return ['success' => true];
    } catch (Exception $e) {
        return ['error' => $e->getMessage()];
    }
}

function getLiveOrders($livreurId) {
    $availableOrders = db()->fetchAll("
        SELECT c.*, co.nom as commerce_nom, r.first_name, r.last_name,
               (6371 * acos(cos(radians(l.latitude)) * cos(radians(c.latitude_livraison)) * 
               cos(radians(c.longitude_livraison) - radians(l.longitude)) + sin(radians(l.latitude)) * 
               sin(radians(c.latitude_livraison)))) AS distance
        FROM vr_commandes c
        JOIN vr_commerces co ON c.commerce_id = co.id
        JOIN residents r ON c.resident_id = r.id
        JOIN vr_livreurs l ON l.id = ?
        WHERE c.statut = 'en_attente' AND c.livreur_id IS NULL
        HAVING distance <= 15
        ORDER BY c.date_commande ASC
    ", [$livreurId]);
    
    $myOrders = db()->fetchAll("
        SELECT c.*, co.nom as commerce_nom, r.first_name, r.last_name
        FROM vr_commandes c
        JOIN vr_commerces co ON c.commerce_id = co.id
        JOIN residents r ON c.resident_id = r.id
        WHERE c.livreur_id = ? AND c.statut IN ('confirmee', 'en_route_commerce', 'commande_recue', 'en_route_client')
        ORDER BY c.date_confirmation ASC
    ", [$livreurId]);
    
    return [
        'available_orders' => $availableOrders,
        'my_orders' => $myOrders
    ];
}

function getUnreadMessages($commandeId, $userType) {
    $expediteur = $userType === 'client' ? 'livreur' : 'client';
    
    $messages = db()->fetchAll("
        SELECT * FROM vr_messages 
        WHERE commande_id = ? AND expediteur = ? AND lu = 0
        ORDER BY created_at ASC
    ", [$commandeId, $expediteur]);
    
    return ['messages' => $messages];
}

function markMessagesRead($commandeId, $userType) {
    try {
        $expediteur = $userType === 'client' ? 'livreur' : 'client';
        
        db()->update('vr_messages', 
            ['lu' => 1], 
            'commande_id = ? AND expediteur = ?', 
            [$commandeId, $expediteur]
        );
        
        return ['success' => true];
    } catch (Exception $e) {
        return ['error' => $e->getMessage()];
    }
}

function getOrderTracking($commandeId) {
    $commande = db()->fetchOne("
        SELECT c.*, l.latitude as livreur_lat, l.longitude as livreur_lon,
               l.prenom, l.nom, l.photo_profil
        FROM vr_commandes c
        LEFT JOIN vr_livreurs l ON c.livreur_id = l.id
        WHERE c.id = ?
    ", [$commandeId]);
    
    if (!$commande) {
        return ['error' => 'Commande introuvable'];
    }
    
    $estimatedTime = null;
    if ($commande['statut'] === 'en_route_client' && $commande['livreur_lat']) {
        $distance = calculateDistance(
            $commande['livreur_lat'], $commande['livreur_lon'],
            $commande['latitude_livraison'], $commande['longitude_livraison']
        );
        $estimatedTime = round($distance * 3);
    }
    
    return [
        'commande' => $commande,
        'estimated_arrival' => $estimatedTime
    ];
}
?>